//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/Model/Model/RealModel.cpp
//! @brief     Implements class RealModel
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#include "GUI/Model/Model/RealModel.h"
#include "Base/Util/Assert.h"
#include "GUI/Model/Device/RealItem.h"
#include "GUI/Support/XML/UtilXML.h"

namespace {
namespace Tag {

const QString RealData("RealData");
const QString SelectedIndex("SelectedIndex");
const QString SelectedRank("SelectedRank");

} // namespace Tag
} // namespace

RealModel::RealModel(InstrumentModel* instrumentModel, QObject* parent)
    : QObject(parent)
    , m_instrumentModel(instrumentModel)
{
    setObjectName("RealModel");
}

RealModel::~RealModel() = default;

QVector<DataItem*> RealModel::dataItems() const
{
    QVector<DataItem*> result;
    for (auto* realData : realItems()) {
        if (auto* intensityItem = realData->dataItem())
            result.push_back(intensityItem);
        if (auto* native_data_item = realData->nativeDataItem())
            result.push_back(native_data_item);
    }
    return result;
}

void RealModel::clear()
{
    for (auto* realItem : realItems())
        removeRealItem(realItem);
}

void RealModel::writeTo(QXmlStreamWriter* w) const
{
    XML::writeAttribute(w, XML::Attrib::version, uint(1));

    // real items
    for (auto* realItem : realItems()) {
        w->writeStartElement(Tag::RealData);
        XML::writeAttribute(w, XML::Attrib::name, realItem->realItemName());
        realItem->writeTo(w);
        w->writeEndElement();
    }

    // selected index
    w->writeStartElement(Tag::SelectedIndex);
    XML::writeAttribute(w, XML::Attrib::value, m_selectedIndex);
    w->writeEndElement();

    // selected rank
    w->writeStartElement(Tag::SelectedRank);
    XML::writeAttribute(w, XML::Attrib::value, m_selectedRank);
    w->writeEndElement();
}

void RealModel::readFrom(QXmlStreamReader* r)
{
    clear();

    const uint version = XML::readUIntAttribute(r, XML::Attrib::version);
    Q_UNUSED(version)

    while (r->readNextStartElement()) {
        QString tag = r->name().toString();

        // real item
        if (tag == Tag::RealData) {
            pushNewItem()->readFrom(r);
            XML::gotoEndElementOfTag(r, tag);

            // selected index
        } else if (tag == Tag::SelectedIndex) {
            XML::readAttribute(r, XML::Attrib::value, &m_selectedIndex);
            XML::gotoEndElementOfTag(r, tag);

            // selected rank
        } else if (tag == Tag::SelectedRank) {
            XML::readAttribute(r, XML::Attrib::value, &m_selectedRank);
            XML::gotoEndElementOfTag(r, tag);

        } else
            r->skipCurrentElement();
    }

    if (r->hasError())
        throw std::runtime_error(r->errorString().toLatin1());
}

void RealModel::writeDataFiles(const QString& projectDir)
{
    for (const auto* realItem : realItems())
        realItem->writeDataFiles(projectDir);

    dataFilesCleaner.cleanOldFiles(projectDir, dataItems());
}

void RealModel::readDataFiles(const QString& projectDir, MessageService* messageService)
{
    for (auto* realItem : realItems())
        realItem->readDataFiles(projectDir, messageService);

    dataFilesCleaner.recollectDataNames(dataItems());
}

RealItem* RealModel::pushNewItem()
{
    m_realItems.emplace_back(new RealItem);
    return m_realItems.back().get();
}

RealItem* RealModel::insertDataItem(int rank)
{
    RealItem* r = pushNewItem();
    r->initItemWithRank(rank);
    return r;
}

QVector<RealItem*> RealModel::realItems() const
{
    QVector<RealItem*> items(m_realItems.size());
    for (int i = 0; i < items.size(); i++)
        items[i] = m_realItems[i].get();
    return items;
}


QVector<RealItem*> RealModel::realItems(int rank) const
{
    return rank == 1 ? realItems1D() : realItems2D();
}

QVector<RealItem*> RealModel::realItems1D() const
{
    QVector<RealItem*> vec_1d;

    for (auto* item : realItems())
        if (item->isSpecularData())
            vec_1d.push_back(item);

    return vec_1d;
}

QVector<RealItem*> RealModel::realItems2D() const
{
    QVector<RealItem*> vec_2d;

    for (auto* item : realItems())
        if (item->isIntensityData())
            vec_2d.push_back(item);

    return vec_2d;
}

void RealModel::removeRealItem(RealItem* realItem)
{
    ASSERT(realItem);
    int index = realItems().indexOf(realItem);
    m_realItems.erase(m_realItems.begin() + index);
}

QStringList RealModel::realItemNames() const
{
    QStringList result;
    for (auto* item : realItems())
        result.append(item->realItemName());
    return result;
}
