//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/Model/Job/FitParameterItem.cpp
//! @brief     Implements FitParameterItems family of classes
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#include "GUI/Model/Job/FitParameterItem.h"
#include "Base/Util/Assert.h"
#include "GUI/Model/Job/FitParameterLinkItem.h"
#include "GUI/Support/XML/UtilXML.h"

namespace {
namespace Tag {

const QString Value("Value");
const QString IsEnabled("IsEnabled");
const QString DisplayName("DisplayName");
const QString Type("Type");
const QString TypeItem("TypeItem");
const QString StartValue("StartValue");
const QString MinValue("MinValue");
const QString MaxValue("MaxValue");
const QString FitLinkItem("FitLinkItem");
const QString BaseData("BaseData");

} // namespace Tag
} // namespace

FitTypeItem::FitTypeItem(const ComboProperty& type, QObject* parent)
    : QObject(parent)
    , m_type(type)
{
    setObjectName("Type");
}

void FitTypeItem::writeTo(QXmlStreamWriter* w) const
{
    XML::writeAttribute(w, XML::Attrib::version, uint(1));

    // type
    w->writeStartElement(Tag::Type);
    m_type.writeTo(w);
    w->writeEndElement();
}

void FitTypeItem::readFrom(QXmlStreamReader* r)
{
    const uint version = XML::readUIntAttribute(r, XML::Attrib::version);
    Q_UNUSED(version)

    while (r->readNextStartElement()) {
        QString tag = r->name().toString();

        // type
        if (tag == Tag::Type) {
            m_type.readFrom(r);
            XML::gotoEndElementOfTag(r, tag);

        } else
            r->skipCurrentElement();
    }
}

//------------------------------------------------------------------------------------------------

FitDoubleItem::FitDoubleItem(double value, QObject* parent)
    : QObject(parent)
    , m_value(value)
    , m_decimals(3)
    , m_limits(RealLimits::limitless())
{
}

void FitDoubleItem::writeTo(QXmlStreamWriter* w) const
{
    // no need to write m_limits and m_decimals
    XML::writeAttribute(w, XML::Attrib::version, uint(1));

    // value
    w->writeStartElement(Tag::Value);
    XML::writeAttribute(w, XML::Attrib::value, m_value);
    w->writeEndElement();
}

void FitDoubleItem::readFrom(QXmlStreamReader* r)
{
    // no need to read m_limits and m_decimals
    const uint version = XML::readUIntAttribute(r, XML::Attrib::version);
    Q_UNUSED(version)

    while (r->readNextStartElement()) {
        QString tag = r->name().toString();

        // value
        if (tag == Tag::Value) {
            XML::readAttribute(r, XML::Attrib::value, &m_value);
            XML::gotoEndElementOfTag(r, tag);

        } else
            r->skipCurrentElement();
    }
}

//------------------------------------------------------------------------------------------------

FitEditableDoubleItem::FitEditableDoubleItem(double value, bool isEnabled, QObject* parent)
    : FitDoubleItem(value, parent)
    , m_isEnabled(isEnabled)
{
}

void FitEditableDoubleItem::writeTo(QXmlStreamWriter* w) const
{
    XML::writeAttribute(w, XML::Attrib::version, uint(1));

    // parameters from base class
    w->writeStartElement(Tag::BaseData);
    FitDoubleItem::writeTo(w);
    w->writeEndElement();

    // is enabled
    w->writeStartElement(Tag::IsEnabled);
    XML::writeAttribute(w, XML::Attrib::value, m_isEnabled);
    w->writeEndElement();
}

void FitEditableDoubleItem::readFrom(QXmlStreamReader* r)
{
    const uint version = XML::readUIntAttribute(r, XML::Attrib::version);
    Q_UNUSED(version)

    while (r->readNextStartElement()) {
        QString tag = r->name().toString();

        // parameters from base class
        if (tag == Tag::BaseData) {
            FitDoubleItem::readFrom(r);
            XML::gotoEndElementOfTag(r, tag);

            // is enabled?
        } else if (tag == Tag::IsEnabled) {
            XML::readAttribute(r, XML::Attrib::value, &m_isEnabled);
            XML::gotoEndElementOfTag(r, tag);

        } else
            r->skipCurrentElement();
    }
}

//------------------------------------------------------------------------------------------------

namespace {

ComboProperty fitParameterTypeCombo()
{
    QStringList tooltips = QStringList() << "Fixed at given value"
                                         << "Limited in the range [min, max]"
                                         << "Limited at lower bound [min, inf]"
                                         << "Limited at upper bound [-inf, max]"
                                         << "No limits imposed to parameter value";

    ComboProperty result;
    result << "fixed"
           << "limited"
           << "lower limited"
           << "upper limited"
           << "free";
    result.setCurrentValue("limited");
    result.setToolTips(tooltips);
    return result;
}

const double range_factor = 0.5;

} // namespace


FitParameterItem::FitParameterItem(QObject* parent)
    : QObject(parent)
    , m_typeItem(std::make_unique<FitTypeItem>(fitParameterTypeCombo(), this))
    , m_startValueItem(std::make_unique<FitDoubleItem>(0.0, this))
    , m_minItem(std::make_unique<FitEditableDoubleItem>(0.0, true, this))
    , m_maxItem(std::make_unique<FitEditableDoubleItem>(0.0, false, this))
{
    setObjectName("FitParameter");
    m_startValueItem->setObjectName("Value");
    m_minItem->setObjectName("Min");
    m_maxItem->setObjectName("Max");

    onTypeChange();
}

//! Inits P_MIN and P_MAX taking into account current value and external limits

void FitParameterItem::initMinMaxValues(const RealLimits& limits)
{
    double value = startValue();

    double dr(0);
    if (value == 0.0)
        dr = 1.0 * range_factor;
    else
        dr = std::abs(value) * range_factor;

    double min = value - dr;
    double max = value + dr;

    if (limits.hasLowerLimit() && min < limits.lowerLimit())
        min = limits.lowerLimit();

    if (limits.hasUpperLimit() && max > limits.upperLimit())
        max = limits.upperLimit();

    setMinimum(min);
    m_minItem->setLimits(limits);
    setMaximum(max);
    m_maxItem->setLimits(limits);

    m_startValueItem->setLimits(limits);
}

//! Constructs Limits corresponding to current GUI settings.

AttLimits FitParameterItem::attLimits() const
{
    if (isFixed())
        return AttLimits::fixed();
    if (isLimited())
        return AttLimits::limited(minimum(), maximum());
    if (isLowerLimited())
        return AttLimits::lowerLimited(minimum());
    if (isUpperLimited())
        return AttLimits::upperLimited(maximum());
    if (isFree())
        return AttLimits::limitless();
    ASSERT(false);
}

bool FitParameterItem::isValid() const
{
    if (isFixed() || isFree())
        return true;

    if (isLowerLimited())
        return minimum() <= startValue();
    if (isUpperLimited())
        return startValue() <= maximum();
    return minimum() <= startValue() && startValue() <= maximum();
}

QString FitParameterItem::displayName() const
{
    return m_displayName;
}

void FitParameterItem::setDisplayName(QString displayName)
{
    m_displayName = displayName;
}

double FitParameterItem::startValue() const
{
    return m_startValueItem->value();
}

void FitParameterItem::setStartValue(double start_value)
{
    m_startValueItem->setValue(start_value);
}

QObject* FitParameterItem::startValueItem() const
{
    return m_startValueItem.get();
}

double FitParameterItem::minimum() const
{
    return m_minItem->value();
}

void FitParameterItem::setMinimum(double minimum)
{
    m_minItem->setValue(minimum);
}

QObject* FitParameterItem::minimumItem() const
{
    return m_minItem.get();
}

double FitParameterItem::maximum() const
{
    return m_maxItem->value();
}

void FitParameterItem::setMaximum(double maximum)
{
    m_maxItem->setValue(maximum);
}

QObject* FitParameterItem::maximumItem() const
{
    return m_maxItem.get();
}

FitParameterLinkItem* FitParameterItem::addLinkItem(const QString& title, const QString& link)
{
    FitParameterLinkItem* newLink = new FitParameterLinkItem(this);
    m_links.emplace_back(newLink);

    newLink->setTitle(title);
    newLink->setLink(link);
    return newLink;
}

void FitParameterItem::removeLink(const QString& link)
{
    for (FitParameterLinkItem* linkItem : m_links)
        if (linkItem->link() == link)
            m_links.delete_element(linkItem);
}

QVector<FitParameterLinkItem*> FitParameterItem::linkItems() const
{
    return QVector<FitParameterLinkItem*>(m_links.begin(), m_links.end());
}

QStringList FitParameterItem::links() const
{
    QStringList links;
    for (FitParameterLinkItem* linkItem : linkItems())
        links << linkItem->link();
    return links;
}

void FitParameterItem::setTypeCombo(const ComboProperty& type)
{
    m_typeItem->setType(type);
    onTypeChange();
}

QString FitParameterItem::currentType() const
{
    return m_typeItem->type().currentValue();
}

QObject* FitParameterItem::typeItem() const
{
    return m_typeItem.get();
}

void FitParameterItem::writeTo(QXmlStreamWriter* w) const
{
    XML::writeAttribute(w, XML::Attrib::version, uint(1));

    // displayName
    w->writeStartElement(Tag::DisplayName);
    XML::writeAttribute(w, XML::Attrib::value, m_displayName);
    w->writeEndElement();

    // type
    w->writeStartElement(Tag::TypeItem);
    m_typeItem->writeTo(w);
    w->writeEndElement();

    // start value
    w->writeStartElement(Tag::StartValue);
    m_startValueItem->writeTo(w);
    w->writeEndElement();

    // min value
    w->writeStartElement(Tag::MinValue);
    m_minItem->writeTo(w);
    w->writeEndElement();

    // max value
    w->writeStartElement(Tag::MaxValue);
    m_maxItem->writeTo(w);
    w->writeEndElement();

    // parameter links
    for (const auto* fitLink : linkItems()) {
        w->writeStartElement(Tag::FitLinkItem);
        w->writeAttribute(XML::Attrib::name, fitLink->title());
        fitLink->writeTo(w);
        w->writeEndElement();
    }
}

void FitParameterItem::readFrom(QXmlStreamReader* r)
{
    const uint version = XML::readUIntAttribute(r, XML::Attrib::version);
    Q_UNUSED(version)

    while (r->readNextStartElement()) {
        QString tag = r->name().toString();

        // displayName
        if (tag == Tag::DisplayName) {
            XML::readAttribute(r, XML::Attrib::value, &m_displayName);
            XML::gotoEndElementOfTag(r, tag);

            // type
        } else if (tag == Tag::TypeItem) {
            m_typeItem->readFrom(r);
            XML::gotoEndElementOfTag(r, tag);

            // start value
        } else if (tag == Tag::StartValue) {
            m_startValueItem->readFrom(r);
            XML::gotoEndElementOfTag(r, tag);

            // min value
        } else if (tag == Tag::MinValue) {
            m_minItem->readFrom(r);
            XML::gotoEndElementOfTag(r, tag);

            // max value
        } else if (tag == Tag::MaxValue) {
            m_maxItem->readFrom(r);
            XML::gotoEndElementOfTag(r, tag);

            // parameter links
        } else if (tag == Tag::FitLinkItem) {
            addLinkItem("", "")->readFrom(r);
            XML::gotoEndElementOfTag(r, tag);

        } else
            r->skipCurrentElement();
    }
}

//! Enables/disables min, max properties on FitParameterItem's type

void FitParameterItem::onTypeChange()
{
    if (isFixed()) {
        setLimitEnabled(m_minItem.get(), false);
        setLimitEnabled(m_maxItem.get(), false);
    }

    else if (isLimited()) {
        setLimitEnabled(m_minItem.get(), true);
        setLimitEnabled(m_maxItem.get(), true);
    }

    else if (isLowerLimited()) {
        setLimitEnabled(m_minItem.get(), true);
        setLimitEnabled(m_maxItem.get(), false);
    }

    else if (isUpperLimited()) {
        setLimitEnabled(m_minItem.get(), false);
        setLimitEnabled(m_maxItem.get(), true);
    }

    else if (isFree()) {
        setLimitEnabled(m_minItem.get(), false);
        setLimitEnabled(m_maxItem.get(), false);
    }
}

//! Set limit property with given name to the enabled state

void FitParameterItem::setLimitEnabled(FitEditableDoubleItem* propertyItem, bool enabled)
{
    ASSERT(propertyItem);
    propertyItem->setEnabled(enabled);
}

bool FitParameterItem::isLimited() const
{
    return currentType() == "limited";
}

bool FitParameterItem::isFree() const
{
    return currentType() == "free";
}

bool FitParameterItem::isLowerLimited() const
{
    return currentType() == "lower limited";
}

bool FitParameterItem::isUpperLimited() const
{
    return currentType() == "upper limited";
}

bool FitParameterItem::isFixed() const
{
    return currentType() == "fixed";
}
