//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/Model/Beam/BeamAngleItems.cpp
//! @brief     Implements class BeamAngleItems
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#include "GUI/Model/Beam/BeamAngleItems.h"
#include "Base/Const/Units.h"
#include "GUI/Model/Descriptor/DistributionItems.h"

BeamAzimuthalAngleItem::BeamAzimuthalAngleItem()
{
    m_distribution.initWithInitializer("Distribution", "", DistributionItemCatalog::types(),
                                       initDistribution);
}

void BeamAzimuthalAngleItem::readFrom(QXmlStreamReader* r)
{
    // Here we only directly call parental method, so no versioning
    BeamDistributionItem::readFrom(r);

    m_distribution->setUnit(Unit::degree);
}

void BeamAzimuthalAngleItem::initDistribution(DistributionItem* newDistribution,
                                              const DistributionItem* currentDistribution)
{
    newDistribution->setUnit(Unit::degree);

    double meanVal = 0.0;
    if (auto* cd = dynamic_cast<const SymmetricResolutionItem*>(currentDistribution))
        meanVal = cd->mean();

    if (auto* nd = dynamic_cast<SymmetricResolutionItem*>(newDistribution)) {
        nd->setMean(meanVal);
        nd->setMeanDecimals(3);
    }

    newDistribution->initDistribution(meanVal);
}

double BeamAzimuthalAngleItem::azimuthalAngle() const
{
    return BeamDistributionItem::meanValue();
}

double BeamAzimuthalAngleItem::scaleFactor() const
{
    return Units::deg;
}

// ------------------------------------------------------------------------------------------------

BeamInclinationAngleItem::BeamInclinationAngleItem()
{
    m_distribution.initWithInitializer("Distribution", "", DistributionItemCatalog::types(),
                                       initDistribution);
}

void BeamInclinationAngleItem::readFrom(QXmlStreamReader* r)
{
    // Here we only directly call parental method, so no versioning
    BeamDistributionItem::readFrom(r);

    m_distribution->setUnit(Unit::degree);
}

double BeamInclinationAngleItem::inclinationAngle() const
{
    return BeamDistributionItem::meanValue();
}

double BeamInclinationAngleItem::scaleFactor() const
{
    return Units::deg;
}

void BeamInclinationAngleItem::initDistribution(DistributionItem* newDistribution,
                                                const DistributionItem* currentDistribution)
{
    newDistribution->setUnit(Unit::degree);

    double meanVal = 0.2;
    if (auto* cd = dynamic_cast<const SymmetricResolutionItem*>(currentDistribution))
        meanVal = cd->mean();

    if (auto* nd = dynamic_cast<SymmetricResolutionItem*>(newDistribution)) {
        nd->setMean(meanVal);
        nd->setMeanDecimals(3);
    }

    newDistribution->initDistribution(meanVal);
}
