//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      Device/IO/ReadRefsans.cpp
//! @brief     Implements functions readRefsans
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2023
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#include "Device/IO/ReadRefsans.h"
#include "Base/Axis/MakeScale.h"
#include "Base/Util/Assert.h"
#include "Base/Util/StringUtil.h"
#include "Device/Data/DataUtil.h"
#include "Device/Data/Datafield.h"
#include "Device/IO/ParseUtil.h"
#include <algorithm>
#include <vector>

namespace {

const std::vector<std::string> to_replace = {",", "\"", "(", ")", "[", "]"};

bool getNextLine(std::istream& input_stream, std::string& line)
{
    while (std::getline(input_stream, line)) {
        line = Base::String::trim(line);
        Base::String::replaceItemsFromString(line, to_replace, " ");
        if (!line.empty())
            return true;
    }
    return false;
}

} // namespace

// Refsans GISANS format:
// 1. There is no header or metadata in any csv-file.
// 2. Each csv-matrix has 257 rows and 257 columns of comma-separated floating-point values with a
// period as the decimal separator.
// 3. The first row and the first column are Q-values for each matrix element.
// 4. q-units are A^-1
Datafield* Util::RW::readRefsans(std::istream& input_stream)
{
    // Since any unit conversion is available only after linking data to instrument,
    // here Qy and Qz axes are loaded only for better readability of the original data.
    // Internal q-units should be nm^-1, but here the axes remain unscaled.

    // read qx coords from first line
    std::string line;
    getNextLine(input_stream, line);
    std::vector<double> dataInRow = Util::Parse::parse_doubles(line);
    std::vector<double> qy(dataInRow.begin() + 1, dataInRow.end()); // first point is unused
    ASSERT(qy.front() != qy.back());

    // read qz coords from first column and all values
    std::vector<double> qz;
    std::vector<std::vector<double>> matrix;
    while (getNextLine(input_stream, line)) {
        dataInRow = Util::Parse::parse_doubles(line);
        ASSERT(dataInRow.size() == (qy.size() + 1));
        qz.push_back(dataInRow.front());
        std::vector<double> val_row(dataInRow.begin() + 1, dataInRow.end());
        matrix.push_back(val_row);
    }
    ASSERT(qz.front() != qz.back());

    // sort axes
    if (qz.front() > qz.back()) {
        std::reverse(qz.begin(), qz.end());
        matrix = DataUtil::Data::invertAxis(1, matrix);
    }
    if (qy.front() > qy.back()) {
        std::reverse(qy.begin(), qy.end());
        matrix = DataUtil::Data::invertAxis(0, matrix);
    }

    // adjustment. Transpose after sorting!
    matrix = DataUtil::Data::transpose(matrix);

    // flatten values
    std::vector<double> values;
    for (const auto& row : matrix)
        values.insert(values.end(), row.begin(), row.end());

    // create datafield
    std::vector<const Scale*> axes;
    axes.push_back(newListScan("Qy", qy));
    axes.push_back(newListScan("Qz", qz));
    return new Datafield(std::move(axes), values);
}
