# -*- coding: utf-8 -*-

# Bluemindo: A really simple but powerful audio player in Python/PyGTK.
# Copyright (C) 2007-2009  Erwan Briand

# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation version 3 of the License.

# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

from gettext import gettext as _
from gobject import markup_escape_text, timeout_add, idle_add
from gtk.glade import XML as glade_XML
from gtk.gdk import pixbuf_new_from_file, INTERP_BILINEAR, display_get_default
from gtk import (FileChooserDialog, FILE_CHOOSER_ACTION_OPEN, STOCK_CANCEL,
                 RESPONSE_CANCEL, STOCK_OPEN, RESPONSE_OK, FileFilter,
                 STOCK_MEDIA_PAUSE, STOCK_MEDIA_PLAY, MAPPED)
from os.path import join, isfile
from os import remove
from shutil import copyfile
from gst import QueryError as gst_QueryError

from gui.volumemanagement import VolumeManagement

from common.statistics import Statistics
from common.functions import Functions
from common.config import ConfigLoader
from media.gstreamer import GStreamer

from modules.explorer import viewscommon
unescape = viewscommon.unescape

class PlayerGUI:
    def __init__(self, glade_file, config):
        self.glade_file = glade_file
        self.config = config
        self.userconf = ConfigLoader()
        self.functions = Functions()
        self.new_album_state = None
        self.statistics = Statistics()

        # Minimize
        if not config['startminimized']:
            self.glade_file.get_widget('window1').show()

        # GStreamer
        self.gst = GStreamer()
        self.gst.set_playback(config['gstplayback'])
        self.plb = config['gstplayback']

        self.gst.stop()

        if config['gstplayback'] == 'gapless':
            self.gst.player.connect('about-to-finish', self.song_nearly_ended)

        self.glade_file.get_widget('hscale1').set_sensitive(False)
        self.repeat = self.glade_file.get_widget('tool-repeat')

        # VolumeManagement
        volume = VolumeManagement(self.glade_file)
        volume.change_volume(volume.volume)

        datadir = self.functions.datadir
        album_cover = self.glade_file.get_widget('image1')
        self.current_file = [join(datadir, 'image', 'logo_head_big.png'), None]
        self.image_set(album_cover, join(datadir, 'image',
                                                  'logo_head_big.png'), 85)

        alb_btn = self.glade_file.get_widget('button1')
        alb_btn.connect('clicked', self.on_album_clicked)
        alb_btn.connect('enter', self.on_album_enter)
        alb_btn.connect('leave', self.on_album_leave)

        self.glade_file.get_widget('label2').set_alignment(0, 0.5)
        self.glade_file.get_widget('label3').set_alignment(0, 0.5)
        self.glade_file.get_widget('toolbutton1').set_expand(True)
        self.glade_file.get_widget('hscale1').connect('change-value',
                                                      self.on_change_value)

    def reload_config(self, newconf):
        self.config['title'] = newconf['title']
        self.config['popup'] = newconf['popup']

        if self.plb != newconf['gstplayback']:
            # Stop playback
            self.config['__extensions'].load_event('OnStopPressed')

            # Start a new playback
            self.gst.set_playback(newconf['gstplayback'])
            self.plb = newconf['gstplayback']

            # Connect if we are in gapless mode
            if newconf['gstplayback'] == 'gapless':
                self.gst.player.connect('about-to-finish',
                                        self.song_nearly_ended)

    def song_nearly_ended(self, *args):
        if self.plb == 'gapless':
            idle_add(self.handler_next, True)

    def image_set(self, pimg, pfile, pscale):
        """Change album art image from a pixbuf."""
        pixbuf = pixbuf_new_from_file(pfile)
        pixbuf = pixbuf.scale_simple(pscale, pscale, INTERP_BILINEAR)
        pimg.set_from_pixbuf(pixbuf)

    def on_album_clicked(self, widget):
        """Change current album artwork."""
        if self.current_file[1] is not None:
            (artist, album) = self.current_file[1]

            file_selector = FileChooserDialog(_('Change album artwork'), None,
                            FILE_CHOOSER_ACTION_OPEN, (STOCK_CANCEL,
                            RESPONSE_CANCEL, STOCK_OPEN, RESPONSE_OK))

            filter_ = FileFilter()
            filter_.add_mime_type('image/gif')
            filter_.add_mime_type('image/png')
            filter_.add_mime_type('image/jpeg')
            file_selector.set_filter(filter_)

            fld = file_selector.run()

            # Select a file, copy it in the covers directory
            file_ = None
            if fld == RESPONSE_OK:
                file_ = file_selector.get_filename()
                if not file_ == None:

                    _file = join(self.config['__data-dir'], 'covers',
                                 self.functions.get_hash(album, artist))
                    if isfile(_file):
                        remove(_file)

                    copyfile(file_, _file)
                    self.image_set(self.glade_file.get_widget('image1'),
                                   _file, 90)

            file_selector.destroy()

    def on_album_enter(self, widget):
        """Create a popup window with the album cover."""
        def show_cover():
            if self.is_the_cursor_on_album and self.config['popup']:
                try:
                    self.new_album_state.destroy()
                except AttributeError:
                    pass

                img = self.glade_file.get_widget('image1')
                pixbuf = img.get_pixbuf()

                widg = glade_XML(join(self.functions.datadir,
                                 'glade', 'albumpreview.glade'))
                main_window = widg.get_widget('window1')
                self.new_album_state = main_window

                # Position
                dsp = display_get_default()
                scr = dsp.get_default_screen()
                tmp1, x0, y0, tmp2 = dsp.get_pointer()
                main_window.move((x0 + 5), (y0 + 5))

                main_window.show()

                # Change image from pixbuf
                pimg = widg.get_widget('image1')
                pixbuf = pixbuf_new_from_file(self.current_file[0])
                pixbuf = pixbuf.scale_simple(285, 285, INTERP_BILINEAR)
                pimg.set_from_pixbuf(pixbuf)

        self.is_the_cursor_on_album = True
        timeout_add(1000, show_cover)

    def on_album_leave(self, widget):
        """Delete the popup window if exists."""
        self.is_the_cursor_on_album = False

        if self.new_album_state:
            self.new_album_state.destroy()

    def on_change_value(self, widget, scroll, value):
        """Jump to another place in the song."""
        seconds = int(value)
        self.gst.seek(seconds)

    def handler_previous(self):
        """This is the handler for the OnPreviousPressed signal."""
        cur = self.gst.getnow()
        self.gst.stop()
        extensions = self.config['__extensions']

        tree_pls = self.glade_file.get_widget('treeview2')
        text_lyr = self.glade_file.get_widget('textview1')
        if tree_pls.flags() & MAPPED or text_lyr.flags() & MAPPED:
            model = tree_pls.get_model()
        else:
            # TODO: webradios
            return

        if len(model) > 0:
            n = 0
            id_in_playlist = 0
            for song in model:
                songfile = eval(unescape(str(song[8])))
                if songfile[8] == cur:
                    id_in_playlist = n
                n = n + 1

            _p = 0
            _n = id_in_playlist - 1

            if _n < _p and self.repeat.get_active():
                # Load the previous song (last in playlist)
                song_id = len(model) - 1
                song_data = eval(unescape(str(model[song_id][8])))
                extensions.load_event('OnPlayNewSong', song_data)
            elif _n < _p and not self.repeat.get_active():
                # Don't do anything
                pass
            else:
                # Load the previous song
                song_data = eval(unescape(str(model[_n][8])))
                extensions.load_event('OnPlayNewSong', song_data)
        else:
            # The playlist is empty but we want another song
            extensions.load_event('OnPlayPressedWithoutQueue')

    def handler_stop(self):
        """This is the handler for the OnStopPressed signal."""
        # Stop the song
        cur = self.gst.getnow()
        self.gst.stop()

        # Update GUI
        self.glade_file.get_widget('hscale1').set_sensitive(False)
        self.glade_file.get_widget('hscale1').set_value(0)
        self.glade_file.get_widget('tool-play').set_stock_id(STOCK_MEDIA_PLAY)
        self.glade_file.get_widget('hscale1').set_value(0)
        self.glade_file.get_widget('label1').set_text('')
        self.glade_file.get_widget('label2').set_text('')
        self.glade_file.get_widget('label3').set_text('')
        self.glade_file.get_widget('label5').set_text('')
        self.glade_file.get_widget('label6').set_text('')

        # Delete the album art
        datadir = self.functions.datadir
        album_cover = self.glade_file.get_widget('image1')
        self.current_file = [join(datadir, 'image', 'logo_head_big.png'), None]
        self.image_set(album_cover, join(datadir, 'image',
                                                  'logo_head_big.png'), 85)

        # Delete the content of the current playing file
        self.current_playing_file('')

        tree_pls = self.glade_file.get_widget('treeview2')
        text_lyr = self.glade_file.get_widget('textview1')
        if tree_pls.flags() & MAPPED or text_lyr.flags() & MAPPED:
            model = tree_pls.get_model()
        else:
            # TODO: webradios
            return

        # Delete bold in playlist
        u = -1
        for sng in model:
            u += 1
            songfile = eval(unescape(str(sng[8])))
            if songfile[8] == cur:
                sng = model[u]
                model[u] = [self.functions.clear_html(sng[0], True),
                            self.functions.clear_html(sng[1], True),
                            self.functions.clear_html(sng[2], True),
                            self.functions.clear_html(sng[3], True),
                            self.functions.clear_html(sng[4], True),
                            self.functions.clear_html(sng[5], True),
                            self.functions.clear_html(sng[6], True),
                            self.functions.clear_html(sng[7], True),
                            sng[8]
                           ]

    def handler_play(self):
        """This is the handler for the OnPlayPressed signal."""
        # Get status
        rtn = self.gst.playpause(None)
        sta = self.gst.getstatus()
        extensions = self.config['__extensions']

        tree_pls = self.glade_file.get_widget('treeview2')
        text_lyr = self.glade_file.get_widget('textview1')
        if tree_pls.flags() & MAPPED or text_lyr.flags() & MAPPED:
            model = tree_pls.get_model()
        else:
            # TODO: webradios
            return

        if len(model) > 0 and (sta == 'NULL' or sta == 'STOP'):
            # If the playlist is not empty, load the first song
            song_data = eval(unescape(str(model[0][8])))
            extensions.load_event('OnPlayNewSong', song_data)
        elif len(model) == 0 and (sta == 'NULL' or sta == 'STOP'):
            # The playlist is empty but we want another song
            extensions.load_event('OnPlayPressedWithoutQueue')

        # Play button
        tplay = self.glade_file.get_widget('tool-play')
        if self.gst.getstatus() == 'PAUSED':
            tplay.set_stock_id(STOCK_MEDIA_PLAY)
        else:
            tplay.set_stock_id(STOCK_MEDIA_PAUSE)

    def handler_next(self, gapless=False):
        """This is the handler for the OnNextPressed signal."""
        cur = self.gst.getnow()
        if not gapless:
            self.gst.stop()
        extensions = self.config['__extensions']

        tree_pls = self.glade_file.get_widget('treeview2')
        text_lyr = self.glade_file.get_widget('textview1')
        if tree_pls.flags() & MAPPED or text_lyr.flags() & MAPPED:
            model = tree_pls.get_model()
        else:
            # TODO: webradios
            return

        if len(model) > 0:
            n = 0
            id_in_playlist = -1
            for song in model:
                songfile = eval(unescape(str(song[8])))
                if songfile[8] == cur:
                    id_in_playlist = n
                n = n + 1

            _p = len(model) - 1
            _n = id_in_playlist + 1

            if _n > _p and self.repeat.get_active():
                # Load the next song (first in playlist)
                song_data = eval(unescape(str(model[0][8])))
                extensions.load_event('OnPlayNewSong', song_data)
            elif _n > _p and not self.repeat.get_active():
                # Don't do anything
                pass
            else:
                # Load the next song
                song_data = eval(unescape(str(model[_n][8])))
                extensions.load_event('OnPlayNewSong', song_data)
        else:
            # The playlist is empty but we want another song
            extensions.load_event('OnPlayPressedWithoutQueue')

    def handler_play_new_song(self, sg):
        """This is the handler for the OnPlayNewSong signal."""
        song = eval(unescape(str(sg)))
        title = song[0]
        artist = song[1]
        album = song[2]
        length = self.functions.human_length(song[7])

        # Start playing the song
        cur = self.gst.getnow()
        self.gst.playpause(song[8])

        if cur != self.gst.getnow():
            # Update statistics
            self.statistics.set_stats_for_song(song[8], None)
            self.statistics.set_stats_for_artist(artist, None)
            self.statistics.set_stats_for_album(album, None)

        # Update GUI
        self.glade_file.get_widget('hscale1').set_sensitive(True)
        self.glade_file.get_widget('toolbar2').set_sensitive(True)
        self.glade_file.get_widget('label1').set_markup(
                                   '<b>%s</b>' % markup_escape_text(title))
        self.glade_file.get_widget('label2').set_markup(
                                   '<i>%s</i>' % markup_escape_text(artist))
        self.glade_file.get_widget('label3').set_text(album)
        self.glade_file.get_widget('label5').set_text('00:00')
        self.glade_file.get_widget('label6').set_text(' - ' + length)

        # Scale
        self.glade_file.get_widget('hscale1').set_range(0, float(song[7]))

        # Play button
        tplay = self.glade_file.get_widget('tool-play')
        if self.gst.getstatus() == 'PAUSED':
            tplay.set_stock_id(STOCK_MEDIA_PLAY)
        else:
            tplay.set_stock_id(STOCK_MEDIA_PAUSE)

        # Timer
        timeout_add(500, self.scale_timer)

        # Current playing file
        self.current_playing_file('%s - %s (from: %s)' %
                                 (title, artist, album))

        # Window title
        if self.config['title']:
            self.glade_file.get_widget('window1').set_title('Bluemindo '
                                                            '[ %s - %s ]'
                                                            % (title, artist))
        else:
            self.glade_file.get_widget('window1').set_title('Bluemindo')

        # Set album image
        filename = join(self.config['__data-dir'], 'covers',
                        self.functions.get_hash(album, artist))
        if not isfile(filename):
            datadir = self.functions.datadir
            filename = join(datadir, 'image', 'logo_head_big.png')

        album_cover = self.glade_file.get_widget('image1')
        self.current_file = [filename, (artist, album)]
        self.image_set(album_cover, filename, 85)

        # Bold
        tree_pls = self.glade_file.get_widget('treeview2')
        text_lyr = self.glade_file.get_widget('textview1')
        if tree_pls.flags() & MAPPED or text_lyr.flags() & MAPPED:
            model = tree_pls.get_model()
        else:
            # TODO: webradios
            return

        if len(model) > 0:
            # Fetching the playlist and get the current playing song id
            i = 0
            for sng in model:
                if eval(unescape(str(sng[8]))) == song:
                    break
                i += 1

            sng = model[i]
            if (not model[i][0].startswith('<b>')
                and not model[i][0].endswith('</b>')):
                model[i] = ['<b>%s</b>' % sng[0], '<b>%s</b>' % sng[1],
                            '<b>%s</b>' % sng[2], '<b>%s</b>' % sng[3],
                            '<b>%s</b>' % sng[4], '<b>%s</b>' % sng[5],
                            '<b>%s</b>' % sng[6], '<b>%s</b>' % sng[7],
                            sng[8]
                           ]

            # Get the normal text, without bold emphasis
            u = 0
            for sng in model:
                if (sng[1].startswith('<b>') and sng[1].endswith('</b>')
                    and  i != u):
                    sng = model[u]
                    model[u] = [self.functions.clear_html(sng[0], True),
                                self.functions.clear_html(sng[1], True),
                                self.functions.clear_html(sng[2], True),
                                self.functions.clear_html(sng[3], True),
                                self.functions.clear_html(sng[4], True),
                                self.functions.clear_html(sng[5], True),
                                self.functions.clear_html(sng[6], True),
                                self.functions.clear_html(sng[7], True),
                                sng[8]
                               ]

                u += 1

    def handler_play_new_radio(self, radio):
        """This is the handler for the OnPlayNewRadio signal."""
        (title, url) = radio

        # Update GUI
        self.glade_file.get_widget('tool-play').set_stock_id(STOCK_MEDIA_PAUSE)
        self.glade_file.get_widget('toolbar2').set_sensitive(False)
        self.glade_file.get_widget('hscale1').set_value(0)
        self.glade_file.get_widget('label2').set_text('')
        self.glade_file.get_widget('label3').set_text('')
        self.glade_file.get_widget('label5').set_text('')
        self.glade_file.get_widget('label6').set_text('')
        self.glade_file.get_widget('label1').set_markup(
                                   '<b>%s</b>' % markup_escape_text(title))
        self.glade_file.get_widget('label2').set_markup(
                              '<i>%s</i>' % markup_escape_text(_('Webradios')))

        self.gst.playpause(url)

    def current_playing_file(self, data):
        """Update the current-playing file in `datadir`."""
        file_ = open(join(self.userconf.datadir, 'current-playing'), 'wb')
        file_.write(data)
        file_.close()

    def scale_timer(self):
        # This function updates the seek bar
        try:
            label = self.glade_file.get_widget('label5')
            hscale = self.glade_file.get_widget('hscale1')
            pos = self.gst.getposition()
            position = int(self.gst.getposition() / 1000000000)
            label.set_text(self.functions.human_length(position))
            hscale.set_value(position)
            return True
        except gst_QueryError:
            pass

        state = self.gst.getstatus()
        if state == 'NULL' and self.config['gstplayback'] != 'gapless':
            extensions = self.config['__extensions']
            extensions.load_event('OnNextPressed')