/* $Id: pwr.c,v 1.1.2.4 2001/06/18 14:42:41 deti Exp $
    Power Management module
*/

#include <wdm.h>
#include "stdarg.h"
#include "stdio.h"

//
// Include files needed for USB support
//
#include "usbdi.h"
#include "usbdlib.h"

//
// Include file for the dabusb Device
//
#include "bcusb.h"

//#define deti

#define dabusb_StringForDevState( devState )  szDevicePowerState[ devState ] 

#define dabusb_StringForSysState( sysState )  szSystemPowerState[ sysState ] 

static const PCHAR szSystemPowerState[] = 
{
	"PowerSystemUnspecified",
	"PowerSystemWorking",
	"PowerSystemSleeping1",
	"PowerSystemSleeping2",
	"PowerSystemSleeping3",
	"PowerSystemHibernate",
	"PowerSystemShutdown",
	"PowerSystemMaximum"
};

static const PCHAR szDevicePowerState[] = 
{
	"PowerDeviceUnspecified",
	"PowerDeviceD0",
	"PowerDeviceD1",
	"PowerDeviceD2",
	"PowerDeviceD3",
	"PowerDeviceMaximum"
};


NTSTATUS
dabusb_PoSelfRequestCompletion(
	IN PDEVICE_OBJECT       fdo,
	IN UCHAR                MinorFunction,
	IN POWER_STATE          PowerState,
	IN PVOID                Context,
	IN PIO_STATUS_BLOCK     IoStatus
	)
/*++

Routine Description:

    This routine is called when the driver completes a self-originated power IRP
        that was generated by a call to dabusb_SelfSuspendOrActivate().
    We power down whenever the last pipe is closed and power up when the first pipe is opened.

    For power-up , we set an event in our FDO extension to signal this IRP done
    so the power request can be treated as a synchronous call.
    We need to know the device is powered up before opening the first pipe, for example.
    For power-down, we do not set the event, as no caller waits for powerdown complete.

Arguments:

    fdo - Pointer to the device object for the class device. ( Physical Device Object )

    Context - Driver defined context, in this case our FDO ( functional device object )

Return Value:

    The function value is the final status from the operation.

--*/
{
	PDEVICE_OBJECT deviceObject = Context;
	pdabusb_t s = deviceObject->DeviceExtension;
	NTSTATUS ntStatus = IoStatus->Status;

	// we should not be in the midst of handling a system-generated power irp
	ASSERT( NULL == s->PowerIrp );

	// We only need to set the event if we're powering up;
	// No caller waits on power down complete
	if ( (ULONG) PowerState.DeviceState < s->PowerDownLevel ) {

		// Trigger Self-requested power irp completed event;
		//  The caller is waiting for completion
		KeSetEvent(&s->SelfRequestedPowerIrpEvent, 1, FALSE);
	}

	dabusb_decrement_io_count(deviceObject);

	printk( "Exit dabusb_PoSelfRequestCompletion() FAILED, ntStatus = 0x%x\n", ntStatus );

	return ntStatus;
}

NTSTATUS
dabusb_SelfRequestPowerIrp(
	IN PDEVICE_OBJECT fdo,
	IN POWER_STATE PowerState
	)
/*++

Routine Description:

    This routine is called by dabusb_SelfSuspendOrActivate() to
    actually make the system request for a powerdown/up to PowerState.
    It first checks to see if we are already in Powerstate and immediately
    returns  SUCCESS with no further processing if so


Arguments:

    fdo - Pointer to the device object

    PowerState. power state requested, e.g PowerDeviceD0.


Return Value:

    The function value is the final status from the operation.

--*/
{
	NTSTATUS ntStatus = STATUS_SUCCESS;
	pdabusb_t s;
	PIRP pIrp = NULL;

	s =  fdo->DeviceExtension;

	// This should have been reset in completion routine
	ASSERT( !s->SelfPowerIrp );

#ifndef deti
	if (  s->CurrentDevicePowerState ==  PowerState.DeviceState )
#endif
		return STATUS_SUCCESS;  // nothing to do

	printk( "Enter dabusb_SelfRequestPowerIrp() will request power irp to state %s\n",
		dabusb_StringForDevState( PowerState.DeviceState ));

	dabusb_increment_io_count(fdo);

        // flag we're handling a self-generated power irp
	s->SelfPowerIrp = TRUE;

        // actually request the Irp
	ntStatus = PoRequestPowerIrp(s->pdo,
				     IRP_MN_SET_POWER,
				     PowerState,
				     dabusb_PoSelfRequestCompletion,
				     fdo,
				     NULL);


	if  ( ntStatus == STATUS_PENDING ) {
		// status pending is the return code we wanted

		// We only need to wait for completion if we're powering up
		if ( (ULONG) PowerState.DeviceState < s->PowerDownLevel ) {

			NTSTATUS waitStatus;

			waitStatus = KeWaitForSingleObject(
				&s->SelfRequestedPowerIrpEvent,
				Suspended,
				KernelMode,
				FALSE,
				NULL);

		}

		ntStatus = STATUS_SUCCESS;

		s->SelfPowerIrp = FALSE;

		printk( "dabusb_SelfRequestPowerIrp() SUCCESS\n    IRP 0x%x to state %s\n",
			pIrp, dabusb_StringForDevState(PowerState.DeviceState) );


	}
	else {
		// The return status was not STATUS_PENDING; any other codes must be considered in error here;
		//  i.e., it is not possible to get a STATUS_SUCCESS  or any other non-error return from this call;
		printk( "dabusb_SelfRequestPowerIrp() to state %s FAILED, status = 0x%x\n",
			dabusb_StringForDevState( PowerState.DeviceState ),ntStatus);
	}

	return ntStatus;
}

NTSTATUS
dabusb_SelfSuspendOrActivate(
	IN PDEVICE_OBJECT fdo,
	IN BOOLEAN fSuspend
	)
/*++

Routine Description:

        Called on dabusb_PnPAddDevice() to power down until needed (i.e., till a pipe is actually opened).
        Called on dabusb_Create() to power up device to D0 before opening 1st pipe.
        Called on dabusb_Close() to power down device if this is the last pipe.

Arguments:

    fdo - Pointer to the device object

    fSuspend; TRUE to Suspend, FALSE to acivate.


Return Value:

    If the operation is not attemtped, SUCCESS is returned.
    If the operation is attemtped, the value is the final status from the operation.

--*/
{
	NTSTATUS ntStatus = STATUS_SUCCESS;

	POWER_STATE PowerState;
	pdabusb_t s;


	s = fdo->DeviceExtension;
	printk( "Enter dabusb_SelfSuspendOrActivate(),fSuspend = %d\n", fSuspend);


        // Can't accept request if:
	//  1) device is removed,
	//  2) has never been started,
	//  3) is stopped,
	//  4) has a remove request pending,
	//  5) has a stop device pending
	if ( s->device_state<=_stop_pending ) {
		ntStatus = STATUS_DELETE_PENDING;

                printk( "ABORTING dabusb_SelfSuspendOrActivate()\n");
		return ntStatus;
	}


	// don't do anything if any System-generated Device Pnp irps are pending
	if ( NULL != s->PowerIrp ) {
		printk( "Exit dabusb_SelfSuspendOrActivate(),refusing on pending s->PowerIrp 0x%x\n", s->PowerIrp);
		return ntStatus;
	}

	// don't do anything if any self-generated Device Pnp irps are pending
	if ( s->SelfPowerIrp ) {
		printk( "Exit dabusb_SelfSuspendOrActivate(),refusing on pending s->SelfPowerIrp\n" );
		return ntStatus;
	}

	// don't auto-suspend if any pipes are open
	if ( fSuspend && s->opened ) {
		printk( "Exit dabusb_SelfSuspendOrActivate(),refusing to self-suspend on OpenPipeCount %d\n", s->opened);
		return ntStatus;
	}

	// don't auto-activate if no pipes are open
	if ( !fSuspend && !s->opened ) {
		printk( "Exit dabusb_SelfSuspendOrActivate(),refusing to self-activate, no pipes open\n");
		return ntStatus;
	}

	// dont do anything if registry CurrentControlSet\Services\BulkUsb\Parameters\PowerDownLevel
	//  has been set to  zero, PowerDeviceD0 ( 1 ), or a bogus high value
	if ( ( s->PowerDownLevel == PowerDeviceD0 ) ||
	     ( s->PowerDownLevel == PowerDeviceUnspecified)  ||
	     ( s->PowerDownLevel >= PowerDeviceMaximum ) ) {
		printk( "Exit dabusb_SelfSuspendOrActivate(), refusing on s->PowerDownLevel == %d\n", s->PowerDownLevel);
		return ntStatus;
	}

	if ( fSuspend )
		PowerState.DeviceState = s->PowerDownLevel;
	else
		PowerState.DeviceState = PowerDeviceD0;  // power up all the way; we're probably just about to do some IO

	ntStatus = dabusb_SelfRequestPowerIrp( fdo, PowerState );

	printk( "dabusb_SelfSuspendOrActivate() status 0x%x on setting dev state %s\n", ntStatus, dabusb_StringForDevState(PowerState.DeviceState ) );

	return ntStatus;

}

BOOLEAN
dabusb_SetDevicePowerState(
	IN PDEVICE_OBJECT fdo,
	IN DEVICE_POWER_STATE DeviceState
	)
/*++

Routine Description:

    This routine is called when An IRP_MN_SET_POWER of type 'DevicePowerState'
    has been received by dabusb_ProcessPowerIrp().


Arguments:

    fdo - Pointer to the device object for the class device.

    DeviceState - Device specific power state to set the device in to.


Return Value:

    For requests to DeviceState D0 ( fully on ), returns TRUE to signal caller
    that we must set a completion routine and finish there.

--*/
{
	NTSTATUS ntStatus = STATUS_SUCCESS;
	pdabusb_t s;
	BOOLEAN fRes = FALSE;

	s = (pdabusb_t) fdo->DeviceExtension;

	switch (DeviceState) {
	case PowerDeviceD3:

		//
		// Device will be going OFF,
                // TODO: add any needed device-dependent code to save state here.
                //  ( We have nothing to do in this sample )
		//

		printk( "dabusb_SetDevicePowerState() PowerDeviceD3 (OFF)\n");

		s->CurrentDevicePowerState = DeviceState;
		break;

	case PowerDeviceD1:
	case PowerDeviceD2:
		//
		// power states D1,D2 translate to USB suspend

		printk( "dabusb_SetDevicePowerState()  %s\n",
			dabusb_StringForDevState(DeviceState) );

		s->CurrentDevicePowerState = DeviceState;
		break;

	case PowerDeviceD0:


		printk( "dabusb_SetDevicePowerState() PowerDeviceD0 (ON)\n");

		// We'll need to finish the rest in the completion routine;
		//   signal caller we're going to D0 and will need to set a completion routine
		fRes = TRUE;

		// Caller will pass on to PDO ( Physical Device object )
		break;

	default:

		printk( " Bogus DeviceState = %x\n", DeviceState);
	}

	return fRes;
}

NTSTATUS
dabusb_QueryCapabilities(
	IN PDEVICE_OBJECT       Lowerfdo,
	IN PDEVICE_CAPABILITIES DeviceCapabilities
	)

/*++

Routine Description:

    This routine generates an internal IRP from this driver to the lower portion
    of the driver stack to obtain information on the Device Object's
    capabilities. We are most interested in learning which system power states
    are to be mapped to which device power states for honoring
    IRP_MJ_SET_POWER Irps.

    This is a blocking call which waits for the IRP completion routine
    to set an event on finishing.

Arguments:

    Lowerfdo        - fdo beneath this driver in the stack.

Return Value:

    NTSTATUS value from the IoCallDriver() call.

--*/

{
	PIO_STACK_LOCATION nextStack;
	PIRP irp;
	NTSTATUS ntStatus;
	KEVENT event;


	// This is a DDK-defined DBG-only macro that ASSERTS we are not running pageable code
	// at higher than APC_LEVEL.
	PAGED_CODE();


	// Build an IRP for us to generate an internal query request to the PDO
	irp = IoAllocateIrp(Lowerfdo->StackSize, FALSE);

	if (!irp) {
		return STATUS_INSUFFICIENT_RESOURCES;
	}

	//
	// Preinit the device capability structures appropriately.
	//
	RtlZeroMemory( DeviceCapabilities, sizeof(DEVICE_CAPABILITIES) );
	DeviceCapabilities->Size = sizeof(DEVICE_CAPABILITIES);
	DeviceCapabilities->Version = 1;
	DeviceCapabilities->Address = -1;
	DeviceCapabilities->UINumber = -1;

	// IoGetNextIrpStackLocation gives a higher level driver access to the next-lower
	// driver's I/O stack location in an IRP so the caller can set it up for the lower driver.
	nextStack = IoGetNextIrpStackLocation(irp);
	nextStack->MajorFunction= IRP_MJ_PNP;
	nextStack->MinorFunction= IRP_MN_QUERY_CAPABILITIES;

	// init an event to tell us when the completion routine's been called
	KeInitializeEvent(&event, NotificationEvent, FALSE);

	// Set a completion routine so it can signal our event when
	//  the next lower driver is done with the Irp
	IoSetCompletionRoutine(irp,
			       OnRequestComplete,
			       &event,  // pass the event as Context to completion routine
			       TRUE,    // invoke on success
			       TRUE,    // invoke on error
			       TRUE);   // invoke on cancellation of the Irp


	// set our pointer to the DEVICE_CAPABILITIES struct
	nextStack->Parameters.DeviceCapabilities.Capabilities = DeviceCapabilities;

	// preset the irp to report not supported
	irp->IoStatus.Status = STATUS_NOT_SUPPORTED;

	ntStatus = IoCallDriver(Lowerfdo,
				irp);

	printk( " dabusb_QueryCapabilities() ntStatus from IoCallDriver to PCI = 0x%x\n", ntStatus);

	if (ntStatus == STATUS_PENDING) {
		// wait for irp to complete

		KeWaitForSingleObject(
			&event,
			Suspended,
			KernelMode,
			FALSE,
			NULL);

		ntStatus = irp->IoStatus.Status;
	}

	// failed? this is probably a bug
	if(!NT_SUCCESS(ntStatus)) 
		printk( "dabusb_QueryCapabilities() failed\n");

	IoFreeIrp(irp);

	return ntStatus;
}

NTSTATUS
dabusb_PowerIrp_Complete(
	IN PDEVICE_OBJECT Nullfdo,
	IN PIRP Irp,
	IN PVOID Context
	)
/*++

Routine Description:

    This routine is called when An IRP_MN_SET_POWER of type 'DevicePowerState'
    has been received by dabusb_ProcessPowerIrp(), and that routine has  determined
        1) the request is for full powerup ( to PowerDeviceD0 ), and
        2) We are not already in that state
    A call is then made to PoRequestPowerIrp() with this routine set as the completion routine.


Arguments:

    fdo - Pointer to the device object for the class device.

    Irp - Irp completed.

    Context - Driver defined context.

Return Value:

    The function value is the final status from the operation.

--*/
{
	NTSTATUS ntStatus = STATUS_SUCCESS;
	PDEVICE_OBJECT deviceObject;
	PIO_STACK_LOCATION irpStack;
	pdabusb_t s;

	printk( "enter dabusb_PowerIrp_Complete\n");

	deviceObject = (PDEVICE_OBJECT) Context;

	s = (pdabusb_t) deviceObject->DeviceExtension;

	//  If the lower driver returned PENDING, mark our stack location as pending also.
	if (Irp->PendingReturned) {
		IoMarkIrpPending(Irp);
	}

	irpStack = IoGetCurrentIrpStackLocation (Irp);

	// We can assert that we're a  device powerup-to D0 request,
	// because that was the only type of request we set a completion routine
	// for in the first place
	ASSERT(irpStack->MajorFunction == IRP_MJ_POWER);
	ASSERT(irpStack->MinorFunction == IRP_MN_SET_POWER);
	ASSERT(irpStack->Parameters.Power.Type==DevicePowerState);
	ASSERT(irpStack->Parameters.Power.State.DeviceState==PowerDeviceD0);

	// Now that we know we've let the lower drivers do what was needed to power up,
	//  we can set our device extension flags accordingly
	s->CurrentDevicePowerState = PowerDeviceD0;

	Irp->IoStatus.Status = ntStatus;

	dabusb_decrement_io_count(deviceObject);

	printk( "exit dabusb_PowerIrp_Complete Exit IRP_MN_SET_POWER D0 complete\n");
	return ntStatus;
}



NTSTATUS
dabusb_PoRequestCompletion(
	IN PDEVICE_OBJECT       fdo,
	IN UCHAR                MinorFunction,
	IN POWER_STATE          PowerState,
	IN PVOID                Context,
	IN PIO_STATUS_BLOCK     IoStatus
	)
/*++

Routine Description:

        This is the completion routine set in a call to PoRequestPowerIrp()
        that was made in dabusb_ProcessPowerIrp() in response to receiving
    an IRP_MN_SET_POWER of type 'SystemPowerState' when the device was
        not in a compatible device power state. In this case, a pointer to
        the IRP_MN_SET_POWER Irp is saved into the FDO device extension
        (s->PowerIrp), and then a call must be
        made to PoRequestPowerIrp() to put the device into a proper power state,
        and this routine is set as the completion routine.

    We decrement our pending io count and pass the saved IRP_MN_SET_POWER Irp
        on to the next driver

Arguments:

    fdo - Pointer to the device object for the class device.
        Note that we must get our own device object from the Context

    Context - Driver defined context, in this case our own functional device object ( FDO )

Return Value:

    The function value is the final status from the operation.

--*/
{
	PIRP irp;
	pdabusb_t s;
	PDEVICE_OBJECT deviceObject = Context;
	NTSTATUS ntStatus;

	s = deviceObject->DeviceExtension;

        // Get the Irp we saved for later processing in dabusb_ProcessPowerIrp()
        // when we decided to request the Power Irp that this routine
        // is the completion routine for.
	irp = s->PowerIrp;

        // We will return the status set by the PDO for the power request we're completing
	ntStatus = IoStatus->Status;

	printk( "Enter dabusb_PoRequestCompletion()\n");

	// we should not be in the midst of handling a self-generated power irp
	ASSERT( !s->SelfPowerIrp );

	// we must pass down to the next driver in the stack
	IoCopyCurrentIrpStackLocationToNext(irp);

	// Calling PoStartNextPowerIrp() indicates that the driver is finished
	// with the previous power IRP, if any, and is ready to handle the next power IRP.
	// It must be called for every power IRP.Although power IRPs are completed only once,
	// typically by the lowest-level driver for a device, PoStartNextPowerIrp must be called
	// for every stack location. Drivers must call PoStartNextPowerIrp while the current IRP
	// stack location points to the current driver. Therefore, this routine must be called
	// before IoCompleteRequest, IoSkipCurrentStackLocation, and PoCallDriver.

	PoStartNextPowerIrp(irp);

	// PoCallDriver is used to pass any power IRPs to the PDO instead of IoCallDriver.
	// When passing a power IRP down to a lower-level driver, the caller should use
	// IoSkipCurrentIrpStackLocation or IoCopyCurrentIrpStackLocationToNext to copy the IRP to
	// the next stack location, then call PoCallDriver. Use IoCopyCurrentIrpStackLocationToNext
	// if processing the IRP requires setting a completion routine, or IoSkipCurrentStackLocation
	// if no completion routine is needed.
//beep(500);
	PoCallDriver(s->sdo,
		     irp);
//beep(1000);
	dabusb_decrement_io_count(deviceObject);

	printk( "dabusb_PoRequestCompletion() Exit IRP_MN_SET_POWER\n");

	s->PowerIrp = NULL;

	return ntStatus;
}


NTSTATUS
dabusb_dispatch_power(
	IN PDEVICE_OBJECT fdo,
	IN PIRP           Irp
	)
/*++

Routine Description:

    This is our FDO's dispatch table function for IRP_MJ_POWER.
    It processes the Power IRPs sent to the PDO for this device.

    For every power IRP, drivers must call PoStartNextPowerIrp and use PoCallDriver
    to pass the IRP all the way down the driver stack to the underlying PDO.


Arguments:

    fdo - pointer to our device object (FDO)

    Irp          - pointer to an I/O Request Packet

Return Value:

    NT status code

--*/
{

	PIO_STACK_LOCATION irpStack;
	NTSTATUS ntStatus = STATUS_SUCCESS;
	pdabusb_t s;
	BOOLEAN fGoingToD0 = FALSE;
	POWER_STATE sysPowerState, desiredDevicePowerState;
	KEVENT event;

	printk( " dabusb_ProcessPowerIrp() IRP_MJ_POWER\n");

	s = (pdabusb_t) fdo->DeviceExtension;
	irpStack = IoGetCurrentIrpStackLocation (Irp);
	dabusb_increment_io_count(fdo);

	switch (irpStack->MinorFunction) {
#if 1
	case IRP_MN_WAIT_WAKE:
		printk( "dabusb_ProcessPowerIrp() Enter IRP_MN_WAIT_WAKE\n");

		//beep(500);
                // A driver sends IRP_MN_WAIT_WAKE to indicate that the system should
                // wait for its device to signal a wake event. The exact nature of the event
                // is device-dependent.
                // Drivers send this IRP for two reasons:
                // 1) To allow a device to wake the system
                // 2) To wake a device that has been put into a sleep state to save power
                //    but still must be able to communicate with its driver under certain circumstances.
                // When a wake event occurs, the driver completes the IRP and returns
                // STATUS_SUCCESS. If the device is sleeping when the event occurs,
                // the driver must first wake up the device before completing the IRP.
                // In a completion routine, the driver calls PoRequestPowerIrp to send a
                // PowerDeviceD0 request. When the device has powered up, the driver can
                //  handle the IRP_MN_WAIT_WAKE request.

		// s->DeviceCapabilities.DeviceWake specifies the lowest device power state (least powered)
		// from which the device can signal a wake event
		s->PowerDownLevel = s->DeviceCapabilities.DeviceWake;


		if  ( ( PowerDeviceD0 == s->CurrentDevicePowerState )  ||
		      ( s->DeviceCapabilities.DeviceWake > s->CurrentDevicePowerState ) ) {
                        //
                        //    STATUS_INVALID_DEVICE_STATE is returned if the device in the PowerD0 state
                        //    or a state below which it can support waking, or if the SystemWake state
                        //    is below a state which can be supported. A pending IRP_MN_WAIT_WAKE will complete
                        //    with this error if the device's state is changed to be incompatible with the wake
                        //    request.

			//  If a driver fails this IRP, it should complete the IRP immediately without
			//  passing the IRP to the next-lower driver.
			ntStatus = STATUS_INVALID_DEVICE_STATE;
			Irp->IoStatus.Status = ntStatus;
			IoCompleteRequest (Irp,IO_NO_INCREMENT );
			printk(  "Exit dabusb_ProcessPowerIrp(), ntStatus STATUS_INVALID_DEVICE_STATE\n" );
			dabusb_decrement_io_count(fdo);
			return ntStatus;
		}

		// flag we're enabled for wakeup
		s->EnabledForWakeup = TRUE;

		// init an event for our completion routine to signal when PDO is done with this Irp
		KeInitializeEvent(&event, NotificationEvent, FALSE);

		// If not failing outright, pass this on to our PDO for further handling
		IoCopyCurrentIrpStackLocationToNext(Irp);

		// Set a completion routine so it can signal our event when
		//  the PDO is done with the Irp
		IoSetCompletionRoutine(Irp,
				       OnRequestComplete,
				       &event,  // pass the event to the completion routine as the Context
				       TRUE,    // invoke on success
				       TRUE,    // invoke on error
				       TRUE);   // invoke on cancellation

		PoStartNextPowerIrp(Irp);
		ntStatus = PoCallDriver(s->sdo,
					Irp);

		// if PDO is not done yet, wait for the event to be set in our completion routine
		if (ntStatus == STATUS_PENDING) {
			// wait for irp to complete

			NTSTATUS waitStatus = KeWaitForSingleObject(
				&event,
				Suspended,
				KernelMode,
				FALSE,
				NULL);

			printk( "dabusb_ProcessPowerIrp() done waiting for PDO to finish IRP_MN_WAIT_WAKE\n");
		}

                // now tell the device to actually wake up
                dabusb_SelfSuspendOrActivate( fdo, FALSE );

		// flag we're done with wakeup irp
		s->EnabledForWakeup = FALSE;

		dabusb_decrement_io_count(fdo);

		printk( "dabusb_ProcessPowerIrp() Exit IRP_MN_WAIT_WAKE\n");
		break;

	case IRP_MN_SET_POWER:
        {

                // The system power policy manager sends this IRP to set the system power state.
                // A device power policy manager sends this IRP to set the device power state for a device.

		printk( "dabusb_ProcessPowerIrp() Enter IRP_MN_SET_POWER\n");

		// Set Irp->IoStatus.Status to STATUS_SUCCESS to indicate that the device
		// has entered the requested state. Drivers cannot fail this IRP.

		switch (irpStack->Parameters.Power.Type) {
		case SystemPowerState:

			// Get input system power state
			sysPowerState.SystemState = irpStack->Parameters.Power.State.SystemState;

			printk( "dabusb_ProcessPowerIrp() Set Power, type SystemPowerState = %s\n",
				dabusb_StringForSysState( sysPowerState.SystemState ) );

			// If system is in working state always set our device to D0
			//  regardless of the wait state or system-to-device state power map
			if ( sysPowerState.SystemState ==  PowerSystemWorking) {
				desiredDevicePowerState.DeviceState = PowerDeviceD0;

				printk( "dabusb_ProcessPowerIrp() PowerSystemWorking, will set D0, not use state map\n");
				//beep(2000);


			} else {
				// set to corresponding system state if IRP_MN_WAIT_WAKE pending
				if ( s->EnabledForWakeup ) { // got a WAIT_WAKE IRP pending?

					// Find the device power state equivalent to the given system state.
					// We get this info from the DEVICE_CAPABILITIES struct in our device
					// extension (initialized in dabusb_PnPAddDevice() )
					desiredDevicePowerState.DeviceState =
						s->DeviceCapabilities.DeviceState[ sysPowerState.SystemState ];
						
					printk( "dabusb_ProcessPowerIrp() IRP_MN_WAIT_WAKE pending, will use state map\n");

				} else {
					// if no wait pending and the system's not in working state, just turn off
					desiredDevicePowerState.DeviceState = PowerDeviceD3;
						
					printk( "dabusb_ProcessPowerIrp() Not EnabledForWakeup and the system's not in working state,\n  settting PowerDeviceD3 (off )\n");
				}
			}
			//
			// We've determined the desired device state; are we already in this state?
			//

			printk( "dabusb_ProcessPowerIrp() Set Power, desiredDevicePowerState = %s\n",
				dabusb_StringForDevState( desiredDevicePowerState.DeviceState ) );
#ifdef deti
			s->CurrentDevicePowerState = desiredDevicePowerState.DeviceState;
			s->device_state = _removed;
#else
			if (desiredDevicePowerState.DeviceState !=
			    s->CurrentDevicePowerState) {

				// No, request that we be put into this state
				// by requesting a new Power Irp from the Pnp manager
                    
				s->PowerIrp = Irp;
				ntStatus = PoRequestPowerIrp(s->pdo,
							     IRP_MN_SET_POWER,
							     desiredDevicePowerState,
							     // completion routine will pass the Irp down to the PDO
							     dabusb_PoRequestCompletion,
							     fdo,
							     NULL);
//beep(2000);
			} else 
#endif				
				{
					// Yes, just pass it on to PDO (Physical Device Object)
					IoCopyCurrentIrpStackLocationToNext(Irp);
					PoStartNextPowerIrp(Irp);
					ntStatus = PoCallDriver(s->sdo,
								Irp);

					dabusb_decrement_io_count(fdo);
					printk( "dabusb_ProcessPowerIrp() Exit IRP_MN_SET_POWER\n");

				}

			break;

		case DevicePowerState:

			printk( "dabusb_ProcessPowerIrp() Set Power, type DevicePowerState = %s\n",
				dabusb_StringForDevState( irpStack->Parameters.Power.State.DeviceState ) );

			// For requests to D1, D2, or D3 ( sleep or off states ),
                                // sets s->CurrentDevicePowerState to DeviceState immediately.
                                // This enables any code checking state to consider us as sleeping or off
                                // already, as this will imminently become our state.

			// For requests to DeviceState D0 ( fully on ), sets fGoingToD0 flag TRUE
			// to flag that we must set a completion routine and update
                                // s->CurrentDevicePowerState there.
                                // In the case of powering up to fully on, we really want to make sure
                                // the process is completed before updating our CurrentDevicePowerState,
                                // so no IO will be attempted or accepted before we're really ready.
//beep(500);
			fGoingToD0 = dabusb_SetDevicePowerState(fdo,
								irpStack->Parameters.Power.State.DeviceState
				); // returns TRUE for D0
//beep(1000);
			IoCopyCurrentIrpStackLocationToNext(Irp);

			if (fGoingToD0) {
				printk( "dabusb_ProcessPowerIrp() Set PowerIrp Completion Routine, fGoingToD0 =%d\n", fGoingToD0);
				IoSetCompletionRoutine(Irp,
						       dabusb_PowerIrp_Complete,
						       // Always pass FDO to completion routine as its Context;
						       // This is because the DriverObject passed by the system to the routine
						       // is the Physical Device Object ( PDO ) not the Functional Device Object ( FDO )
						       fdo,
						       TRUE,            // invoke on success
						       TRUE,            // invoke on error
						       TRUE);           // invoke on cancellation of the Irp
			}

			PoStartNextPowerIrp(Irp);
			ntStatus = PoCallDriver(s->sdo,
						Irp);

			if ( !fGoingToD0 ) // completion routine will decrement
				dabusb_decrement_io_count(fdo);

			printk( "dabusb_ProcessPowerIrp() Exit IRP_MN_SET_POWER\n");
			break;
		} /* case irpStack->Parameters.Power.Type */

        }
        break; /* IRP_MN_SET_POWER */

	case IRP_MN_QUERY_POWER:
                //
                // A power policy manager sends this IRP to determine whether it can change
                // the system or device power state, typically to go to sleep.
                //

		printk( "dabusb_ProcessPowerIrp() IRP_MN_QUERY_POWER %d %s %s\n",
			irpStack->Parameters.Power.Type,
			dabusb_StringForSysState( irpStack->Parameters.Power.State.SystemState),
			dabusb_StringForDevState( irpStack->Parameters.Power.State.DeviceState ));

		if(!s->opened)
			{
				// We do nothing special here, just let the PDO handle it
				IoCopyCurrentIrpStackLocationToNext(Irp);
				PoStartNextPowerIrp(Irp);
				ntStatus = PoCallDriver(s->sdo, Irp);
			}
		else
			{
				PoStartNextPowerIrp(Irp);
				ntStatus=STATUS_DEVICE_NOT_READY;
				Irp->IoStatus.Status=ntStatus;
				IoCompleteRequest (Irp,IO_NO_INCREMENT );
			}


		dabusb_decrement_io_count(fdo);

		break; /* IRP_MN_QUERY_POWER */
#endif
	default:

		printk( "dabusb_ProcessPowerIrp() UNKNOWN POWER MESSAGE (%x)\n", irpStack->MinorFunction);

		//
		// All unhandled power messages are passed on to the PDO
		//

		IoCopyCurrentIrpStackLocationToNext(Irp);
		PoStartNextPowerIrp(Irp);
		ntStatus = PoCallDriver(s->sdo, Irp);

		dabusb_decrement_io_count(fdo);

	} /* irpStack->MinorFunction */

	printk(  "Exit dabusb_ProcessPowerIrp()  ntStatus = 0x%x\n", ntStatus );
	return ntStatus;
}
