/*
 * Argus Client Software.  Tools to read, analyze and manage Argus data.
 * Copyright (c) 2000-2003 QoSient, LLC
 * All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

/*
 * rahistogram - argus record collector and time bin output processor
 *
 * written by Carter Bullard
 * QoSient, LLC
 *
 */


#include <argus_client.h>
#include <rahistogram.h>
#include <signal.h>
#include <math.h>

#define RA_PRINTPROTO	1
#define RA_PRINTSRCID	2

#define RAMON_TOPN      1
#define RAMON_MATRIX    2
#define RAMON_SVCS      3
#define RAMON_SINGLE    4


int RaPrintMode = 0;
extern int RaHistoStart, RaHistoEnd, RaHistoBins;

struct RaBinStruct {
   struct timeval start, end;
   int status, timeout;
   struct RaQueueStruct *queue;
   struct RaHashTableStruct hashtable;
};

struct RaBinStruct *RaThisBin;

struct RaBinStruct *RaNewBin(struct ArgusRecord *);
void RaDeleteBin (struct RaBinStruct *);

int RaLabelCounter = 0;
int RaFirstArgusRecord = 1;
int RaSeconds = 0;
int RaScaleSeconds = 0;
int RaStartSeconds = 0;
int RaEndSeconds = 0;
struct timeval RaStartTime = {0x7FFFFFFF, 0x7FFFFFFF};
struct timeval RaEndTime = {0,0};
struct timeval RaStartPoint = {0x7FFFFFFF, 0x7FFFFFFF};
struct timeval RaEndPoint = {0,0};

int RaBinCount = 0;
float RaBinSize = 0.0;

void RaInitializeMergeArray(void);
void RaReInitializeMergeArray(int);

#define RACRICKET_HASHTABLESIZE		128
 
void RaProcessArray(struct RaBinStruct **, unsigned char);
struct ArgusRecord * RaAdjustArgusRecord (struct ArgusRecord *);
struct ArgusRecord * RaRmonArgusRecord (struct ArgusRecord *);
void RaProcessThisSrvRecord (struct ArgusServiceRecord *);
int RaCreateEtherFlow (struct ArgusRecord *);
void RaModifyEtherFlow(struct ArgusRecord *);

struct RaBinStruct **RaBinMergeArray = NULL;
struct RaBinStruct **RaBinOldMergeArray = NULL;
struct tm RaThisTmStructBuf, *RaThisTmStruct = &RaThisTmStructBuf;
struct tm RaEndTmStructBuf, *RaEndTmStruct = &RaEndTmStructBuf;
struct tm RaStartTmStructBuf, *RaStartTmStruct = &RaStartTmStructBuf;

int RaAdjustCount = 0;
int RaHoldTime = 0;

int RaNoZeroMode = 0;
int RaStreamMode = 0;
int RaProbeMode  = 0;
int RaNetMode    = 0;
int RaMonMode    = 0;
int RaMonMask    = 0;


extern int ArgusPassNum;

int RaPassNum = 0;
int RaPrintMACAddress = 0;

struct RaQueueStruct *RaProbeQueue = NULL;

struct timeval * RaMaxTime  (struct timeval *, struct timeval *);
struct timeval * RaMinTime  (struct timeval *, struct timeval *);
struct timeval * RaDiffTime (struct timeval *, struct timeval *);

struct timeval *
RaMinTime (struct timeval *s1, struct timeval *s2)
{
   struct timeval *retn = s2;

   if ((s1->tv_sec < s2->tv_sec) || ((s1->tv_sec == s2->tv_sec) && (s1->tv_usec < s2->tv_usec)))
      retn = s1;

   return (retn);
}


struct timeval *
RaMaxTime (struct timeval *s1, struct timeval *s2)
{
   struct timeval *retn = s2;

   if ((s1->tv_sec > s2->tv_sec) || ((s1->tv_sec == s2->tv_sec) && (s1->tv_usec > s2->tv_usec)))
      retn = s1;
  
   return (retn);
}

struct timeval RaDiffTimeBuf;

struct timeval *
RaDiffTime (struct timeval *s1, struct timeval *s2)
{
   struct timeval *retn = NULL;

   bzero ((char *)&RaDiffTimeBuf, sizeof(RaDiffTimeBuf));

   if (s1 && s2) {
      RaDiffTimeBuf.tv_sec  = s1->tv_sec - s2->tv_sec;
      RaDiffTimeBuf.tv_usec = s1->tv_usec - s2->tv_usec;

      if (RaDiffTimeBuf.tv_usec < 0) {
         RaDiffTimeBuf.tv_usec += 1000000;
         RaDiffTimeBuf.tv_sec--;
      }
      retn = &RaDiffTimeBuf;
   }

   return (retn);
}


void RaProcessRecord (struct ArgusRecord *argus)
{
   if (RaFirstArgusRecord && (!(argus->ahdr.type & ARGUS_MAR))) {
      RaStartTime = argus->argus_far.time.start;
      RaInitializeMergeArray();
      RaFirstArgusRecord = 0;
   }

   if (argus->ahdr.type & ARGUS_MAR)
      RaProcessManRecord (argus);

   else {

      switch (argus->ahdr.status & 0xFFFF) {
         case ETHERTYPE_IP:
            switch (argus->argus_far.flow.ip_flow.ip_p) {
               case IPPROTO_TCP:
                  RaProcessTCPRecord (argus);
                  break;

               case IPPROTO_UDP:
                  RaProcessUDPRecord (argus);
                  break;

               case IPPROTO_ICMP:
                  RaProcessICMPRecord (argus);
                  break;

               default:
                  RaProcessIPRecord (argus);
                  break;
            }
            break;

         case ETHERTYPE_ARP:
         case ETHERTYPE_REVARP:
            RaProcessARPRecord (argus);
            break;

         default:
            RaProcessNonIPRecord (argus);
            break;
      }
   }

/*
   if (Nflag > 0) {
      if (Nflag-- == 1) {
         RaParseComplete(SIGINT);
      }
   }
*/
}

char *RaMonTopNModelString   = "Model 200  ip %s  0.0.0.0  %s  %s  %s";
char *RaMonMatrixModelString = "Model 200  ip %s %s  %s  %s  %s";

char *RaMonSingleFlowModelFile [] = {
"RAGATOR_PRESERVE_FIELDS=yes",
"RAGATOR_REPORT_AGGREGATION=yes",

"Flow  100 ip    *        *    *   *   *   200  0   0",
"Model 200 ip 0.0.0.0  0.0.0.0 no  no  no",
NULL,
};

char *RaMonMatrixFlowModelFile [] = {
"RAGATOR_PRESERVE_FIELDS=yes",
"RAGATOR_REPORT_AGGREGATION=yes",

"Flow  100 ip     *                  *         *   *   *   200  0   0",
"Model 200 ip 255.255.255.255  255.255.255.255 no  no  no",
NULL,
};

char *RaMonTopNFlowModelFile [] = {
"RAGATOR_PRESERVE_FIELDS=yes",
"RAGATOR_REPORT_AGGREGATION=yes",
"Flow  100 ip     *                  *         *   *   *   200  0   0",
"Model 200 ip 255.255.255.255     0.0.0.0      no  no  no",
NULL,
};

 
char *RaMonServicesFlowModelFile [] = {
"RAGATOR_PRESERVE_FIELDS=yes",
"RAGATOR_REPORT_AGGREGATION=yes",
"Flow   10 ip     *        *     tcp   20        *    100  0 0",
"Flow  100 ip     *        *     *      *        *    200  0 0",
"Model 100 ip  0.0.0.0  0.0.0.0  yes   yes      yes",
"Model 200 ip  0.0.0.0  0.0.0.0  yes   no       yes",
NULL,
};



void RaProcessSrvRecord (struct ArgusServiceRecord *);
 
int RaInitialized = 0;
int RaTimeout     = 0x1FFFFFFF;
int RaIdleTimeout = 0;

#define RAHISTOGRAMMODENUM      8
#define RASECONDS               0
#define RAMINUTE                1
#define RAHOURLY                2
#define RADAILY                 3
#define RAWEEKLY                4
#define RAMONTHLY               5
#define RANNUALY                6
#define RAONEBIN                7

int RaModeValue = RAONEBIN;

char *RaHistogramModes[RAHISTOGRAMMODENUM] = {
  "second",
  "minute",
  "hourly",
  "dayly",
  "weekly",
  "Monthly",
  "annually",
  "onebin",
};

extern struct timeval RaClientTimeout;
extern int RaFlowMajorModified;
struct RaFlowModelStruct *RaFlowModel = NULL;

char *RaPrintProtoStr   = "no";
char *RaPrintSrcPortStr = "no";
char *RaPrintDstPortStr = "no";

void
ArgusClientInit ()
{
   struct ArgusModeStruct *mode = NULL;
   int i = 0, found = 0;
   char *ptr = NULL;

   if (!(RaInitialized)) {
      RaWriteOut = 0;
      RaCumulativeMerge = 1;
      RaFlowMajorModified = 1;

      if (dflag == 0)
         dflag++;

      if (ArgusFlowModelFile)
         if ((RaFlowModel = RaReadFlowModelFile(NULL)) == NULL)
            ArgusLog(LOG_ERR, "ArgusClientInit RaReadFlowModelFile %s", strerror(errno));

      if ((mode = ArgusModeList) != NULL) {
         while (mode) {
            found = 0;
            if (isdigit((int) *mode->mode)) {
               if (sscanf(mode->mode, "%d", &dflag) == 1)
                  while (isdigit((int) *mode->mode))
                     mode->mode++;
            }
            for (i = 0; i < RAHISTOGRAMMODENUM; i++) {
               if (!(strncasecmp (mode->mode, RaHistogramModes[i], strlen(mode->mode)))) {
                  RaModeValue = i;
                  found++;
                  break;
               }
            }

            if (!found) {
               if (!(strcasecmp (mode->mode, "nozero"))) {
                  found++;
                  RaNoZeroMode++;
               } else
               if (!(strcasecmp (mode->mode, "Single"))) {
                  found++;
                  if (RaMonMode == RAMON_TOPN)
                     ArgusLog(LOG_ERR, "-M Single conflicts with -M Addrs");
                  if (RaMonMode == RAMON_MATRIX)
                     ArgusLog(LOG_ERR, "-M Single conflicts with -M Matrix");
                  if (RaMonMode == RAMON_SVCS)
                     ArgusLog(LOG_ERR, "-M Single conflicts with -M Ports");
 
                  RaMonMode = RAMON_SINGLE;
                  if (RaFlowModelFile) {
                     ArgusLog(LOG_ERR, "multiple models specified");
                  } else {
                     RaFlowModelFile = RaMonSingleFlowModelFile;
                  }
 
                  if (!(RaSOptionIndex)) {
                     bzero ((char *) RaPrintAlgorithms, sizeof(RaPrintAlgorithms));
                     RaPrintAlgorithms[0] = ArgusPrintDate;
                     RaPrintAlgorithms[1] = ArgusPrintCounts;
                     RaPrintAlgorithms[2] = ArgusPrintBytes;
                  }
               } else
               if (!(strcasecmp (mode->mode, "Addrs"))) {
                  found++;
                  if (RaMonMode == RAMON_MATRIX)
                     ArgusLog(LOG_ERR, "-M Addrs conflicts with -M Matrix");

                  RaMonMode = RAMON_TOPN;
                  if (RaFlowModelFile) {
                     ArgusLog(LOG_ERR, "multiple models specified");
                  } else {
                     RaFlowModelFile = RaMonTopNFlowModelFile;
                  }

                  if (!(RaSOptionIndex)) {
                     bzero ((char *) RaPrintAlgorithms, sizeof(RaPrintAlgorithms));
                     RaPrintAlgorithms[0] = ArgusPrintDate;
                     RaPrintAlgorithms[1] = ArgusPrintSrcAddr;
                     RaPrintAlgorithms[2] = ArgusPrintProto;
                     RaPrintAlgorithms[3] = ArgusPrintCounts;
                     RaPrintAlgorithms[4] = ArgusPrintBytes;
                  }

               } else
               if (!(strcasecmp (mode->mode, "Matrix"))) {
                  found++;
                  if (RaMonMode == RAMON_TOPN)
                     ArgusLog(LOG_ERR, "-M Matrix conflicts with -M Addrs");

                  RaMonMode = RAMON_MATRIX;
                  if (RaFlowModelFile) {
                     ArgusLog(LOG_ERR, "multiple models specified");
                  } else {
                     RaFlowModelFile = RaMonMatrixFlowModelFile;
                  }

                  if (!(RaSOptionIndex)) {
                     bzero ((char *) RaPrintAlgorithms, sizeof(RaPrintAlgorithms));
                     RaPrintAlgorithms[0] = ArgusPrintDate;
                     RaPrintAlgorithms[1] = ArgusPrintSrcAddr;
                     RaPrintAlgorithms[2] = ArgusPrintDstAddr;
                     RaPrintAlgorithms[3] = ArgusPrintProto;
                     RaPrintAlgorithms[4] = ArgusPrintCounts;
                     RaPrintAlgorithms[5] = ArgusPrintBytes;
                  }

               } else
               if (!(strcasecmp (mode->mode, "Ports"))) {
                  found++;
                  if (RaMonMode == RAMON_TOPN)
                     ArgusLog(LOG_ERR, "-M Svc conflicts with -M TopN");
                  if (RaMonMode == RAMON_MATRIX)
                     ArgusLog(LOG_ERR, "-M Svc conflicts with -M Matrix");
 
                  RaMonMode = RAMON_SVCS;
                  if (RaFlowModelFile) {
                     ArgusLog(LOG_ERR, "multiple models specified");
                  } else {
                     RaFlowModelFile = RaMonServicesFlowModelFile;
                  }
 
                  if (!(RaSOptionIndex)) {
                     bzero ((char *) RaPrintAlgorithms, sizeof(RaPrintAlgorithms));
                     RaPrintAlgorithms[0] = ArgusPrintDate;
                     RaPrintAlgorithms[1] = ArgusPrintDstPort;
                     RaPrintAlgorithms[2] = ArgusPrintProto;
                     RaPrintAlgorithms[3] = ArgusPrintCounts;
                     RaPrintAlgorithms[4] = ArgusPrintBytes;
                  }
 
               } else
               if (!(strncasecmp (mode->mode, "Net", 3))) {
                  found++;
                  if ((ptr = strchr (mode->mode, '/')) != NULL) {
                     if (sscanf(&ptr[1], "%d", &RaMonMask) != 1)
                        usage();
                  }
       
                  RaNetMode++;

               } else
               if (!(strcasecmp (mode->mode, "Probe"))) {
                  found++;
                  RaProbeMode++;
               } else
               if (!(strcasecmp (mode->mode, "Stream"))) {
                  found++;
                  RaStreamMode++;
               }
            }

            if (!found)
               usage();

            mode = mode->nxt;
         }
      }

      if (Gflag) {
         uflag++;
         RaFieldDelimiter = ',';
      }

      for (i = 0; RaPrintAlgorithms[i] != NULL; i++) {
         if (RaPrintAlgorithms[i] == ArgusPrintProto) {
            RaPrintMode |= RA_PRINTPROTO;
            RaPrintProtoStr = "yes";
            break;
         }
         if (RaPrintAlgorithms[i] == ArgusPrintSrcPort) {
            RaPrintProtoStr = "yes";
            break;
         }
         if (RaPrintAlgorithms[i] == ArgusPrintDstPort) {
            RaPrintProtoStr = "yes";
            break;
         }
         if (RaPrintAlgorithms[i] == ArgusPrintSourceID) {
            RaPrintMode |= RA_PRINTSRCID;
            break;
         }
      }

      if (!RaNetMode)
         RaMonMask = 32;

      switch (RaMonMode) {
         case RAMON_TOPN: {
            char buf[MAXSTRLEN];
            char addrstr[128];

            if (RaMonMask != 0) {
               sprintf (addrstr, "255.255.255.255/%d", RaMonMask);
            } else
               sprintf (addrstr, "class");

            sprintf (buf, RaMonTopNModelString, addrstr, RaPrintProtoStr, RaPrintSrcPortStr, RaPrintDstPortStr);
            RaMonTopNFlowModelFile [1] = strdup(buf);
            RaFlowModel = RaReadFlowModelFile (RaMonTopNFlowModelFile);
            break;
         }

         case RAMON_MATRIX: {
            char buf[MAXSTRLEN];
            char addrstr[128];

            if (RaMonMask != 0) {
               sprintf (addrstr, "255.255.255.255/%d", RaMonMask);
            } else
               sprintf (addrstr, "class");

            sprintf (buf, RaMonMatrixModelString, addrstr, addrstr, RaPrintProtoStr, RaPrintSrcPortStr, RaPrintDstPortStr);
            RaMonMatrixFlowModelFile [1] = strdup(buf);
            RaFlowModel = RaReadFlowModelFile (RaMonMatrixFlowModelFile);
            break;
         }

         case RAMON_SINGLE:
            RaFlowModel = RaReadFlowModelFile (RaMonSingleFlowModelFile);
            break;

         case RAMON_SVCS:
            RaFlowModel = RaReadFlowModelFile (RaMonServicesFlowModelFile);
            break;

         default:
            break;
      }

      if (ArgusFlowModelFile != NULL)
         ArgusFlowModelFile = "dummy";
   
      if ((RaModelerQueue = RaNewQueue()) == NULL)
         exit(0);

      if ((RaProbeQueue = RaNewQueue()) == NULL)
         exit(0);
   
      if ((RaHashTable.array = (struct RaHashTableHeader **) ArgusCalloc (RACRICKET_HASHTABLESIZE,
                                    sizeof (struct RaHashTableHeader *))) != NULL) {
         RaHashTable.size = RACRICKET_HASHTABLESIZE;
      }

      if ((RaProbeTable.array = (struct RaHashTableHeader **) ArgusCalloc (RA_HASHTABLESIZE,
                                    sizeof (struct RaHashTableHeader *))) != NULL) {
         RaProbeTable.size = RA_HASHTABLESIZE;
      }

      if (Hflag) {
         if (!(RaHistoTimeParse (Hstr)))
            usage();
      }

      RaClientTimeout.tv_sec  = 1;
      RaClientTimeout.tv_usec = 0;

      if (nflag > 1) {
         hfield = 15;
         pfield =  5;
      } else {
         pfield =  15;
      }

      RaInitialized++;
   }
}

int RaParseCompleting = 0;

void
RaParseComplete (int sig)
{
   if (sig >= 0) {
      if ((ArgusInput != NULL) && (RaMonMode == RAMON_TOPN))
         ArgusInput->ArgusManStart.ahdr.type |= ARGUS_RMON;

      if (RaPassNum++ == 1) {
         RaFirstArgusRecord = 1;
         totalrecords = 0;

      } else {
         if (!RaParseCompleting) {
            RaParseCompleting++;
            if (RaModelerQueue)
               RaProcessArray (RaBinMergeArray, ARGUS_STOP);
         }

         if (sig == SIGINT)
            _exit(0);
      }
   }

#ifdef ARGUSDEBUG
   ArgusDebug (1, "RaParseComplete: returning ArgusPassNum %d RaPassNum %d\n", ArgusPassNum, RaPassNum);
#endif
}

struct timeval ArgusClientLastTimeout = {0,0};
void
ArgusClientTimeout ()
{
   struct timeval tvpbuf, *tvp = &tvpbuf;

   gettimeofday(tvp, 0L);
   ArgusAdjustGlobalTime (tvp);

   if (Sflag || Cflag || Bflag)
      RaProcessArray (RaBinMergeArray, ARGUS_STATUS);

#ifdef ARGUSDEBUG
   if ((tvp->tv_sec % 60) == 0) {
      if (ArgusClientLastTimeout.tv_sec < tvp->tv_sec) {
         struct RaBinStruct *bin = NULL;
         int i, qcount = 0, hcount = 0;

         for (i = 0; i < RaBinCount; i++) {
            if ((bin = RaBinMergeArray[i]) != NULL) {
               if (bin->queue && (bin->queue->count > 0))
                  qcount += bin->queue->count;
            
               if (bin->hashtable.count > 0)
                  hcount += bin->hashtable.count;
            }
         }

         ArgusDebug (2, "ArgusClientTimeout: RaBinCount %d RaHoldTime %d Hash %d HashHdrs %d Queue %d\n",
                        RaBinCount, RaHoldTime, hcount, RaAllocHashTableHeaders, qcount);
      }

      ArgusClientLastTimeout = *tvp;
   }

   ArgusDebug (7, "ArgusClientTimeout: done\n");
#endif
}

void
parse_arg (int argc, char**argv)
{ 
}

void
usage ()
{
   extern char version[];

   fprintf (stderr, "Ragator Version %s\n", version);
   fprintf (stderr, "usage:  %s [-f flowfile]\n", ArgusProgramName);
   fprintf (stderr, "usage:  %s [-f flowfile] [ra-options] [- filter-expression]\n", ArgusProgramName);
   fprintf (stderr, "options:    -f <flowfile>     read flow model from <flowfile>.\n");
   fprintf (stderr, "ra-options: -a                print record summaries on termination.\n");
   fprintf (stderr, "            -A                print application bytes.\n");
   fprintf (stderr, "            -b                dump packet-matching code.\n");
   fprintf (stderr, "            -C                treat the remote source as a Cisco Netflow source.\n");
   fprintf (stderr, "            -D <level>        specify debug level\n");
   fprintf (stderr, "            -F <conffile>     read configuration from <conffile>.\n");
   fprintf (stderr, "            -h                print help.\n");
   fprintf (stderr, "            -n                don't convert numbers to names.\n");
   fprintf (stderr, "            -p <digits>       print fractional time with <digits> precision.\n");
   fprintf (stderr, "            -q                quiet mode. don't print record outputs.\n");
   fprintf (stderr, "            -r <file>         read argus data <file>. '-' denotes stdin.\n");
   fprintf (stderr, "            -R                print out response data when availabile.\n");
   fprintf (stderr, "            -S <host[:port}>  specify remote argus <host> and optional port number.\n");
   fprintf (stderr, "            -t <timerange>    specify <timerange> for reading records.\n");
   fprintf (stderr, "                     format:  timeSpecification[-timeSpecification]\n");
   fprintf (stderr, "                              timeSpecification: [mm/dd[/yy].]hh[:mm[:ss]]\n");
   fprintf (stderr, "                                                  mm/dd[/yy]\n");
   fprintf (stderr, "                                                  -%%d{yMhdms}\n");
   fprintf (stderr, "            -T <secs>         attach to remote server for T seconds.\n");
   fprintf (stderr, "            -u                print time in Unix time format.\n");
#ifdef ARGUS_SASL
   fprintf (stderr, "            -U <user/auth>    specify <user/auth> authentication information.\n");
#endif
   fprintf (stderr, "            -w <file>         write output to <file>. '-' denotes stdout.\n");
   fprintf (stderr, "            -z                print Argus TCP state changes.\n");
   fprintf (stderr, "            -Z <s|d|b>        print actual TCP flag values.<'s'rc | 'd'st | 'b'oth>\n");
   exit(1);
}


int RaBinsCheckTimeout (struct RaBinStruct *, struct ArgusRecord *);

int
RaBinsCheckTimeout (struct RaBinStruct *bin, struct ArgusRecord *argus)
{
   int retn = 0, lapseTime;
   struct timeval *tvp;

   if (Sflag || Cflag)
      tvp = &ArgusRealTime;
   else
      tvp = &ArgusGlobalTime;

   if (bin->timeout > 0) {
      lapseTime = bin->end.tv_sec + bin->timeout;

      if ((tvp->tv_sec > lapseTime) || ((tvp->tv_sec == lapseTime) && (tvp->tv_usec > bin->end.tv_usec)))
         retn++;
   }

#ifdef ARGUSDEBUG
   ArgusDebug (6, "RaCheckTimeout: returning %d \n", retn);
#endif

   return (retn);
}


void
RaProcessSrvRecord (struct ArgusServiceRecord *srv)
{
   struct ArgusRecord *argus = NULL;
   struct ArgusRecord *arg2 = NULL;
   struct RaHashTableHeader *rtblhdr = NULL;
   unsigned int key = 0;

   if (srv && ((argus = srv->argus) != NULL)) {
      unsigned int probe = srv->argus->ahdr.argusid;
      unsigned short proto = srv->argus->argus_far.flow.ip_flow.ip_p;
      struct RaHashStruct ArgusHash;
      struct ArgusRecordStore *store;

      bzero ((char *)&ArgusHash, sizeof(ArgusHash));
      ArgusHash.len = sizeof(ArgusHash.buf);
      ArgusHash.hash = 0;
 
      switch (RaPrintMode) {
         case RA_PRINTSRCID: key = probe; break;
         case RA_PRINTPROTO: key = proto; break;
      }

      ArgusHash.buf[0] = key;

      if ((rtblhdr = RaFindHash(&RaProbeTable, &ArgusHash)) == NULL) {
         if ((store = RaNewArgusStore(argus)) == NULL)
            ArgusLog (LOG_ERR, "RaProcessSrvRecord: RaNewArgusStore %s", strerror(errno));
         else {
            if ((store->data[RaThisActiveIndex] = RaNewArgusData(argus)) == NULL)
               ArgusLog (LOG_ERR, "RaProcessSrvRecord: RaNewArgusData %s", strerror(errno));
         }

         if ((store->rahtblhdr = RaAddHashObject (&RaProbeTable, (void *) key, &ArgusHash)) == NULL)
            ArgusLog (LOG_ERR, "RaProcessSrvRecord: RaAddHashObject %s", strerror(errno));

         RaAddToQueue(RaProbeQueue, &store->qhdr);
      }

      if (RaMonMode == RAMON_TOPN) {
         if ((arg2 = RaRmonArgusRecord(argus)) != NULL) {
            RaProcessThisSrvRecord(srv);
            srv->argus = arg2;
            RaProcessThisSrvRecord(srv);
            ArgusFree(arg2);
         }

      } else {
         if (RaProbeMode && RaMonitorTable) {
            switch (srv->argus->ahdr.status & 0xFFFF) {
               case ETHERTYPE_IP: {
                  unsigned int saddr = srv->argus->argus_far.flow.ip_flow.ip_src;
 
                  if ((RaProbeMonitorsThisAddr (probe, saddr)))
                     ArgusReverseDataRecord(srv->argus);
               }
            }
         }

         RaProcessThisSrvRecord(srv);
      }
   }

#ifdef ARGUSDEBUG
   ArgusDebug (5, "RaProcessSrvRecord: returning.\n");
#endif
}


struct timeval RaThisStartTime;
int RaNumBlanks = 0;

void
RaProcessThisSrvRecord (struct ArgusServiceRecord *srv)
{
   struct ArgusRecord *argus = srv->argus, *thisArgus;
   struct ArgusRecordStore *store;
   struct RaPolicyStruct *rap;
   struct ArgusFlow *flow = &argus->argus_far.flow;
   int RaStartSec = 0;
   int retn = 0, i = 0, index = 0;
   extern struct ArgusCanonicalRecord *RaThisCanon;

   if (xflag && (srv->status == RA_SVCFAILED))
      return;

   if ((retn = argus_filter (ArgusFilterCode.bf_insns, RaThisCanon)) == 0)
      return;

   bcopy((char *) flow, (char *) RaArgusFlow, sizeof(struct ArgusFlow));

   if (!(RaPrintMACAddress)) {
      if ((rap = RaFlowModelOverRides(argus, RaFlowModel)) != NULL) {
         RaModifyFlow(rap, argus, RaFlowModel);
         RaTimeout = rap->ArgusTimeout;
         RaIdleTimeout = rap->ArgusIdleTimeout;
      } else {
         RaModifyDefaultFlow(argus);
      }
   } else
      RaModifyEtherFlow(argus);

   RaThisActiveDuration = RaGetActiveDuration(argus);
   RaThisActiveIndex = RaHistoTimeSeries;

   if (Hflag) {
      if (RaThisActiveDuration < RaHistoStart)
         return;

      for (i = 0; i < RaHistoTimeSeries; i++) {
         if (RaThisActiveDuration < RaHistoTimeValues[i]) {
            RaThisActiveIndex = i;
            break;
         }
      }

      if (i == RaHistoTimeSeries)
         return;

   } else
      RaThisActiveIndex = 0;

   RaTimeout     = RaHoldTime;
   RaIdleTimeout = 0;

   RaThisStartTime = argus->argus_far.time.start;

   if (!(RaStreamMode)) {
      while ((thisArgus = RaAdjustArgusRecord (argus)) != NULL) {
         RaAdjustCount++;

         if ((thisArgus->argus_far.src.count + thisArgus->argus_far.dst.count) > 0) {
            RaStartSec = thisArgus->argus_far.time.start.tv_sec - RaStartSeconds;

            if (RaStartSec < 0) {
               if (!(Sflag || Cflag || Bflag)) {
                  RaReInitializeMergeArray(thisArgus->argus_far.time.start.tv_sec);
                  RaStartSec = thisArgus->argus_far.time.start.tv_sec - RaStartSeconds;
               }
            }

            if (RaEndSeconds < thisArgus->argus_far.time.last.tv_sec) {
               RaReInitializeMergeArray(thisArgus->argus_far.time.last.tv_sec);
            }

            flow = &thisArgus->argus_far.flow;
            bcopy((char *) flow, (char *) RaArgusFlow, sizeof(struct ArgusFlow));
         
            if (RaModeValue == RAONEBIN) {
               index = 0;
            } else {
               index = (int) (RaStartSec/RaBinSize);
            }

            if (index < RaBinCount) {
               if (index >= 0) {
                  if ((RaThisBin = RaBinMergeArray[index]) == NULL) {
                     if ((RaThisBin = RaNewBin(thisArgus)) == NULL) {
                        ArgusLog (LOG_ERR, "RaProcessThisSrvRecord: RaNewBin error %s\n", strerror(errno));
                     } else {
                        RaThisBin->status |= RA_MODIFIED;
                        RaThisBin->timeout = RaTimeout;
                        RaBinMergeArray[index] = RaThisBin;
                     }
                  }
               }

               RaStartPoint = *RaMinTime (&RaStartPoint, &RaThisBin->start);
               RaEndPoint   = *RaMaxTime (&RaEndPoint,   &RaThisBin->end);

               if ((store = RaFindArgusRecord(&RaThisBin->hashtable, thisArgus)) == NULL)
                  if ((store = RaFindRevArgusRecord(&RaThisBin->hashtable, thisArgus)) != NULL) {
                  }

               if (store) {
                  RaThisArgusStore = store;

                  if (!(store->data[RaThisActiveIndex])) {
                     struct ArgusRecordData *data = NULL;

                     if ((data = RaNewArgusData(thisArgus)) != NULL) {
                        data->farhdrstatus = ArgusIndexRecord (data->argus, data->farhdrs);
                        data->status |= RA_MODIFIED;
                        data->status |= srv->status & RA_SVCTEST;

                        if (data->farhdrstatus & ARGUS_AGR_DSR_STATUS) {
                           double sumtime;

                           bcopy((char *)data->farhdrs[ARGUS_AGR_DSR_INDEX], (char *)&data->agr, sizeof(data->agr));
                           data->act.n        = data->agr.act.n;
                           sumtime            = data->agr.act.meanval * data->agr.act.n;
                           data->act.sumtime  = sumtime;
                           data->act.sumsqrd  = (data->agr.act.n * pow(data->agr.act.stdev, 2.0)) + pow(sumtime, 2.0)/data->agr.act.n;

                           data->idle.n       = data->agr.idle.n;
                           sumtime            = data->agr.idle.meanval * data->agr.idle.n;
                           data->idle.sumtime = sumtime;
                           data->idle.sumsqrd = (data->agr.idle.n * pow(data->agr.idle.stdev, 2.0)) + pow(sumtime, 2.0)/data->agr.idle.n;
                        }

                        store->data[RaThisActiveIndex] = data;
                     } else
                        ArgusLog (LOG_ERR, "RaNewArgusData failed %s\n", strerror(errno));

                  } else
                     RaMergeArgusRecord(thisArgus, store, RaThisActiveIndex);

                  RaUpdateArgusStore(thisArgus, store);

               } else {
                  if ((store = RaNewArgusStore(thisArgus)) != NULL) {
                     struct ArgusRecordData *data = NULL;

                     store->ArgusTimeout     = RaTimeout;
                     store->ArgusIdleTimeout = RaIdleTimeout;
                     store->startime         = RaThisBin->start;
                     store->lasttime         = RaThisBin->end;

                     RaThisArgusStore = store;

                     store->status |= RA_MODIFIED;
                     store->status |= srv->status & RA_SVCTEST;

                     if ((data = RaNewArgusData(thisArgus)) != NULL) {
                        data->farhdrstatus = ArgusIndexRecord (data->argus, data->farhdrs);
                        data->status |= RA_MODIFIED;

                        if (data->farhdrstatus & ARGUS_AGR_DSR_STATUS) {
                           double sumtime;

                           bcopy((char *)data->farhdrs[ARGUS_AGR_DSR_INDEX], (char *)&data->agr, sizeof(data->agr));
                           data->act.n        = data->agr.act.n;
                           sumtime            = data->agr.act.meanval * data->agr.act.n;
                           data->act.sumtime  = sumtime;
                           data->act.sumsqrd  = (data->agr.act.n * pow(data->agr.act.stdev, 2.0)) + pow(sumtime, 2.0)/data->agr.act.n;

                           data->idle.n       = data->agr.idle.n;
                           sumtime            = data->agr.idle.meanval * data->agr.idle.n;
                           data->idle.sumtime = sumtime;
                           data->idle.sumsqrd = (data->agr.idle.n * pow(data->agr.idle.stdev, 2.0)) + pow(sumtime, 2.0)/data->agr.idle.n;
                        }

                        store->data[RaThisActiveIndex] = data;

                        if ((store->rahtblhdr = RaAddHashEntry (&RaThisBin->hashtable, store)) != NULL)
                           RaAddToQueue(RaThisBin->queue, &store->qhdr);

                     } else
                        ArgusLog (LOG_ERR, "RaNewArgusData failed %s\n", strerror(errno));
                  } else
                     ArgusLog (LOG_ERR, "RaNewArgusStore failed %s\n", strerror(errno));
               }

            } else 
               ArgusLog (LOG_ERR, "index %d exceeds bin count %d startsec %d\n", index, RaBinCount, RaStartSec);
         }

         ArgusFree(thisArgus);
      }

   } else {
/* 
   streaming mode, no aggregation, just put it in the bin and crank out the
   data as the hold timer allows. so no merging, no hashing, no nada.
*/
      thisArgus = argus;
      RaStartSec = thisArgus->argus_far.time.start.tv_sec - RaStartSeconds;

      if (RaStartSec < 0) {
         RaReInitializeMergeArray(thisArgus->argus_far.time.start.tv_sec);
         RaStartSec = thisArgus->argus_far.time.start.tv_sec - RaStartSeconds;
      }

      if (RaEndSeconds < thisArgus->argus_far.time.last.tv_sec) {
         RaReInitializeMergeArray(thisArgus->argus_far.time.last.tv_sec);
      }

      flow = &thisArgus->argus_far.flow;
      bcopy((char *) flow, (char *) RaArgusFlow, sizeof(struct ArgusFlow));
   
      if (RaModeValue == RAONEBIN) {
         index = 0;
      } else {
         index = (int) (RaStartSec/RaBinSize);
      }

      if (index < RaBinCount) {
         if (index >= 0) {
            if ((RaThisBin = RaBinMergeArray[index]) == NULL) {
               if ((RaThisBin = RaNewBin(thisArgus)) == NULL) {
                  ArgusLog (LOG_ERR, "RaProcessThisSrvRecord: RaNewBin error %s\n", strerror(errno));
               } else {
                  RaThisBin->status |= RA_MODIFIED;
                  RaThisBin->timeout = RaTimeout;
                  RaBinMergeArray[index] = RaThisBin;
               }
            }
         } else {

         }

         RaStartPoint = *RaMinTime (&RaStartPoint, &RaThisBin->start);
         RaEndPoint   = *RaMaxTime (&RaEndPoint,   &RaThisBin->end);

         if ((store = RaNewArgusStore(thisArgus)) != NULL) {
            struct ArgusRecordData *data = NULL;

            store->ArgusTimeout     = RaTimeout;
            store->ArgusIdleTimeout = RaIdleTimeout;
            store->startime         = RaThisBin->start;
            store->lasttime         = RaThisBin->end;

            RaThisArgusStore = store;

            store->status |= RA_MODIFIED;
            store->status |= srv->status & RA_SVCTEST;

            if ((data = RaNewArgusData(thisArgus)) != NULL) {
               data->farhdrstatus = ArgusIndexRecord (data->argus, data->farhdrs);
               data->status |= RA_MODIFIED;

               if (data->farhdrstatus & ARGUS_AGR_DSR_STATUS) {
                  double sumtime;

                  bcopy((char *)data->farhdrs[ARGUS_AGR_DSR_INDEX], (char *)&data->agr, sizeof(data->agr));
                  data->act.n        = data->agr.act.n;
                  sumtime            = data->agr.act.meanval * data->agr.act.n;
                  data->act.sumtime  = sumtime;
                  data->act.sumsqrd  = (data->agr.act.n * pow(data->agr.act.stdev, 2.0)) + pow(sumtime, 2.0)/data->agr.act.n;

                  data->idle.n       = data->agr.idle.n;
                  sumtime            = data->agr.idle.meanval * data->agr.idle.n;
                  data->idle.sumtime = sumtime;
                  data->idle.sumsqrd = (data->agr.idle.n * pow(data->agr.idle.stdev, 2.0)) + pow(sumtime, 2.0)/data->agr.idle.n;
               }

               store->data[RaThisActiveIndex] = data;

               if ((store->rahtblhdr = RaAddHashEntry (&RaThisBin->hashtable, store)) != NULL)
                  RaAddToQueue(RaThisBin->queue, &store->qhdr);

            } else
               ArgusLog (LOG_ERR, "RaNewArgusData failed %s\n", strerror(errno));
         } else
            ArgusLog (LOG_ERR, "RaNewArgusStore failed %s\n", strerror(errno));
      } else 
         ArgusLog (LOG_ERR, "index %d exceeds bin count %d startsec %d\n", index, RaBinCount, RaStartSec);
   }

#ifdef ARGUSDEBUG
   ArgusDebug (7, "RaProcessSrvRecord: done\n");
#endif
}


void
RaProcessManRecord (struct ArgusRecord *argus)
{
   struct ArgusRecordStore *srv = NULL;

   if ((srv = RaNewArgusStore(argus)) != NULL) {
      srv->startime = argus->argus_far.time.start;
      srv->lasttime = argus->argus_far.time.last;

      srv->status |= RA_MODIFIED;
      if ((srv->data[RaThisActiveIndex] = RaNewArgusData(argus)) != NULL) {
         srv->data[RaThisActiveIndex]->status |= RA_MODIFIED;
/*
         RaSendArgusRecord (srv);
*/
      }

      RaDeleteArgusStore(srv);
   }

#ifdef ARGUSDEBUG
   ArgusDebug (2, "RaProcessManRecord (0x%x): done\n", argus);
#endif
}


struct ArgusServiceRecord ArgusThisSrv;

void
RaProcessTCPRecord (struct ArgusRecord *argus)
{
   struct ArgusTCPObject *tcp = NULL;

   ArgusThisSrv.argus = argus;
   ArgusThisSrv.status = RA_SVCPASSED;

   if (Vflag) {
      if (ArgusThisFarStatus & ARGUS_TCP_DSR_STATUS) {
         if ((tcp = (struct ArgusTCPObject *) ArgusThisFarHdrs[ARGUS_TCP_DSR_INDEX]) != NULL) {
            if (tcp->state & (ARGUS_SAW_SYN || ARGUS_SAW_SYN_SENT)) {
               if (tcp->state & ARGUS_RESET) {
                  if (tcp->state & ARGUS_DST_RESET)
                     if (argus->argus_far.src.count && argus->argus_far.dst.count)
                        if (!(argus->argus_far.src.bytes && argus->argus_far.dst.bytes))
                           ArgusThisSrv.status = RA_SVCFAILED;
       
                  if (tcp->state & ARGUS_SRC_RESET)
                     if (argus->argus_far.src.count && !(argus->argus_far.dst.count))
                        ArgusThisSrv.status = RA_SVCFAILED;
               }
            }
    
            if (tcp->state & ARGUS_TIMEOUT)
               if (argus->argus_far.src.count && !(argus->argus_far.dst.count))
                  ArgusThisSrv.status = RA_SVCFAILED;
         }
      }
   }

   RaProcessSrvRecord (&ArgusThisSrv);
}


void
RaProcessICMPRecord (struct ArgusRecord *argus)
{
   ArgusThisSrv.argus = argus;
   ArgusThisSrv.status = RA_SVCPASSED;

   if (Vflag) {
      if ((argus->argus_far.src.count > 0) &&
             (argus->argus_far.dst.count == 0)) {
         ArgusThisSrv.status = RA_SVCFAILED;
 
      } else {
         ArgusThisSrv.status = RA_SVCPASSED;
      }
   }

   RaProcessSrvRecord (&ArgusThisSrv);
}


void
RaProcessUDPRecord (struct ArgusRecord *argus)
{
   int srccount, dstcount;

   ArgusThisSrv.argus = argus;
   ArgusThisSrv.status = RA_SVCPASSED;

   srccount = argus->argus_far.src.count;
   dstcount = argus->argus_far.dst.count;

   if (Vflag)
      if ((srccount == 0) || (dstcount == 0))
         ArgusThisSrv.status = RA_SVCFAILED;

   if (argus->argus_far.flow.ip_flow.tp_p == ARGUS_RTP_FLOWTAG)
      if (!((argus->argus_far.src.count > 3) || (argus->argus_far.dst.count > 3)))
         argus->argus_far.flow.ip_flow.tp_p = 0;

   if (Rflag) {
      struct ArgusAGRStruct *agr;

      if ((agr = (struct ArgusAGRStruct *) ArgusThisFarHdrs[ARGUS_AGR_DSR_INDEX]) != NULL) {
         if (!((srccount == dstcount) && (srccount == agr->count)))
            ArgusThisSrv.status = RA_SVCFAILED;

      } else {
         if (!((srccount == 1) && (dstcount == 1)))
            ArgusThisSrv.status = RA_SVCFAILED;
      }
   }


   RaProcessSrvRecord (&ArgusThisSrv);
}


void
RaProcessIPRecord (struct ArgusRecord *argus)
{
   ArgusThisSrv.argus = argus;
   ArgusThisSrv.status = RA_SVCPASSED;

   if (Vflag) {
      if ((argus->argus_far.src.count > 0) &&
             (argus->argus_far.dst.count == 0)) {
         ArgusThisSrv.status = RA_SVCFAILED;

      } else {
         ArgusThisSrv.status = RA_SVCPASSED;
      }
   }

   if (Rflag)
      if (!((argus->argus_far.src.count == 1) && (argus->argus_far.dst.count == 1)))
         ArgusThisSrv.status = RA_SVCFAILED;

   RaProcessSrvRecord (&ArgusThisSrv);
}


void
RaProcessARPRecord (struct ArgusRecord *argus)
{
   ArgusThisSrv.argus = argus;
   ArgusThisSrv.status = RA_SVCPASSED;

   if (Rflag)
      if (!((argus->argus_far.src.count == 1) && (argus->argus_far.dst.count == 1)))
         ArgusThisSrv.status = RA_SVCFAILED;

   RaProcessSrvRecord (&ArgusThisSrv);
}


void
RaProcessNonIPRecord (struct ArgusRecord *argus)
{
   ArgusThisSrv.argus = argus;
   ArgusThisSrv.status = RA_SVCPASSED;
   RaProcessSrvRecord (&ArgusThisSrv);
}


#include <stdio.h>
#include <errno.h>

void RaAdjustArray(struct RaBinStruct **);

void
RaAdjustArray(struct RaBinStruct **array)
{
   struct ArgusRecord argusbuf, *argus = &argusbuf;
   int i, start;

   if (array[0] == NULL) {
      for (i = 0, start = 0; i < RaBinCount; i++) {
         if (array[i] != NULL) {
            start = i;
            break;
         }
      }

      if (start > 0) {
         for (i = 0; i < RaBinCount; i++) {
            array[i] = array[i + 1];
         }
      }
    
      RaStartTime.tv_sec += RaBinSize;
      RaStartSeconds = RaStartTime.tv_sec;
    
      bzero((char *)argus, sizeof(*argus));
      argus->ahdr.type = ARGUS_FAR;
      argus->ahdr.length = sizeof(*argus);
      argus->argus_far.type   = ARGUS_FAR;
      argus->argus_far.length = sizeof(argus->argus_far);

      argus->argus_far.time.start.tv_sec = RaStartSeconds + (RaBinSize * RaBinCount);
      argus->argus_far.time.last.tv_sec  = RaStartSeconds + (RaBinSize * (RaBinCount + 1));

      if ((array[RaBinCount] = RaNewBin (argus)) == NULL) {
         ArgusLog (LOG_ERR, "RaProcessRecord() RaNewBin %s\n", strerror(errno));
      } else {
         array[RaBinCount]->status |= RA_MODIFIED;
         array[RaBinCount]->timeout = RaHoldTime;   
      }
    
      RaEndTime.tv_sec += RaBinSize;
      RaEndSeconds = RaEndTime.tv_sec;
   }
}

void RaSendBlank(void);

#define RA_MAXQSCAN  25600
#define RA_MAXQSIZE  250000
 
void
RaProcessArray(struct RaBinStruct **array, unsigned char status)
{
   struct RaBinStruct *bin = NULL;
   int cnt = 0, i, retn;

   if (!(RaStreamMode))
      RaSortAlgorithms[0] = RaSortAlgorithmTable[RASORTPKTSCOUNT];
   else
      RaSortAlgorithms[0] = RaSortAlgorithmTable[RASORTSTARTTIME];
 
   if (array != NULL) {
      switch (status) {
         case ARGUS_STOP:
            if (!(Sflag || Cflag || Bflag || Hflag)) {
               if (startime_t > 0) {
                  if (startime_t < RaStartPoint.tv_sec) {
                     RaStartPoint.tv_sec = startime_t;
                  }
                  if (lasttime_t > RaEndPoint.tv_sec) {
                     RaEndPoint.tv_sec = lasttime_t;
                  }
               }
            }

            for (i = 0; i < RaBinCount; i++) {
               if ((bin = array[i]) != NULL) {
                  if (Hflag) {
                  }
                  if ((bin->start.tv_sec >= RaStartPoint.tv_sec) &&
                      (bin->end.tv_sec <= RaEndPoint.tv_sec)) {
                     RaSortQueue(bin->queue);
                     RaProcessQueue(bin->queue, ARGUS_STOP);
                  }

                  RaDeleteBin (bin);
                  array[i] = NULL;

               } else {
                  RaSendBlank();
               }

               RaStartTime.tv_sec += RaBinSize;
               RaStartSeconds = RaStartTime.tv_sec;
               cnt++;
            }
            break;

         default: {
            if (array[0] == NULL)
               RaAdjustArray(array);

            while (((bin = array[0]) !=  NULL) && ((retn = RaBinsCheckTimeout(bin, NULL)) > 0)) {
               RaSortQueue(bin->queue);
               RaProcessQueue(bin->queue, ARGUS_STOP);
               RaDeleteBin (bin);
               array[0] = NULL;
               RaAdjustArray(array);
            }
         }

         break;
      }
   }

#ifdef ARGUSDEBUG
   ArgusDebug (6, "RaProcessArray (0x%x, %d) returning\n", array, status);
#endif
}


extern struct ArgusCanonicalRecord *RaThisCanon;

int
RaSendArgusRecord(struct ArgusRecordStore *store)
{
   unsigned char buf[MAXSTRLEN];
   struct ArgusRecordData *data;
   struct ArgusRecord *argus = NULL;
   struct ArgusFarHeaderStruct *farhdr;
   struct ArgusFlow *flow = NULL;
   struct ArgusAGRStruct *agr = NULL;
   int i, retn = 1;

   for (i = 0; i < RaHistoTimeSeries; i++) {
      if ((data = store->data[i]) != NULL) {

         argus = data->argus;
 
         if ((argus->argus_far.src.count + argus->argus_far.dst.count) != 0) {
            if (aflag == 0) { 
#ifdef _LITTLE_ENDIAN
               ArgusHtoN(argus);
#endif
               ArgusGenerateCanonicalRecord (argus, RaThisCanon);
 
#ifdef _LITTLE_ENDIAN
               ArgusNtoH(argus);
#endif
               if (argus_filter (ArgusFilterCode.bf_insns, RaThisCanon) == 0)
                  return 0;
            }
         }

         if (argus && (data->status & RA_MODIFIED)) {

            if (data->act.n > 0) {
               data->agr.act.n = data->act.n;
               data->agr.act.meanval = data->act.sumtime/data->act.n;
               data->agr.act.stdev = sqrt (data->act.sumsqrd/data->act.n - pow (data->act.sumtime/data->act.n, 2.0));
            }
            if (data->idle.n > 0) {
               data->agr.idle.n = data->idle.n;
               data->agr.idle.meanval = data->idle.sumtime/data->idle.n;
               data->agr.idle.stdev = sqrt (data->idle.sumsqrd/data->idle.n - pow (data->idle.sumtime/data->idle.n, 2.0));
            }

            ArgusThisFarStatus = ArgusIndexRecord(argus, ArgusThisFarHdrs);

            if ((agr = (struct ArgusAGRStruct *) ArgusThisFarHdrs[ARGUS_AGR_DSR_INDEX]) != NULL) {
               bcopy ((char *)&data->agr, (char *)agr, data->agr.length);
            } else {
               bcopy ((char *) argus, buf, argus->ahdr.length);
               argus = (struct ArgusRecord *) buf;
               ArgusThisFarStatus = ArgusIndexRecord(argus, ArgusThisFarHdrs);

               bcopy ((char *)&data->agr, &buf[argus->ahdr.length], data->agr.length);
               argus->ahdr.length += data->agr.length;
               argus->ahdr.status |= ARGUS_MERGED;
               ArgusFree (data->argus);
               data->argus = RaCopyArgusRecord(argus);
               ArgusThisFarStatus = ArgusIndexRecord(data->argus, ArgusThisFarHdrs);
            }

            flow = &argus->argus_far.flow;

            if (!(data->status & ARGUS_FAR_SRCADDR_MODIFIED))
               flow->ip_flow.ip_src = data->flow.ip_flow.ip_src;

            if (!(data->status & ARGUS_FAR_DSTADDR_MODIFIED))
               flow->ip_flow.ip_dst = data->flow.ip_flow.ip_dst;

            if (!(data->status & ARGUS_FAR_PROTO_MODIFIED))
               flow->ip_flow.ip_p = data->flow.ip_flow.ip_p;

            if (!(data->status & ARGUS_FAR_SRCPORT_MODIFIED))
               flow->ip_flow.sport = data->flow.ip_flow.sport;

            if (!(data->status & ARGUS_FAR_DSTPORT_MODIFIED))
               flow->ip_flow.dport = data->flow.ip_flow.dport;

            if (!(data->status & ARGUS_FAR_TPVAL_MODIFIED))
               flow->ip_flow.tp_p = data->flow.ip_flow.tp_p;

            if (!(RaStreamMode)) {
               argus->argus_far.time.start = store->startime;
               argus->argus_far.time.last  = store->lasttime;
            }

            if (!(RaNoZeroMode) || (RaNoZeroMode &&
                  ((argus->argus_far.src.count + argus->argus_far.dst.count) > 0))) {

               if ((ArgusWfileList != NULL) && (!(ArgusListEmpty(ArgusWfileList)))) {
                  struct ArgusWfileStruct *wfile = NULL, *start = NULL;
    
                  if ((wfile = ArgusFrontList(ArgusWfileList)) != NULL) { 
                     start = wfile; 
                     do { 
                        if ((exceptfile == NULL) || strcmp(wfile->filename, exceptfile)) { 
#ifdef _LITTLE_ENDIAN
                           ArgusHtoN(argus); 
#endif
                           ArgusWriteNewLogfile (wfile, argus); 
    
#ifdef _LITTLE_ENDIAN
                           ArgusNtoH(argus); 
#endif
                        }   
                        ArgusPopFrontList(ArgusWfileList); 
                        ArgusPushBackList(ArgusWfileList, wfile); 
                        wfile = ArgusFrontList(ArgusWfileList); 
    
                     } while (wfile != start); 
                  }   

               } else {
         
                  if (argus->ahdr.type & ARGUS_MAR)
                     printf ("%s\n", get_man_string (argus));
                     
                  else {
                     ArgusThisFarStatus = ArgusIndexRecord(argus, ArgusThisFarHdrs);

                     if (Hflag) {
                        RaThisActiveDuration = RaGetuSecAvgDuration(argus);
                        for (i = 0; i < RaHistoTimeSeries; i++) {
                           if (RaThisActiveDuration < RaHistoTimeValues[i]) {
                              RaThisActiveIndex = i;
                              break;
                           }
                        }
                        printf ("%1.*f", pflag, (RaHistoStart + (RaThisActiveIndex + 1) * ((RaHistoEnd - RaHistoStart)/(RaHistoBins * 1.0)))/1000000.0);
                     }

                     switch (argus->ahdr.status & 0xFFFF) {
                        case ETHERTYPE_IP:
                           switch (argus->argus_far.flow.ip_flow.ip_p) {
                              case IPPROTO_TCP:              
                                 printf ("%s", get_tcp_string (argus));
                                 break;
               
                              case IPPROTO_ICMP:              
                                 printf ("%s", get_icmp_string (argus));
                                 break;
               
                              default:
                                 printf ("%s", get_ip_string (argus));
                                 break;
                           }
                           break;
               
                        case ETHERTYPE_ARP:
                        case ETHERTYPE_REVARP:
                           printf ("%s", get_arp_string (argus));
                           break;
               
                        default:
                           printf ("%s", get_nonip_string (argus));
                           break;
                     }

                     printf ("\n");
                  }
                  fflush (stdout);
               }
            }
      
            argus = data->argus;

            if (argus->ahdr.type & ARGUS_FAR) {
               int farlen, length = argus->ahdr.length - sizeof(argus->ahdr);
               farhdr = (struct ArgusFarHeaderStruct *)((char *)argus + sizeof(argus->ahdr));
      
               while (length > 0) {
                  switch (farhdr->type) {
                     case ARGUS_FAR: {
                        struct ArgusFarStruct *far = (struct ArgusFarStruct *) farhdr;
                        far->time.start.tv_sec = 0x7FFFFFFF; far->time.start.tv_usec = 0;
                        far->time.last.tv_sec = 0; far->time.last.tv_usec = 0;
                        far->src.count = 0; far->src.bytes = 0;
                        far->dst.count = 0; far->dst.bytes = 0;
                        break;
                     }
      
                     case ARGUS_TCP_DSR: {
                        struct ArgusTCPObject *tcp = (struct ArgusTCPObject *) farhdr;
      
                        tcp->state = 0;
                        tcp->src.seqbase = 0; tcp->src.ackbytes = 0;
                        tcp->src.rpkts = 0; tcp->src.win = 0; tcp->src.flags = 0;
                        tcp->dst.seqbase = 0; tcp->dst.ackbytes = 0;
                        tcp->dst.rpkts = 0; tcp->dst.win = 0; tcp->dst.flags = 0;
      
                        break;
                     }
      
                     case ARGUS_TIME_DSR: {
                        struct ArgusTimeStruct *time = (struct ArgusTimeStruct *) farhdr;

                        time->src.act.n = 0;
                        time->src.act.meanval = 0;
                        time->src.act.stdev = 0;
                        time->src.act.maxval = 0;
                        time->src.act.minval = 0x7FFFFFFF;
                        time->src.idle.n = 0;
                        time->src.idle.meanval = 0;
                        time->src.idle.stdev = 0;
                        time->src.idle.maxval = 0;
                        time->src.idle.minval = 0x7FFFFFFF;

                        time->dst.act.n = 0;
                        time->dst.act.meanval = 0;
                        time->dst.act.stdev = 0;
                        time->dst.act.maxval = 0;
                        time->dst.act.minval = 0x7FFFFFFF;
                        time->dst.idle.n = 0;
                        time->dst.idle.meanval = 0;
                        time->dst.idle.stdev = 0;
                        time->dst.idle.maxval = 0;
                        time->dst.idle.minval = 0x7FFFFFFF;

                        break;
                     }

                     case ARGUS_VLAN_DSR: {
                        struct ArgusVlanStruct *vlan = (struct ArgusVlanStruct *) farhdr;
                        vlan->status &= ~(ARGUS_SRC_CHANGED | ARGUS_DST_CHANGED);
                        break;
                     }

                     case ARGUS_MPLS_DSR: {
                        struct ArgusMplsStruct *mpls = (struct ArgusMplsStruct *) farhdr;
                        mpls->status &= ~(ARGUS_SRC_CHANGED | ARGUS_DST_CHANGED);
                        break;
                     }

                     case ARGUS_AGR_DSR: {
                        struct ArgusAGRStruct *agr = (struct ArgusAGRStruct *) farhdr;
      
                        agr->count = 0;
                        agr->act.n = 0;
                        agr->act.minval = 0x7FFFFFFF;   agr->act.meanval = 0;
                        agr->act.stdev = 0;  agr->act.maxval = 0;
                        agr->idle.n = 0;
                        agr->idle.minval = 0x7FFFFFFF; agr->idle.meanval = 0;
                        agr->idle.stdev = 0; agr->idle.maxval = 0;
                        break;
                     }
                  }

                  if ((farlen = farhdr->length) == 0)
                     break;
                  if ((farhdr->type == ARGUS_SRCUSRDATA_DSR) ||
                      (farhdr->type == ARGUS_DSTUSRDATA_DSR))
                     farlen = farlen * 4;
                  length -= farlen;
                  farhdr = (struct ArgusFarHeaderStruct *)((char *)farhdr + farlen);
               }
            }

            data->agr.count = 0;
            data->agr.act.n = 0;
            data->agr.act.meanval = 0;
            data->agr.act.stdev = 0;
            data->agr.act.maxval = 0;
            data->agr.act.minval = 0x7FFFFFFF;
            data->agr.idle.n = 0;
            data->agr.idle.meanval = 0;
            data->agr.idle.stdev = 0;
            data->agr.idle.maxval = 0;
            data->agr.idle.minval = 0x7FFFFFFF;
      
            data->act.n = 0;
            data->act.sumtime = 0;
            data->act.sumsqrd = 0;
            data->idle.n = 0;
            data->idle.sumtime = 0;
            data->idle.sumtime = 0;
      
            data->argus->ahdr.status &= ~ARGUS_MERGED;
            data->status &= ~RA_MODIFIED;
         }
      }
   }

   store->status &= ~RA_MODIFIED;
   store->qhdr.logtime = ArgusGlobalTime;

#ifdef ARGUSDEBUG
   ArgusDebug (4, "RaSendArgusRecord(0x%x) done.\n", store);
#endif
   return (retn);
}


#define RA_BASE_GMT_YEAR	70

void
RaReInitializeMergeArray(int secs)
{
   int offset, oldbin = RaBinCount, bins;

   if (secs < RaStartTime.tv_sec) {
      offset = secs - RaStartTime.tv_sec;
      RaStartTime.tv_sec = secs;
      *RaStartTmStruct = *localtime((time_t *) &RaStartTime.tv_sec);

   } else {
      offset = secs - RaEndTime.tv_sec;
      if (RaModeValue == RAONEBIN) {
         bins = 1;
      } else {
         bins = (offset/RaBinSize);
      }
      RaEndTime.tv_sec += (bins < 1000) ? (1000 * RaBinSize) : (bins * RaBinSize);
      *RaEndTmStruct   = *localtime((time_t *) &RaEndTime.tv_sec);
   }

   switch (RaModeValue) {
      case RASECONDS: {
         if (offset < 0) {
            RaStartTmStruct->tm_sec = ((RaStartTmStruct->tm_sec/dflag)*dflag);
         } else {
            RaEndTmStruct->tm_sec = ((RaEndTmStruct->tm_sec/dflag)*dflag);
         }
         break;
      }

      case RAMINUTE: {
         if (offset < 0) {
            RaStartTmStruct->tm_sec = 0;
            RaStartTmStruct->tm_min = ((RaStartTmStruct->tm_min/dflag)*dflag);

         } else {
            if (RaEndTmStruct->tm_sec > 0) {
               RaEndTmStruct->tm_sec = 0;
               if (++RaEndTmStruct->tm_min == 0)
                  if (++RaEndTmStruct->tm_hour == 0)
                     if (++RaEndTmStruct->tm_mday == 0)
                        if (++RaEndTmStruct->tm_mon == 0)
                           RaEndTmStruct->tm_year++;
            }
         }

         break;
      }

      case RAHOURLY: {
         if (offset < 0) {
            RaStartTmStruct->tm_sec  = 0;
            RaStartTmStruct->tm_min  = 0;
            RaStartTmStruct->tm_hour = ((RaStartTmStruct->tm_hour/dflag)*dflag);
         } else {

            if ((RaEndTmStruct->tm_sec > 0) || (RaEndTmStruct->tm_min > 0)) {
               RaEndTmStruct->tm_sec = 0;
               RaEndTmStruct->tm_min = 0;
               if (++RaEndTmStruct->tm_hour == 0)
                  if (++RaEndTmStruct->tm_mday == 0)
                     if (++RaEndTmStruct->tm_mon == 0)
                        RaEndTmStruct->tm_year++;
            }
         }

         break;
      }

      case RADAILY: {
         if (offset < 0) {
            RaStartTmStruct->tm_sec  = 0;
            RaStartTmStruct->tm_min  = 0;
            RaStartTmStruct->tm_hour = 0;
         } else {

            if ((RaEndTmStruct->tm_sec > 0) || (RaEndTmStruct->tm_min > 0) ||
                (RaEndTmStruct->tm_hour > 0)) {
               RaEndTmStruct->tm_sec  = 0;
               RaEndTmStruct->tm_min  = 0;
               RaEndTmStruct->tm_hour = 0;

               if (++RaEndTmStruct->tm_mday == 0)
                  if (++RaEndTmStruct->tm_mon == 0)
                     RaEndTmStruct->tm_year++;
            }
         }

         break;
      }

/*
      case RAWEEKLY: {
         if ((RaEndTmStruct->tm_sec > 0)  || (RaEndTmStruct->tm_min > 0) ||
             (RaEndTmStruct->tm_hour > 0) || (RaEndTmStruct->tm_{
            if (++RaEndTmStruct->tm_mday == 0)
               if (++RaThisTmStruct->tm_mon == 0)
                  RaThisTmStruct->tm_year++;
         }

         break;
      }
*/

      case RAMONTHLY: {
         if (offset < 0) {
            RaStartTmStruct->tm_sec  = 0;
            RaStartTmStruct->tm_min  = 0;
            RaStartTmStruct->tm_hour = 0;
            RaStartTmStruct->tm_mday = 0;

         } else {
            if ((RaEndTmStruct->tm_sec > 0)  || (RaEndTmStruct->tm_min > 0) ||
                (RaEndTmStruct->tm_hour > 0) || (RaEndTmStruct->tm_mday > 0)) {
               RaEndTmStruct->tm_sec  = 0;
               RaEndTmStruct->tm_min  = 0;
               RaEndTmStruct->tm_hour = 0;
               RaEndTmStruct->tm_mday = 0;

               if (++RaEndTmStruct->tm_mon == 0)
                  RaEndTmStruct->tm_year++;
            }
         }


         break;
      }

      case RANNUALY: {
         if (offset < 0) {
            RaStartTmStruct->tm_sec  = 0;
            RaStartTmStruct->tm_min  = 0;
            RaStartTmStruct->tm_hour = 0;
            RaStartTmStruct->tm_mday = 0;
            RaStartTmStruct->tm_mon  = 0;

         } else {
            if ((RaEndTmStruct->tm_sec > 0)  || (RaEndTmStruct->tm_min > 0) ||
                (RaEndTmStruct->tm_hour > 0) || (RaEndTmStruct->tm_mday > 0) ||
                (RaThisTmStruct->tm_mon > 0)) {
               RaEndTmStruct->tm_sec  = 0;
               RaEndTmStruct->tm_min  = 0;
               RaEndTmStruct->tm_hour = 0;
               RaEndTmStruct->tm_mday = 0;
               RaEndTmStruct->tm_mon  = 0;
               RaEndTmStruct->tm_year++;
            }
         }
         break;
      }

      case RAONEBIN: {
         break;
      }
   }

   if (offset < 0) {
      RaStartSeconds = mktime(RaStartTmStruct);
      RaStartTime.tv_sec = RaStartSeconds;

   } else {
      RaEndSeconds   = mktime(RaEndTmStruct);
      RaEndTime.tv_sec = RaEndSeconds;
      RaSeconds = RaEndTime.tv_sec - RaStartTime.tv_sec;
   }

   *RaThisTmStruct  = *localtime((time_t *) &RaSeconds);
 
   if ((Nflag == 0)) {
      RaScaleSeconds = RaEndSeconds - RaStartSeconds;
      if (RaModeValue == RAONEBIN) {
         RaBinCount = 1;
      } else {
         RaBinCount = (RaScaleSeconds/RaBinSize);
      }
 
   } else {
      RaBinCount = Nflag;
      RaEndSeconds = RaStartSeconds + (Nflag * RaBinSize);
      RaScaleSeconds = RaEndSeconds - RaStartSeconds;
   }

   RaBinOldMergeArray = RaBinMergeArray;

   if ((RaBinMergeArray = (struct RaBinStruct **) ArgusCalloc (sizeof(void *), (RaBinCount + 1))) == NULL) {
      ArgusLog (LOG_ERR, "RaInitializeMergeArray() ArgusCalloc %s\n", strerror(errno));
 
   } else {
      int i, cnt = (offset < 0) ? oldbin - RaBinCount : RaBinCount - oldbin;
 
      for (i = 0; i < oldbin; i++) {
         struct ArgusRecord argusbuf, *argus = &argusbuf;
         if (((cnt < 0) && (i < -cnt)) || ((cnt > 0) && (i > (RaBinCount - cnt)))) {
            bzero((char *)argus, sizeof(*argus));
            argus->ahdr.type = ARGUS_FAR;
            argus->ahdr.length = sizeof(*argus);
            argus->argus_far.type   = ARGUS_FAR;
            argus->argus_far.length = sizeof(argus->argus_far);
 
            argus->argus_far.time.start.tv_sec = RaStartSeconds + (RaBinSize * i);
            argus->argus_far.time.last.tv_sec  = RaStartSeconds + (RaBinSize * (i + 1));

            if ((RaBinMergeArray[i] = RaNewBin (argus)) == NULL) {
               ArgusLog (LOG_ERR, "RaInitializeMergeArray() RaNewArgusStore %s\n", strerror(errno));
 
            } else {
               RaBinMergeArray[i]->status |= RA_MODIFIED;
               RaBinMergeArray[i]->timeout = RaHoldTime;
            }

         } else {
            RaBinMergeArray[i] = RaBinOldMergeArray[i + ((cnt < 0) ? cnt : 0)];
         }
      }
   }

   ArgusFree(RaBinOldMergeArray);
}


void
RaInitializeMergeArray()
{
   double RaBinSeconds;
   short intervalTime;
   int reportTime;

   RaStartTime.tv_usec = 0;

   if (RaEndTime.tv_usec > 0) {
      RaEndTime.tv_sec++;
      RaEndTime.tv_usec = 0;
   }

   if (startime_t > 0) {
      if (RaStartTime.tv_sec > startime_t)
         RaStartTime.tv_sec = startime_t;
      if (RaEndTime.tv_sec < lasttime_t)
         RaEndTime.tv_sec = lasttime_t;
   }

   *RaStartTmStruct = *localtime((time_t *) &RaStartTime.tv_sec);
   *RaEndTmStruct   = *localtime((time_t *) &RaEndTime.tv_sec);

   switch (RaModeValue) {
      case RASECONDS: {
         RaStartTmStruct->tm_sec = ((RaStartTmStruct->tm_sec/dflag)*dflag);
         RaBinSize = 1.0 * dflag;
         break;
      }

      case RAMINUTE: {
         RaStartTmStruct->tm_sec = 0;
         RaStartTmStruct->tm_min = ((RaStartTmStruct->tm_min/dflag)*dflag);

         if (RaEndTmStruct->tm_sec > 0) {
            RaEndTmStruct->tm_sec = 0;
            if (++RaEndTmStruct->tm_min == 0)
               if (++RaEndTmStruct->tm_hour == 0)
                  if (++RaEndTmStruct->tm_mday == 0)
                     if (++RaEndTmStruct->tm_mon == 0)
                        RaEndTmStruct->tm_year++;
         }

         RaBinSize = 60.0 * dflag;
         break;
      }

      case RAHOURLY: {
         RaStartTmStruct->tm_sec  = 0;
         RaStartTmStruct->tm_min  = 0;
         RaStartTmStruct->tm_hour = ((RaStartTmStruct->tm_hour/dflag)*dflag);

         if ((RaEndTmStruct->tm_sec > 0) || (RaEndTmStruct->tm_min > 0)) {
            RaEndTmStruct->tm_sec = 0;
            RaEndTmStruct->tm_min = 0;
            if (++RaEndTmStruct->tm_hour == 0)
               if (++RaEndTmStruct->tm_mday == 0)
                  if (++RaEndTmStruct->tm_mon == 0)
                     RaEndTmStruct->tm_year++;
         }

         RaBinSize = 3600.0 * dflag;
         break;
      }

      case RADAILY: {
         RaStartTmStruct->tm_sec  = 0;
         RaStartTmStruct->tm_min  = 0;
         RaStartTmStruct->tm_hour = 0;

         if ((RaEndTmStruct->tm_sec > 0) || (RaEndTmStruct->tm_min > 0) ||
             (RaEndTmStruct->tm_hour > 0)) {
            RaEndTmStruct->tm_sec  = 0;
            RaEndTmStruct->tm_min  = 0;
            RaEndTmStruct->tm_hour = 0;

            if (++RaEndTmStruct->tm_mday == 0)
               if (++RaEndTmStruct->tm_mon == 0)
                  RaEndTmStruct->tm_year++;
         }

         RaBinSize = 3600.0 * 24.0 * dflag;
         break;
      }

/*
      case RAWEEKLY: {
         if ((RaEndTmStruct->tm_sec > 0)  || (RaEndTmStruct->tm_min > 0) ||
             (RaEndTmStruct->tm_hour > 0) || (RaEndTmStruct->tm_{
            if (++RaEndTmStruct->tm_mday == 0)
               if (++RaThisTmStruct->tm_mon == 0)
                  RaThisTmStruct->tm_year++;
         }

         RaBinSize = 3600.0 * 24.0;
         break;
      }
*/

      case RAMONTHLY: {
         RaStartTmStruct->tm_sec  = 0;
         RaStartTmStruct->tm_min  = 0;
         RaStartTmStruct->tm_hour = 0;
         RaStartTmStruct->tm_mday = 0;

         if ((RaEndTmStruct->tm_sec > 0)  || (RaEndTmStruct->tm_min > 0) ||
             (RaEndTmStruct->tm_hour > 0) || (RaEndTmStruct->tm_mday > 0)) {
            RaEndTmStruct->tm_sec  = 0;
            RaEndTmStruct->tm_min  = 0;
            RaEndTmStruct->tm_hour = 0;
            RaEndTmStruct->tm_mday = 0;

            if (++RaEndTmStruct->tm_mon == 0)
               RaEndTmStruct->tm_year++;
         }

         RaBinSize = 3600.0 * 24.0 * 30 * dflag;

         break;
      }

      case RANNUALY: {
         RaStartTmStruct->tm_sec  = 0;
         RaStartTmStruct->tm_min  = 0;
         RaStartTmStruct->tm_hour = 0;
         RaStartTmStruct->tm_mday = 0;
         RaStartTmStruct->tm_mon  = 0;

         if ((RaEndTmStruct->tm_sec > 0)  || (RaEndTmStruct->tm_min > 0) ||
             (RaEndTmStruct->tm_hour > 0) || (RaEndTmStruct->tm_mday > 0) ||
             (RaThisTmStruct->tm_mon > 0)) {
            RaEndTmStruct->tm_sec  = 0;
            RaEndTmStruct->tm_min  = 0;
            RaEndTmStruct->tm_hour = 0;
            RaEndTmStruct->tm_mday = 0;
            RaEndTmStruct->tm_mon  = 0;
            RaEndTmStruct->tm_year++;
         }

         RaBinSize = 3600.0 * 24.0 * 30 * 12 * dflag;
         break;
      }

      case RAONEBIN: {
         RaBinSize = 0.0;
         break;
      }
   }

   if ((intervalTime = ntohs(ArgusInput->ArgusInitCon.argus_mar.reportInterval)) > 0)
      reportTime = intervalTime * 3;
   else
      reportTime = RaBinSize * 3;
 
   if (Bflag)
      RaHoldTime = Bflag;
   else
      RaHoldTime = 0;

   if (RaStartTime.tv_sec) {
      RaStartSeconds = mktime(RaStartTmStruct);
      RaStartTime.tv_sec = RaStartSeconds;
   }

   if (RaEndTime.tv_sec == 0) {
      RaEndTime.tv_usec = 0;
      
      if (RaModeValue == RAONEBIN) {
         RaBinCount = 1;
         RaEndTime.tv_sec = RaStartTime.tv_sec;

      } else {
         RaBinCount = 1000 + (RaBinSize + RaHoldTime)/RaBinSize;
         RaEndTime.tv_sec = RaStartTime.tv_sec + RaBinCount * RaBinSize;
      }

      *RaEndTmStruct   = *localtime((time_t *) &RaEndTime.tv_sec);
   }

   RaEndSeconds   = mktime(RaEndTmStruct);
   RaEndTime.tv_sec = RaEndSeconds;
   RaSeconds = RaEndTime.tv_sec - RaStartTime.tv_sec;
   *RaThisTmStruct  = *localtime((time_t *) &RaSeconds);

   if ((Nflag == 0)) {
      RaScaleSeconds = RaEndSeconds - RaStartSeconds;
      if (RaModeValue == RAONEBIN) {
         RaBinCount = 1;
      } else {
         RaBinCount = (RaScaleSeconds + (RaBinSize - 1))/RaBinSize;
      }

   } else {
      RaBinCount = Nflag;
      RaEndSeconds = RaStartSeconds + (Nflag * RaBinSize);
      RaScaleSeconds = RaEndSeconds - RaStartSeconds;
   }

   if ((RaBinMergeArray = (struct RaBinStruct **) ArgusCalloc (sizeof(void *), (RaBinCount + 1))) == NULL) {
      ArgusLog (LOG_ERR, "RaInitializeMergeArray() ArgusCalloc %s\n", strerror(errno));

   } else {
      int i;

      for (i = 0; i < RaBinCount; i++) {
         struct ArgusRecord argusbuf, *argus = &argusbuf;

         bzero((char *)argus, sizeof(*argus));
         argus->ahdr.type = ARGUS_FAR;
         argus->ahdr.length = sizeof(*argus);
         argus->argus_far.type   = ARGUS_FAR;
         argus->argus_far.length = sizeof(argus->argus_far);

         argus->argus_far.time.start.tv_sec = RaStartSeconds + (RaBinSize * i);
         argus->argus_far.time.last.tv_sec  = RaStartSeconds + (RaBinSize * (i + 1));

         if ((RaBinMergeArray[i] = RaNewBin (argus)) == NULL) {
            ArgusLog (LOG_ERR, "RaInitializeMergeArray() RaNewArgusStore %s\n", strerror(errno));

         } else {
            RaBinMergeArray[i]->status |= RA_MODIFIED;
            RaBinMergeArray[i]->timeout = RaHoldTime;
         }
      }
   }

   modf(RaBinSize, &RaBinSeconds);
   if (RaBinSize != RaBinSeconds)
      RaBinSize = RaBinSeconds + 1.0;

#ifdef ARGUSDEBUG
   ArgusDebug (3, "RaInitializeMergeArray(0x%x) RaStartTime %d RaBinSize %f.\n", RaStartTime.tv_sec, RaBinSize);
#endif
}


int RaQueueCount = 0;
int RaAdjustBinCount = 0;
int RaAdjustLastBinCount = 0;

float RaSrcPeriod = 0.0;
float RaDstPeriod = 0.0;


struct ArgusRecord *
RaAdjustArgusRecord (struct ArgusRecord *argus)
{
   struct ArgusRecord *retn = NULL;

   if (RaNumBlanks > 0) {
      struct timeval tvpbuf, *tvp = &tvpbuf;

      retn = RaCopyArgusRecord (argus);
      ArgusZeroRecord (retn);
      *tvp = argus->argus_far.time.start;
      retn->argus_far.time.start = *tvp;
      tvp->tv_usec = 0;
      tvp->tv_sec += RaBinSize;

      retn->argus_far.time.last = *tvp;
      argus->argus_far.time.start = *tvp;

      RaNumBlanks--;
      return (retn);
   }

   if ((argus->argus_far.src.count != 0) || (argus->argus_far.dst.count != 0)) {
      struct timeval tvpbuf, *tvp = &tvpbuf;
      struct ArgusTCPObject *tcp = NULL, *rtcp = NULL;
      float startsec = 0.0, duration = 0.0, thisduration = 0.0, actduration = 0.0;
      int timeindex = 0, count = 0, bytes = 0, srcABpp, srcBpp, dstABpp, dstBpp, thisCount;
      struct ArgusAGRStruct *agr;
      double RaBinSeconds;

      ArgusThisFarStatus = ArgusIndexRecord(argus, ArgusThisFarHdrs);
      if (ArgusThisFarStatus & ARGUS_TCP_DSR_STATUS)
         tcp = (struct ArgusTCPObject *) ArgusThisFarHdrs[ARGUS_TCP_DSR_INDEX];

      startsec = (float)(argus->argus_far.time.start.tv_sec - RaStartSeconds) + 
                        (argus->argus_far.time.start.tv_usec/1000000.0);

      duration = (float)((argus->argus_far.time.last.tv_sec - argus->argus_far.time.start.tv_sec) + 
                         (argus->argus_far.time.last.tv_usec - argus->argus_far.time.start.tv_usec)/1000000.0);

      if (startsec < 0.0) {
         if (argus->argus_far.time.last.tv_sec < RaStartSeconds)
            return (retn);

         thisduration = (float)((argus->argus_far.time.last.tv_sec - RaStartSeconds) +
                                (argus->argus_far.time.last.tv_usec)/1000000.0);

         if (argus->argus_far.src.count) {

            srcABpp = (argus->argus_far.src.appbytes * thisduration)/duration;
            srcBpp  = (argus->argus_far.src.bytes    * thisduration)/duration;
            thisCount = (argus->argus_far.src.count  * thisduration)/duration;

            argus->argus_far.src.count    = thisCount;
            argus->argus_far.src.bytes    = srcBpp;
            argus->argus_far.src.appbytes = srcABpp;

            if (tcp != NULL) {
               tcp->src.ackbytes = (tcp->src.ackbytes * thisduration)/duration;
               tcp->src.bytes    = (tcp->src.bytes * thisduration)/duration;
               tcp->src.rpkts    = (tcp->src.rpkts * thisduration)/duration;
            }
         }

         if (argus->argus_far.dst.count) {
            dstABpp = (argus->argus_far.dst.appbytes * thisduration)/duration;
            dstBpp  = (argus->argus_far.dst.bytes    * thisduration)/duration;

            thisCount = (argus->argus_far.dst.count  * thisduration)/duration;

            argus->argus_far.dst.count    = thisCount;
            argus->argus_far.dst.bytes    = dstBpp;
            argus->argus_far.dst.appbytes = dstABpp;

            if (tcp != NULL) {
               tcp->dst.ackbytes = (tcp->dst.ackbytes * thisduration)/duration;
               tcp->dst.bytes    = (tcp->dst.bytes * thisduration)/duration;
               tcp->dst.rpkts    = (tcp->dst.rpkts * thisduration)/duration;
            }
         }

         duration -= thisduration;
         startsec = 0.0;
         argus->argus_far.time.start.tv_sec = RaStartSeconds;
         argus->argus_far.time.start.tv_usec = 0;
      }

      if ((agr = (struct ArgusAGRStruct *) ArgusThisFarHdrs[ARGUS_AGR_DSR_INDEX]) != NULL) {
         int ArgusThisMultiplier = 1000;
         if (agr->status & ARGUS_AGR_USECACTTIME)
            ArgusThisMultiplier = 1000000;
         if (agr->act.n)
            actduration = agr->act.meanval / ArgusThisMultiplier;

         retn = RaCopyArgusRecord (argus);

      } else {
         struct ArgusAGRStruct agrbuf;
         char buf[0x10000];
         agr = &agrbuf;

         bzero ((char *) agr, sizeof(*agr));
         bzero ((char *) buf, (argus->ahdr.length + sizeof(*agr) + 1));
 
         agr->laststartime = argus->argus_far.time.start;
         agr->lasttime     = argus->argus_far.time.last;

         agr->type     = ARGUS_AGR_DSR;
         agr->length   = sizeof(*agr);
         agr->act.maxval  = RaThisActiveDuration;
         agr->act.minval  = RaThisActiveDuration;
         agr->act.meanval = RaThisActiveDuration;
         agr->act.n    = 1;
         agr->count    = 1;

         agr->idle.minval = 0x7FFFFFFF;
         agr->status  |= ARGUS_AGR_USECACTTIME;
         agr->status  |= ARGUS_AGR_USECIDLETIME;

         bcopy ((char *) argus, buf, argus->ahdr.length);
         retn = (struct ArgusRecord *) buf;
         bcopy ((char *)agr, &buf[retn->ahdr.length], agr->length);
         retn->ahdr.length += agr->length;
         retn = RaCopyArgusRecord (retn);
      }

      ArgusThisFarStatus = ArgusIndexRecord(retn, ArgusThisFarHdrs);

      if ((RaModeValue == RAONEBIN) || Hflag) {
         ArgusZeroRecord (argus);

      } else {
         timeindex = (int) (startsec/RaBinSize);

         if ((duration > 0.0) && ((startsec + duration) > (RaBinSize * (timeindex + 1)))) { /* we span out of this bin */

            if (ArgusThisFarStatus & ARGUS_TCP_DSR_STATUS)
               rtcp = (struct ArgusTCPObject *) ArgusThisFarHdrs[ARGUS_TCP_DSR_INDEX];

            if ((count = (retn->argus_far.src.count + retn->argus_far.dst.count)) == 2) {
               if (actduration == 0.0) {
                  startsec = (RaBinSize * (timeindex + 1));               /* where is the start of the next bin */
                  modf(startsec, &RaBinSeconds);                          /* where is the start of the next bin */

                  tvp->tv_sec  = RaBinSeconds + RaStartSeconds;
                  tvp->tv_usec = (int)((startsec - RaBinSeconds)*1000000);
                  if (tvp->tv_usec >= 1000000) {
                     tvp->tv_sec++;
                     tvp->tv_usec -= 1000000;
                  }
               } else {
                  modf(actduration, &RaBinSeconds);                          /* where is the start of the next bin */

                  tvp->tv_sec  = RaBinSeconds + retn->argus_far.time.start.tv_sec;
                  tvp->tv_usec = (int)((startsec - RaBinSeconds)*1000000) + retn->argus_far.time.start.tv_usec;
                  if (tvp->tv_usec >= 1000000) {
                     tvp->tv_sec++;
                     tvp->tv_usec -= 1000000;
                  }
               }

               if (RaSrcPeriod == RaDstPeriod) {
                  retn->argus_far.time.last = *tvp;                       /* this is the new end for this record */
                  argus->argus_far.time.start = *tvp;
               } else {
                  retn->argus_far.time.last = retn->argus_far.time.start;
                  argus->argus_far.time.start = argus->argus_far.time.last;

                  if (retn->argus_far.src.count) {
                     if (retn->argus_far.src.count == 1) {
                        retn->argus_far.dst.count = 0;
                        retn->argus_far.dst.bytes = 0;
                     } else {
                        retn->argus_far.src.count = 1;
                        bytes = retn->argus_far.src.bytes;
                        retn->argus_far.src.bytes /= 2;
                        if (bytes & 0x01)
                           retn->argus_far.src.bytes += 1;
                     }
                  } else {
                     retn->argus_far.dst.count = 1;
                     bytes = retn->argus_far.dst.bytes;
                     retn->argus_far.dst.bytes /= 2;
                     if (bytes & 0x01)
                        retn->argus_far.dst.bytes += 1;
                  }
               }

               argus->argus_far.src.count -= retn->argus_far.src.count;
               argus->argus_far.dst.count -= retn->argus_far.dst.count;
               argus->argus_far.src.bytes -= retn->argus_far.src.bytes;
               argus->argus_far.dst.bytes -= retn->argus_far.dst.bytes;

            } else {

               if (RaAdjustCount == 0) {
                  RaAdjustBinCount = 0;
                  RaAdjustLastBinCount = 0;
               }

               startsec = (RaBinSize * (timeindex + 1));         /* where is the start of the next bin */
               modf(startsec, &RaBinSeconds);                    /* where is the start of the next bin */
               tvp->tv_sec  = RaBinSeconds + RaStartSeconds;

               tvp->tv_usec = (int)((startsec - RaBinSeconds)*1000000);
               if (tvp->tv_usec >= 1000000) {
                  tvp->tv_sec++;
                  tvp->tv_usec -= 1000000;
               }

               retn->argus_far.time.last = *tvp;                       /* this is the new end for this record */

               if (((argus->argus_far.time.last.tv_sec - argus->argus_far.time.start.tv_sec) == 1) &&
                   ((argus->argus_far.time.last.tv_usec - argus->argus_far.time.start.tv_usec) == 0)) {
                  thisduration = 1.0;

               } else {
                  thisduration = (float)(((retn->argus_far.time.last.tv_sec  - retn->argus_far.time.start.tv_sec) * 1.0) +
                                          (retn->argus_far.time.last.tv_usec - retn->argus_far.time.start.tv_usec)/1000000.0);
               }
               
               if ((count = (retn->argus_far.src.count > retn->argus_far.dst.count) ?
                            retn->argus_far.src.count : retn->argus_far.dst.count) > 0) {

                  if ((duration / count) > RaBinSize) {               /* we've got fewer pkts than bins */
                     startsec = (duration/count);                     /* where is the start of the next bin (round up)*/
                     modf(startsec, &RaBinSeconds);                   /* where is the start of the next bin */

                     RaNumBlanks = (duration/count)/RaBinSize;

                  } else {
                     RaBinSeconds = RaBinSize;
                  }
               }

               argus->argus_far.time.start = *tvp;
               
               if (argus->argus_far.src.count) {
                  thisCount = (argus->argus_far.src.count  * (thisduration /duration));

                  if (thisCount > argus->argus_far.src.count)
                     thisCount = argus->argus_far.src.count;

                  if ((thisCount == 0) && (argus->argus_far.src.count > 1)) {
                     thisCount = 1;
                  }

                  retn->argus_far.src.count = thisCount;

                  if (retn->argus_far.src.count == argus->argus_far.src.count) {
                     srcBpp  = argus->argus_far.src.bytes;
                     srcABpp = argus->argus_far.src.appbytes;
                  } else {
                     if (retn->argus_far.src.count > 0) {
                        srcBpp  = (argus->argus_far.src.bytes * (thisduration / duration));
                        srcABpp = (argus->argus_far.src.appbytes * (thisduration / duration));
                     } else {
                        srcBpp  = 0;
                        srcABpp = 0;
                     }
                  }

                  retn->argus_far.src.bytes    = srcBpp;
                  retn->argus_far.src.appbytes = srcABpp;

                  if ((tcp != NULL) && (rtcp != NULL)) {
                     rtcp->src.ackbytes = (rtcp->src.ackbytes * (thisduration / duration));
                     rtcp->src.bytes    = (rtcp->src.bytes * (thisduration / duration));
                     rtcp->src.rpkts    = (rtcp->src.rpkts * (thisduration / duration));
                  }
               }

               if (argus->argus_far.dst.count) {
                  thisCount = (argus->argus_far.dst.count * (thisduration /duration));

                  if (thisCount > argus->argus_far.dst.count)
                     thisCount = argus->argus_far.dst.count;

                  if ((thisCount == 0) && (argus->argus_far.dst.count > 1))
                     thisCount = 1;

                  retn->argus_far.dst.count = thisCount;

                  if (retn->argus_far.dst.count == argus->argus_far.dst.count) {
                     dstABpp = argus->argus_far.dst.appbytes;
                     dstBpp  = argus->argus_far.dst.bytes;
                  } else {
                     if (retn->argus_far.dst.count > 0) {
                        dstABpp = (argus->argus_far.dst.appbytes * (thisduration /duration));
                        dstBpp  = (argus->argus_far.dst.bytes * (thisduration /duration));
                     } else {
                        dstABpp = 0;
                        dstBpp  = 0;
                     }
                  }
                  retn->argus_far.dst.bytes    = dstBpp;
                  retn->argus_far.dst.appbytes = dstABpp;

                  if ((tcp != NULL) && (rtcp != NULL)) {
                     rtcp->dst.ackbytes = (rtcp->dst.ackbytes * (thisduration /duration));
                     rtcp->dst.bytes    = (rtcp->dst.bytes * (thisduration /duration));
                     rtcp->dst.rpkts    = (rtcp->dst.rpkts * (thisduration /duration));
                  }
               }

               argus->argus_far.src.count -= retn->argus_far.src.count;
               argus->argus_far.dst.count -= retn->argus_far.dst.count;
               argus->argus_far.src.bytes -= retn->argus_far.src.bytes;
               argus->argus_far.dst.bytes -= retn->argus_far.dst.bytes;
               argus->argus_far.src.appbytes -= retn->argus_far.src.appbytes;
               argus->argus_far.dst.appbytes -= retn->argus_far.dst.appbytes;

               if ((tcp != NULL) && (rtcp != NULL)) {
                  tcp->src.ackbytes -= rtcp->src.ackbytes;
                  tcp->src.bytes    -= rtcp->src.bytes;
                  tcp->src.rpkts    -= rtcp->src.rpkts;
                  tcp->dst.ackbytes -= rtcp->dst.ackbytes;
                  tcp->dst.bytes    -= rtcp->dst.bytes;
                  tcp->dst.rpkts    -= rtcp->dst.rpkts;
               }                             
            }

         } else {
            startsec = (RaBinSize * timeindex);                     /* where is the start of the current bin */
            modf(startsec, &RaBinSeconds);                          /* where is the start of the current bin */
            tvp->tv_sec  = RaBinSeconds + RaStartSeconds;
            tvp->tv_usec = 0;
            tvp->tv_usec = (int)((startsec - RaBinSeconds)*1000000);
            if (tvp->tv_usec >= 1000000) {
               tvp->tv_sec++;
               tvp->tv_usec -= 1000000;
            }

            retn->argus_far.time.start = *tvp;                     /* this is the new start for this record */

            startsec = (RaBinSize * (timeindex + 1));               /* where is the start of the next bin */
            modf(startsec, &RaBinSeconds);                          /* where is the start of the next bin */
            tvp->tv_sec  = RaBinSeconds + RaStartSeconds;
            tvp->tv_usec = 0;
            tvp->tv_usec = (int)((startsec - RaBinSeconds)*1000000);
            if (tvp->tv_usec >= 1000000) {
               tvp->tv_sec++;
               tvp->tv_usec -= 1000000;
            }
            retn->argus_far.time.last = *tvp;                       /* this is the new end for this record */

            argus->argus_far.src.count = 0;
            argus->argus_far.dst.count = 0;
         }
      }
   }

   return(retn);
}

void
RaModifyEtherFlow (struct ArgusRecord *argus)
{
   struct ArgusFlow *flow = &argus->argus_far.flow;
   bzero ((char *) &flow->mac_flow.ehdr.ether_dhost, 6);
}


int
RaCreateEtherFlow (struct ArgusRecord *argus)
{
   struct ArgusFlow *flow = &argus->argus_far.flow;
   struct ArgusMacStruct *mac = NULL;
   struct ArgusETHERObject *ether = NULL;
   int retn = 0;
   
   if (ArgusThisFarStatus & ARGUS_MAC_DSR_STATUS) {
      if ((mac = (struct ArgusMacStruct *) ArgusThisFarHdrs[ARGUS_MAC_DSR_INDEX]) != NULL) {
         ether = &mac->ether_mac;
         bzero ((char *) flow, sizeof(*flow));
         bcopy ((char *) ether, (char *) &flow->mac_flow.ehdr, 12);
         argus->ahdr.status &= ~0xFFFF;
         retn++;
      }

   }

   return (retn);     
}

struct ArgusRecord *
RaRmonArgusRecord (struct ArgusRecord *argus)
{
   struct ArgusRecord *retn = NULL;
   struct ArgusMeter meter;
   struct ArgusFlow *flow1, *flow2;
   struct ArgusFarHeaderStruct *a1farhdr[32], *a2farhdr[32];
   unsigned int a1DSRStatus = 0, a2DSRStatus = 0;


   if (RaPrintMACAddress)
      if (!(RaCreateEtherFlow (argus)))
         return (retn);

   if ((retn = RaCopyArgusRecord (argus)) != NULL) {

      a1DSRStatus = ArgusIndexRecord (argus, a1farhdr);
      a2DSRStatus = ArgusIndexRecord (retn,  a2farhdr);
   
      flow1 = &argus->argus_far.flow;
      flow2 = &retn->argus_far.flow;

      switch (argus->ahdr.status & 0xFFFF) {
         case ETHERTYPE_IP:
            flow2->ip_flow.ip_src = flow1->ip_flow.ip_dst;
            flow2->ip_flow.ip_dst = flow1->ip_flow.ip_src;

            switch (flow1->ip_flow.ip_p) {
               case IPPROTO_UDP:
               case IPPROTO_TCP:
                  break;
            }

            meter = argus->argus_far.src;
            argus->argus_far.src = argus->argus_far.dst;
            argus->argus_far.dst = meter;

            break;

         case ETHERTYPE_ARP:
         case ETHERTYPE_REVARP:
            break;

         default:
            bcopy ((char *)&flow1->mac_flow.ehdr.ether_dhost,
                   (char *)&flow2->mac_flow.ehdr.ether_shost, 6);
            bzero ((char *)&flow1->mac_flow.ehdr.ether_dhost, 6);
            bzero ((char *)&flow2->mac_flow.ehdr.ether_dhost, 6);

            meter = retn->argus_far.src;
            retn->argus_far.src = retn->argus_far.dst;
            retn->argus_far.dst = meter;
            break;
      }

      if ((a1DSRStatus & ARGUS_MAC_DSR_STATUS) && (a2DSRStatus & ARGUS_MAC_DSR_STATUS)) {
         struct ArgusMacStruct *mac1 = NULL, *mac2 = NULL;

         mac1 = (struct ArgusMacStruct *) a1farhdr[ARGUS_MAC_DSR_INDEX];
         mac2 = (struct ArgusMacStruct *) a2farhdr[ARGUS_MAC_DSR_INDEX];

         bcopy ((char *)&mac2->ether_mac.etherdst, (char *)&mac1->ether_mac.ethersrc, 6);
         
         bzero ((char *)&mac1->ether_mac.etherdst, 6);
         bzero ((char *)&mac2->ether_mac.etherdst, 6);
      }

       retn->ahdr.type |= ARGUS_RMON;
      argus->ahdr.type |= ARGUS_RMON;
   }

   return (retn);
}

struct RaBinStruct *
RaNewBin(struct ArgusRecord *argus)
{
   struct RaBinStruct *retn = NULL;

   if ((retn = (struct RaBinStruct *) ArgusCalloc (1, sizeof(*retn))) != NULL) {
      if ((retn->queue = RaNewQueue()) == NULL)
         ArgusLog (LOG_ERR, "RaMergeQueue: RaNewQueue error %s\n", strerror(errno));

      if ((retn->hashtable.array = (struct RaHashTableHeader **) ArgusCalloc (RACRICKET_HASHTABLESIZE,
                                    sizeof (struct RaHashTableHeader *))) == NULL) {
         ArgusLog (LOG_ERR, "RaMergeQueue: ArgusCalloc error %s\n", strerror(errno));
      } else {
         retn->hashtable.size = RACRICKET_HASHTABLESIZE;
      }

      if (argus != NULL) {
         retn->start = argus->argus_far.time.start;
         retn->end   = argus->argus_far.time.last;

      }

   } else
      ArgusLog (LOG_ERR, "RaNewBin: ArgusCalloc error %s\n", strerror(errno));

   return (retn);
}

void
RaDeleteBin (struct RaBinStruct *bin)
{
   struct ArgusRecordStore *obj = NULL;

   if (bin->queue) {
      while ((obj = (struct ArgusRecordStore *) RaPopQueue(bin->queue))) {
         if (obj->rahtblhdr)
            ArgusFree(obj->rahtblhdr);

         ArgusFree(obj);
      }
      
      ArgusFree(bin->queue);
   }

   if (bin->hashtable.array != NULL)
      ArgusFree(bin->hashtable.array);

   ArgusFree(bin);
   return;
}


int RaPrintHeader = 0;
 
void
RaProcessQueue(struct RaQueueStruct *queue, unsigned char status)
{
   struct ArgusRecordStore *obj = NULL;
   struct ArgusRecord argusbuf, *argus = &argusbuf;
   struct ArgusRecordStore *srv = NULL;
   char stimebuf[128], dtimebuf[128];

   strcpy (stimebuf, print_time(&RaStartPoint));
   strcpy (dtimebuf, print_time(&RaEndPoint));

   RaScaleSeconds = RaEndPoint.tv_sec - RaStartPoint.tv_sec;

   if (RaLabel == NULL)
      RaLabel = RaGenerateLabel(argus);

   if (ArgusWfileList == NULL) {
      if (RaPrintHeader++ == 0 ) {
         if (Gflag) {
            if (Hflag) {
               fprintf (stdout, "StartTime=%1.*f\n", pflag, (RaHistoStart/1000000.0));
               fprintf (stdout, "StopTime=%1.*f\n", pflag, (RaHistoEnd/1000000.0));
               fprintf (stdout, "Seconds=%d\n", RaScaleSeconds);
            } else {
               fprintf (stdout, "StartTime=%s\n", stimebuf);
               fprintf (stdout, "StopTime=%s\n", dtimebuf);
               fprintf (stdout, "Seconds=%d\n", RaScaleSeconds);
            }

            if (RaModeValue == RAONEBIN) {
               if (Hflag) {
                  fprintf (stdout, "BinSize=%1.*f\n", pflag, (RaHistoEnd - RaHistoStart)/(RaHistoBins * 1000000.0));
                  fprintf (stdout, "Bins=%d\n", RaHistoBins);
               } else {
                  fprintf (stdout, "BinSize=%d.00\n", (int)((RaEndPoint.tv_sec - RaStartPoint.tv_sec)));
                  fprintf (stdout, "Bins=1\n");
               }
            } else {
               fprintf (stdout, "BinSize=%f\n", RaBinSize);
               fprintf (stdout, "Bins=%d\n", (int)((RaEndPoint.tv_sec - RaStartPoint.tv_sec)/RaBinSize));
            }

            bzero ((char *) RaSortAlgorithms, sizeof(RaSortAlgorithms));

            if (Hflag) {
               fprintf (stdout, "Columns=BinTime,%s\n", RaLabel);
            } else {
               fprintf (stdout, "Columns=%s\n", RaLabel);
            
               switch (RaPrintMode) {
                  case RA_PRINTSRCID:
    
                     RaSortAlgorithms[0] = RaSortAlgorithmTable[RASORTSRCID];
                     RaSortQueue(RaProbeQueue);
                     
                     fprintf (stdout, "Probes=");
                     while ((srv = (struct ArgusRecordStore *) RaPopQueue(RaProbeQueue)) != NULL)
                        fprintf (stdout, "%s ", ipaddr_string((unsigned int *)&srv->rahtblhdr->object));
                     break;

                  case RA_PRINTPROTO: {
                     int i;

                     RaSortAlgorithms[0] = RaSortAlgorithmTable[RASORTPROTOCOL];
                     RaSortQueue(RaProbeQueue);

                     for (i = 0; i < RaProbeQueue->count; i++) {
                        unsigned int proto;
                        srv = (void *) RaProbeQueue->start;
                        if ((proto = (unsigned int) srv->rahtblhdr->object) == IPPROTO_TCP)
                           break;
                        else
                           RaProbeQueue->start = RaProbeQueue->start->nxt;
                     }
                     
                     fprintf (stdout, "Protos=");
                     while ((srv = (struct ArgusRecordStore *) RaPopQueue(RaProbeQueue)) != NULL) {
                        unsigned int proto = (unsigned int) srv->rahtblhdr->object;
                        fprintf (stdout, "%4.4s ", ip_proto_string[(unsigned short) proto]);
                     }
                     break;
                  }
               }
            }
            fprintf (stdout, "\n");
            

         } else {
            if (Lflag) {
               printf ("%s\n", RaLabel);
               fflush (stdout);
            }

            if (Lflag < 0)
               Lflag = 0;
         }
      }
   }

   switch (status) {
      default: {
         int cnt = 0;

         while ((obj = (struct ArgusRecordStore *) RaPopQueue(queue)) != NULL) {
            if (!(Sflag || Cflag || Bflag || Hflag)) {
               if ((argus = obj->data[RaThisActiveIndex]->argus) != NULL) {
                  if ((argus->argus_far.time.start.tv_sec >= RaStartPoint.tv_sec) &&
                      (argus->argus_far.time.last.tv_sec  <= RaEndPoint.tv_sec)) {
                     if (RaTimeoutArgusStore(obj) != 0)
                        cnt++;
                  }
               }

            } else
               if (RaTimeoutArgusStore(obj) != 0)
                  cnt++;
         }

         if (cnt == 0)
            RaSendBlank();
         break;
      }
   }

#ifdef ARGUSDEBUG
   ArgusDebug (6, "RaProcessQueue (0x%x, %d) returning\n", queue, status);
#endif
}

void
RaSendBlank(void)
{
   struct ArgusRecord argusbuf, *argus = &argusbuf;
   struct ArgusRecordStore *srv = NULL;

   bzero((char *)argus, sizeof(*argus));
   argus->ahdr.type = ARGUS_FAR;
   argus->ahdr.length = sizeof(*argus);
   if (!mflag)
      argus->ahdr.status = ETHERTYPE_IP;

   argus->argus_far.type   = ARGUS_FAR;
   argus->argus_far.length = sizeof(argus->argus_far);

   argus->argus_far.time.start.tv_sec = RaStartSeconds;
   argus->argus_far.time.last.tv_sec  = RaStartSeconds + RaBinSize;

   bzero ((char *) RaArgusFlow, sizeof(struct ArgusFlow));

   if ((srv = RaNewArgusStore(argus)) != NULL) {
      srv->startime = argus->argus_far.time.start;
      srv->lasttime = argus->argus_far.time.last;

      srv->status |= RA_MODIFIED;
      if ((srv->data[RaThisActiveIndex] = RaNewArgusData(argus)) != NULL) {
         srv->data[RaThisActiveIndex]->status |= RA_MODIFIED;

         RaSendArgusRecord (srv);
      }

      RaDeleteArgusStore(srv);
   }
}


void *
RaFindHashObject (struct RaHashTableStruct *thisHashTable, struct RaHashStruct *thisHash)
{
   void *retn = NULL;
   struct RaHashTableHeader *rahtblhdr = NULL;
 
   if ((rahtblhdr = RaFindHash(thisHashTable, thisHash)) != NULL)
      retn = rahtblhdr->object;
 
   return (retn);
}


struct RaHashTableHeader *
RaFindHash (struct RaHashTableStruct *thisHashTable, struct RaHashStruct *thisHash)
{
   struct RaHashTableHeader *retn = NULL;
   struct RaHashTableHeader *head = NULL, *target;
   unsigned short *ptr = (unsigned short *) thisHash->buf;
   int i, len = 0;
 
   if (thisHashTable->size > 0) {
      thisHash->hash = 0;
 
      for (i = 0, len = (thisHash->len / sizeof(unsigned short)); i < len; i++)
         thisHash->hash += *ptr++;
 
      if ((target = thisHashTable->array[thisHash->hash % thisHashTable->size]) != NULL) {
         len = (thisHash->len < target->hstruct.len) ?  thisHash->len : target->hstruct.len;
         if (len > 0) {
            head = target;
            do {
               if (!(bcmp ((char *) thisHash->buf, (char *) target->hstruct.buf, len))) {
                  retn = target;
                  break;
               } else
                  target = target->nxt;
            } while (target != head);
         }
      }
   }
 
#ifdef ARGUSDEBUG
   ArgusDebug (7, "RaFindHash: returning 0x%x\n", retn);
#endif
 
   return (retn);
}


struct RaHashTableHeader *
RaAddHashObject (struct RaHashTableStruct *thisHashTable, void *obj, struct RaHashStruct *hstruct)
{
   struct RaHashTableHeader *retn = NULL;
   struct RaHashTableHeader *start = NULL;
   unsigned short *ptr = (unsigned short *) hstruct->buf;
   int i, len;

   hstruct->hash = 0;

   if (thisHashTable) {
      if ((retn = (struct RaHashTableHeader *) ArgusCalloc (1, sizeof (struct RaHashTableHeader))) != NULL) {
         RaAllocHashTableHeaders++;

         for (i = 0, len = (hstruct->len / sizeof(unsigned short)); i < len; i++)
            hstruct->hash += *ptr++;

         retn->object = obj;
         retn->hstruct.len = sizeof(hstruct->buf);
         retn->hstruct.hash = hstruct->hash;
         bcopy((char *)hstruct->buf, (char *)retn->hstruct.buf, hstruct->len);

         if ((start = thisHashTable->array[hstruct->hash % thisHashTable->size]) != NULL) {
            retn->nxt = start;
            retn->prv = start->prv;
            retn->prv->nxt = retn;
            retn->nxt->prv = retn;
         } else
            retn->prv = retn->nxt = retn;

         thisHashTable->array[hstruct->hash % thisHashTable->size] = retn;
         thisHashTable->count++;

         retn->htable = thisHashTable;
      }
   }

#ifdef ARGUSDEBUG
   ArgusDebug (7, "RaAddHashObject (0x%x, 0x%x, 0x%x) returning 0x%x\n", thisHashTable, obj, hstruct, retn);
#endif

   return (retn);
}


int
RaProbeMonitorsThisAddr (unsigned int probe, unsigned int saddr)
{
   int retn = 0;

   struct ArgusListStruct *list = NULL;
   struct RaHashStruct ArgusHash;
   struct RaCIDRAddr *addr, *start;
 
   bzero ((char *)&ArgusHash, sizeof(ArgusHash));
   ArgusHash.len = sizeof(ArgusHash.buf);
   ArgusHash.hash = 0;
 
   ArgusHash.buf[0] = probe;
 
   if (RaMonitorTable != NULL) {
      if ((list = (struct ArgusListStruct *) RaFindHashObject(RaMonitorTable, &ArgusHash)) != NULL) {
         if ((addr = ArgusFrontList(list)) != NULL) {
            start = addr;
            do {
               if (RaCIDRAddrMatches (saddr, addr)) {
                  retn++;
                  break;
               }

               ArgusPopFrontList(list);
               ArgusPushBackList(list, addr);
               addr = ArgusFrontList(list);

            } while (addr != start);
         }
      }
   }

   return (retn);
}


void
ArgusReverseDataRecord (struct ArgusRecord *argus)
{
   struct ArgusRecord *tmp = RaCopyArgusRecord(argus);
   struct ArgusFarHeaderStruct *a1farhdr[32], *a2farhdr[32];
   unsigned int a1DSRStatus = 0, a2DSRStatus = 0, status;
   int i;
 
   a1DSRStatus = ArgusIndexRecord (argus, a1farhdr);
   a2DSRStatus = ArgusIndexRecord (tmp,   a2farhdr);

   status = a1DSRStatus;
 
   for (i = 0; i < 32; i++) {
      if (status & 0x01) {
         switch (i) {
            case ARGUS_FAR_DSR_INDEX: {
               struct ArgusFarStruct *far1 = (struct ArgusFarStruct *) a1farhdr[i];
               struct ArgusFarStruct *far2 = (struct ArgusFarStruct *) a2farhdr[i];

               bcopy ((char *)&far1->src, (char *)&far2->dst, sizeof(struct ArgusMeter));
               bcopy ((char *)&far1->dst, (char *)&far2->src, sizeof(struct ArgusMeter));
               break;
            }

            case ARGUS_MAC_DSR_INDEX:
            case ARGUS_VLAN_DSR_INDEX:
            case ARGUS_MPLS_DSR_INDEX:
            case ARGUS_AGR_DSR_INDEX: {
               break;
            }

            case ARGUS_TIME_DSR_INDEX: {
               struct ArgusTimeStruct *time1 = (struct ArgusTimeStruct *) a1farhdr[i];
               struct ArgusTimeStruct *time2 = (struct ArgusTimeStruct *) a2farhdr[i];

               bcopy ((char *)&time1->src, (char *)&time2->dst, sizeof(struct ArgusTimeEntity));
               bcopy ((char *)&time1->dst, (char *)&time2->src, sizeof(struct ArgusTimeEntity));

               break;
            }

            case ARGUS_SRCUSRDATA_DSR_INDEX:
            case ARGUS_DSTUSRDATA_DSR_INDEX: {
               struct ArgusUserStruct *user2 = (struct ArgusUserStruct *) a2farhdr[i];
               if (user2->type == ARGUS_SRCUSRDATA_DSR)
                  user2->type = ARGUS_DSTUSRDATA_DSR;
               else
                  user2->type = ARGUS_SRCUSRDATA_DSR;
               break;
            }

            case ARGUS_TCP_DSR_INDEX: {
               struct ArgusTCPObject *tcp1 = (struct ArgusTCPObject *) a1farhdr[i];
               struct ArgusTCPObject *tcp2 = (struct ArgusTCPObject *) a2farhdr[i];

               bcopy ((char *)&tcp1->src, (char *)&tcp2->dst, sizeof(struct ArgusTCPObjectMetrics));
               bcopy ((char *)&tcp1->dst, (char *)&tcp2->src, sizeof(struct ArgusTCPObjectMetrics));

               break;
            }

            case ARGUS_ICMP_DSR_INDEX: {
               break;
            }

            case ARGUS_RTP_DSR_INDEX: {
               struct ArgusRTPObject *rtp1 = (struct ArgusRTPObject *) a1farhdr[i];
               struct ArgusRTPObject *rtp2 = (struct ArgusRTPObject *) a2farhdr[i];

               bcopy ((char *)&rtp1->src, (char *)&rtp2->dst, sizeof(struct rtphdr));
               bcopy ((char *)&rtp1->dst, (char *)&rtp2->src, sizeof(struct rtphdr));
               rtp2->sdrop = rtp2->ddrop;
               rtp2->ddrop = rtp2->sdrop;
               rtp2->ssdev = rtp2->dsdev;
               rtp2->dsdev = rtp2->ssdev;

               break;
            }

            case ARGUS_IGMP_DSR_INDEX: {
               break;
            }

            case ARGUS_ARP_DSR_INDEX:
               break;

            case ARGUS_FRG_DSR_INDEX:
               break;

            case ARGUS_ESP_DSR_INDEX: {
               break;
            }
         }
      }

      status >>= 1;
   }

   bcopy ((char *)tmp, (char *)argus, argus->ahdr.length);
   ArgusFree(tmp);
}
