#
# -*- coding: utf-8 -*-
# Copyright 2019 Red Hat
# GNU General Public License v3.0+
# (see COPYING or https://www.gnu.org/licenses/gpl-3.0.txt)
"""
The nxos_lag_interfaces class
It is in this file where the current configuration (as dict)
is compared to the provided configuration (as dict) and the command set
necessary to bring the current configuration to it's desired end-state is
created
"""
from __future__ import absolute_import, division, print_function


__metaclass__ = type

from ansible_collections.ansible.netcommon.plugins.module_utils.network.common.cfg.base import (
    ConfigBase,
)
from ansible_collections.ansible.netcommon.plugins.module_utils.network.common.utils import (
    dict_diff,
    remove_empties,
    search_obj_in_list,
    to_list,
)

from ansible_collections.cisco.nxos.plugins.module_utils.network.nxos.facts.facts import Facts
from ansible_collections.cisco.nxos.plugins.module_utils.network.nxos.utils.utils import (
    normalize_interface,
)


class Lag_interfaces(ConfigBase):
    """
    The nxos_lag_interfaces class
    """

    gather_subset = ["!all", "!min"]

    gather_network_resources = ["lag_interfaces"]

    def __init__(self, module):
        super(Lag_interfaces, self).__init__(module)

    def get_lag_interfaces_facts(self, data=None):
        """Get the 'facts' (the current configuration)

        :rtype: A dictionary
        :returns: The current configuration as a dictionary
        """
        facts, _warnings = Facts(self._module).get_facts(
            self.gather_subset,
            self.gather_network_resources,
            data=data,
        )
        lag_interfaces_facts = facts["ansible_network_resources"].get("lag_interfaces")
        if not lag_interfaces_facts:
            return []
        return lag_interfaces_facts

    def execute_module(self):
        """Execute the module

        :rtype: A dictionary
        :returns: The result from module execution
        """
        result = {"changed": False}
        commands = list()
        warnings = list()

        if self.state in self.ACTION_STATES:
            existing_lag_interfaces_facts = self.get_lag_interfaces_facts()
        else:
            existing_lag_interfaces_facts = []

        if self.state in self.ACTION_STATES or self.state == "rendered":
            commands.extend(self.set_config(existing_lag_interfaces_facts))

        if commands and self.state in self.ACTION_STATES:
            if not self._module.check_mode:
                resp = self._connection.edit_config(commands)
                if "response" in resp:
                    for item in resp["response"]:
                        if item:
                            err_str = item
                            if err_str.lower().startswith("cannot add"):
                                self._module.fail_json(msg=err_str)
            result["changed"] = True

        if self.state in self.ACTION_STATES:
            result["commands"] = commands

        if self.state in self.ACTION_STATES or self.state == "gathered":
            changed_lag_interfaces_facts = self.get_lag_interfaces_facts()

        elif self.state == "rendered":
            result["rendered"] = commands

        elif self.state == "parsed":
            running_config = self._module.params["running_config"]
            if not running_config:
                self._module.fail_json(
                    msg="value of running_config parameter must not be empty for state parsed",
                )
            result["parsed"] = self.get_lag_interfaces_facts(data=running_config)

        if self.state in self.ACTION_STATES:
            result["before"] = existing_lag_interfaces_facts
            if result["changed"]:
                result["after"] = changed_lag_interfaces_facts

        elif self.state == "gathered":
            result["gathered"] = changed_lag_interfaces_facts

        result["warnings"] = warnings
        return result

    def set_config(self, existing_lag_interfaces_facts):
        """Collect the configuration from the args passed to the module,
            collect the current configuration (as a dict from facts)

        :rtype: A list
        :returns: the commands necessary to migrate the current configuration
                  to the desired configuration
        """
        want = self._module.params.get("config")
        if want:
            for w in want:
                w.update(remove_empties(w))
                if "members" in w and w["members"]:
                    for item in w["members"]:
                        item.update({"member": normalize_interface(item["member"])})
        have = existing_lag_interfaces_facts
        resp = self.set_state(want, have)
        return to_list(resp)

    def set_state(self, want, have):
        """Select the appropriate function based on the state provided

        :param want: the desired configuration as a dictionary
        :param have: the current configuration as a dictionary
        :rtype: A list
        :returns: the commands necessary to migrate the current configuration
                  to the desired configuration
        """
        state = self._module.params["state"]
        if state in ("overridden", "merged", "replaced", "rendered") and not want:
            self._module.fail_json(
                msg="value of config parameter must not be empty for state {0}".format(state),
            )
        commands = list()

        if state == "overridden":
            commands.extend(self._state_overridden(want, have))
        elif state == "deleted":
            commands.extend(self._state_deleted(want, have))
        else:
            for w in want:
                if state in ["merged", "rendered"]:
                    commands.extend(self._state_merged(w, have))
                if state == "replaced":
                    commands.extend(self._state_replaced(w, have))
        return commands

    def _state_replaced(self, w, have):
        """The command generator when state is replaced

        :rtype: A list
        :returns: the commands necessary to migrate the current configuration
                  to the desired configuration
        """
        commands = []
        merged_commands = self.set_commands(w, have)
        replaced_commands = self.del_intf_commands(w, have)
        if merged_commands:
            commands.extend(replaced_commands)
            commands.extend(merged_commands)
        return commands

    def _state_overridden(self, want, have):
        """The command generator when state is overridden

        :rtype: A list
        :returns: the commands necessary to migrate the current configuration
                  to the desired configuration
        """
        commands = []
        for h in have:
            obj_in_want = search_obj_in_list(h["name"], want, "name")
            if obj_in_want:
                diff = self.diff_list_of_dicts(h.get("members", []), obj_in_want["members"])
                if not diff:
                    continue
            commands.extend(self.del_all_commands(h))
        for w in want:
            commands.extend(self.set_commands(w, have))
        return commands

    def _state_merged(self, w, have):
        """The command generator when state is merged

        :rtype: A list
        :returns: the commands necessary to merge the provided into
                  the current configuration
        """
        return self.set_commands(w, have)

    def _state_deleted(self, want, have):
        """The command generator when state is deleted

        :rtype: A list
        :returns: the commands necessary to remove the current configuration
                  of the provided objects
        """
        commands = []
        if want:
            for w in want:
                obj_in_have = search_obj_in_list(w["name"], have, "name")
                commands.extend(self.del_all_commands(obj_in_have))
        else:
            if not have:
                return commands
            for h in have:
                commands.extend(self.del_all_commands(h))
        return commands

    def diff_list_of_dicts(self, want, have):
        if not want:
            want = []

        if not have:
            have = []

        diff = []
        for w_item in want:
            h_item = search_obj_in_list(w_item["member"], have, key="member") or {}
            delta = dict_diff(h_item, w_item)
            if delta:
                if h_item:
                    delta["exists"] = True
                    if (
                        "mode" in delta.keys()
                        and delta["mode"] == "on"
                        and "mode" not in h_item.keys()
                    ):
                        # mode = on will not be displayed in running-config
                        continue
                if "member" not in delta.keys():
                    delta["member"] = w_item["member"]
                diff.append(delta)

        return diff

    def intersect_list_of_dicts(self, w, h):
        intersect = []
        wmem = []
        hmem = []
        for d in w:
            wmem.append({"member": d["member"]})
        for d in h:
            hmem.append({"member": d["member"]})
        set_w = set(tuple(sorted(d.items())) for d in wmem)
        set_h = set(tuple(sorted(d.items())) for d in hmem)
        intersection = set_w.intersection(set_h)
        for element in intersection:
            intersect.append(dict((x, y) for x, y in element))
        return intersect

    def add_commands(self, diff, name):
        commands = []
        name = name.strip("port-channel")
        for d in diff:
            sub_cmd = ""
            final_cmd = ""
            member_exists = d.get("exists", False)
            group_cmd = "channel-group {0}".format(name)

            # force does not appear in config
            # will only be applied for a net new member
            if d.get("force") and not member_exists:
                sub_cmd += " force"
            if "mode" in d:
                sub_cmd += " mode %s" % d["mode"]

            if sub_cmd:
                final_cmd = group_cmd + sub_cmd
            elif not member_exists:
                final_cmd = group_cmd

            if final_cmd:
                commands.append("interface" + " " + d["member"])
                commands.append(final_cmd)

        return commands

    def set_commands(self, w, have):
        commands = []
        obj_in_have = search_obj_in_list(w["name"], have, "name")
        if not obj_in_have:
            commands = self.add_commands(w["members"], w["name"])
        else:
            if "members" not in obj_in_have:
                obj_in_have["members"] = None
            diff = self.diff_list_of_dicts(w["members"], obj_in_have["members"])
            commands = self.add_commands(diff, w["name"])
        return commands

    def del_all_commands(self, obj_in_have):
        commands = []
        if not obj_in_have:
            return commands
        for m in obj_in_have.get("members", []):
            commands.append("interface" + " " + m["member"])
            commands.append("no channel-group")
        return commands

    def del_intf_commands(self, w, have):
        commands = []
        obj_in_have = search_obj_in_list(w["name"], have, "name")
        if obj_in_have:
            lst_to_del = self.intersect_list_of_dicts(w["members"], obj_in_have["members"])
            if lst_to_del:
                for item in lst_to_del:
                    commands.append("interface" + " " + item["member"])
                    commands.append("no channel-group")
        return commands
