// ---------------------------------------------------------------------------
// - Part.cpp                                                                -
// - afnix:csm service -  part class implementation                          -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2019 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Part.hpp"
#include "Csmsid.hxx"
#include "Vector.hpp"
#include "Boolean.hpp"
#include "Integer.hpp"
#include "Runnable.hpp"
#include "QuarkZone.hpp"
#include "Exception.hpp"

namespace afnix {

  // -------------------------------------------------------------------------
  // - private section                                                       -
  // -------------------------------------------------------------------------

  // the part request plist
  static const String PN_PRT_NAME = "PN-PRT-NAME";
  static const String PI_PRT_NAME = "PART NAME";
  static const String PN_PRT_INFO = "PN-PRT-INFO";
  static const String PI_PRT_INFO = "PART INFO";
  static const String PN_PRT_UUID = "PN-PRT-UUID";
  static const String PI_PRT_UUID = "PART UUID";

  // -------------------------------------------------------------------------
  // - class section                                                         -
  // -------------------------------------------------------------------------

  // create a default part

  Part::Part (void) {
    d_kid.random ();
    p_plst = nullptr;
  }

  // create an part by name

  Part::Part (const String& name) : Taggable (name) {
    d_kid.random ();
    p_plst = nullptr;
  }

  // create an part by name and info

  Part::Part (const String& name, const String& info) : Taggable (name, info) {
    d_kid.random ();
    p_plst = nullptr;
  }

  // copy construct this part

  Part::Part (const Part& that) {
    that.rdlock ();
    try {
      // copy base taggable
      Taggable::operator = (that);
      // copy locally
      d_kid = that.d_kid;
      Object::iref (p_plst = that.p_plst);
      that.unlock ();
    } catch (...) {
      that.unlock ();
      throw;
    }
  }
  
  // copy move this part

  Part::Part (Part&& that) noexcept : Taggable (static_cast<Taggable&&>(that)) {
    that.wrlock ();
    try {
      d_kid  = that.d_kid;  that.d_kid.reset ();
      p_plst = that.p_plst; that.p_plst = nullptr;
    } catch (...) {
      d_kid.reset ();
    }
    that.unlock ();
  }

  // destroy this part

  Part::~Part (void) {
    Object::dref (p_plst);
  }
  
  // assign an part to this one

  Part& Part::operator = (const Part& that) {
    // check for self assignation
    if (this == &that) return *this;
    // lock and assign
    wrlock ();
    that.rdlock ();
    try {
      // copy base taggable
      Taggable::operator = (that);
      // copy locally
      d_kid = that.d_kid;
      Object::iref (that.p_plst); Object::dref (p_plst); p_plst = that.p_plst;
      unlock ();
      that.unlock ();
      return *this;
    } catch (...) {
      unlock ();
      that.unlock ();
      throw;
    }
  }

  // move a part to this one

  Part& Part::operator = (Part&& that) noexcept {
    // check for self move
    if (this == &that) return *this;
    // lock and assign
    wrlock ();
    that.wrlock ();
    try {
      // assign base object
      Taggable::operator = (static_cast<Taggable&&>(that));
      // move locally
      d_kid  = that.d_kid;  that.d_kid.reset ();
      p_plst = that.p_plst; that.p_plst = nullptr;
    } catch (...) {
      d_kid.reset ();
    }
    unlock ();
    that.unlock ();
    return *this;
  }

  // return the part class name

  String Part::repr (void) const {
    return "Part";
  }

  // return a clone of this object

  Object* Part::clone (void) const {
    return new Part (*this);
  }

  // return the serial did

  t_word Part::getdid (void) const {
    return SRL_DEOD_CSM;
  }

  // return the serial sid

  t_word Part::getsid (void) const {
    return SRL_PART_SID;
  }
  
  // serialize this part

  void Part::wrstream (OutputStream& os) const {
    rdlock ();
    try {
      // serialize the taggable
      Taggable::wrstream (os);
      // serialize locally
      d_kid.wrstream (os);
      if (p_plst == nullptr) {
	Serial::wrnilid (os);
      } else {
	p_plst->serialize (os);
      }
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // deserialize this part

  void Part::rdstream (InputStream& is) {
    wrlock ();
    try {
      // deserialize the taggable
      Taggable::rdstream (is);
      // deserialize locally
      d_kid.rdstream (is);
      p_plst = dynamic_cast <Plist*> (Serial::deserialize (is));
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // validate a part by key id

  bool Part::iskid (const Uuid& kid) const {
    rdlock ();
    try {
      bool result = (d_kid == kid);
      unlock ();
      return result;
    } catch  (...) {
      unlock ();
      throw;
    }
  }

  // validate a part by key id

  bool Part::iskid (const String& kid) const {
    rdlock ();
    try {
      bool result = (d_kid == kid);
      unlock ();
      return result;
    } catch  (...) {
      unlock ();
      throw;
    }
  }

  // get the part kid

  Uuid Part::getkid (void) const {
    rdlock ();
    try {
      Uuid result = d_kid;
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }
    
  // request the part header

  Plist Part::gethead (void) const {
    rdlock ();
    try {
      // get the local plist
      Plist result;
      result.add (PN_PRT_NAME, PI_PRT_NAME, d_name);
      result.add (PN_PRT_INFO, PI_PRT_INFO, d_info);
      result.add (PN_PRT_UUID, PI_PRT_UUID, d_kid.tostring ());
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // request an information list

  Plist Part::getplst (void) const {
    rdlock ();
    try {
      // get the local plist
      Plist result;
      result.add (PN_PRT_NAME, PI_PRT_NAME, d_name);
      result.add (PN_PRT_INFO, PI_PRT_INFO, d_info);
      result.add (PN_PRT_UUID, PI_PRT_UUID, d_kid.tostring ());
      // merge the local one
      if (p_plst != nullptr) result+= *p_plst;
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // check if a property exists in the plist

  bool Part::isprop (const String& name) const {
    rdlock ();
    try {
      bool result = (p_plst == nullptr) ? false : p_plst->exists (name);
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }
  
  // add to the plist by name and value

  void Part::add (const String& name, const Literal& lval) {
    wrlock ();
    try {
      // check for a plist
      if (p_plst == nullptr) p_plst = new Plist;
      // add by name and literal
      p_plst->add (name, lval);
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // add to the plist by name, info and literal value

  void Part::add (const String&  name, const String& info,
		  const Literal& lval) {
    wrlock ();
    try {
      // check for a plist
      if (p_plst == nullptr) p_plst = new Plist;
      // add by name and literal
      p_plst->add (name, info, lval);
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }
    
  // get a plist literal by name

  Literal* Part::toliteral (const String& name) const {
    rdlock ();
    try {
      Literal* result = nullptr;
      if (p_plst != nullptr) result = p_plst->toliteral (name);
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }
    
  // get a plist value by name

  String Part::getpval (const String& name) const {
    rdlock ();
    try {
      String result;
      if (p_plst != nullptr) result = p_plst->getpval (name);
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // -------------------------------------------------------------------------
  // - object section                                                        -
  // -------------------------------------------------------------------------

  // the quark zone
  static const long QUARK_ZONE_LENGTH = 8;
  static QuarkZone  zone (QUARK_ZONE_LENGTH);

  // the object supported quarks
  static const long QUARK_ADD     = zone.intern ("add");
  static const long QUARK_TOLITR  = zone.intern ("to-literal");
  static const long QUARK_ISKIDP  = zone.intern ("kid-p");
  static const long QUARK_GETKID  = zone.intern ("get-kid");
  static const long QUARK_GETHEAD = zone.intern ("get-header");
  static const long QUARK_GETPLST = zone.intern ("get-plist");
  static const long QUARK_GETPVAL = zone.intern ("get-value");
  static const long QUARK_ISPROPP = zone.intern ("property-p");

  // create a new object in a generic way

  Object* Part::mknew (Vector* argv) {
    long argc = (argv == nullptr) ? 0 : argv->length ();
    // create a default part
    if (argc == 0) return new Part;
    // check for 1 argument
    if (argc == 1) {
      String name = argv->getstring (0);
      return new Part (name);
    }
    // check for 2 arguments
    if (argc == 2) {
      String name = argv->getstring (0);
      String info = argv->getstring (1);
      return new Part (name, info);
    }
    throw Exception ("argument-error",
                     "too many argument with part constructor");
  }

  // return true if the given quark is defined

  bool Part::isquark (const long quark, const bool hflg) const {
    rdlock ();
    try {
      if (zone.exists (quark) == true) {
	unlock ();
	return true;
      }
      bool result = hflg ? Taggable::isquark (quark, hflg) : false;
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }
  
  // apply this object with a set of arguments and a quark
  
  Object* Part::apply (Runnable* robj, Nameset* nset, const long quark,
		       Vector* argv) {
    // get the number of arguments
    long argc = (argv == nullptr) ? 0 : argv->length ();
    
    // check for 0 argument
    if (argc == 0) {
      if (quark == QUARK_GETKID)  return new Uuid  (getkid  ());
      if (quark == QUARK_GETHEAD) return new Plist (gethead ());
      if (quark == QUARK_GETPLST) return new Plist (getplst ());
    }
    // check for 1 argument
    if (argc == 1) {
      if (quark == QUARK_ISKIDP) {
	String kid = argv->getstring (0);
	return new Boolean (iskid (kid));
      }
      if (quark == QUARK_ISPROPP) {
	String name = argv->getstring (0);
	return new Boolean (isprop (name));
      }
      if (quark == QUARK_TOLITR) {
        String name = argv->getstring (0);
        return toliteral (name);
      }
      if (quark == QUARK_GETPVAL) {
        String name = argv->getstring (0);
        return new String (getpval (name));
      }
    }
    // dispatch 2 arguments
    if (argc == 2) {
      if (quark == QUARK_ADD) {
        String name = argv->getstring (0);
        Object* obj = argv->get (1);
        auto   lobj = dynamic_cast <Literal*> (obj);
        if (lobj == nullptr) {
	  throw Exception ("type-error", "invalid object with add",
			   Object::repr (obj));
        }
        add (name, *lobj);
        return nullptr;
      }
    }
    // dispatch 3 arguments
    if (argc == 3) {
      if (quark == QUARK_ADD) {
        String name = argv->getstring (0);
        String info = argv->getstring (1);
        Object* obj = argv->get (2);
        auto   lobj = dynamic_cast <Literal*> (obj);
        if (lobj == nullptr) {
	  throw Exception ("type-error", "invalid object with add",
			   Object::repr (obj));
        }
        add (name, info, *lobj);
        return nullptr;
      }
    }
    // call the taggable method
    return Taggable::apply (robj, nset, quark, argv);
  }
}
