// ---------------------------------------------------------------------------
// - Promise.cpp                                                             -
// - afnix engine - promise class implementation                             -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2019 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Engsid.hxx"
#include "Promise.hpp"
#include "Boolean.hpp"
#include "Runnable.hpp"
#include "Exception.hpp"

namespace afnix {

  // -------------------------------------------------------------------------
  // - class section                                                         -
  // -------------------------------------------------------------------------

  // create a default promise

  Promise::Promise (void) {
    p_form   = nullptr;
    p_object = nullptr;
    d_delay  = true;
  }

  // create a new promise with a form

  Promise::Promise (Object* form) {
    p_form   = Object::iref (form);
    p_object = nullptr;
    d_delay  = true;
  }

  // destroy this promise

  Promise::~Promise (void) {
    Object::dref (p_form);
    Object::dref (p_object);
  }

  // return the class name

  String Promise::repr (void) const {
    return "Promise";
  }

  // return the serial did
  
  t_word Promise::getdid (void) const {
    return SRL_DEOD_ENG;
  }

  // return the serial sid

  t_word Promise::getsid (void) const {
    return SRL_PMIS_SID;
  }
  
  // serialize this promise

  void Promise::wrstream (OutputStream& os) const {
    rdlock ();
    try {
      // serialize the form
      if (p_form == nullptr) {
	Serial::wrnilid (os);
      } else {
	Serial* sobj = dynamic_cast <Serial*> (p_form);
	if (sobj == nullptr) {
	  throw Exception ("serial-error", "cannot serialize object", 
			   p_form->repr ());
	}
	sobj->serialize (os);
      }
      // serialize the evaluated object
      if (p_object == nullptr) {
	Serial::wrnilid (os);
      } else {
	Serial* sobj = dynamic_cast <Serial*> (p_object);
	if (sobj == nullptr) {
	  throw Exception ("serial-error", "cannot serialize object", 
			   p_object->repr ());
	}
	sobj->serialize (os);
      }
      // serialize the delay flag
      Boolean bval = d_delay;
      bval.wrstream (os);
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // deserialize this promise

  void Promise::rdstream (InputStream& is) {
    wrlock ();
    try {
      // clean the promise
      Object::dref (p_form);
      Object::dref (p_object);
      // get the form
      Object::iref (p_form = Serial::deserialize (is));
      // get the object
      Object::iref (p_object = Serial::deserialize (is));
      // get the delay flag
      d_delay = Serial::rdbool (is);
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // force the evaluation of this promise

  Object* Promise::force (Runnable* robj, Nameset* nset) {
    wrlock ();
    try {
      if (d_delay == true) {
	p_object = (p_form == nullptr) ? nullptr : p_form->eval (robj, nset);
	Object::iref (p_object);
	d_delay  = false;
      }
      robj->post (p_object);
      unlock ();
      return p_object;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // -------------------------------------------------------------------------
  // - object section                                                        -
  // -------------------------------------------------------------------------

  // evaluate this promise

  Object* Promise::eval (Runnable* robj, Nameset* nset) {
    rdlock ();
    try {
      Object* result = d_delay ? this : p_object;
      robj->post (result);
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }
}
