/*
 *  acm : an aerial combat simulator for X
 *  Classic instruments module
 *  Copyright (C) 2007  Umberto Salsi
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; version 2 dated June, 1991.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program;  if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave., Cambridge, MA 02139, USA.
 */

/**
 * This module takes care to update and draw the upper strip of the controls
 * panel containing the classic instruments, which includes:
 * 
 * - timer
 * - APS lights
 * - turn and slip indicator
 * - anemometer
 * - artificial horizon
 * - altimeter
 * - vertical speed indicator
 * 
 * A data structure holding the state of these instruments is allocated and
 * attached to the aircraft data structure only after a request by the pilot,
 * and hereafter updated.
 * 
 * @file
 */

#ifndef _instruments_h
#define _instruments_h

#include "pm.h"

#ifdef instruments_IMPORT
	#define EXTERN
#else
	#define EXTERN extern
#endif

/**
 * Enables the classic instruments panel. The state of each instrument is created
 * and saved into the aircraft's "instruments" field if does not exist already.
 * @param u
 */
EXTERN void instruments_enable(viewer * u);

/**
 * Disables the classic instruments panel. Does nothing if a classic instruments
 * data structure does not already exist.
 * @param u
 */
EXTERN void instruments_disable(viewer * u);

/**
 * Returns true if the classic instruments panel is currently enabled.
 * @param u
 * @return 
 */
EXTERN _BOOL instruments_isEnabled(viewer * u);

/**
 * The little aircraft frame in front of the attitude indicator can be
 * adjusted in pitch by the pilot moving up and down by delta RAD up to
 * +/-10 DEG.
 */
EXTERN void instruments_attitude_adjust_pitch(viewer * u, double delta);

/**
 * Erects gyro vertical to case orientation.
 */
EXTERN void instruments_attitude_reset(viewer * u);

/**
 * Sets altimeter altitude correction. 'delta' is hundreds of inHg, for
 * example if the current altitude correction is 29.92 inHg (the default)
 * and delta==-1, the new altitude correction will be 29.91 inHg.
 */
EXTERN void instruments_altimeter_correction(viewer *u, int delta);

/**
 * Starts/stops/resets timer.
 */
EXTERN void instruments_timer_toggle(viewer *u);

/**
 * Updates the state the instruments if their data structure has been already
 * created. Does nothing if no instruments data structure not allocated.
 */
EXTERN void instruments_update(viewer * u);

/**
 * Draws the instruments if their data structure has been already created.
 * Does nothing if data structure not allocated or allocated but currently not
 * enabled (that is, HUD displayed instead).
 */
EXTERN void instruments_draw(viewer * u);

/**
 * Releases the data structure u->inst (if not NULL). Data "released"
 * are not actually released from memory, but reused if required.
 */
EXTERN void instruments_free(viewer *u);

#undef EXTERN
#endif
