/**
 * Physical constants, measurement units and conversions.
 * 
 * @file
 */

#ifndef units_H
#define units_H

#include <math.h>

#ifdef units_IMPORT
	#define EXTERN
#else
	#define EXTERN extern
#endif

#define units_DEGtoRAD(x)	((x) * M_PI / 180.0)
#define units_RADtoDEG(x)	((x) * 180.0 / M_PI)
#define units_earth_g 32.17   /* acceleration due to gravity (ft/s^2) */
#define units_NmToFeetFactor  6076.115 /* ft/NM */
#define units_NmToMeterFactor 1853.18
#define units_FootToMeterFactor 0.30480060960
#define units_LbToKgFactor    0.45359237
//#define units_LbfToNFactor    (earth_g * units_LbToKgFactor / FtToMeterFactor)
#define units_FPStoKT(v)      ((v) / units_NmToFeetFactor * 3600.0)
#define units_KTtoFPS(v)      ((v) * units_NmToFeetFactor / 3600.0)
#define units_FEETtoMETERS(v) ((v) * units_FootToMeterFactor)
#define units_METERStoFEET(v) ((v) / units_FootToMeterFactor)
#define units_METERStoNM(x)   ((x) / units_NmToMeterFactor)
//#define units_NMtoFEET(x)     ((x) * units_NmToFeetFactor)
#define units_NMtoMETERS(x)   ((x) * units_NmToMeterFactor)

/*
 * All constants referring to the standard atmosphere conditions
 * T = 15 C, P = 101325 Pa at sea level.
 */
#define units_GM  1.4         /* ratio of specific heats for air */
#define units_P0  2116.22     /* sea-level pressure [lbf/ft^2] */
#define units_TA  459.67      /* Conversion temperature from F to R (Rankine, or abs. F temp.) */
#define units_T0  (units_TA + 59.0)   /* sea-level temperature [R] */
#define units_RU  (1545.31 * units_earth_g) /* universal gas constant */
#define units_MA  28.9644             /* molecular weight for air [lbm/lbm-mole] */
#define units_RA  (units_RU / units_MA)  /* gas constant for air [ft^2/sec^2/R] */
#define units_RHO_0 2.3769e-3    /* air density at sea level, std. atm. [slug/ft^3] */

#define units_FarenheitToCelsius(f) (((f) - 32.0)*5/9)
#define units_RankineToCelsius(r) units_FarenheitToCelsius(r - units_TA)
#define units_SlugToKg(s) (14.59 * (s))

#undef EXTERN
#endif