from code.ListBox import ListBox
from cal.CalEvent import CalEvent
from cal.Date import Date
from cal.RRule import RRule
from code import formats

import gtk
import gobject



#
# Editor for calendar events.
#
class CalEditor(gtk.Dialog):

    __CMD_NEW = 0
    __CMD_REMOVE = 1


    #
    # Constructor.
    #
    def __init__(self):

        # the day
        self.__day = (0, 0, 0)

        # the calendar to use
        self.__calendar = None

        # the events on that day
        self.__events = []

        # the currently selected event
        self.__current_event = None

        # handler for the close action
        self.__close_handler = None

        # flag for indicating event blocking
        self.__events_blocked = 0

        # flag for indicating whether the user has changed an entry
        self.__changed_flag = 0


        gtk.Dialog.__init__(self)
        from code import icon; self.set_icon(icon.ICON)
        self.set_title(_("Edit Appointments"))
        self.set_default_size(400, 300)
        self.connect("delete-event", self.__on_close)

        # close button
        btn = self.add_button(gtk.STOCK_CLOSE, 0)
        btn.connect("clicked", self.__on_close)

        #
        # events frame
        #
        self.__events_frame = gtk.Frame("")
        self.__events_frame.set_border_width(4)
        self.__events_frame.show()
        self.vbox.pack_start(self.__events_frame, gtk.TRUE, gtk.TRUE, 0)

        hbox = gtk.HBox()
        hbox.set_border_width(4)
        hbox.show()
        self.__events_frame.add(hbox)

        # list of appointments
        self.__listbox = ListBox(["Time", "Summary"])
        #self.__listbox.set_headers_visible(gtk.TRUE)
        self.__listbox.connect_select_row(self.__on_select)
        self.__listbox.show()
        hbox.pack_start(self.__listbox, gtk.TRUE, gtk.TRUE, 0)

        # buttons
        vbox = gtk.VBox()
        vbox.show()
        hbox.pack_end(vbox, gtk.FALSE, gtk.FALSE, 4)

        btn1 = self.__create_button(gtk.STOCK_NEW, _("New Appointment"))
        btn1.connect("clicked", self.__on_button, self.__CMD_NEW)
        btn1.show()
        vbox.pack_start(btn1, gtk.FALSE, gtk.FALSE, 4)
        btn2 = self.__create_button(gtk.STOCK_DELETE, _("Remove Appointment"))
        btn2.connect("clicked", self.__on_button, self.__CMD_REMOVE)
        btn2.show()
        vbox.pack_start(btn2, gtk.FALSE, gtk.FALSE, 4)


        #
        # edit frame
        #
        self.__edit_frame = gtk.Frame(_("Edit Appointment"))
        self.__edit_frame.set_border_width(4)
        self.__edit_frame.show()
        self.vbox.pack_start(self.__edit_frame, gtk.FALSE, gtk.FALSE, 0)

        tbl = gtk.Table(3, 2)
        tbl.set_border_width(4)
        tbl.show()
        self.__edit_frame.add(tbl)

        lbl = gtk.Label(_("Summary:"))
        lbl.show()
        align = gtk.Alignment()
        align.show()
        align.add(lbl)
        tbl.attach(align, 0, 1, 0, 1, gtk.FILL, 0, 2)

        self.__entry_summary = gtk.Entry()
        self.__entry_summary.connect("changed", self.__on_text_change)
        self.__entry_summary.show()
        tbl.attach(self.__entry_summary, 1, 2, 0, 1,
                   gtk.EXPAND | gtk.FILL, 0, 0)

        lbl = gtk.Label(_("Time:"))
        lbl.show()
        align = gtk.Alignment()
        align.show()
        align.add(lbl)
        tbl.attach(align, 0, 1, 1, 2, gtk.FILL, 0, 2)

        # time entry
        hbox = gtk.HBox()
        hbox.show()
        tbl.attach(hbox, 1, 2, 1, 2, gtk.EXPAND | gtk.FILL, 0, 0)

        self.__entry_hours = gtk.SpinButton(climb_rate = 1, digits = 0)
        self.__entry_hours.connect("value-changed", self.__on_time_change, 0)
        self.__entry_hours.set_range(0, 23)
        self.__entry_hours.set_increments(1, 5)
        self.__entry_hours.show()
        hbox.pack_start(self.__entry_hours, gtk.FALSE, gtk.FALSE, 0)

        lbl = gtk.Label(" : ")
        lbl.show()
        hbox.pack_start(lbl, gtk.FALSE, gtk.FALSE, 0)

        self.__entry_mins = gtk.SpinButton(climb_rate = 1, digits = 0)
        self.__entry_mins.connect("value-changed", self.__on_time_change, 1)
        self.__entry_mins.set_range(0, 59)
        self.__entry_mins.set_increments(1, 10)
        self.__entry_mins.show()
        hbox.pack_start(self.__entry_mins, gtk.FALSE, gtk.FALSE, 0)

        # recurrence entry
        lbl = gtk.Label(_("Recurrences:"))
        lbl.show()
        align = gtk.Alignment()
        align.show()
        align.add(lbl)
        tbl.attach(align, 0, 1, 2, 3, gtk.FILL, 0, 2)

        text = _("every $INTERVAL$ ${days|weeks|months|years}$ for "
                 "$TIMES$ times")

        hbox = self.__create_recurrences_entry(text)
        tbl.attach(hbox, 1, 2, 2, 3, gtk.EXPAND | gtk.FILL, 0, 0)



    #
    # Creates a button.
    #
    def __create_button(self, icon, label):

        btn = gtk.Button()
        box = gtk.HBox()
        box.show()
        btn.add(box)

        if (icon):
            img = gtk.Image()
            img.set_from_stock(icon, gtk.ICON_SIZE_BUTTON)
            img.show()
            box.pack_start(img, gtk.FALSE, gtk.FALSE, 4)
        #end if

        lbl = gtk.Label(label)
        lbl.show()
        box.pack_start(lbl, gtk.FALSE, gtk.FALSE, 4)

        return btn



    #
    # Creates the recurrences entry.
    #
    def __create_recurrences_entry(self, data):

        hbox = gtk.HBox()
        hbox.show()

        parts = data.split("$")
        for p in parts:
            if (p == "INTERVAL"):
                spin = gtk.SpinButton(climb_rate = 1, digits = 0)
                spin.set_range(1, 1000)
                spin.set_increments(1, 10)
                spin.connect("value-changed", self.__on_rec_change, 0)
                spin.show()
                self.__entry_rec_interval = spin
                widget = spin
                
            elif (p == "TIMES"):
                spin = gtk.SpinButton(climb_rate = 1, digits = 0)
                spin.set_range(1, 1000)
                spin.set_increments(1, 10)
                spin.connect("value-changed", self.__on_rec_change, 2)
                spin.show()
                self.__entry_rec_times = spin
                widget = spin
                
            elif (p and p[0] == "{"):
                entries = p[1:-1].split("|")
                option = gtk.OptionMenu()
                option.show()
                menu = gtk.Menu()
                for e in entries:
                    item = gtk.MenuItem(e)
                    item.show()
                    menu.append(item)
                #end for
                option.set_menu(menu)
                self.__entry_rec_freq = option
                option.connect("changed", self.__on_rec_change, 1)
                widget = option
                
            else:
                lbl = gtk.Label(p)
                lbl.show()
                widget = lbl

            hbox.pack_start(widget, gtk.FALSE, gtk.FALSE, 0)
        #end for
        
        return hbox
        


    #
    # Reacts on closing the window.
    #
    def __on_close(self, *args):
 
        self.destroy()
        if (self.__changed_flag and self.__close_handler):
            self.__close_handler()

        return gtk.TRUE



    #
    # Reacts on selecting an event.
    #
    def __on_select(self, src, row):

        self.__current_event = self.__events[row]
        summary = self.__current_event.get_summary()
        hours, mins, secs = self.__current_event.get_start().get_daytime()

        self.__edit_frame.set_sensitive(gtk.TRUE)
        self.__entry_summary.set_text(self.__current_event.get_summary())
        self.__entry_hours.set_value(hours)
        self.__entry_mins.set_value(mins)

        rrules = self.__current_event.get_recurrences().get_rules()
        if (rrules):
            rrule = rrules[0]
            freq = rrule.get_frequency()
            interval = rrule.get_interval()
            times = rrule.get_count()

            self.__entry_rec_interval.set_value(interval)
            self.__entry_rec_freq.set_history(freq)
            self.__entry_rec_times.set_value(times)
        else:
            self.__entry_rec_interval.set_value(1)
            self.__entry_rec_freq.set_history(0)
            self.__entry_rec_times.set_value(1)
        #end if



    #
    # Reacts on changing text.
    #
    def __on_text_change(self, src):

        if (self.__events_blocked): return

        text = src.get_text()
        if (self.__current_event):
            self.__current_event.set_summary(text)
            self.__changed_flag = 1

        self.__update_list()


    #
    # Reacts on changing time values.
    #
    def __on_time_change(self, src, field):

        if (self.__events_blocked): return
        
        value = src.get_value_as_int()
        cyear, cmonth, cday = self.__current_event.get_start().get_day()
        chour, cmin, csec = self.__current_event.get_start().get_daytime()
        if (field == 0):
            chour = value
        elif (field == 1):
            cmin = value

        dtstart = Date(cyear, cmonth, cday, chour, cmin, csec)
        self.__current_event.set_start(dtstart)

        self.__current_event.invalidate()
        self.__changed_flag = 1
        self.__update_list()



    #
    # Reacts on changing recurrence values.
    #
    def __on_rec_change(self, src, field):

        if (self.__events_blocked): return

        interval = self.__entry_rec_interval.get_value_as_int()
        freq = self.__entry_rec_freq.get_history()
        times = self.__entry_rec_times.get_value_as_int()

        rrules = self.__current_event.get_recurrences().get_rules()
        if (not rrules):
            rrule = RRule()
            self.__current_event.get_recurrences().add_rule(rrule)
        else:
            rrule = rrules[0]

        # reset all delta values
        rrule.set_delta(RRule.DAY, 0)
        rrule.set_delta(RRule.MONTH, 0)
        rrule.set_delta(RRule.YEAR, 0)
                
        if (freq == 0):
            rrule.set_frequency(RRule.DAY)
            rrule.set_delta(RRule.DAY, 1)
        elif (freq == 1):
            rrule.set_frequency(RRule.WEEK)
            rrule.set_delta(RRule.DAY, 7)
        elif (freq == 2):
            rrule.set_frequency(RRule.MONTH)
            rrule.set_delta(RRule.MONTH, 1)
        elif (freq == 3):
            rrule.set_frequency(RRule.YEAR)
            rrule.set_delta(RRule.YEAR, 1)

        rrule.set_interval(interval)
        rrule.set_count(times)
        rrule.set_until(None)

        self.__current_event.invalidate()
        self.__changed_flag = 1

    

    #
    # Reacts on pressing a command button.
    #
    def __on_button(self, src, cmd):

        if (self.__events_blocked): return

        year, month, day = self.__day

        if (cmd == self.__CMD_NEW):
            new_event = CalEvent()
            new_event.set_start(Date(year, month, day, 8, 0, 0))
            self.__calendar.add_event(new_event)
            self.__display_events()
            row = self.__events.index(new_event)
            self.__listbox.set_cursor(row)
            self.__changed_flag = 1

        elif (cmd == self.__CMD_REMOVE and self.__current_event):
            row = self.__listbox.get_cursor()
            self.__calendar.remove_event(self.__current_event)
            self.__current_event = None
            self.__edit_frame.set_sensitive(gtk.FALSE)
            self.__display_events()
            row = min(row, len(self.__events) - 1)
            if (row >= 0): self.__listbox.set_cursor(row)
            self.__changed_flag = 1


    #
    # Displays the existing events.
    #
    def __display_events(self):

        year, month, day = self.__day
        events = self.__calendar.get_events(year, month, day)
        if (events): events.sort(events[0].daytime_comparator)

        self.__listbox.clear()
        self.__events = []
        for e in events:
            hour, min, sec = e.get_start().get_daytime()
            daytime = formats.DAYTIMEFORMAT % vars()
            self.__listbox.append(daytime, e.get_summary())
            self.__events.append(e)
        #end for



    #
    # Updates the events list.
    #
    def __update_list(self):

        summary = self.__current_event.get_summary()
        hour, min, sec = self.__current_event.get_start().get_daytime()
        daytime = formats.DAYTIMEFORMAT % vars() % vars()

        row = self.__listbox.get_cursor()
        self.__listbox.set(row, daytime, summary)
        self.__ensure_order()



    #
    # Makes sure that the events are always listed in chronological order.
    #
    def __ensure_order(self):

        if (self.__events):
            e = self.__events[0]
            l1 = tuple(self.__events)
            self.__events.sort(e.daytime_comparator)
            l2 = tuple(self.__events)

            if (l1 != l2):
                self.__display_events()
                row = self.__events.index(self.__current_event)
                self.__listbox.set_cursor(row)
                
        #end if



    #
    # Sets the day to edit.
    #
    def set_day(self, year, month, day, calendar):

        def unblock(self): self.__events_blocked = 0

        self.__events_blocked = 1
        
        self.__day = (year, month, day)
        self.__calendar = calendar
        self.__current_event = None
        self.__entry_summary.set_text("")
        self.__entry_hours.set_value(0)
        self.__entry_mins.set_value(0)

        date = formats.YEARDATEFORMAT % vars()
        self.__events_frame.set_label(_("Your appointments on %(date)s")
                                      % vars())
        self.__edit_frame.set_sensitive(gtk.FALSE)
        self.__display_events()

        gtk.idle_add(unblock, self)
        



    #
    # Connects an action handler to the close action.
    #
    def connect_close(self, handler):

        self.__close_handler = handler
