/*
 * Copyright (C) 2007 Felipe Weckx <felipe.weckx@gmail.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include "gbemol-library.h"
#include "gbemol-utils.h"

#include <gtk/gtk.h>

static GObjectClass* parent_class = NULL;

enum {
	COLUMN_CONSTRAINT_VALUE,
	COLUMN_CONSTRAINT_ALL,
	COLUMN_CONSTRAINT_BOLD,
	COLUMN_CONSTRAINT_BOLD_SET,
	N_COLUMNS_CONSTRAINT
};

enum
{
	COLUMN_RESULTS_ARTIST,
	COLUMN_RESULTS_TITLE,
	COLUMN_RESULTS_ALBUM,
	COLUMN_RESULTS_TRACK,
	COLUMN_RESULTS_DATE,
	COLUMN_RESULTS_GENRE,
	COLUMN_RESULTS_COMPOSER,
	COLUMN_RESULTS_DISC,
	COLUMN_RESULTS_COMMENT,
	COLUMN_RESULTS_TIME,
	COLUMN_RESULTS_ID,
	COLUMN_RESULTS_BOLD,
	COLUMN_RESULTS_BOLD_SET,
	COLUMN_RESULTS_FILE,
	N_COLUMNS_RESULTS
};


enum {
	CMB_TAG_NAME,
	CMB_TAG_KEY,
	N_COLUMNS_TAG
};

enum {
    COLUMN_SELECTED_TITLE,
    COLUMN_SELECTED_ARTIST,
    COLUMN_SELECTED_TIME,
    COLUMN_SELECTED_BOLD,
    COLUMN_SELECTED_BOLD_SET,
    N_COLUMNS_SELECTED
};

static const gchar* column_results[N_C_RESULTS] = {
	N_("Artist"),
	N_("Title"),
	N_("Album"),
	N_("Track"),
	N_("Date"),
	N_("Genre"),
	N_("Composer"),
	N_("Disc"),
	N_("Comment"),
	N_("Time")
};


struct _GbemolLibraryPrivate {
	GbemolMpd *mpd;

	/* Constraints */
	GtkWidget* cmb_constraint[3];
	GtkWidget* tvw_constraint[3];
	GtkListStore* lst_constraint[3];
	GtkTreeSelection* sel_constraint[3];

	GtkListStore* lst_tags;

	/* Results */
	GtkWidget* tvw_results;
	GtkListStore* lst_results;
	GtkTreeSelection* sel_results;
	GtkTreeModel* fil_results;   /* The filter for searching */
	
	/* Search */
	GtkWidget* ent_search;
	GtkWidget* cmb_search;
	GtkWidget* btn_search_clear;
	GtkListStore* lst_column_search;

	/* Buttons */
	GtkWidget* btn_update;
	GtkWidget* btn_enqueue;
	GtkWidget* btn_enqueue_all;
	GtkWidget* btn_play;	

	GtkWidget* lbl_update;

	GtkUIManager* manager;

	GtkTooltips* tips;

	gboolean dispose_has_run;
};

static void gbemol_library_class_init (GObjectClass *g_class);
static void gbemol_library_init (GbemolLibrary *obj);
static void gbemol_library_finalize (GObject *obj);
static void gbemol_library_dispose (GObject *obj);

static void gbemol_library_populate_constraint (GbemolLibrary* lib, int id, GList* tags);
static void gbemol_library_populate_results (GbemolLibrary* lib, GList* songs);
static gint gbemol_library_add_selected_to_playlist (GbemolLibrary* lib);

/* Callbacks */
static void on_constraint_changed (GtkWidget *widget, GbemolLibrary* lib);
static void on_btn_enqueue_all_clicked (GtkWidget* b, GbemolLibrary* lib);
static void on_btn_enqueue_clicked (GtkWidget* b, GbemolLibrary* lib);
static void on_btn_play_clicked (GtkWidget* b, GbemolLibrary* lib);
static void on_btn_update_clicked (GtkWidget* b, GbemolLibrary* lib);
static void on_mnu_select_all_activate (GtkWidget* w, GbemolLibrary* lib);
static void on_mnu_column_activate (GtkWidget* w, GbemolLibrary* lib);
static void on_column_clicked (GtkTreeViewColumn* column, GbemolLibrary* lib);
static gboolean on_tvw_results_button_press_event (GtkWidget *w, GdkEventButton *event, GbemolLibrary* pls);
static gboolean gbemol_library_is_visible (GtkTreeModel* model, GtkTreeIter *iter, GbemolLibrary* lib);
static void on_ent_search_changed (GtkEditable* entry, GbemolLibrary* lib);
static void on_btn_clear_search_activate (GtkButton* btn, GbemolLibrary *lib);

static GtkActionEntry entries[] = {
	{"Enqueue", GTK_STOCK_ADD, N_("Enqueue"), NULL, NULL,
	 G_CALLBACK (on_btn_enqueue_clicked)},
	{"Play", GTK_STOCK_MEDIA_PLAY, N_("Play"), NULL, NULL,
	 G_CALLBACK (on_btn_play_clicked)},
	{"Column", GTK_STOCK_GO_DOWN, N_("Columns"), NULL, NULL,
	 G_CALLBACK (on_mnu_column_activate)},
	{"Select All", GTK_STOCK_SELECT_ALL, N_("Select All"), NULL, NULL,
	 G_CALLBACK (on_mnu_select_all_activate)}
};

static const char *description = {
	"<ui>"
	"<popup name='ResultsMenu'>"
		"<menuitem action='Enqueue'/>"
		"<menuitem action='Play'/>"
		"<separator/>"
		"<menuitem action='Select All'/>"
		"<separator/>"
		"<menuitem action='Column'/>"
	"</popup>"
	"</ui>"
};

GType
gbemol_library_get_type (void)
{
	static GType type = 0;
	if (type == 0) {
		static const GTypeInfo info = {
			sizeof (GbemolLibraryClass),
			NULL,   /* base_init */
			NULL,   /* base_finalize */
			(GClassInitFunc) gbemol_library_class_init,   /* class_init */
			NULL,   /* class_finalize */
			NULL,   /* class_data */
			sizeof (GbemolLibrary),
			0,      /* n_preallocs */
			(GInstanceInitFunc) gbemol_library_init    /* instance_init */
		};

		type = g_type_register_static (GTK_TYPE_VBOX,
				"GbemolLibrary",
				&info, 0);
	}
	return type;
}

static void
gbemol_library_class_init (GObjectClass *g_class)
{
	parent_class = g_type_class_peek_parent (g_class);
	g_class->finalize = gbemol_library_finalize;
	g_class->dispose = gbemol_library_dispose;

}


static void
gbemol_library_dispose (GObject *obj)
{
	GbemolLibrary* self = (GbemolLibrary *) obj;
	int i;
	gchar *str;
	
	
	if (self->priv->dispose_has_run)
		return;
	
	self->priv->dispose_has_run = TRUE;

	for (i = 0; i < 3; i++)
	{
		str = g_strdup_printf ("constraint%d", i);
		gbemol_cfg_set_int ("library", str, 
			gtk_combo_box_get_active (GTK_COMBO_BOX (self->priv->cmb_constraint[i])));

		g_free (str);
	}

	gbemol_cfg_set_int ("library", "cmb_search", 
			gtk_combo_box_get_active (GTK_COMBO_BOX (self->priv->cmb_search)));


	G_OBJECT_CLASS (parent_class)->dispose (obj);
}

static void 
gbemol_library_finalize (GObject *obj)
{
	g_free (GBEMOL_LIBRARY (obj)->priv);
	G_OBJECT_CLASS (parent_class)->finalize (obj);
}

static void
gbemol_library_init (GbemolLibrary *obj)
{
	obj->priv = g_new0 (GbemolLibraryPrivate, 1);
	obj->priv->dispose_has_run = FALSE;
}

static void gbemol_library_populate_tags (GbemolLibrary* lib)
{
	GtkTreeIter iter;
	GtkListStore* lst;

	lib->priv->lst_tags = gtk_list_store_new (N_COLUMNS_TAG,
			G_TYPE_STRING,
			G_TYPE_INT);
	lst = lib->priv->lst_tags;	

	/* Add the tag types */
	gtk_list_store_append (lst, &iter);
	gtk_list_store_set (lst, &iter,
			CMB_TAG_NAME, _("None"),
			CMB_TAG_KEY, -1,
			-1);
	gtk_list_store_append (lst, &iter);
	gtk_list_store_set (lst, &iter,
			CMB_TAG_NAME, _("Artist"),
			CMB_TAG_KEY, MPD_TAG_ITEM_ARTIST,
			-1);
	gtk_list_store_append (lst, &iter);
	gtk_list_store_set (lst, &iter,
			CMB_TAG_NAME, _("Album"),
			CMB_TAG_KEY, MPD_TAG_ITEM_ALBUM,
			-1);
	gtk_list_store_append (lst, &iter);
	gtk_list_store_set (lst, &iter,
			CMB_TAG_NAME, _("Genre"),
			CMB_TAG_KEY, MPD_TAG_ITEM_GENRE,
			-1);
	gtk_list_store_append (lst, &iter);
	gtk_list_store_set (lst, &iter,
			CMB_TAG_NAME, _("Composer"),
			CMB_TAG_KEY, MPD_TAG_ITEM_COMPOSER,
			-1);
	gtk_list_store_append (lst, &iter);
	gtk_list_store_set (lst, &iter,
			CMB_TAG_NAME, _("Performer"),
			CMB_TAG_KEY, MPD_TAG_ITEM_PERFORMER,
			-1);
	gtk_list_store_append (lst, &iter);
	gtk_list_store_set (lst, &iter,
			CMB_TAG_NAME, _("Date"),
			CMB_TAG_KEY, MPD_TAG_ITEM_DATE,
			-1);
	gtk_list_store_append (lst, &iter);
	gtk_list_store_set (lst, &iter,
			CMB_TAG_NAME, _("Disc"),
			CMB_TAG_KEY, MPD_TAG_ITEM_DISC,
			-1);

}

static void
gbemol_library_build_menus (GbemolLibrary *lib)
{
	GtkActionGroup *action_group;
	GError *error = NULL;

	action_group = gtk_action_group_new ("MenuAction");
	gtk_action_group_add_actions (action_group, entries,
				      G_N_ELEMENTS (entries),
				      lib);

	lib->priv->manager = gtk_ui_manager_new ();
	gtk_ui_manager_insert_action_group (lib->priv->manager,
					    action_group,
					    0);

	if (!gtk_ui_manager_add_ui_from_string (lib->priv->manager,
						description,
						-1,
						&error)) {
		g_message ("building menus failed: %s", error->message);
		g_error_free (error);
	}
}


GbemolLibrary*
gbemol_library_new (GbemolMpd* mpd)
{
	GtkWidget *hpnd, *hpnd2, *vpnd, *vbox, *hbox, *scr, *vbox_main, *lbl, *btn;
	GtkCellRenderer* renderer;
	GtkTreeViewColumn* column;
	GbemolLibrary* lib;
	GtkTreeIter iter;
	int i, index;
	gchar *str;

	lib = GBEMOL_LIBRARY (g_object_new (GBEMOL_TYPE_LIBRARY, NULL));
	lib->priv->mpd = mpd;

	lib->priv->tips = gtk_tooltips_new ();

	vbox_main = gtk_vbox_new (TRUE, 5);
	gtk_box_pack_start (GTK_BOX (lib), vbox_main, TRUE, TRUE, 0);

	hpnd = gtk_hpaned_new ();
	hpnd2 = gtk_hpaned_new ();
	vpnd = gtk_vpaned_new ();
	gtk_box_pack_start (GTK_BOX (vbox_main), vpnd, TRUE, TRUE, 0);

	gtk_paned_pack1 (GTK_PANED (vpnd), hpnd, TRUE, FALSE);
	gtk_paned_pack2 (GTK_PANED (hpnd), hpnd2, TRUE, FALSE);

	gbemol_library_populate_tags (lib);

	/* Constraints */
	for (i = 0; i < 3; i++)
	{
		vbox = gtk_vbox_new (FALSE, 5);

		lib->priv->cmb_constraint[i] = gtk_combo_box_new ();
		gtk_tooltips_set_tip (lib->priv->tips, lib->priv->cmb_constraint[i], 
				_("Select a field to be shown"),
				NULL);

		gtk_combo_box_set_model (GTK_COMBO_BOX (lib->priv->cmb_constraint[i]), 
				GTK_TREE_MODEL (lib->priv->lst_tags));
		renderer = gtk_cell_renderer_text_new ();
		gtk_cell_layout_pack_start (GTK_CELL_LAYOUT (lib->priv->cmb_constraint[i]), renderer, TRUE);
		gtk_cell_layout_set_attributes (GTK_CELL_LAYOUT (lib->priv->cmb_constraint[i]), renderer,
				"text", CMB_TAG_NAME,
				NULL);
		gtk_box_pack_start (GTK_BOX (vbox), lib->priv->cmb_constraint[i], FALSE, FALSE, 3);

		lib->priv->lst_constraint[i] = gtk_list_store_new (N_COLUMNS_CONSTRAINT,
				G_TYPE_STRING,
				G_TYPE_BOOLEAN,
				G_TYPE_INT,
				G_TYPE_BOOLEAN);

		lib->priv->tvw_constraint[i] = gtk_tree_view_new ();
		gtk_tree_view_set_model (GTK_TREE_VIEW (lib->priv->tvw_constraint[i]), 
				GTK_TREE_MODEL (lib->priv->lst_constraint[i]));
		gtk_tree_view_set_search_column (GTK_TREE_VIEW (lib->priv->tvw_constraint[i]), COLUMN_CONSTRAINT_VALUE);
		gtk_tree_view_set_headers_visible (GTK_TREE_VIEW (lib->priv->tvw_constraint[i]), FALSE);
		g_object_unref (G_OBJECT (lib->priv->lst_constraint[i]));

		scr = gtk_scrolled_window_new (NULL, NULL);
		gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scr), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
		gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (scr), GTK_SHADOW_ETCHED_IN);
		gtk_container_add (GTK_CONTAINER (scr), lib->priv->tvw_constraint[i]);

		renderer = gtk_cell_renderer_text_new ();
		column = gtk_tree_view_column_new_with_attributes (_(""), renderer,
				"text", COLUMN_CONSTRAINT_VALUE,
				"weight", COLUMN_CONSTRAINT_BOLD,
				"weight-set", COLUMN_CONSTRAINT_BOLD_SET,
				NULL);
		gtk_tree_view_append_column (GTK_TREE_VIEW (lib->priv->tvw_constraint[i]), column);

		lib->priv->sel_constraint[i] = gtk_tree_view_get_selection (GTK_TREE_VIEW (lib->priv->tvw_constraint[i]));

		gtk_box_pack_start (GTK_BOX (vbox), scr, TRUE, TRUE, 0);

		g_object_set_data (G_OBJECT (lib->priv->sel_constraint[i]), "index", GINT_TO_POINTER(i));
		g_object_set_data (G_OBJECT (lib->priv->cmb_constraint[i]), "index", GINT_TO_POINTER(i));

		str = g_strdup_printf ("constraint%d", i);
		index = gbemol_cfg_get_int ("library", str);
		gtk_combo_box_set_active (GTK_COMBO_BOX (lib->priv->cmb_constraint[i]), index);
		g_free (str);

		
		g_signal_connect (G_OBJECT (lib->priv->sel_constraint[i]), "changed", 
				G_CALLBACK (on_constraint_changed), lib);
		g_signal_connect (G_OBJECT (lib->priv->cmb_constraint[i]), "changed", 
				G_CALLBACK (on_constraint_changed), lib);


		switch (i)
		{
			case 0:
				gtk_paned_pack1 (GTK_PANED (hpnd), vbox, TRUE, FALSE);
				break;
			case 1:
				gtk_paned_pack1 (GTK_PANED (hpnd2), vbox, TRUE, FALSE);
				break;
			case 2:
				gtk_paned_pack2 (GTK_PANED (hpnd2), vbox, TRUE, FALSE);
				break;
		}
	}

	
	vbox = gtk_vbox_new (FALSE, 5);

	/* Search */
	
	hbox = gtk_hbox_new (FALSE, 5);

	lbl = gtk_label_new (_("Search:"));
	gtk_box_pack_start (GTK_BOX (hbox), lbl, FALSE, FALSE, 5);

	lib->priv->ent_search = gtk_entry_new ();
	gtk_tooltips_set_tip (lib->priv->tips, lib->priv->ent_search, _("Enter text to be searched"), NULL);
	gtk_box_pack_start (GTK_BOX (hbox), lib->priv->ent_search, FALSE, FALSE, 0);
	g_signal_connect (G_OBJECT (lib->priv->ent_search), "changed", 
			G_CALLBACK (on_ent_search_changed), lib);

	lib->priv->cmb_search = gtk_combo_box_new ();
	gtk_box_pack_start (GTK_BOX (hbox), lib->priv->cmb_search, FALSE, FALSE, 0);

	btn = gbemol_utils_make_button (NULL, GTK_STOCK_CLEAR);
	gtk_button_set_relief (GTK_BUTTON (btn), GTK_RELIEF_NONE);
	gtk_tooltips_set_tip (lib->priv->tips, btn, _("Clear search results"), NULL);
	gtk_box_pack_start (GTK_BOX (hbox), btn, FALSE, FALSE, 0);
	g_signal_connect (G_OBJECT (btn), "activate", G_CALLBACK (on_btn_clear_search_activate), lib);
	g_signal_connect (G_OBJECT (btn), "clicked", G_CALLBACK (on_btn_clear_search_activate), lib);
	lib->priv->btn_search_clear = btn;


	gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);


	lib->priv->lst_results = gtk_list_store_new (N_COLUMNS_RESULTS,
			G_TYPE_STRING,    /* Title */
			G_TYPE_STRING,    /* Artist */
			G_TYPE_STRING,    /* Album */
			G_TYPE_STRING,    /* Track */
			G_TYPE_STRING,    /* Date */
			G_TYPE_STRING,    /* Genre */
			G_TYPE_STRING,    /* Composer */
			G_TYPE_STRING,    /* Disc */
			G_TYPE_STRING,    /* Comment */
			G_TYPE_STRING,    /* Time */
			G_TYPE_INT,       /* Song id */
			G_TYPE_INT,       /* Bold */
			G_TYPE_BOOLEAN,    /* Bold set? */
			G_TYPE_STRING
			);

	lib->priv->fil_results = gtk_tree_model_filter_new (GTK_TREE_MODEL (lib->priv->lst_results), NULL);
	gtk_tree_model_filter_set_visible_func (GTK_TREE_MODEL_FILTER (lib->priv->fil_results), 
			(GtkTreeModelFilterVisibleFunc) gbemol_library_is_visible, lib, NULL);
	g_object_unref (G_OBJECT (lib->priv->lst_results));

	lib->priv->tvw_results = gtk_tree_view_new_with_model (GTK_TREE_MODEL (lib->priv->fil_results));
	g_object_unref (G_OBJECT (lib->priv->fil_results));

	gtk_tree_view_set_search_column (GTK_TREE_VIEW (lib->priv->tvw_results), COLUMN_RESULTS_TITLE);
	lib->priv->sel_results = gtk_tree_view_get_selection (GTK_TREE_VIEW (lib->priv->tvw_results));
	gtk_tree_selection_set_mode (lib->priv->sel_results, GTK_SELECTION_MULTIPLE);
	scr = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scr), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (scr), GTK_SHADOW_ETCHED_IN);
	gtk_container_add (GTK_CONTAINER (scr), lib->priv->tvw_results);

	lib->priv->lst_column_search = gtk_list_store_new (2, GTK_TYPE_STRING, G_TYPE_INT);
	for (i = 0; i < 10; i++) {
		renderer = gtk_cell_renderer_text_new ();
		column = gtk_tree_view_column_new_with_attributes (column_results[i], renderer,
				"text", i,
				"weight", COLUMN_RESULTS_BOLD,
				"weight-set", COLUMN_RESULTS_BOLD_SET,
				NULL);
		gtk_tree_view_append_column (GTK_TREE_VIEW (lib->priv->tvw_results), column);
		gtk_tree_view_column_set_resizable (column, TRUE);
		gtk_tree_view_column_set_reorderable (column, TRUE);
		gtk_tree_view_column_set_clickable (column, TRUE);
		g_object_set_data (G_OBJECT (column), "id", GINT_TO_POINTER (i));
		g_signal_connect (G_OBJECT (column), "clicked", G_CALLBACK (on_column_clicked), lib);

		str = g_strdup_printf ("column%d", i);
		gtk_tree_view_column_set_visible (column, gbemol_cfg_get_int ("library", str));
		g_free (str);

		/* The search combo box */
		gtk_list_store_append (lib->priv->lst_column_search, &iter);
		gtk_list_store_set (lib->priv->lst_column_search, &iter,
				0, column_results[i],
				1, i,
				-1);
	}

	gtk_combo_box_set_model (GTK_COMBO_BOX (lib->priv->cmb_search), GTK_TREE_MODEL (lib->priv->lst_column_search));
	gtk_combo_box_set_active (GTK_COMBO_BOX (lib->priv->cmb_search), gbemol_cfg_get_int ("library", "cmb_search"));

	renderer = gtk_cell_renderer_text_new ();
	gtk_cell_layout_pack_start (GTK_CELL_LAYOUT (lib->priv->cmb_search), renderer, TRUE);
	gtk_cell_layout_set_attributes (GTK_CELL_LAYOUT (lib->priv->cmb_search), renderer,
				"text", 0,
				NULL);
	g_object_unref (G_OBJECT (lib->priv->lst_column_search));

	gtk_tree_view_set_rules_hint (GTK_TREE_VIEW (lib->priv->tvw_results), TRUE);
	gtk_box_pack_start (GTK_BOX (vbox), scr, TRUE, TRUE, 0);

	hbox = gtk_hbox_new (FALSE, 5);

	/* Update DB button */
	btn = gbemol_utils_make_button_and_label (_("_Update Library"), "gtk-refresh", &(lib->priv->lbl_update));
	gtk_tooltips_set_tip (lib->priv->tips, btn,
		       _("Update the MPD library"), NULL);
	gtk_box_pack_start (GTK_BOX (hbox), btn, FALSE, FALSE, 0);
	g_signal_connect (G_OBJECT (btn), "clicked", G_CALLBACK (on_btn_update_clicked), lib);
	g_signal_connect (G_OBJECT (btn), "activate", G_CALLBACK (on_btn_update_clicked), lib);
	lib->priv->btn_update = btn;

	/* Play button */
	lib->priv->btn_play = gbemol_utils_make_button (_("_Play"), GTK_STOCK_MEDIA_PLAY);
	gtk_tooltips_set_tip (lib->priv->tips, lib->priv->btn_play,
		       _("Replace current playlist with selected items"), NULL);
	gtk_box_pack_end (GTK_BOX (hbox), lib->priv->btn_play, FALSE, FALSE, 0);
	g_signal_connect (G_OBJECT (lib->priv->btn_play), "clicked", G_CALLBACK (on_btn_play_clicked), lib);
	g_signal_connect (G_OBJECT (lib->priv->btn_play), "activate", G_CALLBACK (on_btn_play_clicked), lib);

	/* Enqueue button */
	lib->priv->btn_enqueue = gbemol_utils_make_button (_("_Enqueue"), "gtk-add");
	gtk_tooltips_set_tip (lib->priv->tips, lib->priv->btn_enqueue,
		       _("Add selected items to current playlist"), NULL);
	gtk_box_pack_end (GTK_BOX (hbox), lib->priv->btn_enqueue, FALSE, FALSE, 0);
	g_signal_connect (G_OBJECT (lib->priv->btn_enqueue), "clicked", G_CALLBACK (on_btn_enqueue_clicked), lib);
	g_signal_connect (G_OBJECT (lib->priv->btn_enqueue), "activate", G_CALLBACK (on_btn_enqueue_clicked), lib);

	/* Enqueue all button */
	lib->priv->btn_enqueue_all = gbemol_utils_make_button (_("Enqueue A_ll"), "gtk-add");
	gtk_tooltips_set_tip (lib->priv->tips, lib->priv->btn_enqueue_all,
		       _("Add all songs to current playlist"), NULL);
	gtk_box_pack_end (GTK_BOX (hbox), lib->priv->btn_enqueue_all, FALSE, FALSE, 0);
	g_signal_connect (G_OBJECT (lib->priv->btn_enqueue_all), "clicked", G_CALLBACK (on_btn_enqueue_all_clicked), lib);
	g_signal_connect (G_OBJECT (lib->priv->btn_enqueue_all), "activate", G_CALLBACK (on_btn_enqueue_all_clicked), lib);

	gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);

	gtk_paned_pack2 (GTK_PANED (vpnd), vbox, TRUE, FALSE); 

	gbemol_library_build_menus (lib);

	g_signal_connect (G_OBJECT (lib->priv->tvw_results), "button-press-event", 
			G_CALLBACK (on_tvw_results_button_press_event), lib);

	
	gtk_widget_show_all (GTK_WIDGET (lib));


	return lib;
}

void
gbemol_library_start_constraints (GbemolLibrary* lib)
{
	int i;

	for (i = 0; i < 3; i++)
		gbemol_library_update_constraint (lib, i);
}

void gbemol_library_columns_dialog (GbemolLibrary* lib)
{
	GtkWidget *dlg, *vbox, *chk[N_C_RESULTS];
	GtkTreeViewColumn* column;
	gchar *str;
	int i;

	dlg = gtk_dialog_new_with_buttons (_("Results Columns"), 
			NULL,
			GTK_DIALOG_DESTROY_WITH_PARENT,
			GTK_STOCK_OK,
			GTK_RESPONSE_ACCEPT,
			GTK_STOCK_CANCEL,
			GTK_RESPONSE_REJECT,
			NULL);

	vbox = gtk_vbox_new (TRUE, 5);

	for (i = 0; i < N_C_RESULTS; i++)
	{
		column = gtk_tree_view_get_column (GTK_TREE_VIEW (lib->priv->tvw_results), i);
		chk[i] = gtk_check_button_new_with_label (column_results[i]);
		gtk_box_pack_start (GTK_BOX (vbox), chk[i], FALSE, FALSE, 0);
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(chk[i]), gtk_tree_view_column_get_visible (column));
	}

	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dlg)->vbox), vbox, TRUE, TRUE, 5);
	gtk_widget_show_all (vbox);

	if (gtk_dialog_run (GTK_DIALOG (dlg)) == GTK_RESPONSE_ACCEPT)
	{
		for (i = 0; i < N_C_RESULTS; i++)
		{
			column = gtk_tree_view_get_column (GTK_TREE_VIEW (lib->priv->tvw_results), i);
			gtk_tree_view_column_set_visible (column, 
					gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(chk[i])));
			str = g_strdup_printf ("column%d", i);
			gbemol_cfg_set_int ("library", str, 
					gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (chk[i])));
			g_free (str);

		}
	}

	gtk_widget_destroy (dlg);
}


/*
 * Populates lst with the tags
 */
static void 
gbemol_library_populate_constraint (GbemolLibrary* lib, int id, GList* tags)
{
	GtkTreeIter iter_all, iter;	
	GtkListStore* lst = lib->priv->lst_constraint[id];

	gtk_list_store_clear (lst);
	
	gtk_list_store_append (lst, &iter_all);
	gtk_list_store_set (lst, &iter_all,
			COLUMN_CONSTRAINT_VALUE, _("<All>"),
			COLUMN_CONSTRAINT_ALL, TRUE,
			COLUMN_CONSTRAINT_BOLD, PANGO_WEIGHT_ULTRABOLD,
			COLUMN_CONSTRAINT_BOLD_SET, TRUE,
			-1);
	if (tags)
	{
		do
		{
			gtk_list_store_append (lst, &iter);
			gtk_list_store_set (lst, &iter,
					COLUMN_CONSTRAINT_VALUE, (gchar *)tags->data,
					COLUMN_CONSTRAINT_ALL, FALSE,
					-1);
		} while ((tags = g_list_next (tags)));
	}

}

void 
gbemol_library_set_sensitive (GbemolLibrary* lib, gboolean sensitive)
{
	gint i;

	for (i = 0; i < N_CONSTRAINTS; i++)
	{
		gtk_widget_set_sensitive (lib->priv->cmb_constraint[i], sensitive);
		gtk_widget_set_sensitive (lib->priv->tvw_constraint[i], sensitive);
		if (!sensitive)
			gtk_list_store_clear (lib->priv->lst_constraint[i]);
	}

	gtk_widget_set_sensitive (lib->priv->tvw_results, sensitive);

	if (!sensitive)
		gtk_list_store_clear (lib->priv->lst_results);
	else
		gbemol_library_start_constraints (lib);

	gtk_widget_set_sensitive (lib->priv->ent_search, sensitive);
	gtk_widget_set_sensitive (lib->priv->cmb_search, sensitive);
	gtk_widget_set_sensitive (lib->priv->btn_update, sensitive);
	gtk_widget_set_sensitive (lib->priv->btn_enqueue, sensitive);
	gtk_widget_set_sensitive (lib->priv->btn_enqueue_all, sensitive);
	gtk_widget_set_sensitive (lib->priv->btn_play, sensitive);
}

static void 
gbemol_library_populate_results (GbemolLibrary* lib, GList* songs)
{
	GtkTreeIter iter;
	GbemolMpdSong* song;
	gchar *time, *title;
	gint t_time = 0, t_songs = 0;  /* stats */
	GtkListStore* lst = lib->priv->lst_results;

	gtk_list_store_clear (lst);

	if (songs)
		do
		{
			song = songs->data;
			if (song->time != MPD_SONG_NO_TIME)
			{
				time = g_strdup_printf ("%02d:%02d", 
						song->time / 60, song->time % 60);
				t_time += song->time;
			}
			else
				time = NULL;

			if (song->name != NULL) /* Its a stream */
				title = song->name;
			else
				title = song->title;

			gtk_list_store_append (lst, &iter);			
			gtk_list_store_set (lst, &iter,
					COLUMN_RESULTS_ARTIST, song->artist,
					COLUMN_RESULTS_TITLE, title,
					COLUMN_RESULTS_ALBUM, song->album,
					COLUMN_RESULTS_TRACK, song->track,
					COLUMN_RESULTS_DATE, song->date,
					COLUMN_RESULTS_GENRE, song->genre,
					COLUMN_RESULTS_COMPOSER, song->composer,
					COLUMN_RESULTS_DISC, song->disc,
					COLUMN_RESULTS_COMMENT, song->comment,
					COLUMN_RESULTS_TIME, time,
					COLUMN_RESULTS_FILE, song->file,
					COLUMN_RESULTS_ID, song->id,
					-1);
			g_free (time);
			t_songs++;
			
		} while ((songs = g_list_next (songs)));
	gtk_tree_view_columns_autosize (GTK_TREE_VIEW (lib->priv->tvw_results));

	/* Less than 1 hour */
	if (t_time < 3600)
		time = g_strdup_printf ("%02d:%02d", t_time/60, t_time%60);
	/* Less than 1 day */
	else if (t_time < (3600 * 24))
		time = g_strdup_printf ("%d hours, %d minutes and %d seconds",
				t_time/3600, (t_time%3600)/60, (t_time%3600)%60);
	else 
		time = g_strdup_printf ("%d days, %d hours, %d minutes and %d seconds", 
				t_time/(3600*24), (t_time%(3600*24))/3600, (t_time%3600)/60, (t_time%3600)%60);

	g_free (time);
}

/*
 *  Update the constraint number index
 */
void 
gbemol_library_update_constraint (GbemolLibrary* lib, gint index)
{
	gint i = 0, key[3];
	gchar *str[3] = {NULL, NULL, NULL};
	gboolean all[3];
	GList* l, *tags;
	GtkTreeIter iter;

	/* Get the values of the constraints */
	do
	{
		/* The tag key */
		if (gtk_combo_box_get_active_iter (GTK_COMBO_BOX (lib->priv->cmb_constraint[i]), &iter))
			gtk_tree_model_get (gtk_combo_box_get_model(GTK_COMBO_BOX (lib->priv->cmb_constraint[i])),
					&iter, CMB_TAG_KEY, &key[i], -1);
		else
			key[i] = -1;

		/* The value */
		l = gbemol_utils_get_selected (GTK_TREE_VIEW (lib->priv->tvw_constraint[i]));
		if (!l)
		{
			str[i] = NULL;
			all[i] = FALSE;
		}
		else
		{
			gtk_tree_model_get_iter (GTK_TREE_MODEL (lib->priv->lst_constraint[i]), &iter, 
					(GtkTreePath *) l->data);
			gtk_tree_model_get (GTK_TREE_MODEL (lib->priv->lst_constraint[i]), &iter, 
					COLUMN_CONSTRAINT_VALUE, &str[i],
					COLUMN_CONSTRAINT_ALL, &all[i], -1);
			gbemol_utils_selected_list_free (l);
		}

		i++;
	} while (i <= index);

	/* None is selected */
	if (key[index] == -1)
	{
		gtk_list_store_clear (lib->priv->lst_constraint[index]);
		return;
	}

	/* Get the values for the selected constraint */
	gbemol_mpd_search_field_start (lib->priv->mpd, key[index]);
	for (i = 0; i < index; i++)
	{
		if (!all[i])
			gbemol_mpd_search_add_constraint (lib->priv->mpd, key[i], str[i]);
		g_free (str[i]);
	}

	tags = gbemol_mpd_search_field_get_results (lib->priv->mpd, key[index]);
	gbemol_library_populate_constraint (lib, index, tags);
	gtk_tree_view_columns_autosize (GTK_TREE_VIEW (lib->priv->tvw_constraint[index]));
	gbemol_utils_char_list_free (tags);
}

void 
gbemol_library_resolve_tags (GbemolLibrary *lib)
{
	gint key[3], i;
	gchar *str[3];
	gboolean all[3];
	GtkTreeIter iter;
	GList *l, *songs;

	/* Get the values from each constraint */
	for (i = 0; i < 3; i++)
	{
		/* The tag key */
		if (gtk_combo_box_get_active_iter (GTK_COMBO_BOX (lib->priv->cmb_constraint[i]), &iter))
			gtk_tree_model_get (gtk_combo_box_get_model (GTK_COMBO_BOX (lib->priv->cmb_constraint[i])),
					&iter, CMB_TAG_KEY, &key[i], -1);
		else
			key[i] = -1;

		/* The value */
		l = gbemol_utils_get_selected (GTK_TREE_VIEW (lib->priv->tvw_constraint[i]));
		if (!l)
		{
			str[i] = NULL;
			all[i] = FALSE;
		}
		else
		{
			gtk_tree_model_get_iter (GTK_TREE_MODEL (lib->priv->lst_constraint[i]), &iter, 
					(GtkTreePath *) l->data);
			gtk_tree_model_get (GTK_TREE_MODEL (lib->priv->lst_constraint[i]), &iter, 
					COLUMN_CONSTRAINT_VALUE, &str[i],
					COLUMN_CONSTRAINT_ALL, &all[i], -1);
			gbemol_utils_selected_list_free (l);
		}
	}

	/* Specific case to show all songs */
	if (all[0] && all[1] && all[2])
		songs = gbemol_mpd_database_get_all_songs (lib->priv->mpd);
	else
	{
		gbemol_mpd_search_start (lib->priv->mpd);

		for (i = 0; i < 3; i++)
		{
			if ((key[i] != -1) && (str[i] != NULL) && (!all[i]))
				gbemol_mpd_search_add_constraint (lib->priv->mpd, key[i], str[i]);
		}
		songs = gbemol_mpd_search_get_results (lib->priv->mpd);
	}

	gbemol_library_populate_results (lib, songs);

	gbemol_utils_song_list_free (songs);
}

/*
 * Add the selected songs in the results to the playlist
 * returns the id of the first song added
 */
static gint 
gbemol_library_add_selected_to_playlist (GbemolLibrary* lib)
{
	GList *l;
	GtkTreeIter iter;
	gint i;
	gchar *str;

	l = gbemol_utils_get_selected (GTK_TREE_VIEW (lib->priv->tvw_results));

	if (!l)
		return -1;

	i = 0;
	gbemol_mpd_queue_start (lib->priv->mpd);	

	do
	{
		gtk_tree_model_get_iter (GTK_TREE_MODEL (lib->priv->fil_results), &iter, (GtkTreePath *) l->data);
		gtk_tree_model_get (GTK_TREE_MODEL (lib->priv->fil_results), &iter, 
				COLUMN_RESULTS_FILE, &str, -1);
		gbemol_mpd_queue_add_song (lib->priv->mpd, str);
		g_free (str);
		i++;
		
	} while ((l = g_list_next (l)));

	gbemol_mpd_queue_finish (lib->priv->mpd);

	gbemol_utils_selected_list_free (l);

	return 0;
}

void
gbemol_library_set_updating (GbemolLibrary* lib, gboolean updating)
{
	gboolean sensitive;

	g_object_get (G_OBJECT (lib->priv->btn_update), "sensitive", &sensitive, NULL);

	if (updating)
	{
		if (sensitive)
		{
			gtk_label_set_text (GTK_LABEL (lib->priv->lbl_update), _("Updating Library..."));
			gtk_widget_set_sensitive (lib->priv->btn_update, FALSE);
		}
	}
	else
	{
		if (!sensitive)
		{			
			gtk_label_set_text_with_mnemonic (GTK_LABEL (lib->priv->lbl_update), _("_Update Library"));
			gtk_widget_set_sensitive (lib->priv->btn_update, TRUE);
		}
	}
}


static void 
on_constraint_changed (GtkWidget *widget, GbemolLibrary* lib)
{
	gint index = GPOINTER_TO_INT (g_object_get_data (G_OBJECT (widget), "index"));

	if (GTK_IS_COMBO_BOX (widget))
		gbemol_library_update_constraint (lib, index);
	else
	{
		if (index == 0)
		{
			gbemol_library_update_constraint (lib, 1);
			gbemol_library_update_constraint (lib, 2);
		}
		else if (index == 1)
			gbemol_library_update_constraint (lib, 2);
		gbemol_library_resolve_tags (lib);
	}
}

static void 
on_btn_enqueue_clicked (GtkWidget* b, GbemolLibrary* lib)
{
	gbemol_library_add_selected_to_playlist (lib);
	
}

static void 
on_btn_enqueue_all_clicked (GtkWidget* b, GbemolLibrary* lib)
{
	gtk_tree_selection_select_all (lib->priv->sel_results);
	gbemol_library_add_selected_to_playlist (lib);
}

static void 
on_btn_play_clicked (GtkWidget* b, GbemolLibrary* lib)
{
	gbemol_mpd_playlist_clear (lib->priv->mpd);
	gbemol_library_add_selected_to_playlist (lib);

	/* Auto-play first song */
	gbemol_mpd_player_play_song_by_pos (lib->priv->mpd, 0);
}
				
static void 
on_btn_update_clicked (GtkWidget* b, GbemolLibrary* lib)
{
	gbemol_mpd_database_update (lib->priv->mpd, "/");
}

static void 
on_mnu_select_all_activate (GtkWidget* b, GbemolLibrary* lib)
{
	gtk_tree_selection_select_all (lib->priv->sel_results);
}

static void 
on_mnu_column_activate (GtkWidget* b, GbemolLibrary* lib)
{
	gbemol_library_columns_dialog (lib);
	gtk_tree_view_columns_autosize (GTK_TREE_VIEW (lib->priv->tvw_results));
}
	
static gboolean 
on_tvw_results_button_press_event (GtkWidget *w, GdkEventButton *event, GbemolLibrary* lib)
{
	gtk_tree_model_filter_refilter (GTK_TREE_MODEL_FILTER (lib->priv->fil_results));

	if (event->type == GDK_BUTTON_PRESS)
		if (event->button == 3)
		{
			gtk_menu_popup (GTK_MENU(gtk_ui_manager_get_widget (lib->priv->manager, "/ResultsMenu")),
					NULL, NULL, NULL, NULL, 
					event->button, event->time);
			return TRUE;
		}
	return FALSE;
}

static void on_column_clicked (GtkTreeViewColumn* column, GbemolLibrary* lib)
{
	int old_id, new_id;
	GtkSortType old_sort, new_sort;
	GtkTreeViewColumn* c;

	new_sort = GTK_SORT_ASCENDING;
	new_id = GPOINTER_TO_INT (g_object_get_data (G_OBJECT (column), "id"));

	gtk_tree_sortable_get_sort_column_id (GTK_TREE_SORTABLE (lib->priv->lst_results), &old_id, &old_sort);

	/* Same column, reverse order */
	if (old_id == new_id)
		if (old_sort == GTK_SORT_ASCENDING)
			new_sort = GTK_SORT_DESCENDING;

	gtk_tree_sortable_set_sort_column_id (GTK_TREE_SORTABLE (lib->priv->lst_results), new_id, new_sort);

	c = gtk_tree_view_get_column (GTK_TREE_VIEW (lib->priv->tvw_results), old_id);
	gtk_tree_view_column_set_sort_indicator (c, FALSE);
	gtk_tree_view_column_set_sort_indicator (column, TRUE);
	gtk_tree_view_column_set_sort_order (column, new_sort);

}

static void on_ent_search_changed (GtkEditable* entry, GbemolLibrary* lib)
{
	gtk_tree_model_filter_refilter (GTK_TREE_MODEL_FILTER (lib->priv->fil_results));
}

static gboolean
gbemol_library_is_visible (GtkTreeModel* model, GtkTreeIter *iter, GbemolLibrary* lib)
{
	gchar *str1, *list_str = NULL, *search_str = NULL;
	const gchar *str2, *str3;
	GtkTreeIter cmb_iter;
	gint column;
	gboolean result = TRUE;

	if (!gtk_combo_box_get_active_iter (GTK_COMBO_BOX (lib->priv->cmb_search), &cmb_iter))
		return TRUE;

	/* Get the string in the list */
	gtk_tree_model_get (GTK_TREE_MODEL (lib->priv->lst_column_search), &cmb_iter, 1, &column, -1);
	gtk_tree_model_get (model, iter, column, &str1, -1);
	
	/* Get the search box string */
	str2 = gtk_entry_get_text (GTK_ENTRY (lib->priv->ent_search));

	/* Case sensitive search? */
	if (!gbemol_cfg_get_int ("library", "case-sensitive"))
	{
		if (str1)
			list_str = g_utf8_casefold (str1, -1);
		if (str2)
			search_str = g_utf8_casefold (str2, -1);
	}
	else
	{
		list_str = str1;
		search_str = g_strdup (str2);
	}

	/* Const strs for comparing */
	str2 = list_str;
	str3 = search_str;

	if (str2 && str3)
		if (!g_strrstr (list_str, search_str))
			result = FALSE;

	g_free (str1);
	g_free (search_str);

	return result;

}
			
static void
on_btn_clear_search_activate (GtkButton* btn, GbemolLibrary *lib)
{
	gtk_entry_set_text (GTK_ENTRY (lib->priv->ent_search), "");
/*	on_btn_search_activate (GTK_BUTTON (pls->priv->btn_search), pls); */
}

