/*---------------------------------------------------------------------------*
 |              PDFlib - A library for generating PDF on the fly             |
 +---------------------------------------------------------------------------+
 | Copyright (c) 1997-2002 PDFlib GmbH and Thomas Merz. All rights reserved. |
 +---------------------------------------------------------------------------+
 |    This software is NOT in the public domain.  It can be used under two   |
 |    substantially different licensing terms:                               |
 |                                                                           |
 |    The commercial license is available for a fee, and allows you to       |
 |    - ship a commercial product based on PDFlib                            |
 |    - implement commercial Web services with PDFlib                        |
 |    - distribute (free or commercial) software when the source code is     |
 |      not made available                                                   |
 |    Details can be found in the file PDFlib-license.pdf.                   |
 |                                                                           |
 |    The "Aladdin Free Public License" doesn't require any license fee,     |
 |    and allows you to                                                      |
 |    - develop and distribute PDFlib-based software for which the complete  |
 |      source code is made available                                        |
 |    - redistribute PDFlib non-commercially under certain conditions        |
 |    - redistribute PDFlib on digital media for a fee if the complete       |
 |      contents of the media are freely redistributable                     |
 |    Details can be found in the file aladdin-license.pdf.                  |
 |                                                                           |
 |    These conditions extend to ports to other programming languages.       |
 |    PDFlib is distributed with no warranty of any kind. Commercial users,  |
 |    however, will receive warranty and support statements in writing.      |
 *---------------------------------------------------------------------------*/

/* compile_metrics.c
 *
 * Generate C header file with compiled font metrics from AFM, PFM, or TrueType
 *
 */

/* $Id: compile_metrics.c,v 1.13.2.1 2002/01/07 18:26:30 tm Exp $ */

#include <string.h>
#include <stdlib.h>
#include <ctype.h>

#if defined(__CYGWIN32__)
#include <getopt.h>
#elif defined(WIN32)
int getopt(int argc, char * const argv[], const char *optstring);
extern char *optarg;
extern int optind;
#elif !defined(MAC)
#include <unistd.h>
#endif

#include "p_intern.h"
#include "p_font.h"

void
add_entry(PDF *p, FILE *out, pdf_font font, pdf_encoding enc)
{
    int		i;

    fprintf(out, "{");
    fprintf(out, "\t\"%s\",\t\t\t/* FontName */\n", font.name);
    fprintf(out, "\t%s,\t\t\t/* encoding */\n",
	font.encoding == builtin ?
	    "builtin" :
	    p->encodings[font.encoding]->apiname);
    fprintf(out, "\t0,\t\t\t\t/* used on page */\n");
    fprintf(out, "\t0,\t\t\t\t/* embed */\n");
    fprintf(out, "\tNULL,\t\t\t\t/* font file */\n");
    fprintf(out, "\t0L,\t\t\t\t/* id */\n");
    fprintf(out, "\t%luL,\t\t\t/* flags */\n", font.flags);
    fprintf(out, "\tType1,\t\t\t\t/* font type */\n");

    if (font.ttname)
	fprintf(out, "\t\"%s\",\t\t\t\t/* TT name */\n", font.ttname);
    else
	fprintf(out, "\tNULL,\t\t\t\t/* TT name */\n");

    fprintf(out, "\t\"%s\",\n", font.encodingScheme);

    fprintf(out, "\tcc_none,\n");

    fprintf(out, "\t(float) %2.1f,\t\t\t/* ItalicAngle */\n", font.italicAngle);
    fprintf(out, "\t%d,\t\t\t\t/* isFixedPitch */\n", font.isFixedPitch);
    fprintf(out, "\t%d,\t\t\t\t/* llx */\n", (int) font.llx);
    fprintf(out, "\t%d,\t\t\t\t/* lly */\n", (int) font.lly);
    fprintf(out, "\t%d,\t\t\t\t/* urx */\n", (int) font.urx);
    fprintf(out, "\t%d,\t\t\t\t/* ury */\n", (int) font.ury);
    fprintf(out, "\t%d,\t\t\t\t/* UnderlinePosition */\n",
	font.underlinePosition);
    fprintf(out, "\t%d,\t\t\t\t/* UnderlineThickness */\n",
	font.underlineThickness);
    fprintf(out, "\t%d,\t\t\t\t/* CapHeight */\n", font.capHeight);
    fprintf(out, "\t%d,\t\t\t\t/* xHeight */\n", font.xHeight);
    fprintf(out, "\t%d,\t\t\t\t/* Ascender */\n", font.ascender);
    fprintf(out, "\t%d,\t\t\t\t/* Descender */\n", font.descender);
    fprintf(out, "\t%d,\t\t\t\t/* StdVW */\n", font.StdVW);
    fprintf(out, "\t%d,\t\t\t\t/* StdHW */\n", font.StdHW);
    fprintf(out, "\t0,\n");
    fprintf(out, "\t(CharMetricInfo *) NULL,\n");
    fprintf(out, "\t0,\n");
    fprintf(out, "\t(TrackKernData *) NULL,\n");
    fprintf(out, "\t0,\n");
    fprintf(out, "\t(PairKernData *) NULL,\n");
    fprintf(out, "\t0,\n");
    fprintf(out, "\t(CompCharData *) NULL,\n");

    fprintf(out, "\n{");

    for (i=0; i < 256; i++) {
	fprintf(out, "\t%3d", font.widths[i]);
	if (i == 255)
	    fprintf(out, " ");
	else
	    fprintf(out, ",");

	if (i % 8 == 7)
	    fprintf(out, " /* 0x%02x */\n", i);
    }

    fprintf(out, "}\n");
    fprintf(out, "},\n");

    (void) pdf_cleanup_font_struct(p, &font);
}

void
usage(void)
{
    fprintf(stderr,
"compile_metrics: Compile AFM, PFM, or TrueType data to C code for PDFlib\n");
    fprintf(stderr, 
"Usage: compile_metrics -h -e [winansi|macroman|ebcdic] [-o outfile] files...\n"
);
    exit(1);
}

int
main(int argc, char *argv[])
{
    int   opt;
    char *filename, *outfilename = NULL;
    int enc;
    char *encname;
    FILE *out = stdout;
    PDF  *p;
    pdf_font font;
    size_t len;

    /* This is only a dummy to provide a PDF* for the auxiliary functions */
    if ((p = PDF_new()) == NULL) {
	fprintf(stderr, "Couldn't generate PDF object - aborting\n");
	exit(99);
    }

    PDF_open_file(p, "");

    /* By default, generate compiled metrics for the current platform.
     * This may also be changed since the encoding is recorded in the
     * generated C structs.
     */
#ifdef EBCDIC
    enc = ebcdic;
    encname = "ebcdic";
#elif MAC
    enc = macroman;
    encname = "macroman";
#else
    enc = winansi;
    encname = "winansi";
#endif

    while ((opt = getopt(argc, argv, "e:ho:")) != -1)
	switch (opt) {
	    case 'e':
		if (!strcmp(optarg, "builtin")) {
		    enc = builtin;
		    encname = "builtin";
		} else {
		    for (enc = 0; p->encodings[enc] != NULL; enc++) {
			if (p->encodings[enc]->apiname &&
			    !strcmp(optarg, p->encodings[enc]->apiname)) {
			    encname = p->encodings[enc]->apiname;
			    break;
			}
		    }
		    if (p->encodings[enc] == NULL) {
			fprintf(stderr, "Encoding %s not supported!\n", optarg);
			exit(88);
		    }
		}
		break;

	    case 'h':
		usage();
		break;

	    case 'o':
	    	outfilename = optarg;

		if (!strcmp(outfilename, "-")) {
		    out = stdout;
		} else if ((out = fopen(outfilename, "w")) == NULL) {
		    fprintf(stderr, "Couldn't open output file %s!\n",
		    	outfilename);
		    exit(99);
		}

		break;

	    default:
	    	break;
	}

    if (optind < argc) {
	fprintf(out, "\n");
    } else
	usage();

    while (optind < argc) {
	filename = argv[optind++];
	fprintf(stderr, "Reading %s, encoding = %s\n", filename, encname);

	len = strlen(filename);

	if (len >= 5 && (!strcmp(filename + len - 4, ".pfm") ||
	    !strcmp(filename + len - 4, ".PFM"))) {
	    /* parse PFM file */
	    if (!pdf_get_metrics_pfm(p, &font, NULL,  enc, filename)) {
		fprintf(stderr, "Error parsing PFM file '%s' - skipped!\n",
		filename);
	    } else {
		fprintf(out,
		"\n/* -------- Generated from %s -------- */\n",
		filename);
		add_entry(p, out, font, enc);
	    }
	} else if (len >= 5 && (!strcmp(filename + len - 4, ".afm") ||
	    !strcmp(filename + len - 4, ".AFM"))) {
	    /* parse AFM file */
	    if (!pdf_get_metrics_afm(p, &font, NULL,  enc, filename)) {
		fprintf(stderr, "Error parsing AFM file '%s' - skipped!\n",
		filename);
	    } else {
		fprintf(out,
		"\n/* -------- Generated from %s -------- */\n",
		filename);
		add_entry(p, out, font, enc);
	    }
	} else if (len >= 5 && (!strcmp(filename + len - 4, ".ttf") ||
	    !strcmp(filename + len - 4, ".TTF"))) {
	    /* parse TrueType file */
	    if (!pdf_get_metrics_tt(p, &font, NULL,  enc, filename, 0)) {
	    fprintf(stderr, "Error parsing TrueType file '%s' - skipped!\n",
		filename);
	    } else {
		fprintf(out,
		"\n/* -------- Generated from %s -------- */\n",
		filename);
		add_entry(p, out, font, enc);
	    }
	} else {
	    fprintf(stderr, "Unknown file name suffix in '%s' - skipped!\n",
		filename);
	}
    }

    fclose(out);
    exit(0);
}
