# frozen_string_literal: true

require 'spec_helper'

RSpec.describe('CharacterSet::Range parsing') do
  include_examples 'parse', '[a-z]',
    [0]       => [CharacterSet, count: 1],
    [0, 0]    => [CharacterSet::Range, count: 2],
    [0, 0, 0] => [:literal, text: 'a'],
    [0, 0, 1] => [:literal, text: 'z']

  include_examples 'parse', '[\x00-\x22]',
    [0]       => [CharacterSet, count: 1],
    [0, 0]    => [CharacterSet::Range, count: 2],
    [0, 0, 0] => [:hex, text: '\x00'],
    [0, 0, 1] => [:hex, text: '\x22']

  include_examples 'parse', '[\u{40 42}-\u1234]',
    [0]       => [CharacterSet, count: 1],
    [0, 0]    => [CharacterSet::Range, count: 2],
    [0, 0, 0] => [:codepoint_list, text: '\u{40 42}'],
    [0, 0, 1] => [:codepoint, text: '\u1234']

  include_examples 'parse', '[--z]',
    [0]       => [CharacterSet, count: 1],
    [0, 0]    => [CharacterSet::Range, count: 2],
    [0, 0, 0] => [:literal, text: '-'],
    [0, 0, 1] => [:literal, text: 'z']

  include_examples 'parse', '[!--]',
    [0]       => [CharacterSet, count: 1],
    [0, 0]    => [CharacterSet::Range, count: 2],
    [0, 0, 0] => [:literal, text: '!'],
    [0, 0, 1] => [:literal, text: '-']

  include_examples 'parse', '[!-^]',
    [0]       => [CharacterSet, count: 1],
    [0, 0]    => [CharacterSet::Range, count: 2],
    [0, 0, 0] => [:literal, text: '!'],
    [0, 0, 1] => [:literal, text: '^']

  # edge cases that are NOT treated as range

  include_examples 'parse', '[^-z]',
    [0]    => [CharacterSet, count: 2],
    [0, 0] => [:literal, text: '-'],
    [0, 1] => [:literal, text: 'z']

  include_examples 'parse', '[[\-ab]&&-bc]',
    [0]          => [CharacterSet, count: 1],
    [0, 0]       => [CharacterSet::Intersection, count: 2],
    [0, 0, 0]    => [CharacterSet::IntersectedSequence, count: 1],
    [0, 0, 0, 0] => [CharacterSet, count: 3],
    [0, 0, 1]    => [CharacterSet::IntersectedSequence, count: 3],
    [0, 0, 1, 0] => [:literal, text: '-']

  include_examples 'parse', '[bc-&&[\-ab]]',
    [0]          => [CharacterSet, count: 1],
    [0, 0]       => [CharacterSet::Intersection, count: 2],
    [0, 0, 0]    => [CharacterSet::IntersectedSequence, count: 3],
    [0, 0, 0, 2] => [:literal, text: '-'],
    [0, 0, 1]    => [CharacterSet::IntersectedSequence, count: 1],
    [0, 0, 1, 0] => [CharacterSet, count: 3]

  # Some edge-case patterns are evaluated with #match to make sure that
  # their matching behavior still reflects the way they are parsed.
  # #capturing_stderr is used to skip any warnings generated by this.
  specify('ranges behavior remains unchanged') do
    capturing_stderr do
      expect(Regexp.new('[\x00-\x22]')).to match "\x11"
      expect(Regexp.new('[\u{40 42}-\u1234]')).to match "\u0600"
      expect(Regexp.new('[--z]')).to match 'a'
      expect(Regexp.new('[!--]')).to match '$'
      expect(Regexp.new('[!-^]')).to match '$'

      # edge cases that are NOT treated as ranges
      expect(Regexp.new('[^-z]')).to     match 'a'
      expect(Regexp.new('[^-z]')).not_to match 'z'
      expect(Regexp.new('[[\-ab]&&-bc]')).to     match '-'
      expect(Regexp.new('[[\-ab]&&-bc]')).to     match 'b'
      expect(Regexp.new('[[\-ab]&&-bc]')).not_to match 'a'
      expect(Regexp.new('[[\-ab]&&-bc]')).not_to match 'c'
      expect(Regexp.new('[bc-&&[\-ab]]')).to     match '-'
      expect(Regexp.new('[bc-&&[\-ab]]')).to     match 'b'
      expect(Regexp.new('[bc-&&[\-ab]]')).not_to match 'a'
      expect(Regexp.new('[bc-&&[\-ab]]')).not_to match 'c'
    end
  end
end
