/*
 * main_data.cpp
 */
#include "main_data.h"
#include "misc_utils.h"

#include <stdbool.h>
#include <cstring>
#include <string>
#include <iomanip>
#include <iostream>
#include <glib.h>
#include <glib/gi18n.h>

#include <tag.h>
#include <fileref.h> 

#include "interface_common.h"
#include "main_window_handler.h"
#include "select_frame_handler.h"
#include "status_frame_handler.h"

#define stringToTString(s) 	 TagLib::String(s, TagLib::String::UTF8)

void MainData::Clear()
{
  total_length = 0;
  disc_artist = "";
  disc_title = "";
  disc_year = "";
  disc_category = "";
  track.clear();
}

void MainData::Dump()
{
  std::cout << "MainData dump, addr=" << (long int)this << std::endl;
  std::cout << "total_length = " << total_length << std::endl;
  std::cout << "disc_artist = " << disc_artist << std::endl;
  std::cout << "disc_title = " << disc_title << std::endl;
  std::cout << "disc_year = " << disc_year << std::endl;
  std::cout << "disc_category = " << disc_category << std::endl;
  for (int i=0; i<track.size(); ++i) {
    std::cout << "Track " << i << ":";    
    std::cout << " title=" << track[i].title;
    std::cout << " begin=" << track[i].begin;
    std::cout << " length=" << track[i].length;
    std::cout << " make_wav=" << track[i].make_wav;
    std::cout << " wav_exist=" << track[i].wav_exist;
    std::cout << " make_encoding=" << track[i].make_encoding;
    for (int j=0; j < track[i].encoding_files.size(); ++j)
        std::cout << " track type=" << track[i].encoding_files[j].type << " - encoded_file_exist=" << track[i].encoding_files[j].encoded_file_exist;
    std::cout << " artist=" << track[i].artist;
    std::cout << std::endl;
  }
}

/* 
 * This function finds what to do next. type is WAV or MP3.
 * It returns -1 when there's no job left, 0 when a job's found
 *
 * finds the next track from cur_track. If cur_type is wav, it will find the
 * next wav file to rip. If cur_type is mp3, it will find the next mp3 file
 * to encode 
 */
int MainData::find_next_job(
			int cur_track,
		    CommonEncoderType cur_type,
		    int *next_track,
		    CommonEncoderType *next_type) const
{
    int flag = 1, track, temp_next;
    CommonEncoderType temp_type, encoding_next_type;

    if (cur_type > WAV)
    {
		track = (cur_track < 0) ? 0 : cur_track;
    	while (track < this->num_tracks() && flag)
    	{
    		if (this->track[track].make_wav || this->track[track].make_encoding)
    			flag = 0;
    		else
    			track++;
    	}
    	if (flag)
    		return -1;

    	encoding_next_type = get_type_for_encoding_file();
		if (encoding_next_type > WAV)
        {
			*next_track = track;
            *next_type = encoding_next_type;
            return 0;
    	}
    	if (track < this->num_tracks() - 1)		// Don't get next file from last track, last file
    		encoding_next_type = get_type_for_encoding_file();
	}
	else
		encoding_next_type = WAV;

	flag = 1;
	track = cur_track + 1;

    // Find next track from cur_track
    while(track < this->num_tracks() && flag)
    {
        if(this->track[ track ].make_wav || this->track[ track ].make_encoding)
            flag = 0;
        else
            ++track;
    }

    if (flag)
        return -1;

    // For finding the next wav file, check to see if the file exists
    if (cur_type == WAV
        && this->track[ track ].make_encoding
        && this->track[ track ].wav_exist == true
        && config.encode_from_existing_wav == true)
    {
        if(find_next_job(track, WAV, &temp_next, &temp_type) < 0)
            return -1;

        *next_track = temp_next;
        *next_type = WAV;		// temp_type
    }
    else
    {
        *next_track = track;
        *next_type = encoding_next_type;
    }

    return 0;
};

// Called after all jobs are finshed. Files are cleaned up and renamed
void MainData::finish_job()
{
    using std::string;

    static string status_message;
    static string buffer;
    static string wav_file, enc_file;
    int i;
    bool madewavs = false;
    bool mademp3s = false;
	int track_files = -1;
    int filesdone = 0, tracksdone = 0;
    string artist;

    FILE *fp_playlist = NULL;
    string playlist_filespec;

    buffer = "";

    // Clean up
    for(i = 0; i < this->num_tracks(); i++)
    {
        create_file_names_for_track(this, i, wav_file, enc_file, WAV);

        if(this->track[ i ].wav_exist == false)
        {
            unlink(wav_file.c_str());
        }
        else if(this->track[ i ].make_wav == true)
        {
            madewavs = true;

            //write .inf files, for cdrecord - refactor this
            //should do this whenever we keep the .wavs?
            if(config.cddb_config.create_id3 == true)
            {
                string inf_file(wav_file);
	            inf_file.resize(inf_file.size() - 3);
	            inf_file += "inf";

	      		FILE * inf_fp = fopen(inf_file.c_str(), "w");

	      		if (inf_fp)
				{
                    if (artist.empty() || artist != this->track[i].artist)
		      			artist = this->disc_artist;

					fprintf(inf_fp,
			  			"Albumperformer= '%s'\n", artist.c_str());
					fprintf(inf_fp,
			  			"Performer= '%s'\n", this->track[ i ].artist.c_str());
					fprintf(inf_fp,
			  			"Albumtitle= '%s'\n", this->disc_title.c_str());
					fprintf(inf_fp,
			  			"Tracktitle= '%s'\n", this->track[ i ].title.c_str());
					fprintf(inf_fp,
			  			"Tracknumber= %d\n", i + 1);
					fclose(inf_fp);
				}
	      		else
		  			perror (inf_file.c_str());
	    	}
            buffer = buffer + std::to_string(++tracksdone) + ": " + file_name_without_path(wav_file) + "\n";
        }

        if (this->track[i].make_encoding == true)
        {
            mademp3s = true;

			for (int j=0; j < track[i].encoding_files.size(); ++j)
            {
				if (this->track[i].encoding_files[j].encoded_file_exist == false)
					continue;

				filesdone++;
				if (track_files != i && !this->track[i].make_wav)
				{
					tracksdone++;
					track_files = i;
				}

                // Add ID3 tag if requested
		        create_file_names_for_track(this, i, wav_file, enc_file, track[i].encoding_files[j].type);

                if (config.cddb_config.create_id3 == true)
                {
                    if (artist.empty() || artist != this->track[i].artist)
                        artist = this->disc_artist;

                    TagLib::FileRef file(enc_file.c_str());
				    TagLib::Tag *tag = file.tag();
				    if (tag)
		  		    {
		    			tag->setTitle(stringToTString(this->track[i].title));
		    			tag->setArtist(stringToTString(artist));
			    		tag->setAlbum(stringToTString(this->disc_title));
				    	if (!this->disc_year.empty())
				    		tag->setYear(std::stoi(this->disc_year));
					    tag->setTrack(i+1);
					    tag->setGenre(stringToTString(this->disc_category));
		    			file.save();
		  			}
            	}

	            //dc: strcat() is for sissies!
    	        buffer = buffer + std::to_string(tracksdone) + ": "
					+ file_name_without_path(enc_file) + "\n";

	            // tm: basic playlist support - thanks to Mark Tyler
    	        if(config.cddb_config.create_playlist == true && this->track[i].encoding_files[j].type == MP3)
        	    {
            	    if (fp_playlist == NULL)
                	{
                    	playlist_filespec = file_path_without_name(enc_file) + "/playlist.m3u";
	                    fp_playlist = fopen(playlist_filespec.c_str(), "w");
    	            }

        	        // If we succeeded above, we can now write to this
            	    if (fp_playlist != NULL)
                	    fprintf(fp_playlist, "%s\n", file_name_without_path(enc_file).c_str());
            	}
        	}
        }
    	this->track[ i ].make_wav = false;
        this->track[ i ].make_encoding = false;
    } // End loop over all tracks

    if ((config.cddb_config.create_playlist == true) && (fp_playlist != NULL))
        fclose(fp_playlist);

    // Generate status message
    status_message = string(_("Tracks Completed")) + ": " + std::to_string(tracksdone)
		+ "\n" + string(_("Files Encoded")) + ": " + std::to_string(filesdone)
		+ "\n\n" + _("Artist") + ": " + this->disc_artist + "\n"
		+ _("Album") + ": " + this->disc_title + "\n\n" + buffer;

    // Show status pop up
    if (madewavs)
        status_handler(STAT_FINISH_WAV, status_message.c_str());
    else if (mademp3s)
        status_handler(STAT_FINISH_MP3, status_message.c_str());

    // Clear status bar
    main_window_handler(MW_CLEAR_STATUSBAR, NULL, NULL);

    // Destroy status widget
    wm_status_frame_handler(WIDGET_DESTROY, WAV, NULL, NULL);

    // Create select frame
    select_frame_handler(WIDGET_CREATE, 0, this);
    main_window_handler(MW_MODE_SELECT, NULL, NULL);

    select_frame_handler(SF_SYNC_SELECT_FRAME_TO_DATA, 0, this);

#ifdef HAVE_LINUX_CDROM_H
    if (config.eject_disk_when_finished && madewavs)
		eject_disk();
#endif
}
