// SPDX-FileCopyrightText: 2022 deroad <wargio@libero.it>
// SPDX-License-Identifier: LGPL-3.0-only

#ifndef FNV_HASH_H
#define FNV_HASH_H

/** \file fnv_hash.h
 * This is a precomputed table of the following Fowler–Noll–Vo (FNV) hash
 * function with a base 64 limit.
 *
 * algorithm fnv_hash is
 *    hash := FNV_offset_basis
 *
 *    for each byte_of_data to be hashed do
 *        hash := hash × FNV_prime
 *        hash := hash XOR byte_of_data
 *
 *    return hash & 0x3f
 *
 * The FNV constants used for ssdeep are:
 *  - FNV_offset_basis -> 0x28021967
 *  - FNV_prime        -> 0x01000193
 *
 * C equivalent:
 *
 *   ut8  buffer[...] = { ... };
 *   ut32 FNV_offset_basis = 0x28021967;
 *   ut32 FNV_prime        = 0x01000193;
 *
 *   ut32 hash = FNV_offset_basis;
 *   for (i = 0; i < sizeof(buffer); i++) {
 *     ut32 extend = buffer[i];
 *     hash = (hash * FNV_prime) ^ extend;
 *   }
 *   printf("base64 index: 0x%02x", hash & 0x3f);
 *
 */

#define FNV_INIT 0x27

#define FNV_B64_00 \
	{ \
		0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, \
			0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, \
			0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, \
			0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, \
			0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, \
			0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f, \
			0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, \
			0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f, \
	}
#define FNV_B64_01 \
	{ \
		0x13, 0x12, 0x11, 0x10, 0x17, 0x16, 0x15, 0x14, \
			0x1b, 0x1a, 0x19, 0x18, 0x1f, 0x1e, 0x1d, 0x1c, \
			0x03, 0x02, 0x01, 0x00, 0x07, 0x06, 0x05, 0x04, \
			0x0b, 0x0a, 0x09, 0x08, 0x0f, 0x0e, 0x0d, 0x0c, \
			0x33, 0x32, 0x31, 0x30, 0x37, 0x36, 0x35, 0x34, \
			0x3b, 0x3a, 0x39, 0x38, 0x3f, 0x3e, 0x3d, 0x3c, \
			0x23, 0x22, 0x21, 0x20, 0x27, 0x26, 0x25, 0x24, \
			0x2b, 0x2a, 0x29, 0x28, 0x2f, 0x2e, 0x2d, 0x2c, \
	}
#define FNV_B64_02 \
	{ \
		0x26, 0x27, 0x24, 0x25, 0x22, 0x23, 0x20, 0x21, \
			0x2e, 0x2f, 0x2c, 0x2d, 0x2a, 0x2b, 0x28, 0x29, \
			0x36, 0x37, 0x34, 0x35, 0x32, 0x33, 0x30, 0x31, \
			0x3e, 0x3f, 0x3c, 0x3d, 0x3a, 0x3b, 0x38, 0x39, \
			0x06, 0x07, 0x04, 0x05, 0x02, 0x03, 0x00, 0x01, \
			0x0e, 0x0f, 0x0c, 0x0d, 0x0a, 0x0b, 0x08, 0x09, \
			0x16, 0x17, 0x14, 0x15, 0x12, 0x13, 0x10, 0x11, \
			0x1e, 0x1f, 0x1c, 0x1d, 0x1a, 0x1b, 0x18, 0x19, \
	}
#define FNV_B64_03 \
	{ \
		0x39, 0x38, 0x3b, 0x3a, 0x3d, 0x3c, 0x3f, 0x3e, \
			0x31, 0x30, 0x33, 0x32, 0x35, 0x34, 0x37, 0x36, \
			0x29, 0x28, 0x2b, 0x2a, 0x2d, 0x2c, 0x2f, 0x2e, \
			0x21, 0x20, 0x23, 0x22, 0x25, 0x24, 0x27, 0x26, \
			0x19, 0x18, 0x1b, 0x1a, 0x1d, 0x1c, 0x1f, 0x1e, \
			0x11, 0x10, 0x13, 0x12, 0x15, 0x14, 0x17, 0x16, \
			0x09, 0x08, 0x0b, 0x0a, 0x0d, 0x0c, 0x0f, 0x0e, \
			0x01, 0x00, 0x03, 0x02, 0x05, 0x04, 0x07, 0x06, \
	}
#define FNV_B64_04 \
	{ \
		0x0c, 0x0d, 0x0e, 0x0f, 0x08, 0x09, 0x0a, 0x0b, \
			0x04, 0x05, 0x06, 0x07, 0x00, 0x01, 0x02, 0x03, \
			0x1c, 0x1d, 0x1e, 0x1f, 0x18, 0x19, 0x1a, 0x1b, \
			0x14, 0x15, 0x16, 0x17, 0x10, 0x11, 0x12, 0x13, \
			0x2c, 0x2d, 0x2e, 0x2f, 0x28, 0x29, 0x2a, 0x2b, \
			0x24, 0x25, 0x26, 0x27, 0x20, 0x21, 0x22, 0x23, \
			0x3c, 0x3d, 0x3e, 0x3f, 0x38, 0x39, 0x3a, 0x3b, \
			0x34, 0x35, 0x36, 0x37, 0x30, 0x31, 0x32, 0x33, \
	}
#define FNV_B64_05 \
	{ \
		0x1f, 0x1e, 0x1d, 0x1c, 0x1b, 0x1a, 0x19, 0x18, \
			0x17, 0x16, 0x15, 0x14, 0x13, 0x12, 0x11, 0x10, \
			0x0f, 0x0e, 0x0d, 0x0c, 0x0b, 0x0a, 0x09, 0x08, \
			0x07, 0x06, 0x05, 0x04, 0x03, 0x02, 0x01, 0x00, \
			0x3f, 0x3e, 0x3d, 0x3c, 0x3b, 0x3a, 0x39, 0x38, \
			0x37, 0x36, 0x35, 0x34, 0x33, 0x32, 0x31, 0x30, \
			0x2f, 0x2e, 0x2d, 0x2c, 0x2b, 0x2a, 0x29, 0x28, \
			0x27, 0x26, 0x25, 0x24, 0x23, 0x22, 0x21, 0x20, \
	}
#define FNV_B64_06 \
	{ \
		0x32, 0x33, 0x30, 0x31, 0x36, 0x37, 0x34, 0x35, \
			0x3a, 0x3b, 0x38, 0x39, 0x3e, 0x3f, 0x3c, 0x3d, \
			0x22, 0x23, 0x20, 0x21, 0x26, 0x27, 0x24, 0x25, \
			0x2a, 0x2b, 0x28, 0x29, 0x2e, 0x2f, 0x2c, 0x2d, \
			0x12, 0x13, 0x10, 0x11, 0x16, 0x17, 0x14, 0x15, \
			0x1a, 0x1b, 0x18, 0x19, 0x1e, 0x1f, 0x1c, 0x1d, \
			0x02, 0x03, 0x00, 0x01, 0x06, 0x07, 0x04, 0x05, \
			0x0a, 0x0b, 0x08, 0x09, 0x0e, 0x0f, 0x0c, 0x0d, \
	}
#define FNV_B64_07 \
	{ \
		0x05, 0x04, 0x07, 0x06, 0x01, 0x00, 0x03, 0x02, \
			0x0d, 0x0c, 0x0f, 0x0e, 0x09, 0x08, 0x0b, 0x0a, \
			0x15, 0x14, 0x17, 0x16, 0x11, 0x10, 0x13, 0x12, \
			0x1d, 0x1c, 0x1f, 0x1e, 0x19, 0x18, 0x1b, 0x1a, \
			0x25, 0x24, 0x27, 0x26, 0x21, 0x20, 0x23, 0x22, \
			0x2d, 0x2c, 0x2f, 0x2e, 0x29, 0x28, 0x2b, 0x2a, \
			0x35, 0x34, 0x37, 0x36, 0x31, 0x30, 0x33, 0x32, \
			0x3d, 0x3c, 0x3f, 0x3e, 0x39, 0x38, 0x3b, 0x3a, \
	}
#define FNV_B64_08 \
	{ \
		0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, \
			0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, \
			0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, \
			0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, \
			0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f, \
			0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, \
			0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f, \
			0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, \
	}
#define FNV_B64_09 \
	{ \
		0x2b, 0x2a, 0x29, 0x28, 0x2f, 0x2e, 0x2d, 0x2c, \
			0x23, 0x22, 0x21, 0x20, 0x27, 0x26, 0x25, 0x24, \
			0x3b, 0x3a, 0x39, 0x38, 0x3f, 0x3e, 0x3d, 0x3c, \
			0x33, 0x32, 0x31, 0x30, 0x37, 0x36, 0x35, 0x34, \
			0x0b, 0x0a, 0x09, 0x08, 0x0f, 0x0e, 0x0d, 0x0c, \
			0x03, 0x02, 0x01, 0x00, 0x07, 0x06, 0x05, 0x04, \
			0x1b, 0x1a, 0x19, 0x18, 0x1f, 0x1e, 0x1d, 0x1c, \
			0x13, 0x12, 0x11, 0x10, 0x17, 0x16, 0x15, 0x14, \
	}
#define FNV_B64_0A \
	{ \
		0x3e, 0x3f, 0x3c, 0x3d, 0x3a, 0x3b, 0x38, 0x39, \
			0x36, 0x37, 0x34, 0x35, 0x32, 0x33, 0x30, 0x31, \
			0x2e, 0x2f, 0x2c, 0x2d, 0x2a, 0x2b, 0x28, 0x29, \
			0x26, 0x27, 0x24, 0x25, 0x22, 0x23, 0x20, 0x21, \
			0x1e, 0x1f, 0x1c, 0x1d, 0x1a, 0x1b, 0x18, 0x19, \
			0x16, 0x17, 0x14, 0x15, 0x12, 0x13, 0x10, 0x11, \
			0x0e, 0x0f, 0x0c, 0x0d, 0x0a, 0x0b, 0x08, 0x09, \
			0x06, 0x07, 0x04, 0x05, 0x02, 0x03, 0x00, 0x01, \
	}
#define FNV_B64_0B \
	{ \
		0x11, 0x10, 0x13, 0x12, 0x15, 0x14, 0x17, 0x16, \
			0x19, 0x18, 0x1b, 0x1a, 0x1d, 0x1c, 0x1f, 0x1e, \
			0x01, 0x00, 0x03, 0x02, 0x05, 0x04, 0x07, 0x06, \
			0x09, 0x08, 0x0b, 0x0a, 0x0d, 0x0c, 0x0f, 0x0e, \
			0x31, 0x30, 0x33, 0x32, 0x35, 0x34, 0x37, 0x36, \
			0x39, 0x38, 0x3b, 0x3a, 0x3d, 0x3c, 0x3f, 0x3e, \
			0x21, 0x20, 0x23, 0x22, 0x25, 0x24, 0x27, 0x26, \
			0x29, 0x28, 0x2b, 0x2a, 0x2d, 0x2c, 0x2f, 0x2e, \
	}
#define FNV_B64_0C \
	{ \
		0x24, 0x25, 0x26, 0x27, 0x20, 0x21, 0x22, 0x23, \
			0x2c, 0x2d, 0x2e, 0x2f, 0x28, 0x29, 0x2a, 0x2b, \
			0x34, 0x35, 0x36, 0x37, 0x30, 0x31, 0x32, 0x33, \
			0x3c, 0x3d, 0x3e, 0x3f, 0x38, 0x39, 0x3a, 0x3b, \
			0x04, 0x05, 0x06, 0x07, 0x00, 0x01, 0x02, 0x03, \
			0x0c, 0x0d, 0x0e, 0x0f, 0x08, 0x09, 0x0a, 0x0b, \
			0x14, 0x15, 0x16, 0x17, 0x10, 0x11, 0x12, 0x13, \
			0x1c, 0x1d, 0x1e, 0x1f, 0x18, 0x19, 0x1a, 0x1b, \
	}
#define FNV_B64_0D \
	{ \
		0x37, 0x36, 0x35, 0x34, 0x33, 0x32, 0x31, 0x30, \
			0x3f, 0x3e, 0x3d, 0x3c, 0x3b, 0x3a, 0x39, 0x38, \
			0x27, 0x26, 0x25, 0x24, 0x23, 0x22, 0x21, 0x20, \
			0x2f, 0x2e, 0x2d, 0x2c, 0x2b, 0x2a, 0x29, 0x28, \
			0x17, 0x16, 0x15, 0x14, 0x13, 0x12, 0x11, 0x10, \
			0x1f, 0x1e, 0x1d, 0x1c, 0x1b, 0x1a, 0x19, 0x18, \
			0x07, 0x06, 0x05, 0x04, 0x03, 0x02, 0x01, 0x00, \
			0x0f, 0x0e, 0x0d, 0x0c, 0x0b, 0x0a, 0x09, 0x08, \
	}
#define FNV_B64_0E \
	{ \
		0x0a, 0x0b, 0x08, 0x09, 0x0e, 0x0f, 0x0c, 0x0d, \
			0x02, 0x03, 0x00, 0x01, 0x06, 0x07, 0x04, 0x05, \
			0x1a, 0x1b, 0x18, 0x19, 0x1e, 0x1f, 0x1c, 0x1d, \
			0x12, 0x13, 0x10, 0x11, 0x16, 0x17, 0x14, 0x15, \
			0x2a, 0x2b, 0x28, 0x29, 0x2e, 0x2f, 0x2c, 0x2d, \
			0x22, 0x23, 0x20, 0x21, 0x26, 0x27, 0x24, 0x25, \
			0x3a, 0x3b, 0x38, 0x39, 0x3e, 0x3f, 0x3c, 0x3d, \
			0x32, 0x33, 0x30, 0x31, 0x36, 0x37, 0x34, 0x35, \
	}
#define FNV_B64_0F \
	{ \
		0x1d, 0x1c, 0x1f, 0x1e, 0x19, 0x18, 0x1b, 0x1a, \
			0x15, 0x14, 0x17, 0x16, 0x11, 0x10, 0x13, 0x12, \
			0x0d, 0x0c, 0x0f, 0x0e, 0x09, 0x08, 0x0b, 0x0a, \
			0x05, 0x04, 0x07, 0x06, 0x01, 0x00, 0x03, 0x02, \
			0x3d, 0x3c, 0x3f, 0x3e, 0x39, 0x38, 0x3b, 0x3a, \
			0x35, 0x34, 0x37, 0x36, 0x31, 0x30, 0x33, 0x32, \
			0x2d, 0x2c, 0x2f, 0x2e, 0x29, 0x28, 0x2b, 0x2a, \
			0x25, 0x24, 0x27, 0x26, 0x21, 0x20, 0x23, 0x22, \
	}
#define FNV_B64_10 \
	{ \
		0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, \
			0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f, \
			0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, \
			0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f, \
			0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, \
			0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, \
			0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, \
			0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, \
	}
#define FNV_B64_11 \
	{ \
		0x03, 0x02, 0x01, 0x00, 0x07, 0x06, 0x05, 0x04, \
			0x0b, 0x0a, 0x09, 0x08, 0x0f, 0x0e, 0x0d, 0x0c, \
			0x13, 0x12, 0x11, 0x10, 0x17, 0x16, 0x15, 0x14, \
			0x1b, 0x1a, 0x19, 0x18, 0x1f, 0x1e, 0x1d, 0x1c, \
			0x23, 0x22, 0x21, 0x20, 0x27, 0x26, 0x25, 0x24, \
			0x2b, 0x2a, 0x29, 0x28, 0x2f, 0x2e, 0x2d, 0x2c, \
			0x33, 0x32, 0x31, 0x30, 0x37, 0x36, 0x35, 0x34, \
			0x3b, 0x3a, 0x39, 0x38, 0x3f, 0x3e, 0x3d, 0x3c, \
	}
#define FNV_B64_12 \
	{ \
		0x16, 0x17, 0x14, 0x15, 0x12, 0x13, 0x10, 0x11, \
			0x1e, 0x1f, 0x1c, 0x1d, 0x1a, 0x1b, 0x18, 0x19, \
			0x06, 0x07, 0x04, 0x05, 0x02, 0x03, 0x00, 0x01, \
			0x0e, 0x0f, 0x0c, 0x0d, 0x0a, 0x0b, 0x08, 0x09, \
			0x36, 0x37, 0x34, 0x35, 0x32, 0x33, 0x30, 0x31, \
			0x3e, 0x3f, 0x3c, 0x3d, 0x3a, 0x3b, 0x38, 0x39, \
			0x26, 0x27, 0x24, 0x25, 0x22, 0x23, 0x20, 0x21, \
			0x2e, 0x2f, 0x2c, 0x2d, 0x2a, 0x2b, 0x28, 0x29, \
	}
#define FNV_B64_13 \
	{ \
		0x29, 0x28, 0x2b, 0x2a, 0x2d, 0x2c, 0x2f, 0x2e, \
			0x21, 0x20, 0x23, 0x22, 0x25, 0x24, 0x27, 0x26, \
			0x39, 0x38, 0x3b, 0x3a, 0x3d, 0x3c, 0x3f, 0x3e, \
			0x31, 0x30, 0x33, 0x32, 0x35, 0x34, 0x37, 0x36, \
			0x09, 0x08, 0x0b, 0x0a, 0x0d, 0x0c, 0x0f, 0x0e, \
			0x01, 0x00, 0x03, 0x02, 0x05, 0x04, 0x07, 0x06, \
			0x19, 0x18, 0x1b, 0x1a, 0x1d, 0x1c, 0x1f, 0x1e, \
			0x11, 0x10, 0x13, 0x12, 0x15, 0x14, 0x17, 0x16, \
	}
#define FNV_B64_14 \
	{ \
		0x3c, 0x3d, 0x3e, 0x3f, 0x38, 0x39, 0x3a, 0x3b, \
			0x34, 0x35, 0x36, 0x37, 0x30, 0x31, 0x32, 0x33, \
			0x2c, 0x2d, 0x2e, 0x2f, 0x28, 0x29, 0x2a, 0x2b, \
			0x24, 0x25, 0x26, 0x27, 0x20, 0x21, 0x22, 0x23, \
			0x1c, 0x1d, 0x1e, 0x1f, 0x18, 0x19, 0x1a, 0x1b, \
			0x14, 0x15, 0x16, 0x17, 0x10, 0x11, 0x12, 0x13, \
			0x0c, 0x0d, 0x0e, 0x0f, 0x08, 0x09, 0x0a, 0x0b, \
			0x04, 0x05, 0x06, 0x07, 0x00, 0x01, 0x02, 0x03, \
	}
#define FNV_B64_15 \
	{ \
		0x0f, 0x0e, 0x0d, 0x0c, 0x0b, 0x0a, 0x09, 0x08, \
			0x07, 0x06, 0x05, 0x04, 0x03, 0x02, 0x01, 0x00, \
			0x1f, 0x1e, 0x1d, 0x1c, 0x1b, 0x1a, 0x19, 0x18, \
			0x17, 0x16, 0x15, 0x14, 0x13, 0x12, 0x11, 0x10, \
			0x2f, 0x2e, 0x2d, 0x2c, 0x2b, 0x2a, 0x29, 0x28, \
			0x27, 0x26, 0x25, 0x24, 0x23, 0x22, 0x21, 0x20, \
			0x3f, 0x3e, 0x3d, 0x3c, 0x3b, 0x3a, 0x39, 0x38, \
			0x37, 0x36, 0x35, 0x34, 0x33, 0x32, 0x31, 0x30, \
	}
#define FNV_B64_16 \
	{ \
		0x22, 0x23, 0x20, 0x21, 0x26, 0x27, 0x24, 0x25, \
			0x2a, 0x2b, 0x28, 0x29, 0x2e, 0x2f, 0x2c, 0x2d, \
			0x32, 0x33, 0x30, 0x31, 0x36, 0x37, 0x34, 0x35, \
			0x3a, 0x3b, 0x38, 0x39, 0x3e, 0x3f, 0x3c, 0x3d, \
			0x02, 0x03, 0x00, 0x01, 0x06, 0x07, 0x04, 0x05, \
			0x0a, 0x0b, 0x08, 0x09, 0x0e, 0x0f, 0x0c, 0x0d, \
			0x12, 0x13, 0x10, 0x11, 0x16, 0x17, 0x14, 0x15, \
			0x1a, 0x1b, 0x18, 0x19, 0x1e, 0x1f, 0x1c, 0x1d, \
	}
#define FNV_B64_17 \
	{ \
		0x35, 0x34, 0x37, 0x36, 0x31, 0x30, 0x33, 0x32, \
			0x3d, 0x3c, 0x3f, 0x3e, 0x39, 0x38, 0x3b, 0x3a, \
			0x25, 0x24, 0x27, 0x26, 0x21, 0x20, 0x23, 0x22, \
			0x2d, 0x2c, 0x2f, 0x2e, 0x29, 0x28, 0x2b, 0x2a, \
			0x15, 0x14, 0x17, 0x16, 0x11, 0x10, 0x13, 0x12, \
			0x1d, 0x1c, 0x1f, 0x1e, 0x19, 0x18, 0x1b, 0x1a, \
			0x05, 0x04, 0x07, 0x06, 0x01, 0x00, 0x03, 0x02, \
			0x0d, 0x0c, 0x0f, 0x0e, 0x09, 0x08, 0x0b, 0x0a, \
	}
#define FNV_B64_18 \
	{ \
		0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, \
			0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, \
			0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, \
			0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, \
			0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f, \
			0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, \
			0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f, \
			0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, \
	}
#define FNV_B64_19 \
	{ \
		0x1b, 0x1a, 0x19, 0x18, 0x1f, 0x1e, 0x1d, 0x1c, \
			0x13, 0x12, 0x11, 0x10, 0x17, 0x16, 0x15, 0x14, \
			0x0b, 0x0a, 0x09, 0x08, 0x0f, 0x0e, 0x0d, 0x0c, \
			0x03, 0x02, 0x01, 0x00, 0x07, 0x06, 0x05, 0x04, \
			0x3b, 0x3a, 0x39, 0x38, 0x3f, 0x3e, 0x3d, 0x3c, \
			0x33, 0x32, 0x31, 0x30, 0x37, 0x36, 0x35, 0x34, \
			0x2b, 0x2a, 0x29, 0x28, 0x2f, 0x2e, 0x2d, 0x2c, \
			0x23, 0x22, 0x21, 0x20, 0x27, 0x26, 0x25, 0x24, \
	}
#define FNV_B64_1A \
	{ \
		0x2e, 0x2f, 0x2c, 0x2d, 0x2a, 0x2b, 0x28, 0x29, \
			0x26, 0x27, 0x24, 0x25, 0x22, 0x23, 0x20, 0x21, \
			0x3e, 0x3f, 0x3c, 0x3d, 0x3a, 0x3b, 0x38, 0x39, \
			0x36, 0x37, 0x34, 0x35, 0x32, 0x33, 0x30, 0x31, \
			0x0e, 0x0f, 0x0c, 0x0d, 0x0a, 0x0b, 0x08, 0x09, \
			0x06, 0x07, 0x04, 0x05, 0x02, 0x03, 0x00, 0x01, \
			0x1e, 0x1f, 0x1c, 0x1d, 0x1a, 0x1b, 0x18, 0x19, \
			0x16, 0x17, 0x14, 0x15, 0x12, 0x13, 0x10, 0x11, \
	}
#define FNV_B64_1B \
	{ \
		0x01, 0x00, 0x03, 0x02, 0x05, 0x04, 0x07, 0x06, \
			0x09, 0x08, 0x0b, 0x0a, 0x0d, 0x0c, 0x0f, 0x0e, \
			0x11, 0x10, 0x13, 0x12, 0x15, 0x14, 0x17, 0x16, \
			0x19, 0x18, 0x1b, 0x1a, 0x1d, 0x1c, 0x1f, 0x1e, \
			0x21, 0x20, 0x23, 0x22, 0x25, 0x24, 0x27, 0x26, \
			0x29, 0x28, 0x2b, 0x2a, 0x2d, 0x2c, 0x2f, 0x2e, \
			0x31, 0x30, 0x33, 0x32, 0x35, 0x34, 0x37, 0x36, \
			0x39, 0x38, 0x3b, 0x3a, 0x3d, 0x3c, 0x3f, 0x3e, \
	}
#define FNV_B64_1C \
	{ \
		0x14, 0x15, 0x16, 0x17, 0x10, 0x11, 0x12, 0x13, \
			0x1c, 0x1d, 0x1e, 0x1f, 0x18, 0x19, 0x1a, 0x1b, \
			0x04, 0x05, 0x06, 0x07, 0x00, 0x01, 0x02, 0x03, \
			0x0c, 0x0d, 0x0e, 0x0f, 0x08, 0x09, 0x0a, 0x0b, \
			0x34, 0x35, 0x36, 0x37, 0x30, 0x31, 0x32, 0x33, \
			0x3c, 0x3d, 0x3e, 0x3f, 0x38, 0x39, 0x3a, 0x3b, \
			0x24, 0x25, 0x26, 0x27, 0x20, 0x21, 0x22, 0x23, \
			0x2c, 0x2d, 0x2e, 0x2f, 0x28, 0x29, 0x2a, 0x2b, \
	}
#define FNV_B64_1D \
	{ \
		0x27, 0x26, 0x25, 0x24, 0x23, 0x22, 0x21, 0x20, \
			0x2f, 0x2e, 0x2d, 0x2c, 0x2b, 0x2a, 0x29, 0x28, \
			0x37, 0x36, 0x35, 0x34, 0x33, 0x32, 0x31, 0x30, \
			0x3f, 0x3e, 0x3d, 0x3c, 0x3b, 0x3a, 0x39, 0x38, \
			0x07, 0x06, 0x05, 0x04, 0x03, 0x02, 0x01, 0x00, \
			0x0f, 0x0e, 0x0d, 0x0c, 0x0b, 0x0a, 0x09, 0x08, \
			0x17, 0x16, 0x15, 0x14, 0x13, 0x12, 0x11, 0x10, \
			0x1f, 0x1e, 0x1d, 0x1c, 0x1b, 0x1a, 0x19, 0x18, \
	}
#define FNV_B64_1E \
	{ \
		0x3a, 0x3b, 0x38, 0x39, 0x3e, 0x3f, 0x3c, 0x3d, \
			0x32, 0x33, 0x30, 0x31, 0x36, 0x37, 0x34, 0x35, \
			0x2a, 0x2b, 0x28, 0x29, 0x2e, 0x2f, 0x2c, 0x2d, \
			0x22, 0x23, 0x20, 0x21, 0x26, 0x27, 0x24, 0x25, \
			0x1a, 0x1b, 0x18, 0x19, 0x1e, 0x1f, 0x1c, 0x1d, \
			0x12, 0x13, 0x10, 0x11, 0x16, 0x17, 0x14, 0x15, \
			0x0a, 0x0b, 0x08, 0x09, 0x0e, 0x0f, 0x0c, 0x0d, \
			0x02, 0x03, 0x00, 0x01, 0x06, 0x07, 0x04, 0x05, \
	}
#define FNV_B64_1F \
	{ \
		0x0d, 0x0c, 0x0f, 0x0e, 0x09, 0x08, 0x0b, 0x0a, \
			0x05, 0x04, 0x07, 0x06, 0x01, 0x00, 0x03, 0x02, \
			0x1d, 0x1c, 0x1f, 0x1e, 0x19, 0x18, 0x1b, 0x1a, \
			0x15, 0x14, 0x17, 0x16, 0x11, 0x10, 0x13, 0x12, \
			0x2d, 0x2c, 0x2f, 0x2e, 0x29, 0x28, 0x2b, 0x2a, \
			0x25, 0x24, 0x27, 0x26, 0x21, 0x20, 0x23, 0x22, \
			0x3d, 0x3c, 0x3f, 0x3e, 0x39, 0x38, 0x3b, 0x3a, \
			0x35, 0x34, 0x37, 0x36, 0x31, 0x30, 0x33, 0x32, \
	}
#define FNV_B64_20 \
	{ \
		0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, \
			0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f, \
			0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, \
			0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f, \
			0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, \
			0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, \
			0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, \
			0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, \
	}
#define FNV_B64_21 \
	{ \
		0x33, 0x32, 0x31, 0x30, 0x37, 0x36, 0x35, 0x34, \
			0x3b, 0x3a, 0x39, 0x38, 0x3f, 0x3e, 0x3d, 0x3c, \
			0x23, 0x22, 0x21, 0x20, 0x27, 0x26, 0x25, 0x24, \
			0x2b, 0x2a, 0x29, 0x28, 0x2f, 0x2e, 0x2d, 0x2c, \
			0x13, 0x12, 0x11, 0x10, 0x17, 0x16, 0x15, 0x14, \
			0x1b, 0x1a, 0x19, 0x18, 0x1f, 0x1e, 0x1d, 0x1c, \
			0x03, 0x02, 0x01, 0x00, 0x07, 0x06, 0x05, 0x04, \
			0x0b, 0x0a, 0x09, 0x08, 0x0f, 0x0e, 0x0d, 0x0c, \
	}
#define FNV_B64_22 \
	{ \
		0x06, 0x07, 0x04, 0x05, 0x02, 0x03, 0x00, 0x01, \
			0x0e, 0x0f, 0x0c, 0x0d, 0x0a, 0x0b, 0x08, 0x09, \
			0x16, 0x17, 0x14, 0x15, 0x12, 0x13, 0x10, 0x11, \
			0x1e, 0x1f, 0x1c, 0x1d, 0x1a, 0x1b, 0x18, 0x19, \
			0x26, 0x27, 0x24, 0x25, 0x22, 0x23, 0x20, 0x21, \
			0x2e, 0x2f, 0x2c, 0x2d, 0x2a, 0x2b, 0x28, 0x29, \
			0x36, 0x37, 0x34, 0x35, 0x32, 0x33, 0x30, 0x31, \
			0x3e, 0x3f, 0x3c, 0x3d, 0x3a, 0x3b, 0x38, 0x39, \
	}
#define FNV_B64_23 \
	{ \
		0x19, 0x18, 0x1b, 0x1a, 0x1d, 0x1c, 0x1f, 0x1e, \
			0x11, 0x10, 0x13, 0x12, 0x15, 0x14, 0x17, 0x16, \
			0x09, 0x08, 0x0b, 0x0a, 0x0d, 0x0c, 0x0f, 0x0e, \
			0x01, 0x00, 0x03, 0x02, 0x05, 0x04, 0x07, 0x06, \
			0x39, 0x38, 0x3b, 0x3a, 0x3d, 0x3c, 0x3f, 0x3e, \
			0x31, 0x30, 0x33, 0x32, 0x35, 0x34, 0x37, 0x36, \
			0x29, 0x28, 0x2b, 0x2a, 0x2d, 0x2c, 0x2f, 0x2e, \
			0x21, 0x20, 0x23, 0x22, 0x25, 0x24, 0x27, 0x26, \
	}
#define FNV_B64_24 \
	{ \
		0x2c, 0x2d, 0x2e, 0x2f, 0x28, 0x29, 0x2a, 0x2b, \
			0x24, 0x25, 0x26, 0x27, 0x20, 0x21, 0x22, 0x23, \
			0x3c, 0x3d, 0x3e, 0x3f, 0x38, 0x39, 0x3a, 0x3b, \
			0x34, 0x35, 0x36, 0x37, 0x30, 0x31, 0x32, 0x33, \
			0x0c, 0x0d, 0x0e, 0x0f, 0x08, 0x09, 0x0a, 0x0b, \
			0x04, 0x05, 0x06, 0x07, 0x00, 0x01, 0x02, 0x03, \
			0x1c, 0x1d, 0x1e, 0x1f, 0x18, 0x19, 0x1a, 0x1b, \
			0x14, 0x15, 0x16, 0x17, 0x10, 0x11, 0x12, 0x13, \
	}
#define FNV_B64_25 \
	{ \
		0x3f, 0x3e, 0x3d, 0x3c, 0x3b, 0x3a, 0x39, 0x38, \
			0x37, 0x36, 0x35, 0x34, 0x33, 0x32, 0x31, 0x30, \
			0x2f, 0x2e, 0x2d, 0x2c, 0x2b, 0x2a, 0x29, 0x28, \
			0x27, 0x26, 0x25, 0x24, 0x23, 0x22, 0x21, 0x20, \
			0x1f, 0x1e, 0x1d, 0x1c, 0x1b, 0x1a, 0x19, 0x18, \
			0x17, 0x16, 0x15, 0x14, 0x13, 0x12, 0x11, 0x10, \
			0x0f, 0x0e, 0x0d, 0x0c, 0x0b, 0x0a, 0x09, 0x08, \
			0x07, 0x06, 0x05, 0x04, 0x03, 0x02, 0x01, 0x00, \
	}
#define FNV_B64_26 \
	{ \
		0x12, 0x13, 0x10, 0x11, 0x16, 0x17, 0x14, 0x15, \
			0x1a, 0x1b, 0x18, 0x19, 0x1e, 0x1f, 0x1c, 0x1d, \
			0x02, 0x03, 0x00, 0x01, 0x06, 0x07, 0x04, 0x05, \
			0x0a, 0x0b, 0x08, 0x09, 0x0e, 0x0f, 0x0c, 0x0d, \
			0x32, 0x33, 0x30, 0x31, 0x36, 0x37, 0x34, 0x35, \
			0x3a, 0x3b, 0x38, 0x39, 0x3e, 0x3f, 0x3c, 0x3d, \
			0x22, 0x23, 0x20, 0x21, 0x26, 0x27, 0x24, 0x25, \
			0x2a, 0x2b, 0x28, 0x29, 0x2e, 0x2f, 0x2c, 0x2d, \
	}
#define FNV_B64_27 \
	{ \
		0x25, 0x24, 0x27, 0x26, 0x21, 0x20, 0x23, 0x22, \
			0x2d, 0x2c, 0x2f, 0x2e, 0x29, 0x28, 0x2b, 0x2a, \
			0x35, 0x34, 0x37, 0x36, 0x31, 0x30, 0x33, 0x32, \
			0x3d, 0x3c, 0x3f, 0x3e, 0x39, 0x38, 0x3b, 0x3a, \
			0x05, 0x04, 0x07, 0x06, 0x01, 0x00, 0x03, 0x02, \
			0x0d, 0x0c, 0x0f, 0x0e, 0x09, 0x08, 0x0b, 0x0a, \
			0x15, 0x14, 0x17, 0x16, 0x11, 0x10, 0x13, 0x12, \
			0x1d, 0x1c, 0x1f, 0x1e, 0x19, 0x18, 0x1b, 0x1a, \
	}
#define FNV_B64_28 \
	{ \
		0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f, \
			0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, \
			0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f, \
			0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, \
			0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, \
			0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, \
			0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, \
			0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, \
	}
#define FNV_B64_29 \
	{ \
		0x0b, 0x0a, 0x09, 0x08, 0x0f, 0x0e, 0x0d, 0x0c, \
			0x03, 0x02, 0x01, 0x00, 0x07, 0x06, 0x05, 0x04, \
			0x1b, 0x1a, 0x19, 0x18, 0x1f, 0x1e, 0x1d, 0x1c, \
			0x13, 0x12, 0x11, 0x10, 0x17, 0x16, 0x15, 0x14, \
			0x2b, 0x2a, 0x29, 0x28, 0x2f, 0x2e, 0x2d, 0x2c, \
			0x23, 0x22, 0x21, 0x20, 0x27, 0x26, 0x25, 0x24, \
			0x3b, 0x3a, 0x39, 0x38, 0x3f, 0x3e, 0x3d, 0x3c, \
			0x33, 0x32, 0x31, 0x30, 0x37, 0x36, 0x35, 0x34, \
	}
#define FNV_B64_2A \
	{ \
		0x1e, 0x1f, 0x1c, 0x1d, 0x1a, 0x1b, 0x18, 0x19, \
			0x16, 0x17, 0x14, 0x15, 0x12, 0x13, 0x10, 0x11, \
			0x0e, 0x0f, 0x0c, 0x0d, 0x0a, 0x0b, 0x08, 0x09, \
			0x06, 0x07, 0x04, 0x05, 0x02, 0x03, 0x00, 0x01, \
			0x3e, 0x3f, 0x3c, 0x3d, 0x3a, 0x3b, 0x38, 0x39, \
			0x36, 0x37, 0x34, 0x35, 0x32, 0x33, 0x30, 0x31, \
			0x2e, 0x2f, 0x2c, 0x2d, 0x2a, 0x2b, 0x28, 0x29, \
			0x26, 0x27, 0x24, 0x25, 0x22, 0x23, 0x20, 0x21, \
	}
#define FNV_B64_2B \
	{ \
		0x31, 0x30, 0x33, 0x32, 0x35, 0x34, 0x37, 0x36, \
			0x39, 0x38, 0x3b, 0x3a, 0x3d, 0x3c, 0x3f, 0x3e, \
			0x21, 0x20, 0x23, 0x22, 0x25, 0x24, 0x27, 0x26, \
			0x29, 0x28, 0x2b, 0x2a, 0x2d, 0x2c, 0x2f, 0x2e, \
			0x11, 0x10, 0x13, 0x12, 0x15, 0x14, 0x17, 0x16, \
			0x19, 0x18, 0x1b, 0x1a, 0x1d, 0x1c, 0x1f, 0x1e, \
			0x01, 0x00, 0x03, 0x02, 0x05, 0x04, 0x07, 0x06, \
			0x09, 0x08, 0x0b, 0x0a, 0x0d, 0x0c, 0x0f, 0x0e, \
	}
#define FNV_B64_2C \
	{ \
		0x04, 0x05, 0x06, 0x07, 0x00, 0x01, 0x02, 0x03, \
			0x0c, 0x0d, 0x0e, 0x0f, 0x08, 0x09, 0x0a, 0x0b, \
			0x14, 0x15, 0x16, 0x17, 0x10, 0x11, 0x12, 0x13, \
			0x1c, 0x1d, 0x1e, 0x1f, 0x18, 0x19, 0x1a, 0x1b, \
			0x24, 0x25, 0x26, 0x27, 0x20, 0x21, 0x22, 0x23, \
			0x2c, 0x2d, 0x2e, 0x2f, 0x28, 0x29, 0x2a, 0x2b, \
			0x34, 0x35, 0x36, 0x37, 0x30, 0x31, 0x32, 0x33, \
			0x3c, 0x3d, 0x3e, 0x3f, 0x38, 0x39, 0x3a, 0x3b, \
	}
#define FNV_B64_2D \
	{ \
		0x17, 0x16, 0x15, 0x14, 0x13, 0x12, 0x11, 0x10, \
			0x1f, 0x1e, 0x1d, 0x1c, 0x1b, 0x1a, 0x19, 0x18, \
			0x07, 0x06, 0x05, 0x04, 0x03, 0x02, 0x01, 0x00, \
			0x0f, 0x0e, 0x0d, 0x0c, 0x0b, 0x0a, 0x09, 0x08, \
			0x37, 0x36, 0x35, 0x34, 0x33, 0x32, 0x31, 0x30, \
			0x3f, 0x3e, 0x3d, 0x3c, 0x3b, 0x3a, 0x39, 0x38, \
			0x27, 0x26, 0x25, 0x24, 0x23, 0x22, 0x21, 0x20, \
			0x2f, 0x2e, 0x2d, 0x2c, 0x2b, 0x2a, 0x29, 0x28, \
	}
#define FNV_B64_2E \
	{ \
		0x2a, 0x2b, 0x28, 0x29, 0x2e, 0x2f, 0x2c, 0x2d, \
			0x22, 0x23, 0x20, 0x21, 0x26, 0x27, 0x24, 0x25, \
			0x3a, 0x3b, 0x38, 0x39, 0x3e, 0x3f, 0x3c, 0x3d, \
			0x32, 0x33, 0x30, 0x31, 0x36, 0x37, 0x34, 0x35, \
			0x0a, 0x0b, 0x08, 0x09, 0x0e, 0x0f, 0x0c, 0x0d, \
			0x02, 0x03, 0x00, 0x01, 0x06, 0x07, 0x04, 0x05, \
			0x1a, 0x1b, 0x18, 0x19, 0x1e, 0x1f, 0x1c, 0x1d, \
			0x12, 0x13, 0x10, 0x11, 0x16, 0x17, 0x14, 0x15, \
	}
#define FNV_B64_2F \
	{ \
		0x3d, 0x3c, 0x3f, 0x3e, 0x39, 0x38, 0x3b, 0x3a, \
			0x35, 0x34, 0x37, 0x36, 0x31, 0x30, 0x33, 0x32, \
			0x2d, 0x2c, 0x2f, 0x2e, 0x29, 0x28, 0x2b, 0x2a, \
			0x25, 0x24, 0x27, 0x26, 0x21, 0x20, 0x23, 0x22, \
			0x1d, 0x1c, 0x1f, 0x1e, 0x19, 0x18, 0x1b, 0x1a, \
			0x15, 0x14, 0x17, 0x16, 0x11, 0x10, 0x13, 0x12, \
			0x0d, 0x0c, 0x0f, 0x0e, 0x09, 0x08, 0x0b, 0x0a, \
			0x05, 0x04, 0x07, 0x06, 0x01, 0x00, 0x03, 0x02, \
	}
#define FNV_B64_30 \
	{ \
		0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, \
			0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, \
			0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, \
			0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, \
			0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, \
			0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f, \
			0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, \
			0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f, \
	}
#define FNV_B64_31 \
	{ \
		0x23, 0x22, 0x21, 0x20, 0x27, 0x26, 0x25, 0x24, \
			0x2b, 0x2a, 0x29, 0x28, 0x2f, 0x2e, 0x2d, 0x2c, \
			0x33, 0x32, 0x31, 0x30, 0x37, 0x36, 0x35, 0x34, \
			0x3b, 0x3a, 0x39, 0x38, 0x3f, 0x3e, 0x3d, 0x3c, \
			0x03, 0x02, 0x01, 0x00, 0x07, 0x06, 0x05, 0x04, \
			0x0b, 0x0a, 0x09, 0x08, 0x0f, 0x0e, 0x0d, 0x0c, \
			0x13, 0x12, 0x11, 0x10, 0x17, 0x16, 0x15, 0x14, \
			0x1b, 0x1a, 0x19, 0x18, 0x1f, 0x1e, 0x1d, 0x1c, \
	}
#define FNV_B64_32 \
	{ \
		0x36, 0x37, 0x34, 0x35, 0x32, 0x33, 0x30, 0x31, \
			0x3e, 0x3f, 0x3c, 0x3d, 0x3a, 0x3b, 0x38, 0x39, \
			0x26, 0x27, 0x24, 0x25, 0x22, 0x23, 0x20, 0x21, \
			0x2e, 0x2f, 0x2c, 0x2d, 0x2a, 0x2b, 0x28, 0x29, \
			0x16, 0x17, 0x14, 0x15, 0x12, 0x13, 0x10, 0x11, \
			0x1e, 0x1f, 0x1c, 0x1d, 0x1a, 0x1b, 0x18, 0x19, \
			0x06, 0x07, 0x04, 0x05, 0x02, 0x03, 0x00, 0x01, \
			0x0e, 0x0f, 0x0c, 0x0d, 0x0a, 0x0b, 0x08, 0x09, \
	}
#define FNV_B64_33 \
	{ \
		0x09, 0x08, 0x0b, 0x0a, 0x0d, 0x0c, 0x0f, 0x0e, \
			0x01, 0x00, 0x03, 0x02, 0x05, 0x04, 0x07, 0x06, \
			0x19, 0x18, 0x1b, 0x1a, 0x1d, 0x1c, 0x1f, 0x1e, \
			0x11, 0x10, 0x13, 0x12, 0x15, 0x14, 0x17, 0x16, \
			0x29, 0x28, 0x2b, 0x2a, 0x2d, 0x2c, 0x2f, 0x2e, \
			0x21, 0x20, 0x23, 0x22, 0x25, 0x24, 0x27, 0x26, \
			0x39, 0x38, 0x3b, 0x3a, 0x3d, 0x3c, 0x3f, 0x3e, \
			0x31, 0x30, 0x33, 0x32, 0x35, 0x34, 0x37, 0x36, \
	}
#define FNV_B64_34 \
	{ \
		0x1c, 0x1d, 0x1e, 0x1f, 0x18, 0x19, 0x1a, 0x1b, \
			0x14, 0x15, 0x16, 0x17, 0x10, 0x11, 0x12, 0x13, \
			0x0c, 0x0d, 0x0e, 0x0f, 0x08, 0x09, 0x0a, 0x0b, \
			0x04, 0x05, 0x06, 0x07, 0x00, 0x01, 0x02, 0x03, \
			0x3c, 0x3d, 0x3e, 0x3f, 0x38, 0x39, 0x3a, 0x3b, \
			0x34, 0x35, 0x36, 0x37, 0x30, 0x31, 0x32, 0x33, \
			0x2c, 0x2d, 0x2e, 0x2f, 0x28, 0x29, 0x2a, 0x2b, \
			0x24, 0x25, 0x26, 0x27, 0x20, 0x21, 0x22, 0x23, \
	}
#define FNV_B64_35 \
	{ \
		0x2f, 0x2e, 0x2d, 0x2c, 0x2b, 0x2a, 0x29, 0x28, \
			0x27, 0x26, 0x25, 0x24, 0x23, 0x22, 0x21, 0x20, \
			0x3f, 0x3e, 0x3d, 0x3c, 0x3b, 0x3a, 0x39, 0x38, \
			0x37, 0x36, 0x35, 0x34, 0x33, 0x32, 0x31, 0x30, \
			0x0f, 0x0e, 0x0d, 0x0c, 0x0b, 0x0a, 0x09, 0x08, \
			0x07, 0x06, 0x05, 0x04, 0x03, 0x02, 0x01, 0x00, \
			0x1f, 0x1e, 0x1d, 0x1c, 0x1b, 0x1a, 0x19, 0x18, \
			0x17, 0x16, 0x15, 0x14, 0x13, 0x12, 0x11, 0x10, \
	}
#define FNV_B64_36 \
	{ \
		0x02, 0x03, 0x00, 0x01, 0x06, 0x07, 0x04, 0x05, \
			0x0a, 0x0b, 0x08, 0x09, 0x0e, 0x0f, 0x0c, 0x0d, \
			0x12, 0x13, 0x10, 0x11, 0x16, 0x17, 0x14, 0x15, \
			0x1a, 0x1b, 0x18, 0x19, 0x1e, 0x1f, 0x1c, 0x1d, \
			0x22, 0x23, 0x20, 0x21, 0x26, 0x27, 0x24, 0x25, \
			0x2a, 0x2b, 0x28, 0x29, 0x2e, 0x2f, 0x2c, 0x2d, \
			0x32, 0x33, 0x30, 0x31, 0x36, 0x37, 0x34, 0x35, \
			0x3a, 0x3b, 0x38, 0x39, 0x3e, 0x3f, 0x3c, 0x3d, \
	}
#define FNV_B64_37 \
	{ \
		0x15, 0x14, 0x17, 0x16, 0x11, 0x10, 0x13, 0x12, \
			0x1d, 0x1c, 0x1f, 0x1e, 0x19, 0x18, 0x1b, 0x1a, \
			0x05, 0x04, 0x07, 0x06, 0x01, 0x00, 0x03, 0x02, \
			0x0d, 0x0c, 0x0f, 0x0e, 0x09, 0x08, 0x0b, 0x0a, \
			0x35, 0x34, 0x37, 0x36, 0x31, 0x30, 0x33, 0x32, \
			0x3d, 0x3c, 0x3f, 0x3e, 0x39, 0x38, 0x3b, 0x3a, \
			0x25, 0x24, 0x27, 0x26, 0x21, 0x20, 0x23, 0x22, \
			0x2d, 0x2c, 0x2f, 0x2e, 0x29, 0x28, 0x2b, 0x2a, \
	}
#define FNV_B64_38 \
	{ \
		0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f, \
			0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, \
			0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f, \
			0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, \
			0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, \
			0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, \
			0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, \
			0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, \
	}
#define FNV_B64_39 \
	{ \
		0x3b, 0x3a, 0x39, 0x38, 0x3f, 0x3e, 0x3d, 0x3c, \
			0x33, 0x32, 0x31, 0x30, 0x37, 0x36, 0x35, 0x34, \
			0x2b, 0x2a, 0x29, 0x28, 0x2f, 0x2e, 0x2d, 0x2c, \
			0x23, 0x22, 0x21, 0x20, 0x27, 0x26, 0x25, 0x24, \
			0x1b, 0x1a, 0x19, 0x18, 0x1f, 0x1e, 0x1d, 0x1c, \
			0x13, 0x12, 0x11, 0x10, 0x17, 0x16, 0x15, 0x14, \
			0x0b, 0x0a, 0x09, 0x08, 0x0f, 0x0e, 0x0d, 0x0c, \
			0x03, 0x02, 0x01, 0x00, 0x07, 0x06, 0x05, 0x04, \
	}
#define FNV_B64_3A \
	{ \
		0x0e, 0x0f, 0x0c, 0x0d, 0x0a, 0x0b, 0x08, 0x09, \
			0x06, 0x07, 0x04, 0x05, 0x02, 0x03, 0x00, 0x01, \
			0x1e, 0x1f, 0x1c, 0x1d, 0x1a, 0x1b, 0x18, 0x19, \
			0x16, 0x17, 0x14, 0x15, 0x12, 0x13, 0x10, 0x11, \
			0x2e, 0x2f, 0x2c, 0x2d, 0x2a, 0x2b, 0x28, 0x29, \
			0x26, 0x27, 0x24, 0x25, 0x22, 0x23, 0x20, 0x21, \
			0x3e, 0x3f, 0x3c, 0x3d, 0x3a, 0x3b, 0x38, 0x39, \
			0x36, 0x37, 0x34, 0x35, 0x32, 0x33, 0x30, 0x31, \
	}
#define FNV_B64_3B \
	{ \
		0x21, 0x20, 0x23, 0x22, 0x25, 0x24, 0x27, 0x26, \
			0x29, 0x28, 0x2b, 0x2a, 0x2d, 0x2c, 0x2f, 0x2e, \
			0x31, 0x30, 0x33, 0x32, 0x35, 0x34, 0x37, 0x36, \
			0x39, 0x38, 0x3b, 0x3a, 0x3d, 0x3c, 0x3f, 0x3e, \
			0x01, 0x00, 0x03, 0x02, 0x05, 0x04, 0x07, 0x06, \
			0x09, 0x08, 0x0b, 0x0a, 0x0d, 0x0c, 0x0f, 0x0e, \
			0x11, 0x10, 0x13, 0x12, 0x15, 0x14, 0x17, 0x16, \
			0x19, 0x18, 0x1b, 0x1a, 0x1d, 0x1c, 0x1f, 0x1e, \
	}
#define FNV_B64_3C \
	{ \
		0x34, 0x35, 0x36, 0x37, 0x30, 0x31, 0x32, 0x33, \
			0x3c, 0x3d, 0x3e, 0x3f, 0x38, 0x39, 0x3a, 0x3b, \
			0x24, 0x25, 0x26, 0x27, 0x20, 0x21, 0x22, 0x23, \
			0x2c, 0x2d, 0x2e, 0x2f, 0x28, 0x29, 0x2a, 0x2b, \
			0x14, 0x15, 0x16, 0x17, 0x10, 0x11, 0x12, 0x13, \
			0x1c, 0x1d, 0x1e, 0x1f, 0x18, 0x19, 0x1a, 0x1b, \
			0x04, 0x05, 0x06, 0x07, 0x00, 0x01, 0x02, 0x03, \
			0x0c, 0x0d, 0x0e, 0x0f, 0x08, 0x09, 0x0a, 0x0b, \
	}
#define FNV_B64_3D \
	{ \
		0x07, 0x06, 0x05, 0x04, 0x03, 0x02, 0x01, 0x00, \
			0x0f, 0x0e, 0x0d, 0x0c, 0x0b, 0x0a, 0x09, 0x08, \
			0x17, 0x16, 0x15, 0x14, 0x13, 0x12, 0x11, 0x10, \
			0x1f, 0x1e, 0x1d, 0x1c, 0x1b, 0x1a, 0x19, 0x18, \
			0x27, 0x26, 0x25, 0x24, 0x23, 0x22, 0x21, 0x20, \
			0x2f, 0x2e, 0x2d, 0x2c, 0x2b, 0x2a, 0x29, 0x28, \
			0x37, 0x36, 0x35, 0x34, 0x33, 0x32, 0x31, 0x30, \
			0x3f, 0x3e, 0x3d, 0x3c, 0x3b, 0x3a, 0x39, 0x38, \
	}
#define FNV_B64_3E \
	{ \
		0x1a, 0x1b, 0x18, 0x19, 0x1e, 0x1f, 0x1c, 0x1d, \
			0x12, 0x13, 0x10, 0x11, 0x16, 0x17, 0x14, 0x15, \
			0x0a, 0x0b, 0x08, 0x09, 0x0e, 0x0f, 0x0c, 0x0d, \
			0x02, 0x03, 0x00, 0x01, 0x06, 0x07, 0x04, 0x05, \
			0x3a, 0x3b, 0x38, 0x39, 0x3e, 0x3f, 0x3c, 0x3d, \
			0x32, 0x33, 0x30, 0x31, 0x36, 0x37, 0x34, 0x35, \
			0x2a, 0x2b, 0x28, 0x29, 0x2e, 0x2f, 0x2c, 0x2d, \
			0x22, 0x23, 0x20, 0x21, 0x26, 0x27, 0x24, 0x25, \
	}
#define FNV_B64_3F \
	{ \
		0x2d, 0x2c, 0x2f, 0x2e, 0x29, 0x28, 0x2b, 0x2a, \
			0x25, 0x24, 0x27, 0x26, 0x21, 0x20, 0x23, 0x22, \
			0x3d, 0x3c, 0x3f, 0x3e, 0x39, 0x38, 0x3b, 0x3a, \
			0x35, 0x34, 0x37, 0x36, 0x31, 0x30, 0x33, 0x32, \
			0x0d, 0x0c, 0x0f, 0x0e, 0x09, 0x08, 0x0b, 0x0a, \
			0x05, 0x04, 0x07, 0x06, 0x01, 0x00, 0x03, 0x02, \
			0x1d, 0x1c, 0x1f, 0x1e, 0x19, 0x18, 0x1b, 0x1a, \
			0x15, 0x14, 0x17, 0x16, 0x11, 0x10, 0x13, 0x12, \
	}

#endif /* FNV_HASH_H */
