'use strict';

var glob = require('glob');
var promises = require('node:fs/promises');
var node_path = require('node:path');

// A list of common glob patterns for the primary language ('en') or ('dev') translation files.
const HEURISTIC_PATTERNS = [
    'public/locales/dev/*.json',
    'locales/dev/*.json',
    'src/locales/dev/*.json',
    'src/assets/locales/dev/*.json',
    'app/i18n/locales/dev/*.json',
    'src/i18n/locales/dev/*.json',
    'public/locales/en/*.json',
    'locales/en/*.json',
    'src/locales/en/*.json',
    'src/assets/locales/en/*.json',
    'app/i18n/locales/en/*.json',
    'src/i18n/locales/en/*.json',
    'public/locales/en/*.json5',
    'locales/en/*.json5',
    'src/locales/en/*.json5',
    'src/assets/locales/en/*.json5',
    'app/i18n/locales/en/*.json5',
    'src/i18n/locales/en/*.json5',
    'public/locales/en-*/*.json',
    'locales/en-*/*.json',
    'src/locales/en-*/*.json',
    'src/assets/locales/en-*/*.json',
    'app/i18n/locales/en-*/*.json',
    'src/i18n/locales/en-*/*.json',
];
/**
 * Attempts to automatically detect the project's i18n structure by searching for
 * common translation file locations.
 *
 * @returns A promise that resolves to a partial I18nextToolkitConfig if detection
 * is successful, otherwise null.
 */
async function detectConfig() {
    for (const pattern of HEURISTIC_PATTERNS) {
        const files = await glob.glob(pattern, { ignore: 'node_modules/**' });
        if (files.length > 0) {
            // Prefer .json5 if present
            const json5File = files.find(f => node_path.extname(f) === '.json5');
            const firstFile = json5File || files[0];
            const basePath = node_path.dirname(node_path.dirname(firstFile));
            const extension = node_path.extname(firstFile);
            // Infer outputFormat from the file extension
            let outputFormat = 'json';
            if (extension === '.ts') {
                outputFormat = 'ts';
            }
            else if (extension === '.js') {
                // We can't know if it's ESM or CJS, so we default to a safe choice.
                // The tool's file loaders can handle both.
                outputFormat = 'js';
            }
            else if (extension === '.json5') {
                outputFormat = 'json5';
            }
            try {
                const allDirs = await promises.readdir(basePath);
                let locales = allDirs.filter(dir => /^(dev|[a-z]{2}(-[A-Z]{2})?)$/.test(dir));
                if (locales.length > 0) {
                    // Prioritization Logic
                    locales.sort();
                    if (locales.includes('dev')) {
                        locales = ['dev', ...locales.filter(l => l !== 'dev')];
                    }
                    if (locales.includes('en')) {
                        locales = ['en', ...locales.filter(l => l !== 'en')];
                    }
                    return {
                        locales,
                        extract: {
                            input: [
                                'src/**/*.{js,jsx,ts,tsx}',
                                'app/**/*.{js,jsx,ts,tsx}',
                                'pages/**/*.{js,jsx,ts,tsx}',
                                'components/**/*.{js,jsx,ts,tsx}'
                            ],
                            output: node_path.join(basePath, '{{language}}', `{{namespace}}${extension}`),
                            outputFormat,
                            primaryLanguage: locales.includes('en') ? 'en' : locales[0],
                        },
                    };
                }
            }
            catch {
                continue;
            }
        }
    }
    return null;
}

exports.detectConfig = detectConfig;
