'use client';
import { jsx } from 'react/jsx-runtime';
import { forwardRef, useState, useEffect } from 'react';
import { useDebouncedCallback } from '@mantine/hooks';
import { useScrollAreaContext } from '../ScrollArea.context.mjs';
import { composeEventHandlers } from '../utils/compose-event-handlers.mjs';
import { ScrollAreaScrollbarVisible } from './ScrollAreaScrollbarVisible.mjs';

const ScrollAreaScrollbarScroll = forwardRef(
  (props, red) => {
    const { forceMount, ...scrollbarProps } = props;
    const context = useScrollAreaContext();
    const isHorizontal = props.orientation === "horizontal";
    const [state, setState] = useState("hidden");
    const debounceScrollEnd = useDebouncedCallback(() => setState("idle"), 100);
    useEffect(() => {
      if (state === "idle") {
        const hideTimer = window.setTimeout(() => setState("hidden"), context.scrollHideDelay);
        return () => window.clearTimeout(hideTimer);
      }
      return void 0;
    }, [state, context.scrollHideDelay]);
    useEffect(() => {
      const { viewport } = context;
      const scrollDirection = isHorizontal ? "scrollLeft" : "scrollTop";
      if (viewport) {
        let prevScrollPos = viewport[scrollDirection];
        const handleScroll = () => {
          const scrollPos = viewport[scrollDirection];
          const hasScrollInDirectionChanged = prevScrollPos !== scrollPos;
          if (hasScrollInDirectionChanged) {
            setState("scrolling");
            debounceScrollEnd();
          }
          prevScrollPos = scrollPos;
        };
        viewport.addEventListener("scroll", handleScroll);
        return () => viewport.removeEventListener("scroll", handleScroll);
      }
      return void 0;
    }, [context.viewport, isHorizontal, debounceScrollEnd]);
    if (forceMount || state !== "hidden") {
      return /* @__PURE__ */ jsx(
        ScrollAreaScrollbarVisible,
        {
          "data-state": state === "hidden" ? "hidden" : "visible",
          ...scrollbarProps,
          ref: red,
          onPointerEnter: composeEventHandlers(props.onPointerEnter, () => setState("interacting")),
          onPointerLeave: composeEventHandlers(props.onPointerLeave, () => setState("idle"))
        }
      );
    }
    return null;
  }
);

export { ScrollAreaScrollbarScroll };
//# sourceMappingURL=ScrollAreaScrollbarScroll.mjs.map
