#ifndef AVOGADRO_CORE_ELEMENTS_DATA
#define AVOGADRO_CORE_ELEMENTS_DATA

#include "elements.h"
#include <vector>

using isotope = std::pair<unsigned short, double>;

namespace Avogadro::Core {

const char* const element_symbols[element_count] = {
  "Xx", "H",  "He", "Li", "Be", "B",  "C",  "N",  "O",  "F",  "Ne", "Na",
  "Mg", "Al", "Si", "P",  "S",  "Cl", "Ar", "K",  "Ca", "Sc", "Ti", "V",
  "Cr", "Mn", "Fe", "Co", "Ni", "Cu", "Zn", "Ga", "Ge", "As", "Se", "Br",
  "Kr", "Rb", "Sr", "Y",  "Zr", "Nb", "Mo", "Tc", "Ru", "Rh", "Pd", "Ag",
  "Cd", "In", "Sn", "Sb", "Te", "I",  "Xe", "Cs", "Ba", "La", "Ce", "Pr",
  "Nd", "Pm", "Sm", "Eu", "Gd", "Tb", "Dy", "Ho", "Er", "Tm", "Yb", "Lu",
  "Hf", "Ta", "W",  "Re", "Os", "Ir", "Pt", "Au", "Hg", "Tl", "Pb", "Bi",
  "Po", "At", "Rn", "Fr", "Ra", "Ac", "Th", "Pa", "U",  "Np", "Pu", "Am",
  "Cm", "Bk", "Cf", "Es", "Fm", "Md", "No", "Lr", "Rf", "Db", "Sg", "Bh",
  "Hs", "Mt", "Ds", "Rg", "Cn", "Nh", "Fl", "Mc", "Lv", "Ts", "Og"
};

const char* const element_names[element_count] = {
  "Dummy",        "Hydrogen",     "Helium",      "Lithium",     "Beryllium",
  "Boron",        "Carbon",       "Nitrogen",    "Oxygen",      "Fluorine",
  "Neon",         "Sodium",       "Magnesium",   "Aluminium",   "Silicon",
  "Phosphorus",   "Sulfur",       "Chlorine",    "Argon",       "Potassium",
  "Calcium",      "Scandium",     "Titanium",    "Vanadium",    "Chromium",
  "Manganese",    "Iron",         "Cobalt",      "Nickel",      "Copper",
  "Zinc",         "Gallium",      "Germanium",   "Arsenic",     "Selenium",
  "Bromine",      "Krypton",      "Rubidium",    "Strontium",   "Yttrium",
  "Zirconium",    "Niobium",      "Molybdenum",  "Technetium",  "Ruthenium",
  "Rhodium",      "Palladium",    "Silver",      "Cadmium",     "Indium",
  "Tin",          "Antimony",     "Tellurium",   "Iodine",      "Xenon",
  "Caesium",      "Barium",       "Lanthanum",   "Cerium",      "Praseodymium",
  "Neodymium",    "Promethium",   "Samarium",    "Europium",    "Gadolinium",
  "Terbium",      "Dysprosium",   "Holmium",     "Erbium",      "Thulium",
  "Ytterbium",    "Lutetium",     "Hafnium",     "Tantalum",    "Tungsten",
  "Rhenium",      "Osmium",       "Iridium",     "Platinum",    "Gold",
  "Mercury",      "Thallium",     "Lead",        "Bismuth",     "Polonium",
  "Astatine",     "Radon",        "Francium",    "Radium",      "Actinium",
  "Thorium",      "Protactinium", "Uranium",     "Neptunium",   "Plutonium",
  "Americium",    "Curium",       "Berkelium",   "Californium", "Einsteinium",
  "Fermium",      "Mendelevium",  "Nobelium",    "Lawrencium",  "Rutherfordium",
  "Dubnium",      "Seaborgium",   "Bohrium",     "Hassium",     "Meitnerium",
  "Darmstadtium", "Roentgenium",  "Copernicium", "Nihonium",    "Flerovium",
  "Moscovium",    "Livermorium",  "Tennessine",  "Oganesson"
};

const double element_masses[element_count] = {
  // from IUPAC  (2021 set) https://doi.org/10.1515/pac-2019-0603
  //    https://iupac.org/what-we-do/periodic-table-of-elements/
  //    https://iupac.qmul.ac.uk/AtWt/
  // For Ar and Pb which are now ranges, we use the "abridged value"
  0,         1.00784,   4.002602,   6.94,      9.012183,  10.81,     12.011,
  14.007,    15.999,    18.998403,  20.1797,   22.989769, 24.305,    26.981538,
  28.085,    30.973762, 32.06,      35.45,     39.94,     39.0983,   40.078,
  44.955908, 47.867,    50.9415,    51.9961,   54.938043, 55.845,    58.933194,
  58.6934,   63.546,    65.38,      69.723,    72.630,    74.921595, 78.971,
  79.904,    83.798,    85.4678,    87.62,     88.90584,  91.224,    92.90637,
  95.95,     97,        101.07,     102.90549, 106.42,    107.8682,  112.414,
  114.818,   118.710,   121.760,    127.60,    126.90447, 131.293,   132.90447,
  137.327,   138.90547, 140.116,    140.90766, 144.242,   145,       150.36,
  151.964,   157.25,    158.92535,  162.500,   164.93033, 167.259,   168.93422,
  173.045,   174.9668,  178.486,    180.94788, 183.84,    186.207,   190.23,
  192.217,   195.084,   196.966570, 200.592,   204.38,    207.2,     208.9804,
  209,       210,       222,        223,       226,       227,       232.0377,
  231.0358,  238.02891, 237,        244,       243,       247,       247,
  251,       252,       257,        258,       259,       262,       267,
  270,       269,       270,        270,       278,       281,       281,
  285,       286,       289,        289,       293,       293,       294
};

const unsigned char valence_electrons[element_count] = {
  // some of these are debatable, but are intended as a reasonable start
  0, 1, 0,                                              // He
  1, 2, 3, 4, 5, 6, 7, 8,                               // Ne
  1, 2, 3, 4, 5, 6, 7, 8,                               // Ar
  1, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 3, 4, 5, 6, 7, 8, // Kr
  1, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 3, 4, 5, 6, 7, 8, // Xe
  1, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2,
  2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 1, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2,
  2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2
};

const double element_VDW[element_count] = {
  // From Alvarez doi: 10.1039/C3DT50599E
  // Dalton Trans., 2013,42, 8617-8636
  // Dummy, 1st row
  0.69,
  1.2,
  1.43,
  // 2nd row (Li..Ne)
  2.12,
  1.98,
  1.91,
  1.77,
  1.66,
  1.50,
  1.46,
  1.58,
  // 3rd row (Na .. Ar)
  2.50,
  2.51,
  2.25,
  2.19,
  1.90,
  1.89,
  1.82,
  1.83,
  // 4th row (K, Ca)
  2.73,
  2.62,
  // 1st row TM (Sc.. Zn)
  2.58,
  2.46,
  2.42,
  2.45,
  2.45,
  2.44,
  2.40,
  2.40,
  2.38,
  2.39,
  // 4th row p-block (Ga .. Kr)
  2.32,
  2.29,
  1.88,
  1.82,
  1.86,
  2.25,
  // 5th row Rb, Sr
  3.21,
  2.84,
  // 2nd row TM (Y .. Cd)
  2.75,
  2.52,
  2.56,
  2.45,
  2.44,
  2.46,
  2.44,
  2.15,
  2.53,
  2.49,
  // 5th row p-block (Sn .. Xe)
  2.43,
  2.42,
  2.47,
  1.99,
  2.04,
  2.06,
  // 6th row Cs, Ba
  3.48,
  3.03,
  // Lanthanides (La..Gd)
  2.98,
  2.88,
  2.92,
  2.95,
  2.90,
  2.87,
  2.83,
  // Lanthanides (Tb..Yb)
  2.79,
  2.87,
  2.81,
  2.83,
  2.79,
  2.80,
  // 3rd row TM (Lu..Hg)
  2.74,
  2.63,
  2.53,
  2.57,
  2.49,
  2.48,
  2.41,
  2.29,
  2.32,
  2.45,
  // 6th row p-block (Tl.. Bi)
  // 2.5 is a default here
  2.47,
  2.60,
  2.54,
  2.5,
  2.5,
  2.5,
  // 7th row
  // 2.5 is a default here
  2.5,
  2.5,
  // Actinides
  2.8,
  2.93,
  2.88,
  2.71,
  2.82,
  2.81,
  2.83,
  3.05,
  3.38,
  3.05,
  3.,
  3.,
  3.,
  3.,
  // Trans-actinides
  3.,
  3.,
  3.,
  3.,
  3.,
  3.,
  3.,
  3.,
  3.,
  3.,
  // 7th row p-block
  3.,
  3.,
  3.,
  3.,
  3.,
  3.,
  3.,
};

const double element_covalent[element_count] = {
  // From Pyykko doi: 10.1002/chem.200800987
  // Dummy, 1st row
  0.18, 0.32, 0.46,
  // 2nd row
  1.33, 1.02, 0.85, 0.75, 0.71, 0.63, 0.64, 0.67,
  // 3rd row
  1.55, 1.39, 1.26, 1.16, 1.11, 1.03, 0.99, 0.96,
  // 4th row K, Ca
  1.96, 1.71,
  // 1st row TM (Sc.. Zn)
  1.48, 1.36, 1.34, 1.22, 1.19, 1.16, 1.11, 1.10, 1.12, 1.18,
  // 4th row p-block (Ga..Kr)
  1.24, 1.21, 1.21, 1.16, 1.14, 1.17,
  // 5th row Rb, Sr
  2.10, 1.85,
  // 2nd row TM (Y..Cd)
  1.63, 1.54, 1.47, 1.38, 1.28, 1.25, 1.25, 1.20, 1.28, 1.36,
  // 5th row p-block (In..Xe)
  1.42, 1.40, 1.40, 1.36, 1.33, 1.31,
  // 6th row Cs, Ba
  2.32, 1.96,
  // Lanthanides La..Gd
  1.80, 1.63, 1.76, 1.74, 1.73, 1.72, 1.68,
  // Lanthanides Tb..Yb
  1.69, 1.68, 1.67, 1.66, 1.65, 1.64, 1.70,
  // 3rd row TM (Lu..Hg)
  1.62, 1.52, 1.46, 1.37, 1.31, 1.29, 1.22, 1.23, 1.24, 1.33,
  // 6th row p-block (Tl..Rn)
  1.44, 1.44, 1.51, 1.45, 1.47, 1.42,
  // 7th row Fr, Ra
  2.23, 2.01,
  // Actinides (Ac.. Am)
  1.86, 1.75, 1.69, 1.70, 1.71, 1.72, 1.66,
  // Actinides (Cm..No)
  1.66, 1.68, 1.68, 1.65, 1.67, 1.73, 1.76,
  // Trans-actinides
  1.61, 1.57, 1.49, 1.43, 1.41, 1.34, 1.29, 1.28, 1.21, 1.22, 1.36, 1.43, 1.62,
  1.75, 1.65, 1.57
};

const unsigned char element_color[element_count][3] = {
  // See, for example http://jmol.sourceforge.net/jscolors/index.en.html
  // Changes - H is not completely white to add contrast on light backgrounds
  //         - C is slightly darker (i.e. 50% gray - consistent with Avo1)
  //         - F is bluer to add contrast with Cl (e.g. CFC compounds)
  // # Du             # H                 # He
  { 17, 127, 178 },
  { 240, 240, 240 },
  { 217, 255, 255 },
  { 204, 128, 255 },
  // # Be, B, C, N
  { 194, 255, 0 },
  { 255, 181, 181 },
  { 127, 127, 127 },
  { 48, 80, 255 },
  // # O, F, Ne, Na
  { 255, 13, 13 },
  { 178, 255, 255 },
  { 178, 227, 245 },
  { 171, 91, 242 },
  // # Mg
  { 138, 255, 0 },
  { 191, 166, 166 },
  { 240, 200, 160 },
  { 255, 128, 0 },
  // # S
  { 255, 255, 48 },
  { 31, 240, 31 },
  { 128, 209, 227 },
  { 143, 64, 212 },
  // # Ca
  { 61, 255, 0 },
  { 230, 230, 230 },
  { 191, 194, 199 },
  { 166, 166, 171 },
  // # Cr
  { 138, 153, 199 },
  { 156, 122, 199 },
  { 224, 102, 51 },
  { 240, 144, 160 },
  // # Ni
  { 80, 208, 80 },
  { 200, 128, 51 },
  { 125, 128, 176 },
  { 194, 143, 143 },
  // # Ge
  { 102, 143, 143 },
  { 189, 128, 227 },
  { 255, 161, 0 },
  { 166, 41, 41 },
  // # Kr
  { 92, 184, 209 },
  { 112, 46, 176 },
  { 0, 255, 0 },
  { 148, 255, 255 },
  // # Zr
  { 148, 224, 224 },
  { 115, 194, 201 },
  { 84, 181, 181 },
  { 59, 158, 158 },
  // # Ru
  { 36, 143, 143 },
  { 10, 125, 140 },
  { 0, 105, 133 },
  { 192, 192, 192 },
  // # Cd
  { 255, 217, 143 },
  { 166, 117, 115 },
  { 102, 128, 128 },
  { 158, 99, 181 },
  // # Te
  { 211, 122, 0 },
  { 148, 0, 148 },
  { 66, 158, 176 },
  { 87, 23, 143 },
  // # Ba
  { 0, 201, 0 },
  { 112, 212, 255 },
  { 255, 255, 199 },
  { 217, 255, 199 },
  // # Nd
  { 199, 255, 199 },
  { 163, 255, 199 },
  { 143, 255, 199 },
  { 97, 255, 199 },
  // # Gd
  { 69, 255, 199 },
  { 48, 255, 199 },
  { 31, 255, 199 },
  { 0, 255, 156 },
  // # Er
  { 0, 230, 117 },
  { 0, 212, 82 },
  { 0, 191, 56 },
  { 0, 171, 36 },
  // # Hf
  { 77, 194, 255 },
  { 77, 166, 255 },
  { 33, 148, 214 },
  { 38, 102, 150 },
  // # Os
  { 38, 102, 150 },
  { 23, 84, 135 },
  { 208, 208, 224 },
  { 255, 209, 35 },
  // # Hg
  { 184, 194, 208 },
  { 166, 84, 77 },
  { 87, 89, 97 },
  { 158, 79, 181 },
  // # Po
  { 171, 92, 0 },
  { 117, 79, 69 },
  { 66, 130, 150 },
  { 66, 0, 102 },
  { 0, 124, 0 },
  { 112, 170, 249 },
  { 0, 186, 255 },
  { 0, 160, 255 },
  { 0, 142, 255 },
  { 0, 127, 255 },
  { 0, 107, 255 },
  { 84, 91, 242 },
  { 119, 91, 226 },
  { 137, 79, 226 },
  { 160, 53, 211 },
  { 178, 30, 211 },
  { 178, 30, 186 },
  { 178, 12, 165 },
  { 188, 12, 135 },
  { 198, 0, 102 },
  { 204, 0, 89 },
  { 209, 0, 79 },
  { 216, 0, 68 },
  { 224, 0, 56 },
  { 229, 0, 45 },
  { 232, 0, 38 },
  { 234, 0, 35 },
  { 237, 0, 33 },
  { 239, 0, 30 },
  { 242, 0, 28 },
  { 244, 0, 25 },
  { 247, 0, 22 },
  { 249, 0, 20 },
  { 252, 0, 17 },
  { 255, 0, 15 }
};

// Note: Using const std::vector instead of constexpr std::initializer_list
// because std::initializer_list doesn't own its backing storage, which causes
// access violations on MSVC when the temporary arrays are not preserved.
const std::vector<isotope> isotopes[element_count] = {
  // Common isotopes for each element from NIST
  // https://www.nist.gov/pml/atomic-weights-and-isotopic-compositions-relative-atomic-masses
  // for elements with multiple isotopes, the most common is listed as isotope
  // 0
  // clang-format off
{ {0, 0.0} },
{ {0, 1.00782503223},{1, 1.00782503223}, {2, 2.01410177812}, {3, 3.0160492779} },
{ {0, 4.00260325413},{3, 3.0160293201}, {4, 4.00260325413} },
{ {0, 7.0160034366},{6, 6.0151228874}, {7, 7.0160034366} },
{ {9, 9.012183065} },
{ {0, 11.00930536},{10, 10.01293695}, {11, 11.00930536} },
{ {0, 12.0},{12, 12.0}, {13, 13.00335483507}, {14, 14.0032419884} },
{ {0, 14.00307400443},{14, 14.00307400443}, {15, 15.00010889888} },
{ {0, 15.99491461957},{16, 15.99491461957}, {17, 16.9991317565}, {18, 17.99915961286} },
{ {19, 18.99840316273} },
{ {0, 19.9924401762},{20, 19.9924401762}, {21, 20.993846685}, {22, 21.991385114} },
{ {23, 22.989769282} },
{ {0, 23.985041697},{24, 23.985041697}, {25, 24.985836976}, {26, 25.982592968} },
{ {27, 26.98153853} },
{ {0, 27.97692653465},{28, 27.97692653465}, {29, 28.9764946649}, {30, 29.973770136} },
{ {31, 30.97376199842} },
{ {0, 31.9720711744},{32, 31.9720711744}, {33, 32.9714589098}, {34, 33.967867004}, {36, 35.96708071} },
{ {0, 34.968852682},{35, 34.968852682}, {37, 36.965902602} },
{ {0, 39.9623831237},{36, 35.967545105}, {38, 37.96273211}, {40, 39.9623831237} },
{ {0, 38.9637064864},{39, 38.9637064864}, {40, 39.963998166}, {41, 40.9618252579} },
{ {0, 39.962590863},{40, 39.962590863}, {42, 41.95861783}, {43, 42.95876644}, {44, 43.95548156}, {46, 45.953689}, {48, 47.95252276} },
{ {45, 44.95590828} },
{ {0, 47.94794198},{46, 45.95262772}, {47, 46.95175879}, {48, 47.94794198}, {49, 48.94786568}, {50, 49.94478689} },
{ {0, 50.94395704},{50, 49.94715601}, {51, 50.94395704} },
{ {0, 51.94050623},{50, 49.94604183}, {52, 51.94050623}, {53, 52.94064815}, {54, 53.93887916} },
{ {55, 54.93804391} },
{ {0, 55.93493633},{54, 53.93960899}, {56, 55.93493633}, {57, 56.93539284}, {58, 57.93327443} },
{ {59, 58.93319429} },
{ {0, 57.93534241},{58, 57.93534241}, {60, 59.93078588}, {61, 60.93105557}, {62, 61.92834537}, {64, 63.92796682} },
{ {0, 62.92959772},{63, 62.92959772}, {65, 64.9277897} },
{ {0, 63.92914201},{64, 63.92914201}, {66, 65.92603381}, {67, 66.92712775}, {68, 67.92484455}, {70, 69.9253192} },
{ {0, 68.9255735},{69, 68.9255735}, {71, 70.92470258} },
{ {0, 73.921177761},{70, 69.92424875}, {72, 71.922075826}, {73, 72.923458956}, {74, 73.921177761}, {76, 75.921402726} },
{ {75, 74.92159457} },
{ {0, 79.9165218},{74, 73.922475934}, {76, 75.919213704}, {77, 76.919914154}, {78, 77.91730928}, {80, 79.9165218}, {82, 81.9166995} },
{ {0, 78.9183376},{79, 78.9183376}, {81, 80.9162897} },
{ {0, 83.9114977282},{78, 77.92036494}, {80, 79.91637808}, {82, 81.91348273}, {83, 82.91412716}, {84, 83.9114977282}, {86, 85.9106106269} },
{ {0, 84.9117897379},{85, 84.9117897379}, {87, 86.909180531} },
{ {0, 87.9056125},{84, 83.9134191}, {86, 85.9092606}, {87, 86.9088775}, {88, 87.9056125} },
{ {89, 88.9058403} },
{ {0, 89.9046977},{90, 89.9046977}, {91, 90.9056396}, {92, 91.9050347}, {94, 93.9063108}, {96, 95.9082714} },
{ {93, 92.906373} },
{ {0, 97.90540482},{92, 91.90680796}, {94, 93.9050849}, {95, 94.90583877}, {96, 95.90467612}, {97, 96.90601812}, {98, 97.90540482}, {100, 99.9074718} },
{ {0, 96.9063667},{97, 96.9063667}, {98, 97.9072124}, {99, 98.9062508} },
{ {0, 101.9043441},{96, 95.90759025}, {98, 97.9052868}, {99, 98.9059341}, {100, 99.9042143}, {101, 100.9055769}, {102, 101.9043441}, {104, 103.9054275} },
{ {103, 102.905498} },
{ {0, 105.9034804},{102, 101.9056022}, {104, 103.9040305}, {105, 104.9050796}, {106, 105.9034804}, {108, 107.9038916}, {110, 109.9051722} },
{ {0, 106.9050916},{107, 106.9050916}, {109, 108.9047553} },
{ {0, 113.90336509},{106, 105.9064599}, {108, 107.9041834}, {110, 109.90300661}, {111, 110.90418287}, {112, 111.90276287}, {113, 112.90440813}, {114, 113.90336509}, {116, 115.90476315} },
{ {0, 114.903878776},{113, 112.90406184}, {115, 114.903878776} },
{ {0, 119.90220163},{112, 111.90482387}, {114, 113.9027827}, {115, 114.903344699}, {116, 115.9017428}, {117, 116.90295398}, {118, 117.90160657}, {119, 118.90331117}, {120, 119.90220163}, {122, 121.9034438}, {124, 123.9052766} },
{ {0, 120.903812},{121, 120.903812}, {123, 122.9042132} },
{ {0, 129.906222748},{120, 119.9040593}, {122, 121.9030435}, {123, 122.9042698}, {124, 123.9028171}, {125, 124.9044299}, {126, 125.9033109}, {128, 127.90446128}, {130, 129.906222748} },
{ {127, 126.9044719} },
{ {0, 131.9041550856},{124, 123.905892}, {126, 125.9042983}, {128, 127.903531}, {129, 128.9047808611}, {130, 129.903509349}, {131, 130.90508406}, {132, 131.9041550856}, {134, 133.90539466}, {136, 135.907214484} },
{ {133, 132.905451961} },
{ {0, 137.905247},{130, 129.9063207}, {132, 131.9050611}, {134, 133.90450818}, {135, 134.90568838}, {136, 135.90457573}, {137, 136.90582714}, {138, 137.905247} },
{ {0, 138.9063563},{138, 137.9071149}, {139, 138.9063563} },
{ {0, 139.9054431},{136, 135.90712921}, {138, 137.905991}, {140, 139.9054431}, {142, 141.9092504} },
{ {141, 140.9076576} },
{ {0, 141.907729},{142, 141.907729}, {143, 142.90982}, {144, 143.910093}, {145, 144.9125793}, {146, 145.9131226}, {148, 147.9168993}, {150, 149.9209022} },
{ {0, 144.9127559},{145, 144.9127559}, {147, 146.915145} },
{ {0, 151.9197397},{144, 143.9120065}, {147, 146.9149044}, {148, 147.9148292}, {149, 148.9171921}, {150, 149.9172829}, {152, 151.9197397}, {154, 153.9222169} },
{ {0, 152.921238},{151, 150.9198578}, {153, 152.921238} },
{ {0, 157.9241123},{152, 151.9197995}, {154, 153.9208741}, {155, 154.9226305}, {156, 155.9221312}, {157, 156.9239686}, {158, 157.9241123}, {160, 159.9270624} },
{ {159, 158.9253547} },
{ {0, 163.9291819},{156, 155.9242847}, {158, 157.9244159}, {160, 159.9252046}, {161, 160.9269405}, {162, 161.9268056}, {163, 162.9287383}, {164, 163.9291819} },
{ {165, 164.9303288} },
{ {0, 165.9302995},{162, 161.9287884}, {164, 163.9292088}, {166, 165.9302995}, {167, 166.9320546}, {168, 167.9323767}, {170, 169.9354702} },
{ {169, 168.9342179} },
{ {0, 173.9388664},{168, 167.9338896}, {170, 169.9347664}, {171, 170.9363302}, {172, 171.9363859}, {173, 172.9382151}, {174, 173.9388664}, {176, 175.9425764} },
{ {0, 174.9407752},{175, 174.9407752}, {176, 175.9426897} },
{ {0, 179.946557},{174, 173.9400461}, {176, 175.9414076}, {177, 176.9432277}, {178, 177.9437058}, {179, 178.9458232}, {180, 179.946557} },
{ {0, 180.9479958},{180, 179.9474648}, {181, 180.9479958} },
{ {0, 183.95093092},{180, 179.9467108}, {182, 181.94820394}, {183, 182.95022275}, {184, 183.95093092}, {186, 185.9543628} },
{ {0, 186.9557501},{185, 184.9529545}, {187, 186.9557501} },
{ {0, 191.961477},{184, 183.9524885}, {186, 185.953835}, {187, 186.9557474}, {188, 187.9558352}, {189, 188.9581442}, {190, 189.9584437}, {192, 191.961477} },
{ {0, 192.9629216},{191, 190.9605893}, {193, 192.9629216} },
{ {0, 194.9647917},{190, 189.9599297}, {192, 191.9610387}, {194, 193.9626809}, {195, 194.9647917}, {196, 195.96495209}, {198, 197.9678949} },
{ {197, 196.96656879} },
{ {0, 201.9706434},{196, 195.9658326}, {198, 197.9667686}, {199, 198.96828064}, {200, 199.96832659}, {201, 200.97030284}, {202, 201.9706434}, {204, 203.97349398} },
{ {0, 204.9744278},{203, 202.9723446}, {205, 204.9744278} },
{ {0, 207.9766525},{204, 203.973044}, {206, 205.9744657}, {207, 206.9758973}, {208, 207.9766525} },
{ {209, 208.9803991} },
{ {0, 208.9824308},{209, 208.9824308}, {210, 209.9828741} },
{ {0, 209.9871479},{210, 209.9871479}, {211, 210.9874966} },
{ {0, 210.9906011},{211, 210.9906011}, {220, 220.0113941}, {222, 222.0175782} },
{ {223, 223.019736} },
{ {0, 223.0185023},{223, 223.0185023}, {224, 224.020212}, {226, 226.0254103}, {228, 228.0310707} },
{ {227, 227.0277523} },
{ {0, 230.0331341},{230, 230.0331341}, {232, 232.0380558} },
{ {231, 231.0358842} },
{ {0, 233.0396355},{233, 233.0396355}, {234, 234.0409523}, {235, 235.0439301}, {236, 236.0455682}, {238, 238.0507884} },
{ {0, 236.04657},{236, 236.04657}, {237, 237.0481736} },
{ {0, 238.0495601},{238, 238.0495601}, {239, 239.0521636}, {240, 240.0538138}, {241, 241.0568517}, {242, 242.0587428}, {244, 244.0642053} },
{ {0, 241.0568293},{241, 241.0568293}, {243, 243.0613813} },
{ {0, 243.0613893},{243, 243.0613893}, {244, 244.0627528}, {245, 245.0654915}, {246, 246.0672238}, {247, 247.0703541}, {248, 248.0723499} },
{ {0, 247.0703073},{247, 247.0703073}, {249, 249.0749877} },
{ {0, 249.0748539},{249, 249.0748539}, {250, 250.0764062}, {251, 251.0795886}, {252, 252.0816272} },
{ {252, 252.08298} },
{ {257, 257.0951061} },
{ {0, 258.0984315},{258, 258.0984315}, {260, 260.10365} },
{ {259, 259.10103} },
{ {262, 262.10961} },
{ {267, 267.12179} },
{ {268, 268.12567} },
{ {271, 271.13393} },
{ {272, 272.13826} },
{ {270, 270.13429} },
{ {276, 276.15159} },
{ {281, 281.16451} },
{ {280, 280.16514} },
{ {285, 285.17712} },
{ {284, 284.17873} },
{ {289, 289.19042} },
{ {288, 288.19274} },
{ {293, 293.20449} },
{ {292, 292.20746} },
{ {294, 294.21392} },
  // clang-format on
};

} // namespace Avogadro::Core

#endif
