// -*- C++ -*-
#include "Rivet/Analysis.hh"
#include "Rivet/Projections/Beam.hh"
#include "Rivet/Projections/Sphericity.hh"
#include "Rivet/Projections/UnstableParticles.hh"
#include "Rivet/Projections/ChargedFinalState.hh"

namespace Rivet {


  /// @brief baryons at 34.8 and 42.1 GeV
  class TASSO_1989_I266893 : public Analysis {
  public:

    /// Constructor
    RIVET_DEFAULT_ANALYSIS_CTOR(TASSO_1989_I266893);

    /// @name Analysis methods
    /// @{

    /// Book histograms and initialise projections before the run
    void init() {

      // Initialise and register projections
      declare(Beam(), "Beams");
      declare(UnstableParticles(), "UFS");
      const ChargedFinalState cfs;
      declare(cfs, "CFS");
      declare(Sphericity(cfs), "Sphericity");

      // Book histograms
      size_t ih = 0;
      for (double eVal : allowedEnergies()) {
        const string en = toString(round(eVal/MeV));
        if (isCompatibleWithSqrtS(eVal))  _sqs = en;
        book(_h[en+"lam_p"],     6*ih+3, 1, 1);
        book(_h[en+"lam_pL"],    6*ih+4, 1, 1);
        book(_h[en+"lam_pTIn"],  6*ih+5, 1, 1);
        book(_h[en+"lam_pTOut"], 6*ih+6, 1, 1);
        book(_h[en+"lam_rap"],   6*ih+7, 1, 1);
        book(_h[en+"lam_x"],     6*ih+8, 1, 1);
        book(_p[en+"lam_S_1"],  15+ih,   1, 1);
        book(_p[en+"lam_S_2"],  15+ih,   1, 2);
        if (en == "34800"s) {
          book(_h["xi_p"],     18, 1, 1);
          book(_h["xi_pL"],    19, 1, 1);
          book(_h["xi_pTIn"],  20, 1, 1);
          book(_h["xi_pTOut"], 21, 1, 1);
          book(_h["xi_rap"],   22, 1, 1);
          book(_h["xi_x"],     23, 1, 1);
        }
        ++ih;
      }
      raiseBeamErrorIf(_sqs.empty());
    }


    /// Perform the per-event analysis
    void analyze(const Event& event) {
      const ChargedFinalState& cfs = apply<ChargedFinalState>(event, "CFS");
      const size_t numParticles = cfs.particles().size();

      // Even if we only generate hadronic events, we still need a cut on numCharged >= 2.
      if (numParticles < 2)  vetoEvent;

      // Get beams and average beam momentum
      const ParticlePair& beams = apply<Beam>(event, "Beams").beams();
      const double meanBeamMom = 0.5*(beams.first.p3().mod() + beams.second.p3().mod());
      const Sphericity& sphericity = apply<Sphericity>(event, "Sphericity");
      unsigned int nLam(0);
      UnstableParticles ufs = apply<UnstableParticles>(event,"UFS");
      for (const Particle& p : ufs.particles(Cuts::abspid==3122 || Cuts::abspid==3312)) {
      	unsigned int id = p.abspid();
      	double xE = p.E()/meanBeamMom;
      	Vector3 mom3 = p.p3();
        const double energy = p.E();
      	double modp = mom3.mod();
      	double beta = modp/energy;
        const double momS = dot(sphericity.sphericityAxis(), mom3);
        const double pTinS = dot(mom3, sphericity.sphericityMajorAxis());
        const double pToutS = dot(mom3, sphericity.sphericityMinorAxis());
        const double rapidityS = 0.5 * std::log((energy + momS) / (energy - momS));
      	if (id==3122) {
      	  _h[_sqs+"lam_x"]->fill(xE,1./beta);
      	  _h[_sqs+"lam_p"]->fill(modp/GeV);
      	  _h[_sqs+"lam_pL"]->fill(abs(momS)/GeV);
      	  _h[_sqs+"lam_pTIn"]->fill(abs(pTinS)/GeV);
      	  _h[_sqs+"lam_pTOut"]->fill(abs(pToutS)/GeV);
      	  _h[_sqs+"lam_rap"]->fill(abs(rapidityS));
          ++nLam;
      	}
        else if (_sqs == "34800"s) {
      	  _h["xi_x"]->fill(xE,1./beta);
      	  _h["xi_p"]->fill(modp/GeV);
      	  _h["xi_pL"]->fill(abs(momS)/GeV);
      	  _h["xi_pTIn"]->fill(abs(pTinS)/GeV);
      	  _h["xi_pTOut"]->fill(abs(pToutS)/GeV);
      	  _h["xi_rap"]->fill(abs(rapidityS));
      	}
      }
      const double sphere = sphericity.sphericity();
      _p[_sqs+"lam_S_1"]->fill(sphere,nLam);
      _p[_sqs+"lam_S_2"]->fill(sphere,cfs.particles().size());
    }


    /// Normalise histograms etc., after the run
    void finalize() {
      size_t ih = 0;
      Estimate1DPtr temp;
      scale(_h, crossSection()/nanobarn/sumOfWeights());
      for (double eVal : allowedEnergies()) {
        const string en = toString(round(eVal/MeV));
        scale(_h[en+"lam_x"], sqr(eVal));
        if (en == "34800"s)  scale(_h["xi_x"], sqr(eVal));
        book(temp, 15+ih, 1, 3);
        divide(_p[en+"lam_S_1"], _p[en+"lam_S_2"], temp);
        ++ih;
      }
    }

    /// @}


    /// @name Histograms
    /// @{
    map<string,Histo1DPtr> _h;
    map<string,Profile1DPtr> _p;
    string _sqs = "";
    /// @}


  };


  RIVET_DECLARE_PLUGIN(TASSO_1989_I266893);
}
