/**********************************************************************
**   Copyright (C) 2003, International Business Machines Corp.
**   Copyright (C) 2003, Suse Linux AG 
**
**   This program is free software;  you can redistribute it and/or modify
**   it under the terms of the GNU General Public License as published by
**   the Free Software Foundation; either version 2 of the License, or
**   (at your option) any later version.
**
**   This program is distributed in the hope that it will be useful,
**   but WITHOUT ANY WARRANTY;  without even the implied warranty of
**   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
**   the GNU General Public License for more details.
**
**   You should have received a copy of the GNU General Public License
**   along with this pronram;  if not, write to the Free Software
**   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
**
**	Author: 
**		Jerone Young  <jyoung5@us.ibm.com>
**		Olaf Kirch <okir@suse.de>
**
************************************************************************/

#define _XOPEN_SOURCE	/* needed to get strptime prototype */
#define _GNU_SOURCE	/* ... but we do need caddr_t and some other stuff */

#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <getopt.h>
#include <errno.h>
#include <string.h>
#include <pwd.h>
#include <sys/types.h>
#include <sys/syscall.h>
#include <time.h>

#include <laus.h>
#include <linux/laus_audit.h>
#include <laussrv.h>

/* aud_message */
#define	AUDIT_ID	0x0001
#define LOGIN_UID	0x0002
#define EVENT_TYPE	0x0004
#define PID		0x0008
#define SEQUENCE_NUM    0x0010
#define TIME_BEGIN	0x0020
#define TIME_END	0x0040
#define EUID            0x0080
#define RUID		0x0100
#define SUID		0x0200
#define FSUID		0x0400
#define EGID		0x0800
#define RGID		0x1000
#define SGID		0x2000
#define FSGID		0x4000

/* aud_meg_text */
#define TEXT_DATA	0x0001
#define	TEXT_AUD_TAG	0x0002

/* aud_msg_netlink */
#define NETLINK_GROUPS 		0x0001
#define NETLINK_DST_GROUPS	0x0002
#define NETLINK_RESULT		0x0004

/* aud_msg_login */
#define LOGIN_HOSTNAME 	0x0001
#define LOGIN_ADDRESS	0x0002
#define LOGIN_TERMINAL 	0x0004
#define LOGIN_EXECUTE	0x0008

/* aud_msg_syscall */
#define SYSCALL_MAJOR 	0x0001
#define SYSCALL_MINOR 	0x0002
#define SYSCALL_RESULT	0x0004
#define SYSCALL_DATA	0x0008

/* aud_msg_exit */
#define EXIT_CODE 	0x0001

/* Defaults */
static unsigned int grep_opt_flags = 0;
static char *	opt_log_file = PATH_LOGFILE;
static int 	opt_print_caption = 0;
static int      opt_print_flags = AUDPR_TIME_FMT_ISO8601
				| AUDPR_PROCESSID
				| AUDPR_LOGINID
				| AUDPR_EVNAME
				| AUDPR_SEQNR;

/* aud message variables */
static unsigned int uid;
static unsigned int event_type;
static unsigned int audit_id;
static unsigned int pid;
static unsigned int seqnr;
static unsigned int euid;
static unsigned int suid;
static unsigned int ruid;
static unsigned int fsuid;
static unsigned int egid;
static unsigned int rgid;
static unsigned int sgid;
static unsigned int fsgid; 

/* stuff for syscalls */
static unsigned int text_opt=0;
static unsigned int netlink_opt=0;
static unsigned int login_opt=0;
static unsigned int syscall_opt=0;
static unsigned int exit_opt=0;

static char * syscall_name;

static char * text_data;
static char * text_aud_tag;

static unsigned int groups;
static unsigned int dst_groups;
static int netlink_result;

static struct tm start_tm;
static time_t start_time;
static struct tm end_tm;
static time_t end_time;
static struct passwd * pw;

static char * login_hostname;
static char * login_terminal;
static char * login_address;
static char * login_execute;

static unsigned int syscall_major;
static unsigned int syscall_minor;
static int syscall_result;
static char * syscall_data_arg;

static long exit_code;


/* Special Handlers for syscall arguments */
#define MAX_ARGS        8

struct syscall_data {
	unsigned int	major, minor;
	int		result;
	unsigned int	nargs;
	struct syscall_arg {
		int		type;
		const void	*data;
		size_t		len;
	}			args[MAX_ARGS];	
};


/* function protypes */
static audit_callback_fn_t handle_record;
static int grep_record (const struct aud_message *msg);
static int grep_login_opt (const struct aud_message *msg);
static int grep_syscall_opt (const struct aud_message *msg);
static int check_syscall_args (const unsigned char *args, unsigned int size, const struct aud_msg_syscall *aud);
static int check_syscall_args_default(struct syscall_arg *arg);
static int check_syscall_args_ioctl(struct syscall_data *data);
static int grep_netlink_opt (const struct aud_message *msg);
static int grep_text_opt (const struct aud_message *msg);
static int grep_exit_opt (const struct aud_message *msg);
static int check_start_time(time_t timestamp);
static int check_end_time(time_t timestamp);
static void usage(int exit_status);

/*
 * MAIN
 */

int main ( int argc, char ** argv ) {
	
	int c;
	int option_index = 0;

	static struct option long_options[] = {
		{"help",0,0,'?'},
		{"filename",1,0,'f'},
		{"header",0,0,'h'},
		{"time",1,0,'t'},
		{"starttime",1,0,'s'},
		{"endtime",1,0,'x'},
		{"uid",1,0,'u'},
		{"loginid",1,0,'l'},
		{"event",1,0,'e'},
		{"auditid",1,0,'a'},
		{"pid",1,0,'p'},
		{"sequencenum",1,0,'n'},
		{"verboseall",1,0,'v'},
		{"hostname",1,0,'H'},
		{"address",1,0,'A'},
		{"terminal",1,0,'T'},
		{"execute",1,0,'E'},
		{"major",1,0,'M'},
		{"minor",1,0,'N'},
		{"syscall",1,0,'S'},
		{"sysresult",1,0,'R'},
		{"sysdata",1,0,'D'},
		{"group",1,0,'G'},
		{"dstgroup",1,0,'I'},
		{"netresult",1,0,'L'},
		{"audtag",1,0,'U'},
		{"textdata",1,0,'X'},
		{"exitcode",1,0,'C'},
		{"euid",1,0,1},
		{"suid",1,0,2},
		{"ruid",1,0,3},
		{"fsuid",1,0,4},
		{"egid",1,0,5},
		{"rgid",1,0,6},
		{"sgid",1,0,7},
		{"fsgid",1,0,8},
		{"follow",1,0,'F'},
		{0, 0, 0, 0}
	};
	
	const char * short_options = "h?vf:t:u:l:r:e:a:n:p:x:s:H:A:T:E:M:N:S:R:D:"
				     "G:I:L:U:X:C:F";

	audit_check_biarch(argv);

	while ((c=getopt_long(argc,argv, short_options, 
		long_options, &option_index)) != -1) {
		
		switch(c){
		case 'F':
			opt_print_flags |= AUDPR_FOLLOW;
			break;
		case 'f':
			opt_log_file=optarg;	
			break;	
		case 'h':
			opt_print_caption = 1;
			break;
		case 't':
			opt_print_flags &= ~AUDPR_TIME;
			if (!strcasecmp(optarg, "unix")) {
				opt_print_flags |= AUDPR_TIME_FMT_UNIX;
			} else
			if (!strcasecmp(optarg, "iso8601")) {
				opt_print_flags |= AUDPR_TIME_FMT_ISO8601;
			} else
			if (!strcasecmp(optarg, "raw")) {
				opt_print_flags |= AUDPR_TIME_FMT_RAW;
			} else
			if (!strcasecmp(optarg, "none")) {
				/* nothing */
			} else {
				fprintf(stderr, "Invalid time format \"%s\"\n",optarg); 
			}
			break;
		case 1:
			if (!atoi(optarg)  && strcmp(optarg, "0")) {
                                fprintf (stderr,"Invalid euid %s\n",optarg);
                                exit(1);
                        }
			euid=atoi(optarg);
			grep_opt_flags |= EUID;
			break;
		case 2:
			if (!atoi(optarg)  && strcmp(optarg, "0")) {
                                fprintf (stderr,"Invalid suid %s\n",optarg);
                                exit(1);
                        }
			suid=atoi(optarg);
			grep_opt_flags |=SUID;
			break;
		case 3:
			if (!atoi(optarg)  && strcmp(optarg, "0")) {
				fprintf (stderr,"Invalid ruid %s\n",optarg);
                                exit(1);
                        }
			ruid=atoi(optarg);
			grep_opt_flags |=RUID;
			break;
		case 4:
			if (!atoi(optarg) && strcmp(optarg, "0")) {
				fprintf (stderr, "Invalid fsuid %s\n",optarg);
				exit(1);
			}
			fsuid=atoi(optarg);
			grep_opt_flags |=FSUID;
			break;
		case 5:
			 if (!atoi(optarg) && strcmp(optarg, "0")) {
                                fprintf (stderr, "Invalid egid %s\n",optarg);
                                exit(1);
			}
			egid=atoi(optarg);
			grep_opt_flags |=EGID;
			break;
		case 6:
			if (!atoi(optarg) && strcmp(optarg, "0")) {
				fprintf (stderr, "Invalid rgid %s\n",optarg);
                                exit(1);
			}
			rgid=atoi(optarg);
			grep_opt_flags |=RGID;
			break;
		case 7:
			if (!atoi(optarg) && strcmp(optarg, "0")) {
                                fprintf (stderr, "Invalid sgid %s\n",optarg);
                                exit(1);
                        }
			sgid=atoi(optarg);
			grep_opt_flags |=SGID;
			break;
		case 8:
			if (!atoi(optarg) && strcmp(optarg, "0")) {
				fprintf (stderr, "Invalid fsgid %s\n",optarg);
				exit(1);
			}
			fsgid=atoi(optarg);
			grep_opt_flags |=FSGID;
			break;
		case 'u':
			if (!atoi(optarg)  && strcmp(optarg, "0")) {
				fprintf (stderr,"Invalid uid %s\n",optarg);
				exit(1);
			}
			uid=atoi(optarg);
			grep_opt_flags |= LOGIN_UID;
			break;
		case 'l':
			if (grep_opt_flags & LOGIN_UID) {
				fprintf (stderr,"A uid has already been specified.\n");
				exit(1); 
			}
			if ((pw = getpwnam(optarg)) == NULL) {
				fprintf (stderr, "Unable to associate uid with login name\n");
				exit(1);
			}
			uid=(unsigned int)pw->pw_uid;
			grep_opt_flags |= LOGIN_UID;
			break;
		case 'e':
			if (grep_opt_flags & EVENT_TYPE) {
				fprintf (stderr, "You may only specify one event type.\n");
				exit(1);
			}

			if (!strcasecmp("SYSCALL", optarg)) {
				event_type=AUDIT_MSG_SYSCALL;
			}
			else if (!strcasecmp("NETLINK", optarg)) {
				event_type=AUDIT_MSG_NETLINK;
			}	
			else if (!strcasecmp("LOGIN", optarg)) {
				event_type=AUDIT_MSG_LOGIN; 
			}
			else if (!strcasecmp("TEXT",optarg)) {
				event_type=AUDIT_MSG_TEXT;
			}
			else if (!strcasecmp("EXIT",optarg)) {
				event_type=AUDIT_MSG_EXIT;
			}
			else {
				fprintf(stderr,"%s is not a valid event type.\n", optarg);
				exit(1);
			}
	
			grep_opt_flags |=EVENT_TYPE;
			break;
		case 'a':
			if (!atoi(optarg)  && strcmp(optarg, "0")) {
                                fprintf (stderr,"Invalid auditid %s\n",optarg);
                                exit(1);
                        }
			audit_id= atoi(optarg);
			grep_opt_flags |=AUDIT_ID;
			break;
		case 'p':
			if (!atoi(optarg)  && strcmp(optarg, "0")) {
				fprintf (stderr,"Invalid pid %s\n",optarg);
				exit(1);
			}
			pid = atoi(optarg);
			grep_opt_flags |= PID;
			break;
		case 'n':
			if (!atoi(optarg)  && strcmp(optarg, "0")) {
                                fprintf (stderr,"Invalid sequence number %s\n",optarg);
                                exit(1);
                       	}
			seqnr=atoi(optarg);
			grep_opt_flags |= SEQUENCE_NUM; 
			break;
		case 's':
			if (!strptime(optarg, "%Y-%m-%dT%H:%M:%S", &start_tm)) {
				fprintf (stderr, "Invalid start time format %s\n", optarg);
				exit(1);
			}	
			start_tm.tm_gmtoff = start_tm.tm_isdst = -1; /* mktime bug workaround! */ 
			start_time = mktime( &start_tm );					
			/* Correct for DST - convert to UTC: */
			start_time -= start_tm.tm_gmtoff; 
			grep_opt_flags |= TIME_BEGIN;
			break;
		case 'v':
			opt_print_flags |= AUDPR_PRINT_ALL;
			break;
		case 'x':
			if (!strptime(optarg, "%Y-%m-%dT%H:%M:%S", &end_tm)) {
				fprintf (stderr, "Invalid end time format %s\n", optarg);
				exit(1);
			}
			end_tm.tm_gmtoff = end_tm.tm_isdst = -1; /* mktime bug workaround! */ 
			end_time = mktime( &end_tm );
			/* Correct for DST - convert to UTC: */
			end_time -= end_tm.tm_gmtoff; 
			grep_opt_flags |= TIME_END;
			break;
		case 'H':
			if (event_type != AUDIT_MSG_LOGIN ) {
				fprintf (stderr, "Must specify event type LOGIN to specify a hostname option .\n");
				exit(1);
			}
			login_hostname=optarg;
			login_opt |= LOGIN_HOSTNAME;
			break;	
		case 'A':
			if (event_type != AUDIT_MSG_LOGIN) {
				fprintf (stderr, "Must specify event type LOGIN to specify an address option.\n");
				exit(1);
			}
			login_address=optarg;
			login_opt |= LOGIN_ADDRESS;
			break;
		case 'T':
			if (event_type != AUDIT_MSG_LOGIN) {
				fprintf (stderr, "Must specify event type LOGIN to specify a terminal option.\n");
				exit(1);
			}
			login_terminal=optarg;
			login_opt |= LOGIN_TERMINAL;
			break;
		case 'E':
			if (event_type != AUDIT_MSG_LOGIN) {
				fprintf (stderr, "Must specify event type LOGIN to specify execute option.\n");
				exit(1);
			}
			login_execute=optarg;
			login_opt |= LOGIN_EXECUTE;
			break;
		case 'M':
			if (event_type != AUDIT_MSG_SYSCALL) {
				fprintf(stderr, "Must specify event type SYSCALL to specify major number.\n");
				exit(1);
			}
			if (syscall_major) {
                                fprintf (stderr,"You may only specify one syscall, you have one or more specified.\n");
                                exit(1);
                        }
			if (!atoi(optarg)  && strcmp(optarg, "0")) {
				fprintf (stderr,"Invalid major number %s\n",optarg);
				exit(1);
			}
			syscall_major= (unsigned int) atoi(optarg);
			syscall_opt |= SYSCALL_MAJOR;
			break;
		case 'N':
			if (event_type != AUDIT_MSG_SYSCALL) {
				fprintf(stderr, "Must specify event type SYSCALL to specify minor number.\n");
				exit(1);
			}
			if (!atoi(optarg)  && strcmp(optarg, "0")) {
				fprintf(stderr, "Invalid minor number %s\n",optarg);
				exit(1);
			}
			syscall_minor=atoi(optarg);
			syscall_opt |= SYSCALL_MINOR;
			break;
		case 'S':
			
			if (event_type != AUDIT_MSG_SYSCALL) {
				fprintf (stderr, "Must specify event type SYSCALL to specify a syscallname option.\n");
				exit(1);
			}

                        syscall_name = optarg;

			syscall_opt |= SYSCALL_MAJOR|SYSCALL_MINOR;
			break;
		case 'R':
			if (event_type != AUDIT_MSG_SYSCALL) {
				fprintf(stderr, "Must specify event type SYSCALL to specify a result option.\n");
				exit(1);
			}
			if (!atoi(optarg)  && strcmp(optarg, "0")) {
				fprintf(stderr, "Invalid result %s\n", optarg); 
				exit(1);
			}
			syscall_result=atoi(optarg);
			syscall_opt |= SYSCALL_RESULT;		
			break;
		case 'D':
			if (event_type != AUDIT_MSG_SYSCALL) {
				fprintf(stderr, "Must specify event type SYSCALL to specify sysdata option.\n");
				exit(1);
			}
			syscall_data_arg=optarg;
			syscall_opt |= SYSCALL_DATA;		
			break;
		case 'G':
			if (event_type !=AUDIT_MSG_NETLINK) {
				fprintf(stderr, "Must specify event type NETLINK to specify group option.\n");
				exit(1);
			}
			if (!atoi(optarg)  && strcmp(optarg, "0")) {
                                fprintf (stderr,"Invalid group %s\n",optarg);
                                exit(1);
                        }
			groups=atoi(optarg);
			netlink_opt |= NETLINK_GROUPS;
			break;
		case 'I':
			if (event_type !=AUDIT_MSG_NETLINK) {
				fprintf(stderr, "Must specify event type NETLINK to specify dstgroup option.\n");
				exit(1);
			}
			if (!atoi(optarg)  && strcmp(optarg, "0")) {
                                fprintf (stderr,"Invalid groupdst %s\n",optarg);
                                exit(1);
                        }
			dst_groups = atoi(optarg);
			netlink_opt |= NETLINK_DST_GROUPS;
			break;
		case 'L':
			if (event_type !=AUDIT_MSG_NETLINK) {
				fprintf(stderr, "Must specify event type NETLINK to specify netresult option.\n");
				exit(1);
			}
			if (!atoi(optarg)  && strcmp(optarg, "0")) {
                                fprintf (stderr,"Invalid net result %s\n",optarg);
                                exit(1);
                        }
			netlink_result = atoi(optarg);
			netlink_opt |= NETLINK_RESULT;
			break;
		case 'U':
			if (event_type != AUDIT_MSG_TEXT) {
				fprintf(stderr, "Must specify event type TEXT to specify textdata option. \n");
				exit(1);
			}
			text_aud_tag = optarg;
			text_opt |= TEXT_AUD_TAG;
			break;
		case 'X':
			if (event_type != AUDIT_MSG_TEXT) {
				fprintf(stderr, "Must specify event type TEXT to specify textdata option. \n");
				exit(1);
			}
			text_data = optarg;
			text_opt |= TEXT_DATA;
			break;
		case 'C':
			if (event_type != AUDIT_MSG_EXIT) {
				fprintf (stderr,"Must specify event type EXIT to specify exitcode option.\n");
				exit(1);
			}
			if (!atoi(optarg)  && strcmp(optarg, "0")) {
				fprintf (stderr,"Invalid exit code %s\n",optarg);
				exit(1);
			}
			exit_code = atoi(optarg);
			exit_opt |= EXIT_CODE;
			break;	
		case '?':
			usage(0);
			break;
		default:
			usage(1);
			break;	
		}
	}

	if (audit_process_log(opt_log_file, handle_record, opt_print_flags) < 0)
		return 1;
	return 0;
}

/*
 * Handles Records
 */
int
handle_record(const struct aud_message *msg, int flags)
{
	/* Resolve syscall names to major & minor number */
        /* Must be done late because of the new  nature of *_name_to_code functions*/
 	if (syscall_name) {
                     
 			int c=0;
                       
                 /* Get syscall major/minor by name */
                        if ((c = syscall_name_to_code(syscall_name)) >= 0) {
                                syscall_major = c;
                                syscall_minor = 0;
                        } else
#ifdef __NR_socketcall
                        if ((c = socketcall_name_to_code(syscall_name)) >= 0) {
                                syscall_major = __NR_socketcall;
                                syscall_minor = c;
                        } else
#endif
#ifdef __NR_ipc
                        if ((c = ipccall_name_to_code(syscall_name)) >= 0) {
                                syscall_major = __NR_ipc;
                                syscall_minor = c;
                        } else
#endif
                        {
                                fprintf(stderr,
                                        "Unknown system call name \"%s\"\n",
                                        syscall_name);
                                exit(1);
                        }

			syscall_name = NULL;
         }


	if (!(grep_record(msg))) {
		/* See if we should print the header */
		if (opt_print_caption) {
			audit_print_caption(flags | opt_print_flags);
			opt_print_caption = 0;
		}


		/* Just print everything we got */
		audit_print(msg, flags | opt_print_flags);
	}
	return 0;
}

/*
 * Grep Record
 */
int
grep_record(const struct aud_message *msg) {

	int tmp_grep = grep_opt_flags;
	time_t timestamp = msg->msg_timestamp;

	/* start time */
	if (tmp_grep & TIME_BEGIN) {
		if (!check_start_time(timestamp))
			tmp_grep &= ~TIME_BEGIN;
	}

	/* end time */
	if (tmp_grep & TIME_END) {
		if (!check_end_time(timestamp))
			tmp_grep &= ~TIME_END;
	}

	/* uid */
	if (tmp_grep & LOGIN_UID) {
		if (msg->msg_login_uid == uid)
			tmp_grep &= ~LOGIN_UID;
	}


	/* audit_id */
	if (tmp_grep & AUDIT_ID) {
		if (msg->msg_audit_id == audit_id)
			tmp_grep &= ~AUDIT_ID;
	}

	/* pid */
	if (tmp_grep & PID) {
		if (msg->msg_pid == (pid_t) pid)
			tmp_grep &= ~PID;
	}

	/* event type */
	if (tmp_grep & EVENT_TYPE) {
		if (msg->msg_type == event_type) {
			
			/* login */
			if (event_type == AUDIT_MSG_LOGIN) {
				if (!grep_login_opt(msg))
					tmp_grep &= ~EVENT_TYPE;
			}
			/* syscall */
			else if (event_type == AUDIT_MSG_SYSCALL) {
				if (!grep_syscall_opt(msg))
					tmp_grep &= ~EVENT_TYPE;		
			}
			/* netlink */
			else if (event_type == AUDIT_MSG_NETLINK) {
				if (!grep_netlink_opt(msg))
					tmp_grep &= ~EVENT_TYPE;
			}
			/* text */
			else if (event_type == AUDIT_MSG_TEXT) {
				if (!grep_text_opt(msg))
					tmp_grep &= ~EVENT_TYPE;
			}
			/* exit */
			else if (event_type == AUDIT_MSG_EXIT) {
				if (!grep_exit_opt(msg))
					tmp_grep &= ~EVENT_TYPE;
			}
		}
	}

	/* sequence number */
	if ( tmp_grep & SEQUENCE_NUM) {
		if (msg->msg_seqnr == seqnr)
			tmp_grep &= ~SEQUENCE_NUM;
	}

	/* euid */
	if (tmp_grep & EUID) {
		if (msg->msg_euid == euid)
			tmp_grep &= ~EUID;
	}

	/* ruid */
	if (tmp_grep & RUID) {
		if (msg->msg_ruid == ruid)
			tmp_grep &= ~RUID;
	}

	/* suid */
	if (tmp_grep & SUID) {
		if (msg->msg_suid == suid)
			tmp_grep &= ~SUID;
	}

	/* fsuid */
	if (tmp_grep & FSUID) {
		if (msg->msg_fsuid == fsuid)
			tmp_grep &= ~FSUID;
	}

	/* egid */
	if (tmp_grep & EGID) {
		if (msg->msg_egid == egid)
			tmp_grep &= ~EGID;
	}
	
	/* rgid */
	if (tmp_grep & RGID) {
		if (msg->msg_rgid == rgid)
			tmp_grep &= ~RGID;
	}

	/* sgid */
	if (tmp_grep & SGID) {
		if (msg->msg_sgid == sgid)
			tmp_grep &= ~SGID;
	}

	/* fsgid */
	if (tmp_grep & FSGID) {
		if (msg->msg_fsgid == fsgid)
			tmp_grep &= ~FSGID;
	}

	return tmp_grep;
}

/*
 * Grep LOGIN options
 */
int
grep_login_opt (const struct aud_message *msg) {
	
	unsigned int tmp_login = login_opt;
	
	const struct aud_msg_login * lgn = (const struct aud_msg_login *) &msg->msg_data;
	
	/* Check hostname */	
	if (tmp_login & LOGIN_HOSTNAME && lgn->hostname[0]) {
		if (!strcmp(lgn->hostname, login_hostname))
			tmp_login &= ~LOGIN_HOSTNAME;
	}
	
	/* Check address */
	if (tmp_login & LOGIN_ADDRESS && lgn->address[0]) {
		if (!strcmp(lgn->address, login_address))
			tmp_login &= ~LOGIN_ADDRESS;
	}

	/* Check terminal */
	if (tmp_login & LOGIN_TERMINAL && lgn->terminal[0]) {
		if (!strcmp(lgn->terminal, login_terminal))
			tmp_login &= ~LOGIN_TERMINAL;
	}

	/* Check execute */
	if (tmp_login & LOGIN_EXECUTE && lgn->executable[0]) {
		if (!strcasecmp(lgn->executable,login_execute ))
			tmp_login &= ~LOGIN_EXECUTE;
	}
		
	return tmp_login;
}

/*
 * Grep SYSCALL options
 */
int
grep_syscall_opt (const struct aud_message *msg) {
	
	const struct aud_msg_syscall	*sym;
	const unsigned char		*args;
	unsigned int		size;

	unsigned int tmp_syscall = syscall_opt;
	
	sym = (const struct aud_msg_syscall *) &msg->msg_data;
	args = sym->data;
	size = sym->length;
	
	/* Check major number */
	if (tmp_syscall & SYSCALL_MAJOR) {
		if ( sym->major == (int) syscall_major)
			tmp_syscall &= ~SYSCALL_MAJOR;
	}

	/* Check minor number */
	if (tmp_syscall & SYSCALL_MINOR) {
		if ( sym->minor == (int) syscall_minor)
			tmp_syscall &= ~SYSCALL_MINOR;
	}
	
	/* Check result number */
	if (tmp_syscall & SYSCALL_RESULT) {
		if (sym->result == syscall_result)	
			tmp_syscall &= ~SYSCALL_RESULT;
	}


	/* Check data (arguments) of syscall */
	if (tmp_syscall & SYSCALL_DATA) {
		if (!check_syscall_args(args,size,sym)){
			tmp_syscall &= ~SYSCALL_DATA;		
		}	
	}

	return tmp_syscall;
}

/*
 * Check SYSCALL arguments
 */
int 
check_syscall_args (const unsigned char *args, unsigned int size, const struct aud_msg_syscall *aud) {
	
	struct syscall_data 	data;
	u_int32_t 		type, len;
	unsigned int		nargs,j;
	
	memset(&data, 0, sizeof(data));
	data.major  = aud->major;
	data.minor  = aud->minor;
	data.result = aud->result;

	for (nargs = 0; size >= 8 && nargs < MAX_ARGS; nargs++) {
		struct syscall_arg	*arg = &data.args[nargs];

		memcpy(&type, args, 4); args += 4;
		memcpy(&len,  args, 4); args += 4;
		size -= 8;

		if (type == AUDIT_ARG_END)
			break;

		arg->type = type;
		arg->data = args;
		arg->len  = len;

		args += len;
		size -= len;
	}
	data.nargs = nargs;

	switch (data.major) {
	case __NR_ioctl:
		if (!check_syscall_args_ioctl(&data))
			return 0;
		break;
	default:
		for (j = 0; j < data.nargs; j++){ 
			if (!check_syscall_args_default(&data.args[j]))
				return 0;
		}
		break;	
	}

	return 1;
}

/*
 * Check regular syscall arguments
 */
int
check_syscall_args_default(struct syscall_arg *arg) {
	
	int type = arg->type;
	size_t len = arg->len;
	const void * data = arg->data;

	u_int64_t       value = 0,tmp_64;
	u_int32_t	tmp_32;
	u_int16_t	tmp_16;
	const unsigned char *addr;

	void * data1;

	char * beg = "[data, len=";
	char * begerr = "[errror=";
	char * end = "]";
	char num[1024];
	char word[1024];

	switch (arg->type) {
	case AUDIT_ARG_IMMEDIATE:
		addr = (const unsigned char *) arg->data;
		if (len == 1) {
			value = *addr;
		} else if (len == 2) {
			memcpy(&tmp_16, addr, 2);
			value = tmp_16;
		} else if (len == 4) {
			memcpy(&tmp_32, addr, 4);
			value = tmp_32;
		} else if (len == 8) {
			memcpy(&tmp_64, addr, 8);
			value = tmp_64;
		}
		
		if (atoi(syscall_data_arg)) { 
			if (value && value == atoi(syscall_data_arg)) 
				return 0;
		}
		break;
	case AUDIT_ARG_STRING:
	case AUDIT_ARG_PATH:
		if (0 < len < 1024) {
			data1 = malloc(len);
			memcpy(data1,data,len);
			if (strstr((char *)data1,syscall_data_arg) != NULL) {
				free (data1);
				return 0;
			}
			free (data1);
		}
		break;
	case AUDIT_ARG_POINTER:
		memset(num,0,sizeof(num));
		memset(word,0,sizeof(word));
		sprintf(num,"%lu", (unsigned long) arg->len);
		strcat(word,beg);
		strcat(word,num);
		strcat(word,end);
		if (strstr(word,syscall_data_arg) != NULL) 
			return 0;
		break; 
	case AUDIT_ARG_NULL:
		if (!strcasecmp("NULL", syscall_data_arg))
			return 0;
		break;
	case AUDIT_ARG_ERROR:
		addr = (const unsigned char *) arg->data;
		if (len == 1) {
			value = *addr;
		} else if (len == 2) {
			memcpy(&tmp_16, addr, 2);
			value = tmp_16;
		} else if (len == 4) {
			memcpy(&tmp_32, addr, 4);
			value = tmp_32;
		} else if (len == 8) {
			memcpy(&tmp_64, addr, 8);
			value = tmp_64;
		}

		memset(num,0,sizeof(num));
		memset(word,0,sizeof(word));
		strcat(word,begerr);
		strcat(word,strerror(value));
		strcat(word,end);
		if (strstr(word,syscall_data_arg) != NULL)
			return 0;		
		break;
	default:
		if (!strcasecmp("???", syscall_data_arg))
			return 0;
		break;
	}

	return 1;
}

/*
 * Check sycall ioctl arguments
 */
int
check_syscall_args_ioctl(struct syscall_data *data) {
	return 1;
}

/*
 * Check NETLINK options
 */
int 
grep_netlink_opt (const struct aud_message *msg) {
	
	unsigned int tmp_netlink = netlink_opt;

	const struct aud_msg_netlink * nk = (const struct aud_msg_netlink *) &msg->msg_data;
	
	/* groups */
	if (tmp_netlink &  NETLINK_GROUPS) {	 
		if ( nk->groups == groups) 
			tmp_netlink &= ~NETLINK_GROUPS;
	}

	/* dst_groups */
	if (tmp_netlink & NETLINK_DST_GROUPS) {
		if ( nk->dst_groups == dst_groups)
			tmp_netlink &= ~NETLINK_DST_GROUPS;
	}
	
	/* result */
	if (tmp_netlink & NETLINK_RESULT) {
		if (nk->result == netlink_result)
			tmp_netlink &= ~NETLINK_RESULT;
	}

	return tmp_netlink;
}

/*
 * Check TEXT options
 */
int
grep_text_opt (const struct aud_message *msg) {
	unsigned int tmp_text = text_opt;
	char * txt = (char *) &msg->msg_data;

	/* check for audit tag */
	if ((tmp_text & TEXT_AUD_TAG)
	 && msg->msg_evname[0]
	  && strstr(msg->msg_evname,text_aud_tag) != NULL)
		tmp_text &= ~TEXT_AUD_TAG;
	
	/* check for match */
	if ((tmp_text & TEXT_DATA)
	 && strstr(txt,text_data) != NULL)
		tmp_text &= ~TEXT_DATA;	
	
	return tmp_text;	
}

/*
 * Check EXIT options
 */
int
grep_exit_opt (const struct aud_message *msg) {
	unsigned int tmp_exit = exit_opt;
	const struct aud_msg_exit * ex = (const struct aud_msg_exit *) &msg->msg_data;
	
	if (tmp_exit & EXIT_CODE) {
		if (exit_code == ex->code) 
			tmp_exit &= ~EXIT_CODE;	
	}
	
	return tmp_exit;
}

/*
 * Check start time
 */
int
check_start_time(time_t timestamp) {
	struct tm rec_tm ;

	/* correct for DST - convert to UTC:  */
	localtime_r(&timestamp, &rec_tm);
	timestamp -= rec_tm.tm_gmtoff;

	return( ( start_time <= timestamp ) ? 0 : -1);
}

/*
 */
int
check_end_time(time_t timestamp) {
	struct tm rec_tm ;

	/* correct for DST - convert to UTC:  */
	localtime_r(&timestamp, &rec_tm);
	timestamp -= rec_tm.tm_gmtoff;

	return( ( end_time >= timestamp ) ? 0 : -1);
}

/*
 * Print out how to use this prog.
 */
void 
usage (int exit_status) {

	printf(
"Usage: augrep [OPTIONS]\n"
"Outputs data of an  audit record into a readable format.\n"
"Example: augrep -v -h -t iso8601 -e LOGIN -H localhost\n"
"\n"
"General Options:\n"
"	-? --help		Print usage screen\n"
"	-f --filename=FILENAME	Choose audit file that you want to read.\n"
"				Default \"/var/log/audit\"\n"
"		\n"
"Format Options:\n"
"	-h --header		Print header.\n"
"	-t --time=TIMEFORMAT	Change format of time.	\n"
"				Options: \"iso8601\", \"unix\",\"raw\", \"none\"\n"
"	-v --verboseall		Print out all variables in message, not all are\n"
"				printed by default.\n"
"\n"
"Grep Options:\n"
"	-a --auditid=SESS_ID	Find a specific session id. \n"
"	-l --loginid=LOGIN_NAME	Find a specific user who logged in with login name. \n"
"				(cannont use if uid has already been specified)	\n"
"	-n --seqencenum=SEQ	Find record with specific sequence number.\n"
"	-p --pid=PID		Find a specific pid.\n"
"	-u --uid=UID		Find a specific login uid.\n"
"				(cannot use if loginid has already been specified)\n"
"	\n"
"	--euid=EUID		Find specific euid.\n"
"	--egid=EGID		Find specific egid. \n"
"	--fsuid=FSUID		Find specific fsuid. \n"
"	--fsgid=FSGID		Find specific fsgid.  \n"
"	--ruid=RUID		Find specific ruid.    \n"
"	--rgid=RGID		Find specific rgid.     \n"
"	--suid=SUID		Find specific suid.      \n"
"	--sgid=SGID		Find specific sgid.       \n"
"	                                                   \n"
"	-s --startime=STARTT	Find any record that started at or after a \n"
"				specific start time. \n"
"				(Time must be in iso8601 Format\n"
"				\"YYYY-MM-DDThh:mm:ss\")\n"
"	-x --endtime=ENDT	Find any record that started before or at a\n"
"				scpecific end time.\n"
"				(Time must be in iso8601 Format \n"
"				 \"YYYY-MM-DDThh:mm:ss\")\n"
"\n"
"	-e --event=EVENT	Find a specific event.\n"
"				Options: \"LOGIN\",\"NETLINK\",\"SYSCALL\"\n"
"					  \"TEXT\", \"EXIT\" \n"
"	\n"
"	LOGIN options:\n"
"		-A --address=IP_ADDRESS Find specific ip address.\n"
"		-E --execute=EXECUTE    Find specific exectable.\n"
"		-H --hostname=HOSTNAME	Find specific hostname.\n"
"		-T --terminal=TERMINAL	Find specific terminal.		\n"
"	\n"
"	NETLINK options:\n"
"		-G --group=GROUP	Find specific group.\n"
"		-I --dstgroup=DSTGROUP	Find specific dstgroup.\n"
"		-L --netresult=RESULT	Find a specific result.\n"
"		\n"
"	TEXT options: 		\n"
"		(User space tools use this, ex. cron & at)\n"
"\n"
"		-U --audtag=AUDIT_TAG Find specific audit tag.\n"
"		-X --textdata=DATA	Find DATA with in text.\n"
"\n"
"	SYSCALL options:\n"
"		-S --syscall=NAME	Find specific syscall by name.\n"
"		-M --major=MAJOR_NUMBER	Find specific major number.\n"
"		-N --minor=MINOR_NUMBER	Find specific minor number.\n"
"		-R --sysresult=RESULT	Find a specific result.\n"
"		-D --sysdata=DATA	Find DATA in arguments of\n"
"					system call.\n"
"\n"
"	EXIT options:\n"
"		-C --exitcode=EXIT_CODE  Find specific exit code.\n"
"\n"
);
	exit (exit_status);
}
