/*
 * aucat - display contents of audit log
 *
 * Copyright (C) 2003, SuSE Linux AG
 * Written by okir@suse.de
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <sys/poll.h>

#include <linux/laus_audit.h>
#include <laussrv.h>

static char *	opt_logfile = PATH_LOGFILE;
static int	opt_print_caption = 0;
static int	opt_print_flags = AUDPR_TIME_FMT_ISO8601
				| AUDPR_SEQNR
				| AUDPR_PROCESSID
				| AUDPR_LOGINID
				| AUDPR_EVNAME;
static void	usage(int);

static audit_callback_fn_t handle_record;

int
main(int argc, char **argv)
{
	int	c;

	audit_check_biarch(argv);

	while ((c = getopt(argc, argv, "?hvFf:t:")) != -1) {
		switch (c) {
		case 'h':
			opt_print_caption = 1;
			break;
		case 'F':
			opt_print_flags |= AUDPR_FOLLOW;
			break;
		case 'f':
			opt_logfile = optarg;
			break;
		case 'v':
			opt_print_flags |= AUDPR_PRINT_ALL;
			break;
		case 't':
			opt_print_flags &= ~AUDPR_TIME;
			if (!strcasecmp(optarg, "unix")) {
				opt_print_flags |= AUDPR_TIME_FMT_UNIX;
			} else
			if (!strcasecmp(optarg, "iso8601")) {
				opt_print_flags |= AUDPR_TIME_FMT_ISO8601;
			} else
			if (!strcasecmp(optarg, "raw")) {
				opt_print_flags |= AUDPR_TIME_FMT_RAW;
			} else
			if (!strcasecmp(optarg, "none")) {
				/* nothing */
			} else {
				fprintf(stderr, "Invalid time format \"%s\"\n",
						optarg);
				usage(1);
			}
			break;
		case '?':
			usage(0);
		default:
			usage(1);
		}
	}

	if (audit_process_log(opt_logfile, handle_record, opt_print_flags) < 0)
		return 1;
	return 0;
}

int
handle_record(const struct aud_message *msg, int flags)
{
	/* See if we should print the header */
	if (opt_print_caption) {
		audit_print_caption(flags | opt_print_flags);
		opt_print_caption = 0;
	}

	/* Just print everything we got */
	audit_print(msg, flags | opt_print_flags);
	return 0;
}

void
usage(int exval)
{
	fprintf(stderr,
		"usage: aucat [-Fhv?] [-t timefmt] [-f logfile]\n");
	exit(exval);
}
