/*
** PAM module for Linux Audit-Subsystem (LAuS)
** written by: Thomas Biege <thomas@suse.de> SuSE Linux AG
** version: 0.1
** last change: 2003-06-16
** licence: GPL
** based on: pam_limits.c
** todo: - make code thread-safe!!!
*/

#include <stdio.h>
#include <stdarg.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>
#include <pwd.h>
#include <netdb.h>	
#include <sys/types.h>
#include <arpa/inet.h>
#include <errno.h>

#include <laus.h>

#define PAM_SM_AUTH
#define PAM_SM_SESSION
#define PAM_SM_ACCOUNT
#include <security/pam_modules.h>

/*
** Set by options
*/
#define CTRL_NONE		0x0000L
#define CTRL_DETACH		0x0001L

/*
** own routines
*/
static void
_pam_log(int err, const char *format, ...)
{
	va_list args;

	va_start(args, format);
	openlog("pam_laus", LOG_CONS|LOG_PID, LOG_AUTH);
	vsyslog(err, format, args);
	va_end(args);
	closelog();
}

/*
** Initialize audit session for user
*/
static int
_pam_audit_login(pam_handle_t *pamh, int flags, int argc, const char **argv)
{
	char		addrbuf[INET6_ADDRSTRLEN];
	char		*user = NULL,
			*host = NULL,
			*addr = NULL,
			*term = NULL;
	int		ctrl_flags;
	uid_t		login_id;
	struct passwd	*pwd;
	struct hostent	*hent;


	/* process arguments */
	ctrl_flags = CTRL_NONE;
	while (argc-- > 0) {
		if (!strcmp(argv[0], "detach"))
			ctrl_flags |= CTRL_DETACH;
		argv++;
	}

	/* get user name */
	if (pam_get_item((const pam_handle_t *) pamh, PAM_USER, (const void **) &user) != PAM_SUCCESS)
	{
		_pam_log(LOG_CRIT, "PAM error: recovering user-name");
		return(PAM_SESSION_ERR);
	}

        /* get user info */
	if ((pwd = getpwnam(user)) == NULL) {
		_pam_log(LOG_CRIT,
			 "error: user-name '%s' does not exist", user);
		return(PAM_SESSION_ERR);
	}
	login_id = pwd->pw_uid;
	endpwent();

	/* get terminal */
	if (pam_get_item((const pam_handle_t *) pamh, PAM_TTY, (const void **)&term) != PAM_SUCCESS) {
		_pam_log(LOG_WARNING, "PAM error: recovering terminal");
		return(PAM_SESSION_ERR);
	}

        /* get remote host */
	if (pam_get_item((const pam_handle_t *) pamh, PAM_RHOST, (const void **)&host) != PAM_SUCCESS) {
		_pam_log(LOG_WARNING, "PAM warning: recovering remote host-name");
		host = NULL;
	}

	/* Resolve hostname and pretty-print address */
	addr = NULL;
       	if (host != NULL) {
		if (!(hent = gethostbyname(host)) || !hent->h_addr) {
			_pam_log(LOG_WARNING,
				 "warning: cannot resolve hostname \"%s\"",
				 host);
		} else
		if (inet_ntop(hent->h_addrtype, hent->h_addr, addrbuf, sizeof(addrbuf))) {
			addr = addrbuf;
		}
	}
	

	/* Fire up the audit subsystem */
	if (laus_init() < 0
	 || laus_open(NULL) < 0)
	 	goto failed;

	if (ctrl_flags & CTRL_DETACH) {
		if (laus_detach() < 0 && errno != EUNATCH)
			goto failed;
	}

	if (laus_attach() < 0
	 || laus_setauditid() < 0
	 || laus_setsession(login_id, host, addr, term) < 0)
		goto failed;

	laus_close();
	return PAM_SUCCESS;

failed:
	_pam_log(LOG_WARNING,
		"Cannot enable auditing: %s\n",
		laus_strerror(errno));
	laus_close();
	return PAM_SESSION_ERR;
}

/*
** PAM routines
*/
PAM_EXTERN int
pam_sm_authenticate(pam_handle_t *pamh, int flags, int argc, const char **argv)
{
	return PAM_SUCCESS;
}

PAM_EXTERN int
pam_sm_setcred(pam_handle_t *pamh, int flags, int argc, const char **argv)
{
	return _pam_audit_login(pamh, flags, argc, argv);
}

PAM_EXTERN int
pam_sm_acct_mgmt(pam_handle_t *pamh, int flags, int argc, const char **argv)
{
	return _pam_audit_login(pamh, flags, argc, argv);
}

PAM_EXTERN int
pam_sm_open_session(pam_handle_t *pamh, int flags, int argc, const char **argv)
{
	return _pam_audit_login(pamh, flags, argc, argv);
}

PAM_EXTERN int
pam_sm_close_session(pam_handle_t *pamh, int flags, int argc, const char **argv)
{
	return(PAM_SUCCESS);
}

/* static module data */
#ifdef PAM_STATIC
struct pam_module _pam_laus_modstruct = {
    "pam_laus",
    pam_sm_authenticate,
    pam_sm_setcred,
    pam_sm_acct_mgmt,
    pam_sm_open_session,
    pam_sm_close_session,
    NULL
};
#endif
