/*
 * process.c - process audit log file
 *
 * Copyright (C) 2003, SuSE Linux AG
 * Written by okir@suse.de
 */

#define _GNU_SOURCE /* for MREMAP_MAYMOVE */
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/mman.h>

#include <laus.h>
#include <laussrv.h>

struct buffer {
	caddr_t	p, end;
};

static void*	audit_next_record(struct buffer *buf, struct laus_record_header *hdr);

static int api_version;

int
audit_process_log(const char *filename, audit_callback_fn_t *func, int flags)
{
	struct laus_file_header *header;
	struct stat	stb;
	struct buffer	buffer;
	caddr_t		data = NULL;
	off_t		offset = 0;
	u_int32_t	header_count;
	int		fd, result = -1;
	struct aud_message *msg = NULL;
	size_t		msg_bufsize = 0;

new_file:
	if ((fd = open(filename, O_RDONLY)) < 0) {
		fprintf(stderr, "Unable to open %s: %m\n", filename);
		return -1;
	}
	if (fstat(fd, &stb) < 0) {
		fprintf(stderr, "Failed to stat %s: %m\n", filename);
		goto done;
	}

	if (stb.st_size == 0) {
		fprintf(stderr, "Empty log file: %s\n", filename);
		result = 0;
		goto done;
	}

	data = (caddr_t) mmap(NULL, stb.st_size, PROT_READ, MAP_SHARED, fd, 0);
	if (data == MAP_FAILED) {
		fprintf(stderr, "Failed to map %s: %m\n", filename);
		goto done;
	}

	/* Initially, we start parsing the data right after the header */
	offset = sizeof(*header);

new_mmap:
	header = (struct laus_file_header *) data;
	if (header->h_version == 0 && header->h_msgversion == 0) {
		fprintf(stderr, "Empty audit log file: %s\n", filename);
		goto done;
	}

	if (header->h_version != LAUS_VERSION) {
		fprintf(stderr,
			"Warning: Audit file version %x not compatible "
			"with library version %x\n"
			"Output may be garbled.\n",
			header->h_version, LAUS_VERSION);
	}
	if (header->h_msgversion != AUDIT_API_VERSION && header->h_msgversion != AUDIT_API_VERSION_OLD) {
		fprintf(stderr,
			"Warning: Audit record format (version %x) not compatible "
			"with library version %x\n"
			"Output may be garbled.\n",
			header->h_msgversion, AUDIT_API_VERSION);
	}
	api_version = header->h_msgversion;
	if (flags & AUDPR_FILEHDR) {
		printf("Audit trail generated on host %s\n", header->h_hostname);
	}

	buffer.p = data + offset;
	buffer.end = data + stb.st_size;

	/* If the file header specifies a length, don't read beyond
	 * the end */
new_records:
	header_count = header->h_count;
	if (header_count != (u_int32_t) -1)
		buffer.end = buffer.p + header_count;

	while (buffer.p < buffer.end) {
		struct laus_record_header	header;
		void				*msg_raw;
		size_t				msg_len;
		struct aud_message		*copy;

		if (!(msg_raw = audit_next_record(&buffer, &header))) {
			/* In follow mode, it happens frequently that
			 * the last record isn't complete yet. */
			if (flags & AUDPR_FOLLOW)
				break;
			fprintf(stderr, "Truncated log file: %s\n", filename);
			goto done;
		}
		/* nul timestamp indicates end of records (this happens
		 * with bin files) */
		if (header.r_time == 0)
			break;

		/* make sure we have enough buffer space for the aud_message, and copy it */
		msg_len = header.r_size - sizeof(header);
		if (msg_len > msg_bufsize) {
			msg = realloc(msg, msg_len);
			if (!msg) {
				fprintf(stderr, "Out of memory (allocating %d bytes)\n", msg_len);
				goto done;
			}
		}
		memcpy(msg, msg_raw, msg_len);

		if (header.r_size >= sizeof(*msg) && header.r_size >= msg->msg_size)
			flags &= ~AUDREC_TRUNC;
		else	
			flags |= AUDREC_TRUNC;
		if ((flags & AUDREC_TRUNC) || ((uintptr_t)msg & (__alignof__(*msg) - 1))) {
			copy = malloc(header.r_size);
			if (copy) {
				memcpy(copy, msg, header.r_size);
				if (flags & AUDREC_TRUNC)
					copy->msg_size = header.r_size;
			}
		}
		else
			copy = NULL;

		func(copy ?: msg, flags);
		free(copy);
	}

	if (flags & AUDPR_FOLLOW) {
		struct stat	stb2;
		int		complained = 0;
		void		*new_data;

		while (1) {
			if (header->h_count != header_count)
				goto new_records;
			sleep(1);
			while (stat(filename, &stb2) < 0) {
				if (!complained) {
					perror(filename);
					complained++;
				}
				sleep(1);
			}

			if (stb.st_dev != stb2.st_dev
			 || stb.st_ino != stb2.st_ino) {
				munmap(data, stb.st_size);
				close(fd);
				goto new_file;
			}
			if (stb2.st_size == stb.st_size)
				continue;
			if (stb2.st_size < stb.st_size) {
				fprintf(stderr, "%s: file was truncated\n", filename);
				munmap(data, stb.st_size);
				close(fd);
				goto new_file;
			}

			offset = buffer.p - data;
			new_data = mremap(data, stb.st_size, stb2.st_size, MREMAP_MAYMOVE);
			if (new_data == MAP_FAILED) {
				perror("mremap failed");
				goto done;
			}
			data = new_data;
			stb = stb2;
			goto new_mmap;
		}
	}

	result = 0;

done:
	if (fd >= 0)
		close(fd);
	if (data != NULL)
		munmap(data, stb.st_size);
	if (msg != NULL)
		free(msg);
	return result;
}

/*
 * Get the next audit record from file
 *
 * Returns a pointer to the next struct aud_message in the mmap region.
 * This is not guaranteed to be aligned suitably for direct access, so
 * the caller must copy it instead of casting it to a (struct aud_message *).
 */
void * /* struct aud_message * */
audit_next_record(struct buffer *buf, struct laus_record_header *hdr)
{
	void	*msg;

	if (buf->p + sizeof(*hdr) > buf->end)
		return NULL;
	memcpy(hdr, buf->p, sizeof(*hdr));
	buf->p += sizeof(*hdr);

	msg = (void *) buf->p;
	if (buf->p + hdr->r_size > buf->end)
		return NULL;

	buf->p += hdr->r_size;

	return msg;
}

int __attribute__((__weak__))
laus_api_version() {
	return api_version;
}

int __attribute__((__weak__))
laus_version() {
	return LAUS_VERSION;
}
