/*
 * Print audit records
 *
 * Copyright (C) 2003, SuSE Linux AG
 * Written by okir@suse.de
 */

#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <stddef.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <pwd.h>

#include <linux/unistd.h>
#include <linux/sys.h>

#include <laus.h>
#include <laussrv.h>
#include "syscall.h"

static void	print_syscall_msg(const struct aud_message *msg);
static void	print_login_msg(const struct aud_message *msg);
static void	print_text_msg(const struct aud_message *msg);
static void	print_netlink_msg(const struct aud_message *msg);
static void	print_exit_msg(const struct aud_message *msg);

int
audit_print(const struct aud_message *msg, int flags)
{
	audit_print_header(msg, flags);

	if (msg->msg_size >= sizeof(*msg)) {
		switch (msg->msg_type) {
		case AUDIT_MSG_LOGIN:
			print_login_msg(msg); break;
		case AUDIT_MSG_TEXT:
			print_text_msg(msg); break;
		case AUDIT_MSG_SYSCALL:
			print_syscall_msg(msg); break;
		case AUDIT_MSG_NETLINK:
			print_netlink_msg(msg); break;
		case AUDIT_MSG_EXIT:
			print_exit_msg(msg); break;
		default:
			printf("[unknown message type %u]", msg->msg_type); break;
		}
	}

	printf("\n");
	return 0;
}

void
audit_print_caption(int flags)
{
	switch (flags & AUDPR_TIME) {
	case AUDPR_TIME_FMT_ISO8601:
		printf("     TIME            "); break;
	case AUDPR_TIME_FMT_UNIX:
		printf("     TIME           "); break;
	case AUDPR_TIME_FMT_RAW:
		printf("   TIME      "); break;
	}
	if (flags & AUDPR_SEQNR)
		printf("SEQNR   ");
	if (flags & AUDPR_PROCESSID)
		printf("PID     ");
	if (flags & AUDPR_LOGINID)
		printf("LOGIN   ");
	if (flags & AUDPR_UID_ALL){
		printf("EUID   ");
		printf("SUID   ");
		printf("RUID   ");
		printf("FSUID  ");
	}
	if (flags & AUDPR_GID_ALL) {
		printf("EGID   ");
		printf("RGID   ");
		printf("SGID   ");
		printf("FSGID   ");
	}
	if (flags & AUDPR_AUDITID)
		printf("SESS   ");
	printf("DATA\n");
}

void
audit_print_header(const struct aud_message *msg, int flags)
{
	if (msg->msg_size < sizeof(*msg)) {
		printf("\n[truncated message header - got only %u bytes]\n", msg->msg_size);
		return;
	}

	if (flags & AUDPR_TIME) {
		time_t		timestamp = msg->msg_timestamp;
		struct tm	*tm;
		char		timestr[64];

		tm = localtime(&timestamp);

		switch (flags & AUDPR_TIME) {
		case AUDPR_TIME_FMT_ISO8601:
			strftime(timestr, sizeof(timestr), "%Y-%m-%dT%H:%M:%S", tm);
			break;
		case AUDPR_TIME_FMT_UNIX:
			strftime(timestr, sizeof(timestr), "%d %b %y %H:%M:%S", tm);
			break;
		case AUDPR_TIME_FMT_RAW:
			snprintf(timestr, sizeof(timestr), "%-8lu ",
					(unsigned long) timestamp);
			break;
		default:
			strcpy(timestr, "-");
		}
		printf("%s ", timestr);
	}

	if (flags & AUDPR_SEQNR)
		printf("%6d ",msg->msg_seqnr);
	if (flags & AUDPR_PROCESSID)
		printf("%6d ", msg->msg_pid);
	if (flags & AUDPR_LOGINID) {
		struct passwd	*pw;

		if ((pw = getpwuid(msg->msg_login_uid)) == NULL)
			printf("%8d ", msg->msg_login_uid);
		else
			printf("%8s ", pw->pw_name);
	}

	if (flags & AUDPR_UID_ALL){
		printf("%6u ", msg->msg_euid);
		printf("%6u ", msg->msg_suid);
		printf("%6u ", msg->msg_ruid);
		printf("%6u ", msg->msg_fsuid);
	}
	if (flags & AUDPR_GID_ALL){
		printf("%6u ", msg->msg_egid);
		printf("%6u ", msg->msg_rgid);
		printf("%6u ", msg->msg_sgid);
		printf("%6u  ", msg->msg_fsgid);
	}
	if (flags & AUDPR_AUDITID)
		printf("%5u    ", msg->msg_audit_id);

	/* XXX handle other flags */

	if ((flags & AUDPR_EVNAME) && msg->msg_evname[0])
		printf("[%.*s] ", AUD_MAX_EVNAME, msg->msg_evname);
}
	

void
print_login_msg(const struct aud_message *msg)
{
	const struct aud_msg_login	*logmsg;
	typedef char logmsg_aligned_t[-(ssize_t)(offsetof(struct aud_message, msg_data) & (__alignof__(*logmsg) - 1))];

	if (msg->msg_size != sizeof(*msg) + sizeof(*logmsg)) {
		printf("login message, strange message size\n");
		return;
	}

	logmsg = (const struct aud_msg_login *) msg->msg_data;
	printf("LOGIN: uid=%u", logmsg->uid);
	if (logmsg->hostname[0])
		printf(", hostname=%s", logmsg->hostname);
	if (logmsg->address[0])
		printf(", address=%s", logmsg->address);
	if (logmsg->terminal[0])
		printf(", terminal=%s", logmsg->terminal);
	if (logmsg->executable[0])
		printf(", executable=%s", logmsg->executable);
}

void
print_text_msg(const struct aud_message *msg)
{
	int	len;

	len = msg->msg_size - sizeof(*msg);
	if (len <= 0) {
		printf("(empty text message)");
		return;
	}

	printf("%.*s", len, (char *) msg->msg_data);
}

void
print_netlink_msg(const struct aud_message *msg)
{
	const struct aud_msg_netlink *nlmsg;
	typedef char nlmsg_aligned_t[-(ssize_t)(offsetof(struct aud_message, msg_data) & (__alignof__(*nlmsg) - 1))];
	int		len;

	len = msg->msg_size - sizeof(*msg);
	if (len <= sizeof(*nlmsg)) {
		printf("(empty netlink message)");
		return;
	}

	nlmsg = (const struct aud_msg_netlink *) &msg->msg_data;
	len -= sizeof(*nlmsg);

	printf("NETLINK");
	rtnetlink_print(nlmsg, len);
}

/*
 * System call audit message
 */
void
print_syscall_msg(const struct aud_message *msg)
{
	struct syscall_data	data;
	const struct aud_msg_syscall	*scmsg;
	typedef char scmsg_aligned_t[-(ssize_t)(offsetof(struct aud_message, msg_data) & (__alignof__(*scmsg) - 1))];
	const unsigned char	*args;
	unsigned int		size;
	u_int32_t		type, len;
	unsigned int		nargs;

	scmsg  = (const struct aud_msg_syscall *) msg->msg_data;
	args = scmsg->data;
	size = scmsg->length;
	if (size > msg->msg_size - sizeof(*msg)) {
		printf("\n[truncated system call message - got %08X of %08X required bytes]\n", msg->msg_size, size);
		size = msg->msg_size - sizeof(*msg);
		if (size < sizeof(*scmsg))
			return;
	}

	memset(&data, 0, sizeof(data));
	data.major  = scmsg->major;
	data.minor  = scmsg->minor;
	data.result = scmsg->result;

	for (nargs = 0; size > 0 && nargs < MAX_ARGS; nargs++) {
		struct syscall_arg	*arg = &data.args[nargs];

		if (size < 8) {
			printf("\n[truncated argument %u - only %u header bytes available]\n", nargs + 1, size);
			break;
		}

		memcpy(&type, args, 4); args += 4;
		memcpy(&len,  args, 4); args += 4;
		size -= 8;

		if (type == AUDIT_ARG_END)
			break;
		if (len > size) {
			printf("\n[truncated argument %u - got %08X of %08X required bytes (%08X %08X)]\n", nargs + 1, size, len, msg->msg_size, scmsg->length);
			len = size;
		}
		/* Do not display the NUL byte */
		if (type == AUDIT_ARG_STRING && len && args[len-1] == '\0')
			len--;

		arg->type = type;
		arg->data = args;
		arg->len  = len;

		args += len;
		size -= len;
	}
	data.nargs = nargs;

	syscall_print(&data);
}

void
print_exit_msg(const struct aud_message *msg) 
{
	const struct aud_msg_exit *exitmsg;
	typedef char exitmsg_aligned_t[-(ssize_t)(offsetof(struct aud_message, msg_data) & (__alignof__(*exitmsg) - 1))];

	exitmsg  = (const struct aud_msg_exit *) msg->msg_data;
	printf ("EXIT: %ld", exitmsg->code);
}
