/***************************************************************************
                          liblaus.c  -  description
                             -------------------
    copyright            : (C) 2003 by Thomas Biege / SuSE Linux AG
    email                : thomas@suse.de
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <limits.h>
#include <errno.h>

#include <sys/stat.h>
#include <sys/ioctl.h>
#include <sys/wait.h>
#include <sys/types.h>

#include <laus.h>

#define FAIL(err) \
	do { \
		__laus_failed_func = __FUNCTION__; \
		errno = err; \
		return -1; \
	} while (0)

static int			__laus_global_fd = -1;
static int			__laus_version = 0;
volatile static u_int		__laus_open_ctr = 0;
static const char *		__laus_failed_func;

/*
** public objects
*/
int laus_init(void)
{
	return 0;
}

int laus_open(const char *dev_file)
{
	int	old_val;

	if (__laus_global_fd >= 0)
		goto __laus_counter;

	if (!dev_file)
		dev_file = PATH_AUDIT;

	if ((__laus_global_fd = open(dev_file, O_RDWR)) < 0)
		FAIL(errno);

	if ((old_val = fcntl(__laus_global_fd, F_GETFD, 0L)) < 0)
		old_val = 0;
	(void) fcntl(__laus_global_fd, F_SETFD, old_val | FD_CLOEXEC);

__laus_counter:
	if(__laus_open_ctr >= UINT_MAX)
		return -1;
	__laus_open_ctr++;
	return 0;
}

int laus_api_version(void)
{
	int version;

	if ((version = __laus_version) == 0) {
		int err;

		laus_open(NULL);
		if (__laus_global_fd < 0)
			return -1;

		version = ioctl(__laus_global_fd, AUIOCVERSION);
		err = errno;
		if (version < 0 && err == EINVAL)
			version = AUDIT_API_VERSION_OLD;

		laus_close();

		if (version < 0)
			FAIL(err);
	}

	return version;
}

int laus_version(void)
{
	return LAUS_VERSION;
}

int laus_setauditdaemon(void)
{
	if (__laus_global_fd < 0)
		FAIL(EBADFD);

	if (ioctl(__laus_global_fd, AUIOCIAMAUDITD) < 0)
		FAIL(errno);

	return 0;
}

int laus_read(void *buffer, size_t size)
{
	int	n;

	if (__laus_global_fd < 0)
		FAIL(EBADFD);

	if ((n = read(__laus_global_fd, buffer, size)) < 0)
		FAIL(errno);

	return n;
}

pid_t laus_exec(int flags, char *prog, ...)
{
	va_list	arg_list;
	int	status, result, cnt = 0;
	char	**argv = NULL, *arg;
	pid_t	pid;

	/* Copy arguments into argv array */
	va_start(arg_list, prog);
	do {
		argv = (char **) realloc(argv,
				(cnt+2) * sizeof(char *));
		if (argv == NULL)
			FAIL(ENOMEM);

		arg = va_arg(arg_list, char *);
		argv[cnt++] = arg;
	} while (arg != NULL);
	va_end(arg_list);

	if ((pid = fork()) < 0)
		FAIL(errno);
	if (pid == 0) {
		/* child process */
		execvp(prog, argv);
		abort();
	}

	free(argv);

	if ((flags & LAUS_FLG_DETACH)) {
		result = pid;
	} else {
		waitpid(pid, &status, 0);
		/* check status */
		if (WIFEXITED(status)) {
			result = WEXITSTATUS(status);
		} else if (WIFSIGNALED(status)) {
			/* XXX keep WTERMSIG? */
			errno = -ECHILD; /* hm... not very smart choice */
			result = -1;
		} else {
			result = 0; /* XXX */
		}
	}

	return result;
}

int laus_attach(void)
{
	if (__laus_global_fd < 0)
		FAIL(EBADFD);

	if (ioctl(__laus_global_fd, AUIOCATTACH) < 0)
		FAIL(errno);

	return 0;
}

int laus_detach(void)
{
	if (__laus_global_fd < 0)
		FAIL(EBADFD);

	if (ioctl(__laus_global_fd, AUIOCDETACH) < 0)
		FAIL(errno);

	return 0;
}

int laus_suspend(void)
{
	if (__laus_global_fd < 0)
		FAIL(EBADFD);

	if (ioctl(__laus_global_fd, AUIOCSUSPEND) < 0)
		FAIL(errno);

	return 0;
}

int laus_resume(void)
{
	if (__laus_global_fd < 0)
		FAIL(EBADFD);

	if (ioctl(__laus_global_fd, AUIOCRESUME) < 0)
		FAIL(errno);

	return 0;
}

int laus_setauditid(void)
{
	if (__laus_global_fd < 0)
		FAIL(EBADFD);

	if (ioctl(__laus_global_fd, AUIOCSETAUDITID) < 0)
		FAIL(errno);

	return 0;
}

int laus_setsession(uid_t uid,
		const char *hostname,
		const char *address,
		const char *terminal)
{
#if 0
	struct stat		stat_buf;
#endif
	struct aud_msg_login	login;

	if (__laus_global_fd < 0)
		FAIL(EBADFD);

	/* do sanity checks on terminal */
#if 0
        /* XXX thomas: ssh sets terminal to "NODEVssh" */
	if (lstat(terminal, &stat_buf) < 0)
		FAIL(errno);
	if (!S_ISCHR(stat_buf.st_mode))
		FAIL(ENOTTY);
	if (stat_buf.st_mode & S_IRWXO)		/* XXX thomas: check group tty */
		FAIL(EACCES);
#endif

	/* prepare login info */
	memset(&login, 0, sizeof(login));
	login.uid = uid;
	if (hostname && hostname[0])
		strncpy(login.hostname, hostname, sizeof(login.hostname)-1);
	if (address && address[0])
		strncpy(login.address, address, sizeof(login.address)-1);
	if (terminal && terminal[0])
		strncpy(login.terminal, terminal, sizeof(login.terminal)-1);

	/* emit message */
	if (ioctl(__laus_global_fd, AUIOCLOGIN, &login) < 0)
		FAIL(errno);

	return 0;
}

int laus_clrpolicy(void)
{
	if (__laus_global_fd < 0)
		FAIL(EBADFD);

	if (ioctl(__laus_global_fd, AUIOCCLRPOLICY) < 0)
		FAIL(errno);

	return 0;
}

int laus_setpolicy(int syscall, int action, int filter)
{
	struct audit_policy	policy;

	if (__laus_global_fd < 0)
		FAIL(EBADFD);

	policy.code   = syscall;
	policy.action = action;
	policy.filter = filter;
	if (ioctl(__laus_global_fd, AUIOCSETPOLICY, &policy) < 0)
		FAIL(errno);

	return 0;
}

int laus_getpolicy(int policy_id)
{
	// XXX thomas: verify fd
	return 0;
}

int laus_delpolicy(int policy_id)
{
	// XXX thomas: verify fd
	return 0;
}

int laus_clrfilter(void)
{
	if (__laus_global_fd < 0)
		FAIL(EBADFD);

	if (ioctl(__laus_global_fd, AUIOCCLRFILTER) < 0)
		FAIL(errno);

	return 0;
}

int laus_setfilter(struct audit_filter *filter)
{
	if (__laus_global_fd < 0)
		FAIL(EBADFD);

	if (ioctl(__laus_global_fd, AUIOCSETFILTER, filter) < 0)
		FAIL(errno);

	return 0;
}

int laus_log(const char *audit_tag, const char *fmt, ...)
{
	char			buffer[8*1024] = {0};
	struct audit_message	msg;
	va_list			arg_list;


	if (__laus_global_fd < 0)
		FAIL(EBADFD);

	va_start(arg_list, fmt);
	vsnprintf(buffer, sizeof(buffer), fmt, arg_list);
	va_end(arg_list);

	memset(&msg, 0, sizeof(msg));

	msg.msg_type = AUDIT_MSG_TEXT;
	msg.msg_data = (char *) buffer;
	msg.msg_size = strlen(buffer);

	if(audit_tag != NO_TAG)
		strncpy(msg.msg_evname, audit_tag, sizeof(msg.msg_evname)-1);

	if(ioctl(__laus_global_fd, AUIOCUSERMESSAGE, &msg) < 0)
		FAIL(errno);

        return(0);
}

int laus_textmessage(const char *text)
{
	if (__laus_global_fd < 0)
		FAIL(EBADFD);

	return laus_usermessage(AUDIT_MSG_TEXT, text, strlen(text));
}

int laus_usermessage(int type, const void *data, size_t len)
{
        struct audit_message    msg;

        memset(&msg, 0, sizeof(msg));
        msg.msg_type = type;
        msg.msg_data = (char *) data;
        msg.msg_size = len;

        if (ioctl(__laus_global_fd, AUIOCUSERMESSAGE, &msg) < 0)
		FAIL(errno);
	return 0;
}

int laus_close(void)
{
	int fd;

	if ((fd = __laus_global_fd) < 0)
		return 0;

	if(__laus_open_ctr > 1)
	{
		__laus_open_ctr--;
		return(0);
	}

	__laus_open_ctr = 0;
	__laus_global_fd = -1;
	__laus_version = 0;
	return close(fd);
}

int laus_reset(void)
{
	if (__laus_global_fd < 0)
		FAIL(EBADFD);

	if (ioctl(__laus_global_fd, AUIOCRESET) < 0) {
		ioctl(__laus_global_fd, AUIOCCLRPOLICY);
		ioctl(__laus_global_fd, AUIOCCLRFILTER);
	}

	return 0;
}

const char *
laus_strerror(int code)
{
	static char	msgbuf[256];
	const char *	errmsg;

	switch (code) {
	case EBADFD:
		errmsg = "Invalid audit fd, did you call laus_open()?";
		break;
	case ECHILD:
		errmsg = "Process called via laus_exec() has crashed.";
		break;
	default:
		errmsg = strerror(code);
	}

	snprintf(msgbuf, sizeof(msgbuf), "%s: %s",
		__laus_failed_func, errmsg);

	return msgbuf;
}
