/*
 * Simple config file parser
 *
 * Copyright (C) 2003, SuSE Linux AG
 * Written by okir@suse.de
 */

#include <sys/types.h>
#include <stdio.h>
#include <stdlib.h>
#include <setjmp.h>
#include <string.h>
#include <stdarg.h>
#include <ctype.h>

#include <linux/laus_audit.h>
#include <laussrv.h>
#include "parser.h"
#include "auditd.h"
#include "config.h"

struct parser {
	FILE *		file;
	const char *	name;
	unsigned int	line;
	jmp_buf		errjmp;
};

static parser_t *	current;

int
parse_file(const char *filename, int (*parse_statement)(void))
{
	parser_t	*prev, parser;
	char		namebuf[PATH_MAX];

	if (opt_debug)
		log_dbg("Parsing configuration file %s", filename);

	memset(&parser, 0, sizeof(parser));
	prev = current;

	if (setjmp(parser.errjmp)) {
		if (current->file)
			fclose(current->file);
		current = prev;
		if (current != NULL)
			longjmp(current->errjmp, 1);
		return -1;
	}

	if (filename[0] != '/') {
		snprintf(namebuf, sizeof(namebuf), "%s/%s",
				PATH_CONFIG_DIR, filename);
		filename = namebuf;
	}

	parser.name = strdup(filename);
	parser.line = 1;
	current = &parser;

	if (!(parser.file = fopen(filename, "r"))) {
		cf_error("Unable to open %s: %m", filename);
		return -1;
	}

	while (parse_statement())
		;

	fclose(current->file);
	current = prev;
	return 0;
}

#define SEPA	";,<=>!{}()&|"

/*
 * Get a token, or EOF
 */
int
parser_get_token_or_eof(char *buffer, size_t size)
{
	unsigned int	n = 0, begin_line;
	int		c, quote = 0, first_is_sepa = 0, is_comment = 0;

	begin_line = current->line;
	while (1) {
		do {
			if ((c = getc(current->file)) == EOF) {
				if (quote) {
					cf_error("Unexpected end of file in quoted string "
					     "(string starts in line %d)", begin_line);
				}
				if (n)
					break;
				return 0;
			}
			if (!quote) {
			       if (c == '#')
				       is_comment = 1;
			       else if (c == '\n')
				       is_comment = 0;
			}
		} while (is_comment);

	       	if (quote) {
			if (c == quote) {
				quote = 0;
				continue;
			}
			if (c == '\n')
				cf_error("Unexpected newline in quoted string\n");
		} else {
			if (isspace(c)) {
				if (c == '\n')
					current->line++;
				if (n != 0)
					break;
				continue;
			}

			if (strchr(SEPA, c) != NULL) {
				if (n == 0) {
					/* accept it */
					first_is_sepa = 1;
				} else
				if (n == 1
				 && ((buffer[0] == '>' && c == '=')
				  || (buffer[0] == '<' && c == '=')
				  || (buffer[0] == '!' && c == '=')
				  || (buffer[0] == '&' && c == '&')
				  || (buffer[0] == '|' && c == '|'))) {
				 	/* accept it too */
				} else {
pushback:
					ungetc(c, current->file);
					break;
				}
			} else if (first_is_sepa) {
				goto pushback;
			} else if (c == '"' || c == '\'') {
				quote = c;
				continue;
			}
		}
		if (n + 1 < size)
			buffer[n++] = c;
	}
	buffer[n] = '\0';

// printf(": %s\n", buffer);
	return 1;
}

/*
 * Get a token; fail if we encounter EOF
 */
void
parser_get(char *buffer, size_t len)
{
	if (!parser_get_token_or_eof(buffer, len))
		cf_error("unexpected end of file");
}


/*
 * Push back token
 */
void
parser_unget(const char *token)
{
	if (token[1])
		cf_error("Unable to unget token \"%s\"", token);
	ungetc(token[0], current->file);
}

/*
 * Make sure the next token matches what we expect
 */
void
parser_expect(char expect)
{
	char	token[64];

	if (!parser_get_token_or_eof(token, sizeof(token)))
		cf_error("Unexpected end of file, expected \"%c\"", expect);
	if (token[0] != expect || token[1])
		cf_error("Syntax error, expected \"%c\", got \"%s\"",
			       	expect, token);
}

/*
 * Error handling at other functions called from generic filter code
 */
void
cf_error(const char *fmt, ...)
{
	va_list	ap;

	if (current)
		fprintf(stderr, "Error in %s:%d: ",
			       	current->name, current->line);
	else
		fprintf(stderr, "Error: ");

	va_start(ap, fmt);
	vfprintf(stderr, fmt, ap);
	fprintf(stderr, "\n");
	va_end(ap);

	if (current)
		longjmp(current->errjmp, 1);
	fprintf(stderr, "Aborting.\n");
	exit(1);
}

const char *
current_file(void)
{
	return current? current->name : "<unknown>";
}

unsigned int
current_line(void)
{
	return current? current->line : 0;
}
