/*
 * Array and node handling functions
 *
 * Copyright (C) 2003, SuSE Linux AG
 */

#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include "auditd.h"
#include "parser.h"

struct cf_array {
	const char *	name;
	size_t		obj_size;
	unsigned int	count;
	cf_object_t **	objects;
};

struct cf_node {
	cf_object_t	base;
	cf_node_t *	parent;
	char *		path;
	char *		value;
	cf_array_t *	children;
};

/*
 * Array handling
 */
cf_array_t *
cf_array_new(const char *name, size_t obj_size)
{
	cf_array_t	*array;

	array = (cf_array_t *) calloc(1, sizeof(*array));
	array->name = name;
	array->obj_size = obj_size;
	return array;
}

static cf_object_t *
__cf_array_alloc(cf_array_t *array, const char *name)
{
	cf_object_t	*obj;

	obj = (cf_object_t *) calloc(1, array->obj_size);
	obj->name = name ? strdup(name) : NULL;
	obj->num = array->count;
	obj->defined_in.file = current_file();
	obj->defined_in.line = current_line();

	cf_array_grow(array);
	array->objects[array->count++] = obj;

	return obj;
}

cf_object_t *
cf_array_alloc(cf_array_t *array, const char *name)
{
	cf_object_t	*obj;

	if (name && (obj = cf_array_find(array, name)) != NULL) {
		cf_error("duplicate definition of %s.%s "
		         "(previous definition in %s:%d)",
			 array->name, name,
			 obj->defined_in.file,
			 obj->defined_in.line);
	}

	return __cf_array_alloc(array, name);
}

void
cf_array_grow(cf_array_t *array)
{
	if ((array->count % 16) == 0) {
		array->objects = (cf_object_t **) realloc(array->objects,
				(array->count + 16) * sizeof(cf_object_t *));
	}
}

cf_object_t *
cf_array_find_at(const cf_array_t *array, unsigned int idx, const char *name)
{
	const char	*s;

	if (array == NULL || name == NULL)
		return NULL;
	while (idx < array->count) {
		if ((s = array->objects[idx]->name) && !strcmp(s, name))
			return array->objects[idx];
		idx++;
	}
	return NULL;
}

cf_object_t *
cf_array_find(const cf_array_t *array, const char *name)
{
	return cf_array_find_at(array, 0, name);
}

cf_object_t *
cf_array_at(cf_array_t *array, unsigned int num)
{
	if (!array || array->count <= num)
		return NULL;
	return array->objects[num];
}

unsigned int
cf_array_size(const cf_array_t *array)
{
	return array ? array->count : 0;
}

/*
 * Node handling
 */
cf_node_t *
cf_node_new(cf_node_t *parent, const char *name, const char *value)
{
	cf_node_t	*np;

	if (parent == NULL) {
		/* Allocate root node */
		np = (cf_node_t *) calloc(1, sizeof(*np));
	} else {
		char	numbuf[32];
		size_t	len;

		if (!parent->children)
			parent->children = cf_array_new(parent->base.name,
					sizeof(cf_node_t));
		np = (cf_node_t *) __cf_array_alloc(parent->children, name);
		if (value)
			np->value = strdup(value);

		/* Build the node's path name */
		len = 0;
		if (parent->path)
			len = strlen(parent->path) + 1;
		if (name == NULL) {
			snprintf(numbuf, sizeof(numbuf), "%u", np->base.num);
			name = numbuf;
		}
		len += strlen(name);
		np->path = (char *) calloc(1, len+1);
		if (parent->path) {
			strcpy(np->path, parent->path);
			strcat(np->path, ".");
		}
		strcat(np->path, name);

		np->parent = parent;
	}

	return np;
}

/*
 * Find the named node
 */
cf_node_t *
cf_node_find(cf_node_t *np, const char *name)
{
	if (np == NULL)
		return NULL;
	return (cf_node_t *) cf_array_find_at(np->children, 0, name);
}

cf_node_t *
cf_node_find_next(cf_node_t *np, const char *name)
{
	unsigned int	idx;

	idx = np->base.num;
	if (!(np = np->parent))
		return NULL;
	return (cf_node_t *) cf_array_find_at(np->children, idx + 1, name);
}

/*
 * Get the node's name
 */
const char *
cf_node_name(cf_node_t *np)
{
	return np->base.name;
}

/*
 * Get the node's value
 */
const char *
cf_node_value(cf_node_t *np, const char *name)
{
	if (name && !(np = cf_node_find(np, name)))
		return NULL;
	return np->value;
}

/*
 * Get the node's value, as integer
 */
long
cf_node_atoi(cf_node_t *np, const char *name, long defvalue)
{
	const char	*value;
	long		res;
	char		*end;

	if (!(value = cf_node_value(np, name))) {
		res = defvalue;
	} else {
		res = strtoul(value, &end, 10);
		if (*end)
			cf_error("Bad integer value \"%s\"", value);
	}
	return res;
}

/*
 * Interpret node value as file size
 */
size_t
cf_node_atofs(cf_node_t *np, const char *name, long defvalue)
{
	const char	*value;
	long		res;
	char		*end;

	if (!(value = cf_node_value(np, name))) {
		res = defvalue;
	} else {
		res = strtoul(value, &end, 10);
		switch (tolower(*end)) {
		case 'g': res *= 1024;
		case 'm': res *= 1024;
		case 'k': res *= 1024;
			  end++;
			  break;
		}
		if (*end)
			cf_error("Bad integer value \"%s\"", value);
	}
	return res;
}

/*
 * Interpret node value as boolean
 */
int
cf_node_atob(cf_node_t *np, const char *name, int defvalue)
{
	const char	*value;

	if (!(value = cf_node_value(np, name)))
		return defvalue;
	if (!strcasecmp(value, "yes")
	 || !strcasecmp(value, "true")
	 || !strcasecmp(value, "on"))
		return 1;
	if (!strcasecmp(value, "no")
	 || !strcasecmp(value, "false")
	 || !strcasecmp(value, "off"))
		return 0;
	cf_error("Bad boolean value \"%s\"", value);
}
