/*
 * Audit Daemon for LAuS
 *
 * Copyright (C) 2003 SuSE Linux AG
 * Written by okir@suse.de
 */

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/uio.h>
#include <sys/mman.h>
#include <sys/wait.h>
#include <sys/time.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <stdio.h>
#include <string.h>
#include <signal.h>
#include <stdarg.h>
#include <syslog.h>
#include <stdlib.h>
#include <sched.h>
#include <time.h>

#include <laus.h>
#include <laussrv.h>

#include "auditd.h"
#include "config.h"
#include "filter.h"

int			opt_debug = 0;

static int		opt_daemon = 1;
static int		opt_reload = 0;
static char *		opt_config = PATH_CONFIG_DIR "/audit.conf";
static char *		opt_pidfile = "/var/run/auditd.pid";

static int		have_pidfile = 0;

enum {
	LOG_DEST_STDERR = 0x01,
	LOG_DEST_SYSLOG = 0x02,
};
static int		log_dest = LOG_DEST_STDERR | LOG_DEST_SYSLOG;

static void		configure_syscall(unsigned int);
static void		configure_event(unsigned int);


static void		usage(int exval);
static void		sighup_handler(int);
static void		sigterm_handler(int);
static void		sigchld_handler(int);

int
main(int argc, char **argv)
{
	struct sigaction act;
	int	c, err, n, count;

	audit_check_biarch(argv);

	/* Better safe than sorry: prevent log collecting
	 * commands from doing anything stupid */
	umask(077);

	while ((c = getopt(argc, argv, "dFf:hr")) != -1) {
		switch (c) {
		case 'd':
			opt_debug++;
			break;
		case 'F':
			opt_daemon = 0;
			break;
		case 'f':
			opt_config = optarg;
			break;
		case 'r':
			opt_reload = 1;
			break;
		case 'h':
			usage(0);
		default:
			usage(1);
		}
	}

	if (audit_parse_daemon_config(opt_config) < 0)
		return 1;

	/* Load filter config file(s) */
	if (audit_parse_filter_config(NULL) < 0)
		return 1;

	/* Configure output file(s) */
	if (!opt_reload)
		configure_output();

	/* Configure disk thresholds */
	configure_thresholds();

	/* If not reloading, create the pid file now so
	 * that errors still get logged to stderr */
	if (!opt_reload) {
		if (make_pidfile(opt_pidfile, 0) < 0)
			return 1;
		have_pidfile = 1;
	}

	if (opt_daemon && !opt_reload) {
		int	fd;

		if ((fd = open("/dev/null", O_RDONLY)) < 0) {
			perror("Unable to open /dev/null");
			return 1;
		}

		if (daemon(0, 0) < 0) {
			perror("failed to become a daemon");
			return 1;
		}

		dup2(fd, 0);
		dup2(fd, 1);
		dup2(fd, 2);
		if (fd > 2)
			close(fd);

		log_dest &= ~LOG_DEST_STDERR;
		log_dest |= LOG_DEST_SYSLOG;
		log_open();

		/* Update the pid file with our real PID */
		make_pidfile(opt_pidfile, 1);
	}

	/* Install signal handler to catch SIGHUP */
	memset(&act, 0, sizeof(act));
	act.sa_handler = sighup_handler;
	sigaction(SIGHUP, &act, NULL);

	act.sa_handler = sigterm_handler;
	sigaction(SIGTERM, &act, NULL);
	sigaction(SIGINT, &act, NULL);
	sigaction(SIGPWR, &act, NULL);

	act.sa_handler = sigchld_handler;
	sigaction(SIGCHLD, &act, NULL);

	/* Open the audit device and clear all audit
	 * filters. This will *not* remove any filters
	 * that are attached to a system call via policy,
	 * so auditing is still functional while we're
	 * exporting the new set of filters.
	 */
	if ((err = laus_open(NULL)) < 0) {
		log_err(LOG_ERR, "%s", laus_strerror(errno));
		return 1;
	}
	switch (laus_api_version()) {
	case AUDIT_API_VERSION:
	case AUDIT_API_VERSION_OLD:
		break;
	default:
		log_err(LOG_ERR, "Unsupported API version: %x.\n", laus_api_version());
		return 1;
	}
	if ((err = laus_clrfilter()) < 0) {
		log_err(LOG_ERR, "%s", laus_strerror(errno));
		return 1;
	}

	/* Announce that we're changing the policy table.
	 * If auditd isn't running, this won't do anything */
	if (opt_reload) {
		if (laus_attach() >= 0) {
			laus_setauditid();
			laus_setsession(0, NULL, NULL, ttyname(0));
		}
		laus_log(AUDCONF_reload, "reloading audit policy");
	}

	/* Export policy to kernel */
	count = syscall_max();
	for (n = 0; n < count; n++)
		configure_syscall(n);

	/* Push filters for other events */
	count = event_max();
	for (n = event_min(); n < count; n++)
		configure_event(n);

	if (opt_reload)
		return 0;

	/* Announce the fact that we're the audit daemon */
	(void) laus_detach();
	if ((err = laus_setauditdaemon()) < 0) {
		log_err(LOG_ERR, "%s", laus_strerror(errno));
		return 1;
	}

	/* Synthesize start message */
	log_audit(AUDIT_start, "audit system started");

	/* Run with realtime priority */
	audit_set_priority(1);

	/* Read from the audit device */
	while (1) {
		static unsigned long messages;
		struct aud_message msg;
		unsigned char	buffer[8192];
		unsigned int	copied, total, len;

		if ((n = laus_read(buffer, sizeof(buffer))) < 0) {
			if (errno == EINTR)
				continue;
			log_err(LOG_ERR, "failed to read from audit device: %m");
			return 1;
		}

		total = n;
		for (copied = 0; copied + sizeof(msg) < total; copied += len) {
			/* need to copy msg header to ensure alignment */
			memcpy(&msg, buffer+copied, sizeof(msg));
			if ((len = total - copied) > msg.msg_size)
				len = msg.msg_size;
			output_write_record(buffer + copied, len);
			messages++;
		}

		if (opt_debug) {
			static struct timeval mark;
			struct timeval	now;

			gettimeofday(&now, NULL);
			if (now.tv_sec - mark.tv_sec < 2) {
				long	delta;

				delta = 1000 * (now.tv_sec - mark.tv_sec)
					+ (now.tv_usec - mark.tv_usec) / 1000;
				if (delta >= 1000) {
					log_dbg("%lu messages per second",
						       	messages * 1000 / delta);
					messages = 0;
					mark = now;
				}
			} else {
				/* Have been idle too long, no useful measurement */
				mark = now;
			}
		}
	}

	laus_reset();
	laus_close();
	log_audit(AUDIT_stop, "audit system stopped");

	if (have_pidfile)
		unlink(opt_pidfile);
	return 0;
}

void
configure_syscall(unsigned int code)
{
	int	err;
	const char *name = syscall_code_to_name(code);

	if ((err = cf_syscall_export(code, name)) < 0) {
		log_err(LOG_ERR,
			"Failed to export audit filter for %s(): %s",
			name, laus_strerror(errno));
		exit(1);
	}
}

void
configure_event(unsigned int code)
{
	int	err;
	const char *name = event_code_to_name(code);

	if ((err = cf_syscall_export(code, name)) < 0) {
		log_err(LOG_ERR,
			"Failed to export audit filter for event %s: %s",
			name, laus_strerror(errno));
		exit(1);
	}
}

/*
 * Error handling
 */
void
log_open(void)
{
	if (log_dest & LOG_DEST_SYSLOG)
		openlog("auditd", LOG_PID, LOG_DAEMON);
}

void
log_close(void)
{
	if (log_dest & LOG_DEST_SYSLOG)
		closelog();
}

void
log_err(int lvl, const char *fmt, ...)
{
	va_list	ap;

	va_start(ap, fmt);
	vlog_err(lvl, fmt, ap);
	va_end(ap);
}

void
vlog_err(int lvl, const char *fmt, va_list ap)
{
	if (log_dest & LOG_DEST_STDERR) {
		vfprintf(stderr, fmt, ap);
		fprintf(stderr, "\n");
	}
	if (log_dest & LOG_DEST_SYSLOG)
		vsyslog(lvl, fmt, ap);
}

/*
 * Spit out debug message
 */
void
log_dbg(const char *fmt, ...)
{
	va_list	ap;

	va_start(ap, fmt);
	if (log_dest & LOG_DEST_STDERR) {
		fprintf(stderr, "Debug: ");
		vfprintf(stderr, fmt, ap);
		fprintf(stderr, "\n");
	}
	if (log_dest & LOG_DEST_SYSLOG)
		vsyslog(LOG_DEBUG, fmt, ap);
	va_end(ap);
}

/*
 * Log fatal error
 */
void
log_fatal(const char *fmt, ...)
{
	va_list	ap;

	va_start(ap, fmt);
	if (log_dest & LOG_DEST_STDERR) {
		vfprintf(stderr, fmt, ap);
		fprintf(stderr, "\n");
	}
	if (log_dest & LOG_DEST_SYSLOG)
		vsyslog(LOG_ERR, fmt, ap);
	log_err(LOG_ERR, "Abort");
	if (have_pidfile)
		unlink(opt_pidfile);
	exit(1);
}

/*
 * Write to the audit log
 */
#define offsetof(T, mem)	((caddr_t) &((T *) 0)->mem - (caddr_t) 0)
void
log_audit(const char *evname, const char *fmt, ...)
{
	struct { struct aud_message header; char body[1024]; } msg;
	char	*strbuf;
	va_list	ap;

	va_start(ap, fmt);
	memset(&msg, 0, sizeof(msg));
	msg.header.msg_type = AUDIT_MSG_TEXT;
	msg.header.msg_pid = getpid();
	msg.header.msg_timestamp = time(NULL);
	strncpy(msg.header.msg_evname, evname, sizeof(msg.header.msg_evname));

	strbuf = (char *) msg.header.msg_data;
	vsnprintf(strbuf, sizeof(msg.body), fmt, ap);
	msg.header.msg_size = sizeof(msg.header) + strlen(strbuf);

	output_write_record(&msg, msg.header.msg_size);

	va_end(ap);
}

/*
 * Print usage message
 */
void
usage(int exval)
{
	fprintf(stderr,
	"Usage: auditd [-Fdhr] [-f cfgfile]"/* [-l logfile] [-B count] [-s size]*/"\n");
	exit(exval);
}

/*
 * SIGHUP handler
 */
void
sighup_handler(int sig)
{
	output_reopen();
}

/*
 * SIGCHLD handler
 */
void
sigchld_handler(int sig)
{
	int	status;
	pid_t	pid;

	while ((pid = waitpid(0, &status, WNOHANG)) > 0)
		output_child_complete(pid, status);
}

/*
 * Termination handler
 */
void
sigterm_handler(int sig)
{
	log_audit(AUDIT_stop, "audit system stopped");
	laus_reset();
	laus_close();
	if (have_pidfile)
		unlink(opt_pidfile);
	exit(0);
}

/*
 * Set scheduling priority
 */
void
audit_set_priority(int rt)
{
	struct sched_param param;
	int policy = rt? SCHED_RR : SCHED_OTHER;

	/* We're an arrogant SOB and think no-one is more
	 * important */
	param.sched_priority = sched_get_priority_max(policy);

	if (sched_setscheduler(0, policy, &param) < 0)
		log_err(LOG_ERR, "sched_setscheduler failed: %m");
}
