/*
 * Handle audit filters
 *
 * Copyright (C) 2003, SuSE Linux AG
 * Written by okir@suse.de
 */

#include <sys/types.h>
#include <stdio.h>
#include <stdlib.h>
#include <setjmp.h>
#include <string.h>
#include <stdarg.h>
#include <errno.h>

#include <linux/laus_audit.h>
#include <laus.h>
#include <laussrv.h>
#include "auditd.h"
#include "filter.h"

static cf_array_t	*cf_sets;
static cf_array_t	*cf_exprs;
static cf_array_t	*cf_predicates;
static cf_array_t	*cf_filters;
static cf_array_t	*cf_syscalls;

static int		cf_expr_export(cf_expr_t *expr);

/*
 * Handle filter objects
 */
cf_set_t *
cf_set_new(const char *name)
{
	if (!cf_sets)
		cf_sets = cf_array_new("set", sizeof(cf_set_t));
	return (cf_set_t *) cf_array_alloc(cf_sets, name);
}

cf_set_t *
cf_set_find(const char *name)
{
	return (cf_set_t *) cf_array_find(cf_sets, name);
}

void
cf_set_add(struct cf_set *set, const char *s)
{
	set->values = (char **) realloc(set->values,
		       		(set->count + 1) * sizeof(char *));
	set->values[set->count++] = strdup(s);
}

static cf_filter_t *
__cf_filter_new(cf_array_t *array, const char *name, cf_expr_t *expr)
{
	cf_filter_t	*f;

	f = (cf_filter_t *) cf_array_alloc(array, name);
	if (expr) {
		f->expr = expr;
		expr->base.name = strdup(name);
	}
	return f;
}

cf_filter_t *
cf_predicate_new(const char *name, cf_expr_t *expr)
{
	if (!cf_predicates)
		cf_predicates = cf_array_new("predicate", sizeof(cf_filter_t));
	return __cf_filter_new(cf_predicates, name, expr);
}

cf_filter_t *
cf_predicate_find(const char *name)
{
	return (cf_filter_t *) cf_array_find(cf_predicates, name);
}

cf_filter_t *
cf_filter_new(const char *name, cf_expr_t *expr)
{
	if (!cf_filters)
		cf_filters = cf_array_new("filter", sizeof(cf_filter_t));
	return __cf_filter_new(cf_filters, name, expr);
}

cf_filter_t *
cf_filter_find(const char *name)
{
	return (cf_filter_t *) cf_array_find(cf_filters, name);
}

void
cf_filter_print(FILE *fp, cf_filter_t *f)
{
	cf_expr_print(fp, f->expr);
}

/*
 * System calls
 */
cf_filter_t *
cf_syscall_new(const char *name)
{
	if (!cf_syscalls)
		cf_syscalls = cf_array_new("syscall", sizeof(cf_filter_t));
	if (syscall_name_to_code(name) < 0
	 && socketcall_name_to_code(name) < 0
	 && ipccall_name_to_code(name) < 0
	 && event_name_to_code(name) < 0) {
		log_err(LOG_WARNING,
		        "Unrecognized syscall/event '%s' will be ignored.",
		        name);
	}
	return (cf_filter_t *) cf_array_alloc(cf_syscalls, name);
}

cf_filter_t *
cf_syscall_find(const char *name)
{
	return (cf_filter_t *) cf_array_find(cf_syscalls, name);
}

int
cf_syscall_export(unsigned int code, const char *name)
{
	cf_filter_t	*f;
	int		r, filter_id, policy;

	f = cf_syscall_find(name);
	if (f == NULL || f->expr == FC_FILTER_NEVER) {
		policy = filter_id = 0;
	} else if (f->expr == FC_FILTER_ALWAYS) {
		policy = AUDIT_LOG;
		filter_id = 0;
	} else {
		if ((r = cf_expr_export(f->expr)) < 0)
			return r;
		policy = AUDIT_LOG;
		filter_id = f->expr->base.num;
	}
	if (opt_debug) {
		if (policy == 0) {
			printf("No auditing for syscall #%u (%s)\n",
				code, name? name : "unknown");
		} else {
			printf("Attaching %s (#%u), filter #%u\n",
				name, code, filter_id);
		}
	}
	return laus_setpolicy(code, policy, filter_id);
}

/*
 * Target <-> name mapping
 */
static struct {
	unsigned int	value;
	const char *	name;
} cf_target_names[] = {
	{ AUD_FILT_TGT_MINOR_CODE,	"syscall-minor"	},
	{ AUD_FILT_TGT_RETURN_CODE,	"result"	},
	{ AUD_FILT_TGT_USERMSG_EVNAME,	"event-name"	},
	{ AUD_FILT_TGT_UID,		"uid"		},
	{ AUD_FILT_TGT_GID,		"gid"		},
	{ AUD_FILT_TGT_DUMPABLE,	"dumpable"	},
	{ AUD_FILT_TGT_EXIT_CODE,	"exitcode"	},
	{ AUD_FILT_TGT_LOGIN_UID,	"login-uid"	},
	{ AUD_FILT_TGT_FILE_MODE,	"file-mode"	},
	{ AUD_FILT_TGT_FILE_UID,	"file-uid"	},
	{ AUD_FILT_TGT_FILE_GID,	"file-gid"	},
	{ AUD_FILT_TGT_FILE_DEV,	"file-dev"	},
	{ AUD_FILT_TGT_FILE_INO,	"file-ino"	},
	{ AUD_FILT_TGT_FILE_RDEV_MAJOR,	"dev-major"	},
	{ AUD_FILT_TGT_FILE_RDEV_MINOR,	"dev-minor"	},
	{ AUD_FILT_TGT_SOCK_FAMILY,	"sock-family"	},
	{ AUD_FILT_TGT_SOCK_TYPE,	"sock-type"	},
	{ AUD_FILT_TGT_NETLINK_FAMILY,	"netlink-family"},
	{ AUD_FILT_TGT_NETLINK_TYPE,	"netlink-type"	},
	{ AUD_FILT_TGT_NETLINK_FLAGS,	"netlink-flags"	},

	{ 0, NULL }
};

int
cf_name_to_target(const char *name)
{
	unsigned int	n;

	if (!strncmp(name, "arg", 3)) {
		unsigned int	num;
		char		*end;

		num = strtoul(name+3, &end, 10);
		if (*end || num >= AUD_FILT_TGT_RETURN_CODE)
			return -1;
		return num;
	}

	for (n = 0; cf_target_names[n].name; n++) {
		if (!strcmp(cf_target_names[n].name, name))
			return cf_target_names[n].value;
	}

	return -1;
}

const char *
cf_target_to_name(unsigned int target)
{
	static char	namebuf[32];
	unsigned int	n;

	for (n = 0; cf_target_names[n].name; n++) {
		if (cf_target_names[n].value == target)
			return cf_target_names[n].name;
	}

	if (AUD_FILT_TGT_SYSCALL_ATTR(target)) {
		snprintf(namebuf, sizeof(namebuf), "arg%u", target);
		return namebuf;
	}

	return NULL;
}

/*
 * Build filter expression node
 */
cf_expr_t *
cf_expr_new(int op, ...)
{
	cf_expr_t	*e;
	va_list		ap;

	if (!cf_exprs)
		cf_exprs = cf_array_new("expr", sizeof(cf_expr_t));
	e = (cf_expr_t *) cf_array_alloc(cf_exprs, NULL);
	e->data.op = op;
	e->data.num = e->base.num;

	va_start(ap, op);
	switch (op) {
	case AUD_FILT_OP_AND:
	case AUD_FILT_OP_OR:
		e->data.u.boolean.filt1 = va_arg(ap, int);
		e->data.u.boolean.filt2 = va_arg(ap, int);
		break;
	case AUD_FILT_OP_NOT:
		e->data.u.boolean.filt1 = va_arg(ap, int);
		break;
	case AUD_FILT_OP_APPLY:
		e->data.u.apply.filter = va_arg(ap, int);
		e->data.u.apply.target = va_arg(ap, int);
		break;
	case AUD_FILT_OP_RETURN:
		e->data.u.freturn.action = va_arg(ap, int);
		break;
	case AUD_FILT_OP_TRUE:
	case AUD_FILT_OP_FALSE:
		break;
	case AUD_FILT_OP_EQ:
	case AUD_FILT_OP_NE:
	case AUD_FILT_OP_GT:
	case AUD_FILT_OP_GE:
	case AUD_FILT_OP_LE:
	case AUD_FILT_OP_LT:
		e->data.u.integer.value = va_arg(ap, unsigned long);
		break;
	case AUD_FILT_OP_MASK:
		e->data.u.integer.mask  = va_arg(ap, unsigned long);
		e->data.u.integer.value = va_arg(ap, unsigned long);
		break;
	case AUD_FILT_OP_STREQ:
	case AUD_FILT_OP_PREFIX:
		e->data.u.string.value = va_arg(ap, char *);
		break;
	default:
		return NULL;
	}

	return e;
}

cf_expr_t *
cf_expr_lookup(unsigned int num)
{
	return (cf_expr_t *) cf_array_at(cf_exprs, num);
}

/*
 * Export expression to kernel
 */
int
__fc_expr_export(unsigned int num)
{
	cf_expr_t *expr;
	int	r;

	if (!(expr = cf_expr_lookup(num))) {
		fprintf(stderr,
			"Internal error - no expression #%u\n", num);
		return -1;
	}

	if (expr->exported)
		return 0;

	switch (expr->data.op) {
	case AUD_FILT_OP_AND:
	case AUD_FILT_OP_OR:
		if (__fc_expr_export(expr->data.u.boolean.filt1) < 0
		 || __fc_expr_export(expr->data.u.boolean.filt2) < 0)
			return -1;
		break;
	case AUD_FILT_OP_NOT:
		if (__fc_expr_export(expr->data.u.boolean.filt1) < 0)
			return -1;
		break;
	case AUD_FILT_OP_APPLY:
		if (__fc_expr_export(expr->data.u.apply.filter) < 0)
			return -1;
		break;
	}

	if (opt_debug > 1) {
		printf("Exporting filter rule #%u", expr->data.num);
		if (opt_debug > 2) {
			printf(" = ");
			cf_expr_print(stdout, expr);
		}
		printf("\n");
	}
	if ((r = laus_setfilter(&expr->data)) < 0) {
		fprintf(stderr, "Failed to export filter to kernel: %s\n",
				laus_strerror(errno));
		if (opt_debug) {
			fprintf(stderr,
				"Failed expression (defined in %s:%d): ",
				expr->base.defined_in.file,
				expr->base.defined_in.line);
			cf_expr_print(stderr, expr);
		}
		return -1;
	}

	expr->exported = 1;
	return 0;
}

int
cf_expr_export(cf_expr_t *expr)
{
	return __fc_expr_export(expr->base.num);
}

/*
 * Print filter (debugging)
 */
static void
__fc_expr_print(FILE *fp, unsigned int num,
	       	unsigned int callprec, const char *target)
{
	unsigned int	thisprec = 10;
	cf_expr_t	*e;

	if (!(e = cf_expr_lookup(num))) {
		fprintf(fp, "[[filt %u???]]", num);
		return;
	}

	switch (e->data.op) {
	case AUD_FILT_OP_NOT:
		thisprec = 3;
		break;
	case AUD_FILT_OP_AND:
		thisprec = 2;
		break;
	case AUD_FILT_OP_OR:
		thisprec = 1;
		break;
	}

	if (thisprec < callprec)
		fprintf(fp, "(");

	switch (e->data.op) {
	case AUD_FILT_OP_AND:
		__fc_expr_print(fp, e->data.u.boolean.filt1, thisprec, target);
		fprintf(fp, " && ");
		__fc_expr_print(fp, e->data.u.boolean.filt2, thisprec, target);
		break;
	case AUD_FILT_OP_OR:
		__fc_expr_print(fp, e->data.u.boolean.filt1, thisprec, target);
		fprintf(fp, " || ");
		__fc_expr_print(fp, e->data.u.boolean.filt2, thisprec, target);
		break;
	case AUD_FILT_OP_NOT:
		fprintf(fp, "!");
		__fc_expr_print(fp, e->data.u.boolean.filt1, thisprec, target);
		break;
	case AUD_FILT_OP_APPLY:
		__fc_expr_print(fp, e->data.u.apply.filter,
			       	callprec,
			       	cf_target_to_name(e->data.u.apply.target));
		break;
	case AUD_FILT_OP_TRUE:
		fprintf(fp, "true");
		break;
	case AUD_FILT_OP_FALSE:
		fprintf(fp, "false");
		break;
	case AUD_FILT_OP_EQ:
		fprintf(fp, "%s == %lu", target, (unsigned long) e->data.u.integer.value);
		break;
	case AUD_FILT_OP_NE:
		fprintf(fp, "%s != %lu", target, (unsigned long) e->data.u.integer.value);
		break;
	case AUD_FILT_OP_GT:
		fprintf(fp, "%s >= %lu", target, (unsigned long) e->data.u.integer.value);
		break;
	case AUD_FILT_OP_GE:
		fprintf(fp, "%s >  %lu", target, (unsigned long) e->data.u.integer.value);
		break;
	case AUD_FILT_OP_LE:
		fprintf(fp, "%s <= %lu", target, (unsigned long) e->data.u.integer.value);
		break;
	case AUD_FILT_OP_LT:
		fprintf(fp, "%s <  %lu", target, (unsigned long) e->data.u.integer.value);
		break;
	case AUD_FILT_OP_MASK:
		fprintf(fp, "(%s & %lu) == %lu", target,
			       	(unsigned long) e->data.u.integer.mask,
			       	(unsigned long) e->data.u.integer.value);
		break;
	case AUD_FILT_OP_STREQ:
		fprintf(fp, "%s == \"%s\"", target,
				e->data.u.string.value);
		break;
	case AUD_FILT_OP_PREFIX:
		fprintf(fp, "strprefix(%s, \"%s\")", target,
				e->data.u.string.value);
		break;
	default:
		fprintf(fp, "[[op %u]]", e->data.op);
	}

	if (thisprec < callprec)
		fprintf(fp, ")");
}

void
cf_expr_print(FILE *fp, cf_expr_t *expr)
{
	if (expr == NULL)
		fprintf(fp, "never");
	else if (expr == FC_FILTER_ALWAYS)
		fprintf(fp, "always");
	else
		__fc_expr_print(fp, expr->data.num, 0, "<var>");
}
