/*
 * Parse the filter configuration file
 *
 * Copyright (C) 2003, SuSE Linux AG
 * Written by okir@suse.de
 */

#include <sys/types.h>
#include <stdio.h>
#include <stdlib.h>
#include <setjmp.h>
#include <string.h>
#include <stdarg.h>

#include <linux/laus_audit.h>
#include <laussrv.h>
#include "auditd.h"
#include "filter.h"
#include "config.h"

static int		parse_statement(void);
static void		parse_set(const char *);
static void		parse_predicate(const char *, const char *);
static void		parse_filter(const char *, const char *);
static void		parse_syscall(const char *, const char *);
static void		parse_event(const char *);
static cf_expr_t *	parse_expression(void);
static cf_expr_t *	parse_term(void);
static unsigned int	parse_target(void);
static unsigned long	parse_integer(void);
static unsigned long	__parse_integer(const char *arg);
static cf_expr_t *	build_and(cf_expr_t *, cf_expr_t *);
static cf_expr_t *	build_or(cf_expr_t *, cf_expr_t *);
static cf_expr_t *	build_not(cf_expr_t *);
static cf_expr_t *	build_apply(cf_expr_t *, unsigned int);
static cf_expr_t *	build_predicate(int opt, const char *, const char *);
static void		build_switch(const char *syscall,
	       			unsigned int target,
				unsigned int count,
				const char * (*lookup)(unsigned int));
static cf_expr_t *	expand_set(int opt, const char *, const char *);
static void		attach_filter(const char *name, cf_expr_t *expr);
static cf_expr_t *	set_tag(cf_expr_t *expr, const char *tag);


int
audit_parse_filter_config(const char *filename)
{
	int		r;

	if (filename == NULL
	 && (filename = cf_node_value(cf_root, "filter-config")) == NULL)
		filename = PATH_CONFIG_DIR "/filter.conf";

	if ((r = parse_file(filename, parse_statement)) < 0)
		return r;

	/* Combine all definitions for accept/socket/listen
	 * etc into a general case switch for socketcall() */
	build_switch("socketcall", AUD_FILT_TGT_MINOR_CODE,
		       	socketcall_max(), socketcall_code_to_name);

	/* Same for IPC */
	build_switch("ipc", AUD_FILT_TGT_MINOR_CODE,
		       	ipccall_max(), ipccall_code_to_name);

	return 0;
}

static int
parse_statement(void)
{
	char	kwdbuf[64], namebuf[64];
	char	tagbuf[64], *tag = NULL;


	if (!parser_get_token_or_eof(kwdbuf, sizeof(kwdbuf)))
		return 0;

	while (1) {
		parser_get(namebuf, sizeof(namebuf));

		if (!strcmp(kwdbuf, "include")) {
			parse_file(namebuf, parse_statement);
			goto complete;
		}

		if (strcmp(kwdbuf, "tag"))
			break;
		if (tag)
			cf_error("More than one tag per filter expression");
		strcpy(tag = tagbuf, namebuf);
		parser_get(kwdbuf, sizeof(kwdbuf));
	}

	parser_expect(EQUALS);
	if (!strcmp(kwdbuf, "set")) {
		if (tag)
			goto bad_tag;
		parse_set(namebuf);
	} else
	if (!strcmp(kwdbuf, "predicate")) {
		parse_predicate(namebuf, tag);
	} else
	if (!strcmp(kwdbuf, "filter")) {
		parse_filter(namebuf, tag);
	} else
	if (!strcmp(kwdbuf, "syscall")) {
		parse_syscall(namebuf, tag);
	} else
	if (!strcmp(kwdbuf, "event")) {
		if (tag)
			goto bad_tag;
		parse_event(namebuf);
	} else
		cf_error("unexpected keyword %s", kwdbuf);

complete:
	parser_expect(SEMICOLON);
	return 1;

bad_tag:cf_error("tag not allowed in this context");
}

void
parse_set(const char *name)
{
	cf_set_t	*set;
	char		buffer[8192];

	set = cf_set_new(name);

	parser_expect(LEFT_BRACE);
	parser_get(buffer, sizeof(buffer));
	while (buffer[0] != RIGHT_BRACE) {
		cf_set_add(set, buffer);
		parser_get(buffer, sizeof(buffer));
		if (buffer[0] == COMMA)
			parser_get(buffer, sizeof(buffer));
	}
}

void
parse_predicate(const char *name, const char *tag)
{
	cf_filter_t	*f;

	f = cf_predicate_new(name, parse_expression());
	if (tag)
		f->expr = set_tag(f->expr, tag);
}

void
parse_filter(const char *name, const char *tag)
{
	cf_filter_t	*f;
       
	f = cf_filter_new(name, parse_expression());
	if (tag)
		f->expr = set_tag(f->expr, tag);
}

void
parse_syscall(const char *name, const char *tag)
{
	cf_expr_t	*expr;
	cf_set_t	*set;
	unsigned int	n;

	/* Parse the expression */
	expr = parse_expression();
	if (tag)
		expr = set_tag(expr, tag);

	if (opt_debug > 1) {
		printf("syscall %s = ", name);
		cf_expr_print(stdout, expr);
		printf("\n");
	}

	/* Attach filter to syscall */
	if (name[0] == '@') {
		if (!(set = cf_set_find(name+1)))
			cf_error("Unknown set %s", name+1);
		for (n = 0; n < set->count; n++)
			attach_filter(set->values[n], expr);
	} else {
		attach_filter(name, expr);
	}
}

void
parse_event(const char *name)
{
	cf_expr_t	*expr;

	/* Parse the expression */
	expr = parse_expression();

	if (opt_debug > 1) {
		printf("event %s = ", name);
		cf_expr_print(stdout, expr);
		printf("\n");
	}

	/* Attach filter */
	attach_filter(name, expr);
}

void
attach_filter(const char *name, cf_expr_t *expr)
{
	cf_filter_t	*f;

	if (expr == NULL)
		return;

	if (!(f = cf_syscall_find(name)))
		f = cf_syscall_new(name);
	f->expr = build_or(f->expr, expr);
}


cf_expr_t *
set_tag(cf_expr_t *expr, const char *tag)
{
	cf_filter_t	*f;

	if (expr == FC_FILTER_NEVER) {
		if (!(f = cf_filter_find("never"))) {
			f = cf_filter_new("never",
				cf_expr_new(AUD_FILT_OP_FALSE));
		}
		expr = f->expr;
	} else if (expr == FC_FILTER_ALWAYS) {
		if (!(f = cf_filter_find("always"))) {
			f = cf_filter_new("always",
				cf_expr_new(AUD_FILT_OP_TRUE));
		}
		expr = f->expr;
	}

	if (expr->data.event[0]) {
		/* Oops - expression already has a name */
		if (!strcmp(expr->data.event, tag))
			return expr;
		expr = build_or(expr, expr); /* Hack */
	} else if (expr->base.name) {
		expr = build_or(expr, expr); /* Hack */
	}

	if (strlen(tag) >= sizeof(expr->data.event))
		cf_error("Expression tag \"%s\" too long - max %u characters",
				tag, sizeof(expr->data.event)-1);
	strcpy(expr->data.event, tag);
	return expr;
}

/*
 * Boolean expression parsing
 */
cf_expr_t *
parse_expression(void)
{
	char		token[16];
	cf_expr_t	*and = NULL, *or = NULL;
	cf_expr_t	*term;

	and = parse_term();
	while (1) {
		parser_get(token, sizeof(token));
		if (token[0] == SEMICOLON
		 || token[0] == RIGHT_BRACKET) {
			parser_unget(token);
			break;
		}
		if (!strcmp(token, "&&")) {
			/* nothing right now */
		} else if (!strcmp(token, "||")) {
			or = build_or(or, and);
			and = FC_FILTER_ALWAYS;
		} else {
			cf_error("Unexpected token in expression: \"%s\"", token);
		}

		term = parse_term();
		and = build_and(and, term);
	}

	return build_or(or, and);
}

static cf_expr_t *
parse_term(void)
{
	char		token[64], argstr[8192];
	cf_filter_t	*predicate, *filter;
	cf_expr_t	*expr;
	int		op, nconst;

	parser_get(token, sizeof(token));
	if (!strcmp(token, "!")) {
		return build_not(parse_term());
	}
	if (token[0] == LEFT_BRACKET) {
		expr = parse_expression();
		parser_expect(RIGHT_BRACKET);
		return expr;
	}

	/* See if this is the application of a predicate */
	if ((predicate = cf_predicate_find(token)) != NULL) {
		parser_expect(LEFT_BRACKET);
		expr = build_apply(predicate->expr, parse_target());
		parser_expect(RIGHT_BRACKET);
		return expr;
	}

	/* See if this is a filter */
	if ((filter = cf_filter_find(token)) != NULL)
		return filter->expr;

	/* Okay, this is a predicate */
	nconst = 1;
	if (!strcmp(token, "true") || !strcmp(token, "always"))
		return FC_FILTER_ALWAYS;
	else if (!strcmp(token, "false") || !strcmp(token, "never"))
		return FC_FILTER_NEVER;
	else if (!strcmp(token, "eq"))
		op = AUD_FILT_OP_EQ;
	else if (!strcmp(token, "ne"))
		op = AUD_FILT_OP_NE;
	else if (!strcmp(token, "gt"))
		op = AUD_FILT_OP_GT;
	else if (!strcmp(token, "ge"))
		op = AUD_FILT_OP_GE;
	else if (!strcmp(token, "le"))
		op = AUD_FILT_OP_LE;
	else if (!strcmp(token, "lt"))
		op = AUD_FILT_OP_LT;
	else if (!strcmp(token, "streq"))
		op = AUD_FILT_OP_STREQ;
	else if (!strcmp(token, "prefix"))
		op = AUD_FILT_OP_PREFIX;
	else if (!strcmp(token, "return"))
		op = AUD_FILT_OP_RETURN;
	else if (!strcmp(token, "mask")) {
		unsigned long	arg[2];
		int		i;

		/* special case: two constants */
		parser_expect(LEFT_BRACKET);
		for (i = 0; i < 2; i++) {
			if (i)
				parser_expect(COMMA);
			arg[i] = parse_integer();
		}
		parser_expect(RIGHT_BRACKET);
		return cf_expr_new(AUD_FILT_OP_MASK, arg[0], arg[1]);
	} else {
		cf_error("Unknown filter predicate \"%s\"", token);
	}

	parser_expect(LEFT_BRACKET);
	parser_get(argstr, sizeof(argstr));
	parser_expect(RIGHT_BRACKET);

	return build_predicate(op, token, argstr);
}

unsigned int
parse_target(void)
{
	char	namebuf[64];
	int	target;

	parser_get(namebuf, sizeof(namebuf));
	if ((target = cf_name_to_target(namebuf)) < 0)
		cf_error("Invalid predicate target \"%s\"", namebuf);
	return target;
}

unsigned long
__parse_integer(const char *arg)
{
	unsigned long	value;
	char		*end;

	if (cf_constant(arg, &value) < 0) {
		value = strtol(arg, &end, 0);
		if (*end)
			cf_error("Invalid integer constant \"%s\"", arg);
	}
	return value;
}

unsigned long
parse_integer(void)
{
	char	argstr[128];

	parser_get(argstr, sizeof(argstr));
	return __parse_integer(argstr);
}

cf_expr_t *
build_and(cf_expr_t *term1, cf_expr_t *term2)
{
	if (term1 == FC_FILTER_NEVER || term2 == FC_FILTER_NEVER)
		return FC_FILTER_NEVER;
	if (term1 == FC_FILTER_ALWAYS)
		return term2;
	if (term2 == FC_FILTER_ALWAYS)
		return term1;

	return cf_expr_new(AUD_FILT_OP_AND,
			term1->base.num,
			term2->base.num);
}

cf_expr_t *
build_or(cf_expr_t *term1, cf_expr_t *term2)
{
	if (term1 == FC_FILTER_ALWAYS || term2 == FC_FILTER_ALWAYS)
		return FC_FILTER_ALWAYS;
	if (term1 == FC_FILTER_NEVER)
		return term2;
	if (term2 == FC_FILTER_NEVER)
		return term1;

	return cf_expr_new(AUD_FILT_OP_OR,
			term1->base.num,
			term2->base.num);
}

cf_expr_t *
build_not(cf_expr_t *term)
{
	if (term == FC_FILTER_ALWAYS)
		return FC_FILTER_NEVER;
	if (term == FC_FILTER_NEVER)
		return FC_FILTER_ALWAYS;
	return cf_expr_new(AUD_FILT_OP_NOT, term->base.num);
}

cf_expr_t *
build_apply(cf_expr_t *term, unsigned int target)
{
	if (term == FC_FILTER_ALWAYS || term == FC_FILTER_NEVER)
		return term;
	return cf_expr_new(AUD_FILT_OP_APPLY,
			term->base.num, target);
}

cf_expr_t *
build_predicate(int op, const char *name, const char *arg)
{
	cf_expr_t	*expr = NULL;

	if (op == AUD_FILT_OP_RETURN) {
		int	action;

		if (!strcmp(arg, "log"))
			action = AUDIT_LOG;
		else if (!strcmp(arg, "log-verbose"))
			action = AUDIT_LOG|AUDIT_VERBOSE;
		else if (!strcmp(arg, "ignore"))
			action = AUDIT_IGNORE;
		else
			cf_error("Unknown action in %s(%s)", name, arg);
		return cf_expr_new(op, action);
	}

	if (arg[0] == '@') {
		/* Set expansion */
		return expand_set(op, name, arg+1);
	}
       
	if (AUD_FILT_ARGTYPE_INT(op)) {
		expr = cf_expr_new(op, __parse_integer(arg));
	} else {
		expr = cf_expr_new(op, strdup(arg));
	}

	if (expr == NULL)
		cf_error("Invalid predicate or constant in %s(%s)", name, arg);

	return expr;
}


cf_expr_t *
expand_set(int op, const char *name, const char *set_name)
{
	cf_set_t	*set;
	cf_expr_t	*or = NULL;
	unsigned int	n;

	if (!(set = cf_set_find(set_name)))
		cf_error("Undefined set \"%s\"", set_name);

	for (n = 0; n < set->count; n++) {
		const char	*s = set->values[n];

		or = build_or(build_predicate(op, name, s), or);
	}

	return or;
}

void
build_switch(const char *syscall, unsigned int target,
		unsigned int count,
		const char * (*lookup)(unsigned int))
{
	const char	*name;
	cf_expr_t	*expr = NULL;
	cf_filter_t	*f;
	unsigned int	n;

	if (count <= 0)
		return;
	/* Combine all definitions for accept/socket/listen
	 * etc into a general case switch for socketcall() */
	for (n = 0; n < count; n++) {
		if (!(name = lookup(n))
		 || !(f = cf_syscall_find(name))
		 || !f->expr)
			continue;

		expr = build_or(expr,
			build_and(
			  build_apply(
		            cf_expr_new(AUD_FILT_OP_EQ, n),
			    target),
			  f->expr));
	}

	if (opt_debug > 1) {
		printf("syscall %s = ", syscall);
		cf_expr_print(stdout, expr);
		printf("\n");
	}
	attach_filter(syscall, expr);
}
