/*
 * Process the auditd configuration file.
 *
 * Copyright (C) 2003, SuSE Linux AG
 * Written by okir@suse.de
 */

#include <sys/types.h>
#include <stdio.h>
#include <stdlib.h>
#include <setjmp.h>
#include <string.h>
#include <stdarg.h>
#include <unistd.h>
#include <errno.h>

#include <linux/laus_audit.h>
#include "auditd.h"
#include "filter.h"
#include "config.h"

static int		parse_statement(void);
static void		parse_group(const char *, const char *);

cf_node_t *		cf_root;
static cf_node_t *	current;

int
audit_parse_daemon_config(const char *filename)
{
	int	r;

	/* Allocate the root node */
	current = cf_root = cf_node_new(NULL, NULL, NULL);

	if (access(filename, R_OK) < 0 && errno == ENOENT)
		return 0;

	if ((r = parse_file(filename, parse_statement)) < 0)
		return r;

	return 0;
}

static void
parse_real_statement(const char *keyword)
{
	char	namebuf[1024];

	parser_get(namebuf, sizeof(namebuf));

	switch (namebuf[0]) {
	case EQUALS:
		/* Assignment */
		parser_get(namebuf, sizeof(namebuf));
		cf_node_new(current, keyword, namebuf);
		break;
	case LEFT_BRACE:
		parse_group(keyword, NULL);
		break;
	case RIGHT_BRACE:
	case SEMICOLON:
	case COMMA:
	case LEFT_BRACKET:
	case RIGHT_BRACKET:
		cf_error("Unexpected token \"%s\"", namebuf);
	default:
		if (!strcmp(keyword, "include")) {
			parse_file(namebuf, parse_statement);
		} else {
			/* Named group */
			parser_expect(LEFT_BRACE);
			parse_group(keyword, namebuf);
		}
	}

	parser_expect(SEMICOLON);
}

int
parse_statement(void)
{
	char	keyword[64];

	if (!parser_get_token_or_eof(keyword, sizeof(keyword)))
		return 0;
	parse_real_statement(keyword);
	return 1;
}

int
parse_group_statement(void)
{
	char	keyword[64];

	parser_get(keyword, sizeof(keyword));
	if (keyword[0] == RIGHT_BRACE)
		return 0;
	parse_real_statement(keyword);
	return 1;
}

void
parse_group(const char *name, const char *value)
{
	cf_node_t	*parent;

	parent = current;
	current = cf_node_new(parent, name, value);

	while (parse_group_statement())
		;

	current = parent;
}
