
/*****************************************************************************
 *                                    M L X                                  *
 *                 Rendering Library for Accelerated 3d Hardware             *
 *                           (C) SuSE GmbH 1997, 1998                        *
 *****************************************************************************/
/* author: simon pogarcic, sim@suse.de */

/*
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifndef _MLX_H_
#define _MLX_H_

#include "ACL/aclapi.h"
#include "client_gpm.h"



typedef struct mlx_driver_rec {
    ACLenum hw;
    ACLubyte *drvmsg;
    ACLubyte *drvinfo;
    ACLvoid (*bind_client)(ACLContext);
    ACLenum supmod;
} MLXDriverRec, *MLXDriverRecPtr;



typedef struct mlx_driver_context {
/**********************
 * Driver Information *
 **********************/
/* The hw enumerator */
    ACLenum hw;
/* The driver used for the card */
    MLXDriverRecPtr drv;

/********************
 * Card Information *
 ********************/
/* Card index in the group of cards which use the same driver (first: 0) */
    ACLint tween_idx;
/* Card index as found by scan of supported hardware (first: 0) */
    ACLint card_idx;
/* PCI slot nr. where the card is located */
    ACLubyte card_slot;
/* rendering hardware info */
    ACLbyte *card_info;

/*********************
 * Screen Information
 *********************/
    ACLuint ScreenW;
    ACLuint ScreenH;
    ACLubyte ScreenBpp;
    ACLuint VirtualW;
    ACLuint VirtualH;
    ACLint ScreenPProd;
    ACLuint FrontBuffOffset;
    ACLuint BackBuffOffset;
    ACLuint BackBuffAddrOffset;
    ACLuint DepthBuffOffset;
    ACLuint TexBuffOffset;
    ACLuint TexBuffMax;
/* size in pixels (w/o bpp calculated) */
    ACLuint VirtScreenSize;
    ACLuint ScreenSize;
/* size in bytes of whole video RAM */
    ACLuint VidMemSize;
/* which buffer is currently showed? (back, front, left, right,... */
    ACLuint Buff;
/* screen data already set ? */
    ACLbool screen_set;

/************************
 * HW access Information
 ************************/
    ACLPointer mapCTL;
    ACLPointer mapCTL0;
    ACLPointer mapCTL1;
    ACLPointer mapFB;
    ACLPointer mapLB;
    ACLPointer mapTEX;
    ACLPointer hostTEX;
    ACLPointer mapStartDMA;
    ACLPointer mapCountDMA;
/* hw access info set ? */
    ACLbool hwaccess_set;

    struct mlx_driver_context *next, *prev;

} MLXDriverCtx, *MLXDriverCtxPtr;



/*****************************************************************************
 * MLX library for ACL API (mlxlib.c):
 *****************************************************************************/
/*
 * The folowing macros have to be placed at very first lines
 * of every graphic primitive which works with ACLContext.
 * The intention with 'static' types is to avoid initialization
 * of hardware dependent data every time the primitive is called,
 * because those data, once set, must be changed only when using
 * the primitive function on an other graphic device of the same type.
 *
 * Example:
 * ========
 * ACLvoid fast_primitive_func( ACLcontext aclctx )
 * {
 *	MLX_INIT_ACCESS_DRIVERCTX;
 *	MLX_INIT_ACCESS_CTL;
 *	MLX_INIT_ACCESS_TEX;		...and other INIT_ACCESS_ required
 *	....
 *
 *	MLX_SET_ACCESS_DRIVERCTX(aclctx);
 *
 *	if( olddrvidx ^ newdrvidx )
 *	{
 *	    MLX_SET_NEW_DRIVERCTX;
 *	    MLX_SET_ACCESS_CTL;
 *	    MLX_SET_ACCESS_TEX;		...and other SET_ACCESS_ required
 *	    ....
 *	}
 *	....
 *	....
 * }
 *
 */

#define DRVCTX(member) (drvctx->member)

#define MLX_INIT_ACCESS_DRIVERCTX \
	static ACLint olddrvidx = -1; \
	static MLXDriverCtxPtr drvctx = NULL

#define MLX_INIT_ACCESS_CTL \
	static volatile ACLPointer MMIOBASE = NULL

#define MLX_INIT_ACCESS_TEX \
	static volatile ACLPointer MMTEXBASE = NULL

#define MLX_INIT_ACCESS_DMA \
	static volatile ACLPointer MMSTARTDMA = NULL; \
	static volatile ACLPointer MMCOUNTDMA = NULL



#define MLX_SET_NEW_DRIVERCTX(aclctx) \
	olddrvidx = aclctx->drvidx; \
	drvctx = MLXdriverctx[olddrvidx]

#define MLX_SET_ACCESS_CTL \
	MMIOBASE = drvctx->mapCTL

#define MLX_SET_ACCESS_TEX \
	MMTEXBASE = drvctx->mapTEX

#define MLX_SET_ACCESS_DMA \
	MMSTARTDMA = drvctx->mapStartDMA; \
	MMCOUNTDMA = drvctx->mapCountDMA



MLXDriverCtxPtr mlx_drvidx_to_drvctx( ACLint );
MLXDriverCtxPtr	mlx_search_driver( ACLenum, ACLenum );
void		mlx_list_drivers( ACLvoid );

ACLint		mlx_glint_init();



extern MLXDriverCtxPtr		MLXdriverctx[];
extern MLXDriverCtxPtr		MLXfirstdrvctx;



#endif
