#!/usr/bin/perl -w
$text = <<'EOF';
/*+****************************************************************************



				The ACL API



 This API is defined in order to:
 --------------------------------

 - offer a standard/generic path for Mesa/X11 device driver developers. The
   required information for the development of a Mesa/X11 device driver are
   (should be) all included in this file;

 - have an easy "bootstrap" for Mesa/X11 driver developers. You can start with
   the development of the support of a very small number of functions and
   modules and already get a complete Mesa (=> OpenGL 1.2) support. You can
   than move to a more optimized driver by writting the support for optionals
   functions and modules;

 - remove any driver dependence by the Mesa core internal structures;

 - get for free real (or emulated) GLX (/WGL ?) support;

 - this API is developed mainly with hardware accelerators in mind;

 - this API doesn't require any change to the current Mesa sources because it
   is used by the generic driver aclMesa;

 - this API is easy to extend. You can just add another module (for example in
   order to support a OpenGL extension);

 - share common sources among different drivers (for example texture memory
   managment).



 How to write drivers using this API:
 ------------------------------------

 PURE SOFTWARE RENDERING (DRAWING INTO FB  USING HOST CPU):
 ==========================================================

 1) write the initialisation code for *SW functions;

 2) write the code for the mandatory functions in the Color module
    (ACLModules->ACLColorMod). If the acl(TO DO) function can return a RGB
    visual, you have to write the code for:

	WriteRGBASpan,
	WriteRGBAPixels,
	ReadRGBASpan,
	ReadRGBAPixels.

    If the acl(TO DO) function can return a color Index visual, you have to
    write the code for:

	WriteCI32Span,
	WriteCI32Pixels,
	ReadCI32Span,
	ReadCI32Pixels.

 3) done: you have a slow (but complete) driver that it is able to run any
    OpenGL 1.2 application;

 4) you can highly improve the driver performances writting the code for
    some/all of the optional functions of the Color module.



 Z BUFFER, TRIANGLES AND COLOR INTERPOLATION HARDWARE SUPPORT:
 =============================================================

 1) first implement functions from 'PURE SOFTWARE RENDERING';

 2) the support for pure software rendering is ready and the aclMesa driver can
    fallback to software rendering when hardware support is missing (when
    texture mapping is enabled for example);

 3) write the code for the ACLModules->ACLDepthMod;

 4) the driver still able to do only pure software rendering but now the
    ZBuffer is managed by the driver and is located in the card memory instead
    than in the CPU memory. You can improve the driver performances writting
    the code for some/all of the optional functions of the Depth module;

 5) write the code for the ACLModules->ACLAccelMod. Now the aclMesa driver will
    call the accelerated functions for drawing points/lines/traingles/quads
    when possible (for example it will fallback to pure software rendering if
    texture mapping is enabled and it will not use these functions);

 (TO DO: write other cases)

 ******************************************************************************

                                  History

 V2.00 - second release, currently used by MLX rendering library. The few
 	 definition problems about usage of this API are present, however
	 the current API 'as is' makes my life much simpler for connecting
	 it to MLX library. The clean interface has to be worked out.

 	 David Bucciarelli <tech.hmw@plus.it>:
 		- Modular design
		- Initial texture management module
		- how-to's

	 Simon Pogarcic <sim@suse.de>:
	 	- Initial release (from glx-0.7c)
		- Interface setup for rendering library
		- perl script

 ******************************************************************************/

/******************************************************************************
 ******************************************************************************
         THIS INCLUDE FILE IS AUTOMATICLY GENERATED FROM aclapi.pl FILE !
        IF YOU WANT TO MAKE SOME PERMANENT CHANGES, PLEASE EDIT aclapi.pl,
         SET PERMISSIONS TO +x OF aclapi.pl (IF NOT ALREADY) AND RUN IT.
 ******************************************************************************
 ******************************************************************************/

/*
 * Handling of Mesa sources dependency:
 *     Put in your application '#define _ACL_MESA_'
 *     before including aclapi.h
 */

/*
 * Handling of OpenGL data types dependency
 *     Put in your application '#define _ACL_OPENGL_'
 *     before including aclapi.h
 */



#ifndef _ACLAPI_H_
#define _ACLAPI_H_

#define _ACL_API_VERSION_STRING_	"v2.00"
#define _ACL_API_AUTHORS_STRING_	"David Bucciarelli, Simon Pogarcic"



/*+
 * Internal ACL* typedefs which could vary on different OS and hw
 */
#ifdef _ACL_OPENGL_

    #include "GL/gl.h"

    #define ACLbyte		GLbyte
    #define ACLubyte		GLubyte

    #define ACLshort		GLshort
    #define ACLushort		GLushort

    #define ACLint		GLint
    #define ACLuint		GLuint

    #define ACLfloat		GLfloat
    #define ACLdouble		GLdouble
    #define ACLvoid		GLvoid
    #define ACLbool		GLboolean

    #define ACLclampf		GLclampf
    #define ACLclampd		GLclampd

    #define ACLsizei		GLsizei
    #define ACLglenum		GLenum

    #define ACLbitfield		GLbitfield

#else

    typedef char		ACLbyte;
    typedef unsigned char	ACLubyte;
    
    typedef short		ACLshort;
    typedef unsigned short	ACLushort;
    
    typedef int			ACLint;
    typedef unsigned int	ACLuint;
    
    typedef float		ACLfloat;
    typedef double		ACLdouble;
    typedef void		ACLvoid;
    typedef unsigned char	ACLbool;

    typedef float		ACLclampf;
    typedef double		ACLclampd;

    typedef int			ACLsizei;

    typedef enum {
	GL_FALSE = 		0,
	GL_TRUE =		1,

	GL_FRONT =		0x0404,
	GL_BACK =		0x0405,
	GL_FRONT_AND_BACK = 	0x0408,

	GL_COLOR_BUFFER_BIT =	0x00004000,
	GL_DEPTH_BUFFER_BIT =	0x00000100,

        GL_NEVER =		0x0200,
        GL_LESS =		0x0201,
        GL_GEQUAL =		0x0206,
        GL_LEQUAL =		0x0203,
        GL_GREATER =		0x0204,
        GL_NOTEQUAL =		0x0205,
        GL_EQUAL =		0x0202,
        GL_ALWAYS =		0x0207
    } ACLglenum;

    typedef unsigned int 	ACLbitfield;

#endif



typedef ACLuint			ACLTexAddress;
typedef unsigned int *		ACLPointer;

#define	ACLFALSE		0x0
#define	ACLTRUE 		0x1



typedef enum {

/*
 * All supported drivers are enumerated here. These enums
 * are used by Query function. The library which implements ACL
 * API uses those enums to init the proper driver for client.
 * However, more generic access won't be bad.
 */
	ACL_DRIVER_NONE =	0x0,
	ACL_DRIVER_TXMX =	0x1,
	ACL_DRIVER_PM =		0x2,
	ACL_DRIVER_PM2 =	0x3,
	/**************************/
	ACL_DRIVER_LAST =	0x3,	/* keep it always up-to-date! */

/*
 * How to connect to a particular hardware and driver could be
 * specified with the folowing enums. The _FIRST and _LAST would
 * return the first or last registered resource of specified group.
 * Analoguos, _PREV and _NEXT would return previous and next hw
 * resource within currently used group.
 */
	ACL_GET_FIRST =		0x1,
	ACL_GET_LAST =		0x2,
	ACL_GET_PREV =		0x3,
	ACL_GET_NEXT =		0x4,

/*
 * The folowing defines are used for setting of ACLContext->Flags
 */
	ACL_RENDER_FLAT =	0,
	ACL_SWAP_BITBLT = 	0,

	ACL_RENDER_TWOSIDE =	0x00000001,
	ACL_RENDER_SMOOTH =	0x00000002,
	ACL_RENDER_TEX =	0x00000004,
	ACL_SWAP_SCREEN =	0x00000008,
	ACL_SWAP_PIXDEPTH =	0x00000010,
	ACL_QUAKE_HACK =	0x00000020,

/*
 * Which rendering modules are supported ? (for acl_modules.ModFlags)
 */
	ACL_MODULE_NONE =	0,
	ACL_MODULE_ACCEL =	0x00000001,
	ACL_MODULE_DEPTH =	0x00000002,
	ACL_MODULE_TEX =	0x00000004

} ACLenum;



typedef struct acl_color_mod 		*ACLColorMod;
typedef struct acl_accel_mod 		*ACLAccelMod;
typedef struct acl_depth_mod 		*ACLDepthMod;
typedef struct acl_tex_mod 		*ACLTexMod;

typedef struct acl_modules 		*ACLModules;

typedef struct acl_setup		*ACLSetup;

typedef struct acl_context 		*ACLContext;



/*+*************************************************************
 ***************************************************************
 * DESCRIPTION: Overriding/using core renderer dependence.
 * Currently, the considered core renderer is Mesa, but using
 * the folowing strategy, the enhancments for other 3D standards
 * or renderer are possible, for example for an OpenGL renderer,
 * or for a 3D games engine.
 ***************************************************************
 ***************************************************************/

#ifdef _ACL_MESA_

    #include "vb.h"
    #include "ACL/aclApiDrv.h"

    #define acl_hyper_context		GLcontext
    #define ACLHyperContext		GLcontext*

    #define acl_visual			GLvisual
    #define ACLVisual			GLvisual*

    #define acl_frame_buffer		GLframebuffer
    #define ACLFramebuffer		GLframebuffer*

    #define acl_vertex_buffer		struct vertex_buffer
    #define ACLVertexBuffer		struct vertex_buffer*

    #define acl_tex_image		struct gl_texture_image
    typedef struct gl_texture_image	*ACLTexImage;

    #define aclTexmanCtx		taclTexManagerCtx

#else

/* Set here the size of vertex buffer: */
    #define VB_SIZE			0xff
    #define MAX_TEX_SETS		2

    typedef struct acl_hyper_context	*ACLHyperContext;
    typedef struct acl_visual		*ACLVisual;
    typedef struct acl_frame_buffer	*ACLFramebuffer;
    typedef struct acl_vertex_buffer	*ACLVertexBuffer;

    typedef struct acl_tex_image	*ACLTexImage;

    typedef struct acl_texman_ctx	aclTexmanCtx;

/*+************************
 Vertex buffer for ACL API
 **************************/

typedef struct acl_vertex_buffer {
	ACLfloat Win[VB_SIZE][3];		/* Window coords */
	ACLubyte Fcolor[VB_SIZE][4];		/* Front colors (RGBA) */
	ACLubyte Bcolor[VB_SIZE][4];		/* Back colors (RGBA) */
	ACLubyte (*Color)[4];			/* == Fcolor or Bcolor */
	ACLfloat (*TexCoord)[4];
	ACLfloat MultiTexCoord[MAX_TEX_SETS][VB_SIZE][4];
} acl_vertex_buffer;



/*+***************************
 Visual structure for ACL API
 *****************************/
typedef struct acl_visual {
	ACLbool RGBAflag;	/*+ Is frame buffer in RGBA mode, not CI? */
	ACLbool DBflag;		/*+ Is color buffer double buffered? */
	ACLbool StereoFlag;	/*+ stereo buffer? */
	ACLint RedBits;		/*+ Bits per color component */
	ACLint GreenBits;
	ACLint BlueBits;
	ACLint AlphaBits;
	ACLint IndexBits;	/*+ Bits/pixel if in color index mode */
	ACLint AccumBits;	/*+ Number of bits per color channel, or 0 */
	ACLint DepthBits;	/*+ Number of bits in depth buffer, or 0 */
	ACLint StencilBits;	/*+ Number of bits in stencil buffer, or 0 */
	/*+
	 * Software alpha planes:
	 */
	ACLbool FrontAlphaEnabled;
	ACLbool BackAlphaEnabled;
} acl_visual;



/*+************************************
 Rendering bufer structure for ACL API
 **************************************/
typedef struct acl_frame_buffer {
	ACLVisual Visual;	/*+ The corresponding visual */
	ACLint Width;		/*+ Width of frame buffer in pixels */
	ACLint Height;		/*+ Height of frame buffer in pixels */
#if 0
	ACLDepth Depth;		/*+ array [Width*Height] of GLdepth values */
	ACLStencil Stencil;	/*+ array [Width*Height] of GLstencil values */
	ACLAccum Accum;		/*+ array [4*Width*Height] of GLaccum values */
#endif
	/*+ Software alpha planes: */
	ACLubyte *FrontAlpha;	/*+ array [Width*Height] of GLubyte */
	ACLubyte *BackAlpha;	/*+ array [Width*Height] of GLubyte */
	ACLubyte *Alpha;	/*+ Points to front or back alpha buffer */
	/*+
	 * Drawing bounds: intersection of window size and scissor box
	 */
	ACLint Xmin, Xmax, Ymin, Ymax;
/*+
TODO: add structures for ACLDepth, ACLStencil and ACLAccum buffers
*/
} acl_frame_buffer;



/*+**********************
 Texture Image structure
 ************************/
typedef struct acl_tex_image {
	ACLglenum Format;	/*+ 1 ubyte formats: GL_ALPHA, GL_LUMINANCE,
				 *	GL_INTENSITY, GL_COLOR_INDEX
				 *  2 ubyte formats: GL_LUMINANCE_ALPHA
				 *  3 ubyte formats: GL_RGB
				 *  4 ubyte formats: GL_RGBA
				 */
	ACLglenum IntFormat;	/*+ Internal format as given by the user in the
                                    glTexImage() */
	ACLuint Border;		/*+ 0 or 1 */
	ACLuint Width;		/*+ = 2^WidthLog2 + 2*Border */
	ACLuint Height;		/*+ = 2^HeightLog2 + 2*Border */
	ACLuint Depth;		/*+ = 2^DepthLog2 + 2*Border */
	ACLuint Width2;		/*+ = Width - 2*Border */
	ACLuint Height2;	/*+ = Height - 2*Border */
	ACLuint Depth2;		/*+ = Depth - 2*Border */
	ACLuint WidthLog2;	/*+ = log2(Width2) */
	ACLuint HeightLog2;	/*+ = log2(Height2) */
	ACLuint DepthLog2;	/*+ = log2(Depth2) */
	ACLuint MaxLog2;	/*+ = MAX(WidthLog2, HeightLog2) */
	ACLubyte *Data;		/*+ Image data as unsigned bytes */

	/*+ For device driver: */
	void *DriverData;          /*+ Arbitrary device driver data */

} acl_tex_image;



/*+*****************************************************************
 Context of texture manager, if used without ACL Mesa Device Driver
 *******************************************************************/
typedef struct acl_texman_ctx {
    	ACLuint	CURRENTLY_NOTHING;
} acl_texman_ctx;



/*+*****************************
 Main ACL API rendering context
 *******************************/

typedef struct acl_hyper_context {
	ACLVisual Visual;
	ACLContext DriverCtx;
	ACLVertexBuffer VB;
} acl_hyper_context;

#endif /* _ACL_MESA_ */



/*+*************************************************************
 ***************************************************************
 * DESCRIPTION: Definition of color buffer module
 *
 * OPTIONAL: NO
 ***************************************************************
 ***************************************************************/
typedef struct acl_color_mod {

/*+*************************************************************



    NAME: Index
    ===============================

    DESCRIPTION: Set current color index for drawing
                 flat-shaded/monocolor primitives

    PARAMETERS: index = the color index to use

    OPTIONAL: YES/NO (it can be NULL if CI mode isn't supported)
*/
    ACLvoid (*Index)
	( ACLContext ctx,
	  ACLuint index );

/*+*************************************************************



    NAME: Color
    ===============================

    DESCRIPTION: Set current color for drawing
                 flat-shaded/monocolor primitives

    PARAMETERS: red, green, blue, alpha = color to use 

    OPTIONAL: YES/NO (it can be NULL if RGB mode isn't supported)
*/
    ACLvoid (* Color)
	( ACLContext ctx,
	  ACLubyte red, ACLubyte green, ACLubyte blue, ACLubyte alpha );

/*+*************************************************************



    NAME: WriteRGBASpan
    ===============================

    DESCRIPTION: Draw a span of RGBA pixels

    PARAMETERS:      n = span length
                  x, y = span origin
             rgba[][4] = color of each pixel in the span
	        mask[] = pixel visible (0/1)

    OPTIONAL: YES/NO (it can be NULL if RGB mode isn't supported)
*/
    ACLvoid (* WriteRGBASpan)
	( const ACLHyperContext hctx,
	  ACLuint n, ACLint x, ACLint y,
	  const ACLubyte rgba[][4], const ACLubyte mask[]);

/*+*************************************************************



    NAME: WriteRGBADSpanGeneric
    ===============================

    DESCRIPTION: Draw a span of RGBA pixels

    PARAMETERS:      n = span length
                  x, y = span origin
             rgba[][4] = color of each pixel in the span
	        mask[] = pixel visible (0/1)

    OPTIONAL: YES/NO (it can be NULL if RGB mode isn't supported)
*/
    ACLuint (* WriteRGBADSpanGeneric)(const ACLHyperContext,
			ACLuint n, ACLint x, ACLint y, const GLdepth z[],
			ACLubyte mask[]);

/*+*************************************************************

    NAME: WriteRGBADPixelsGeneric
    ===============================

    DESCRIPTION: Draw a span of RGBA pixels

    PARAMETERS:      n = span length
                  x, y = span origin
             rgba[][4] = color of each pixel in the span
	        mask[] = pixel visible (0/1)

    OPTIONAL: YES/NO (it can be NULL if RGB mode isn't supported)
*/
    ACLuint (* WriteRGBADPixelsGeneric)(const ACLHyperContext,
			ACLuint n, ACLint x[], ACLint y[], const GLdepth z[],
			ACLubyte mask[]);

/*+*************************************************************



    NAME: WriteRGBSpan
    ===============================

    DESCRIPTION: Draw a span of RGB pixels

    PARAMETERS:      n = span length
                  x, y = span origin
              rgb[][3] = color of each pixel in the span
	        mask[] = pixel visible (0/1)

    OPTIONAL: ???
*/
    ACLvoid (* WriteRGBSpan)
	( const ACLHyperContext hctx,
	  ACLuint n, ACLint x, ACLint y,
	  const ACLubyte rgb[][3], const ACLubyte mask[] );

/*+*************************************************************



    NAME: WriteMonoRGBASpan
    ===============================

    DESCRIPTION: Draw a span of RGB pixels

    PARAMETERS:      n = span length
                  x, y = span origin
	        mask[] = pixel visible (0/1)

    OPTIONAL: ???
*/
    ACLvoid (* WriteMonoRGBASpan)
	( const ACLHyperContext hctx,
	  ACLuint n, ACLint x, ACLint y, const ACLubyte mask[] );

/*+*************************************************************



    NAME: WriteCI32Span
    ===============================

    DESCRIPTION: Draw a span of CI pixels

    PARAMETERS:      n = number of pixels
                  x, y = span origin
               index[] = color of each pixel

    OPTIONAL: YES/NO (it can be NULL if CI mode isn't supported)
   */
    ACLvoid (* WriteCI32Span)
	( const ACLHyperContext hctx,
	  ACLuint n, ACLint x, ACLint y, ACLuint index[]);

/*+*************************************************************



    NAME: ReadRGBASpan
    ===============================

    DESCRIPTION: Read a span of RGBA pixels

    PARAMETERS:      n = number of pixels
                  x, y = span origin
             rgba[][4] = color of each pixel

    OPTIONAL: YES/NO (it can be NULL if RGB mode isn't supported)
*/
    ACLvoid (* ReadRGBASpan)
	( const ACLHyperContext hctx,
	  ACLuint n, ACLint x, ACLint y, ACLubyte rgba[][4] );

/*+*************************************************************



    NAME: ReadCI32Span
    ===============================

    DESCRIPTION: Read a span of CI pixels

    PARAMETERS:      n = number of pixels
                  x, y = span origin
               index[] = index of each pixel

    OPTIONAL: YES/NO (it can be NULL if CI mode isn't supported)
*/
    ACLvoid (* ReadCI32Span)
	( const ACLHyperContext hctx,
	  ACLuint n, ACLint x, ACLint y, ACLuint index[] );

/*+*************************************************************



    NAME: WriteRGBAPixels
    ===============================

    DESCRIPTION: Draw a sparse set of RGBA pixels

    PARAMETERS:      n = number of pixels
              x[], y[] = position of each pixel
             rgba[][4] = color of each pixel
	        mask[] = pixel visible (0/1)

    OPTIONAL: YES/NO (it can be NULL if RGB mode isn't supported)
*/
    ACLvoid (* WriteRGBAPixels)
	( const ACLHyperContext hctx,
	  ACLuint n, const ACLint x[], const ACLint y[],
	  const ACLubyte rgba[][4], const ACLubyte mask[] );

/*+*************************************************************



    NAME: WriteMonoRGBAPixels
    ===============================

    DESCRIPTION: Draw a sparse set of RGBA pixels

    PARAMETERS:      n = number of pixels
              x[], y[] = position of each pixel
	        mask[] = pixel visible (0/1)

    OPTIONAL: YES/NO (it can be NULL if RGB mode isn't supported)
*/
    ACLvoid (* WriteMonoRGBAPixels)
	( const ACLHyperContext hctx,
	  ACLuint n, const ACLint x[], const ACLint y[], const ACLubyte mask[]);

/*+*************************************************************



    NAME: WriteCI32Pixels
    ===============================

    DESCRIPTION: Draw a sparse set of CI pixels

    PARAMETERS:      n = number of pixels
              x[], y[] = position of each pixel
               index[] = color of each pixel
	        mask[] = pixel visible (0/1)

    OPTIONAL: YES/NO (it can be NULL if CI mode isn't supported)
*/
    ACLvoid (* WriteCI32Pixels)
	( const ACLHyperContext hctx,
	  ACLuint n, ACLint x[], ACLint y[], ACLuint index[], ACLubyte mask[] );

/*+*************************************************************



    NAME: ReadRGBAPixels
    ===============================

    DESCRIPTION: Read a sparse set of RGBA pixels

    PARAMETERS:      n = number of pixels
              x[], y[] = position of each pixel
             rgba[][4] = color of each pixel
	        mask[] = pixel visible (0/1)

    OPTIONAL: YES/NO (it can be NULL if RGB mode isn't supported)
*/
    ACLvoid (* ReadRGBAPixels)
	( const ACLHyperContext hctx,
	  ACLuint n, const ACLint x[], const ACLint y[],
	  ACLubyte rgba[][4], const ACLubyte mask[] );

/*+*************************************************************



    NAME: ReadCI32Pixels
    ===============================

    DESCRIPTION: Read a sparse set of CI pixels

    PARAMETERS:      n = number of pixels
              x[], y[] = position of each pixel
               index[] = index of each pixel
	        mask[] = pixel visible (0/1)

    OPTIONAL: YES/NO (it can be NULL if CI mode isn't supported)
*/
    ACLvoid (* ReadCI32Pixels)
	( const ACLHyperContext hctx,
	  ACLuint n, ACLint x[], ACLint y[], ACLuint index[], ACLubyte mask[] );

/*+*************************************************************

	TO DO (optional):

	WriteMonoCIPixels
	WriteMonoCISpan
	WriteCI8Span
	and other)
*/
} acl_color_mod;



/*+*************************************************************
 ***************************************************************
 * DESCRIPTION: Definition of accelerated primitives
 *              rendering
 *
 * OPTIONAL: YES (IsAvailable can be GL_FALSE)
 ***************************************************************
 ***************************************************************/
typedef struct acl_accel_mod {

/*+*************************************************************



    NAME: UpdateFuncPointers
    ===============================

    DESCRIPTION: Update all the function pointers in the
                 current module

    PARAMETERS: 

    OPTIONAL: YES (it can be NULL)
*/
    ACLvoid (* UpdateFuncPointers)
	( ACLContext ctx );

/*+*************************************************************



    NAME: ClearIndex
    ===============================

    DESCRIPTION: Set the color for clearing the frame buffer

    PARAMETERS: index = color index to use for clearing the
                        framebuffer when in CI mode

    OPTIONAL: YES (it can be NULL if also the Clear is NULL)
*/
    ACLvoid (* ClearIndex)
	( ACLContext ctx, ACLuint index );

/*+*************************************************************



    NAME: ClearColor
    ===============================

    DESCRIPTION: Set the color for clearing the frame buffer

    PARAMETERS: red, green, blue, alpha = color to use for clearing the
                                          framebuffer when in RGB mode

    OPTIONAL: YES (it can be NULL if also the Clear is NULL)
*/
    ACLvoid (* ClearColor)
	( ACLContext ctx,
	  ACLubyte red, ACLubyte green, ACLubyte blue, ACLubyte alpha );

/*+*************************************************************



    NAME: Clear
    ===============================

    DESCRIPTION: Clear the color/depth/stencil/accum buffer(s).
                 'mask' indicates which buffers need to be cleared. Return
                 a bitmask indicating which buffers weren't cleared by the
                 driver function.
		 If 'all' is true then clear the whole buffer, else clear
                 the region defined by (x,y,width,height).


    PARAMETERS: mask = indicates which buffers need to be cleared
                all, x, y,
		width, height = if 'all' is true then the clear the whole
                                buffer, else clear the region defined by
                                (x,y,width,height)

    OPTIONAL: YES (it can be NULL)
*/
    ACLbitfield (*Clear)
	( ACLHyperContext hctx,
	  ACLbitfield mask, ACLbool all,
	  ACLint x, ACLint y, ACLint width, ACLint height );

/*+*************************************************************



    NAME: SetLogOp
    ===============================

    DESCRIPTION: 

    PARAMETERS: 

    OPTIONAL: YES (it can be NULL)
*/
    ACLvoid (*SetLogOp)(ACLContext aclctx, 
          ACLbool enable, ACLint logop);

/*+*************************************************************



    NAME: SetDither
    ===============================

    DESCRIPTION: 

    PARAMETERS: 

    OPTIONAL: YES (it can be NULL)
*/

   ACLvoid (*SetDither)(ACLContext aclctx, ACLbool enable, 
				  ACLbool modeline);

/*+*************************************************************



    NAME: SetAlpha
    ===============================

    DESCRIPTION: 

    PARAMETERS: 

    OPTIONAL: YES (it can be NULL)
*/
   ACLvoid (*SetAlpha)(ACLContext , ACLbool , ACLint , ACLint );

/*+*************************************************************



    NAME: SetBlend
    ===============================

    DESCRIPTION: 

    PARAMETERS: 

    OPTIONAL: YES (it can be NULL)
*/
   ACLvoid (*SetBlend)(ACLContext , ACLbool , ACLint , ACLint );


/*+*************************************************************



    NAME: SetFog
    ===============================

    DESCRIPTION: 

    PARAMETERS: 

    OPTIONAL: YES (it can be NULL)
*/
   ACLvoid (*SetFog)(ACLContext aclctx,
	ACLbool enable, ACLint mode, ACLfloat density,
	ACLfloat r, ACLfloat g, ACLfloat b, ACLfloat a);

/*+*************************************************************



    NAME: Points
    ===============================

    DESCRIPTION: Draws the set points stored in the
                 ctx->RenderingBuffer

    PARAMETERS: first = index of the first point to draw
                 last = index of the last point to draw

    OPTIONAL: YES (it can be NULL)
*/
    ACLvoid (* Points)
	( ACLHyperContext hctx,
	  ACLuint first, ACLuint last);

/*+*************************************************************



    NAME: Line
    ===============================

    DESCRIPTION: Draws a line stored in the
                 ctx->RenderingBuffer

    PARAMETERS:    v1 = index of the first vertex
                   v2 = index of the second vertex
		   pv = index of the color to use for flat shaded
		        lines

    OPTIONAL: YES (it can be NULL)
*/
    ACLvoid (* Line)
	( ACLHyperContext hctx,
	  ACLuint v1, ACLuint v2,  ACLuint pv);
  
/*+*************************************************************



    NAME: Triangle
    ===============================

    DESCRIPTION: Draws a triangle stored in the
                 ctx->RenderingBuffer

    PARAMETERS:  v1 = index of the first vertex
                 v2 = index of the second vertex
                 v3 = index of the third vertex
		 pv = index of the color to use for flat shaded
		      triangles

    OPTIONAL: YES (it can be NULL)
*/
    ACLvoid (* Triangle)
	( ACLHyperContext hctx,
	  ACLuint v1, ACLuint v2, ACLuint v3, ACLuint pv);

/*+*************************************************************



    NAME: Quad
    ===============================

    DESCRIPTION: Draws a quad stored in the
                 ctx->RenderingBuffer

    PARAMETERS:  v1 = index of the first vertex
                 v2 = index of the second vertex
                 v3 = index of the third vertex
                 v4 = index of the fourth vertex
		 pv = index of the color to use for flat shaded
		      quads

    OPTIONAL: YES (it can be NULL)
*/
    ACLvoid (* Quad)
	( ACLHyperContext hctx,
	  ACLuint v1, ACLuint v2, ACLuint v3, ACLuint v4, ACLuint pv);

  /*+ TODO: RasterSetup, RenderVB */
} acl_accel_mod;



/*+*************************************************************
 ***************************************************************
 * DESCRIPTION: Definition of depth buffer module
 *
 * OPTIONAL: YES (IsAvailable can be GL_FALSE)
 ***************************************************************
 ***************************************************************/
typedef struct acl_depth_mod {

/*+*************************************************************



    NAME: UpdateFuncPointers
    ===============================

    DESCRIPTION: Update all the function pointers in the
                 current module

    PARAMETERS: 

    OPTIONAL: NO
*/
    ACLvoid (* UpdateFuncPointers)
	( ACLContext ctx );

/*+*************************************************************



    NAME: Enabled
    ===============================

    DESCRIPTION: Enable/disable the depth test

    PARAMETERS: GL_TRUE/GL_FALSE

    OPTIONAL: NO
*/
    ACLvoid  (* Enabled)
	( ACLContext ctx,
	  ACLbool state);

/*+*************************************************************



    NAME: ClearDepth
    ===============================

    DESCRIPTION: Set the current clear depth value

    PARAMETERS: 0.0 <= depth <= 1.0

    OPTIONAL: NO
*/
    ACLvoid (* ClearDepth)
	( ACLContext ctx,
	  ACLfloat depth );

/*+*************************************************************



    NAME: Func
    ===============================

    DESCRIPTION: Set the current depth function

    PARAMETERS: GL_NEVER, GL_LESS, GL_EQUAL, GL_LEQUAL, GL_GREATER,
                GL_NOTEQUAL, GL_GEQUAL, GL_ALWAYS

    OPTIONAL: NO
*/
    ACLvoid (* Func)
	( ACLContext ctx,
	  ACLglenum func );

/*+*************************************************************



    NAME: Mask
    ===============================

    DESCRIPTION: Enable/Disable writing to the depth buffer 

    PARAMETERS: GL_TRUE/GL_FALSE

    OPTIONAL: NO
*/
    ACLvoid (* Mask)
	( ACLContext ctx,
	  ACLbool mask );

/*+*************************************************************



    NAME: Range
    ===============================

    DESCRIPTION: Specify the mapping of z values

    PARAMETERS: 0.0 <= near, far <= 1.0

    OPTIONAL: NO
*/
    ACLvoid (* Range)
	( ACLContext ctx,
	  ACLclampd near, ACLclampd far );

/*+*************************************************************



    NAME: ReadSpan
    ===============================

    DESCRIPTION: Read a span of depth values from the depth buffer

    PARAMETERS:       n = span length
                   x, y = coordinates of the first depth value
                depth[] = array where the read values are stored

    OPTIONAL: NO
*/
    ACLvoid (*ReadSpan)
	( ACLContext ctx,
	  ACLuint n, ACLint x, ACLint y, ACLfloat depth[] );

/*+*************************************************************



    NAME: WriteSpan
    ===============================

    DESCRIPTION: Write a span of depth values from the depth buffer

    PARAMETERS:       n = span length
                   x, y = coordinate of the first depth value
                depth[] = array where the values to write are stored

    OPTIONAL: NO
*/
    ACLvoid (*WriteSpan)
	( ACLContext ctx,
	  ACLuint n, ACLint x, ACLint y, ACLfloat depth[] );

/*+*************************************************************



    NAME: DepthTestSpan
    ===============================

    DESCRIPTION: Apply the depth buffer test to an span of
                 pixels and return an updated pixel mask. The depth
		 buffer is updated with new depth values that have
		 not failed the depth test.

    PARAMETERS:       n = span length
                   x, y = coordinate of the first depth value
                depth[] = array where the values to write are stored

    OPTIONAL: YES (it can be NULL)
*/
    ACLuint (*DepthTestSpan)
	( ACLContext ctx,
	  ACLuint n, ACLint x, ACLint y, ACLfloat z[], ACLubyte mask[] );

/*+*************************************************************



    NAME: DepthTestPixels
    ===============================

    DESCRIPTION: Apply the depth buffer test to an array of
                 pixels and return an updated pixel mask. The depth
		 buffer is updated with new depth values that have
		 not failed the depth test.

    PARAMETERS:        n = number of pixels
                x[], y[] = coordinates of the depth values
                 depth[] = array of new depth values
                  mask[] = mask of updated pixels (GL_TRUE/GL_FALSE)

    OPTIONAL: YES (it can be NULL)
*/
    ACLvoid (*DepthTestPixels)
	( ACLContext ctx,
	  ACLuint n, ACLint x[], ACLint y[], ACLfloat z[], ACLubyte mask[] );
} acl_depth_mod;



/*+*************************************************************
 ***************************************************************
 * DESCRIPTION: Definition of function pointers for
 *              texture functions.
 *
 *              The current definition make several
 *              assumptions:
 *               - mipmapping is silently ignored
 *               - GL_TEXTURE_1D and GL_TEXTURE_3D are
 *                 handled with software emulation
 *
 * OPTIONAL: YES
 ***************************************************************
 ***************************************************************/
typedef struct acl_tex_mod {

/*+*************************************************************



    NAME: UpdateFuncPointers
    ===============================

    DESCRIPTION: Update all the function pointers in the
                 current module

    PARAMETERS: 

    OPTIONAL: NO
*/
    ACLvoid (* UpdateFuncPointers)
	( ACLContext ctx );

/*+*************************************************************



    NAME: Enabled
    ===============================

    DESCRIPTION: Enable/Disable texture mapping

    PARAMETERS:     state = GL_TRUE/GL_FALSE

    OPTIONAL: NO
*/
    ACLvoid  (* Enabled)
	( ACLContext ctx,
	  ACLbool state);

/*+*************************************************************



    NAME: GetMinTexMemAddress
    ===============================

    DESCRIPTION: Return the minimum address in the texture memory
                 of the graphics accelerator usable by the Mesa
                 ACL texture manager.

    PARAMETERS: 

    OPTIONAL: NO

    ACLTexAddress (* GetMinTexMemAddress)
	( ACLContext ctx );
*/

/*+*************************************************************



    NAME: GetMaxTexMemAddress
    ===============================

    DESCRIPTION: Return the maximum address in the texture memory
                 of the graphics accelerator usable by the Mesa
                 ACL texture manager.

    PARAMETERS: 

    OPTIONAL: NO

    ACLTexAddress (* GetMaxTexMemAddress)
	( ACLContext ctx );
*/

/*+*************************************************************



    NAME: RequiredTexMem
    ===============================

    DESCRIPTION: Return the amount of texture memory required in
                 order to store the texture image described by "tex"
                 in the texture memory of the graphics accelerator.
                 If the texture images have to be aligned to a
                 specific amount of bytes (i.e: 16), the returned
                 number must be a multiple of the specific amount
                 of bytes.

    PARAMETERS:          tex = a pointer to a ACLTexImage describing
                               an image

    OPTIONAL: NO

    ACLuint (* RequiredTexMem)
	( ACLContext ctx,
	  ACLTexImage tex );
*/
/*+*************************************************************



    NAME: TranslateImage
    ===============================

    DESCRIPTION: Translate from the internal Mesa core representation
                 of a texture map to a format suitable for the hardware.
                 This function can be also a NOP if the Mesa core
                 representation can be used directly. The translated
                 image should be stored in a CPU memory area pointed
                 by the translatedImage field. Returns GL_FALSE if failed
                 (i.e. not supported by the hardware), GL_TRUE otherwise.

    PARAMETERS:          tex = a pointer to a ACLTexImage describing
                               the image
             translatedImage = initialized with a pointer to the main memory
                               are where the translated image is stored

    OPTIONAL: NO
*/
    ACLbool (* TranslateImage)
	( ACLContext ctx,
	  ACLTexImage tex, ACLint format, ACLint level, ACLuint name, 
          ACLvoid *dd);

/*+*************************************************************
    NAME: SubImage
    ===============================

    DESCRIPTION:

    PARAMETERS:

    OPTIONAL: NO
*/
    ACLvoid (* SubImage)
       ( ACLContext * ctx, ACLenum target,
	 ACLvoid *dd, ACLint level,
	 ACLint xoffset, ACLint yoffset, ACLint width, ACLint height,
	 ACLint internalFormat, ACLTexImage image);

/*+*************************************************************



    NAME: DownloadToTexMem
    ===============================

    DESCRIPTION: Download to the texture memory (lb) the maps described
                 by "tex". The MLX Library should have only to copy
                 the already translated image stored in ACLTexImage->DriverData
                 into the texture memory starting from address
                 "memAddress"

    PARAMETERS:          tex = a pointer to a ACLTexImage describing
                               an image
                  memAddress = texture memory address where to upload
                               the image

    OPTIONAL: NO
*/
    ACLvoid (* DownloadToTexMem)
	( ACLContext ctx,
	  ACLint level, ACLvoid * dd);

/*+*************************************************************



    NAME: Bind
    ===============================

    DESCRIPTION: Select the texture memory address to use as source
                 for texture mapping

    PARAMETERS:   memAddress = texture memory address where the image
                               was previously uploaded

    OPTIONAL: NO
*/
    ACLvoid (* Bind)
	( ACLContext ctx, ACLuint name, 
	  ACLvoid * dd );

/*+*************************************************************



    NAME: Env
    ===============================

    DESCRIPTION: Select texture function to use for texture
                 mapping. If the function isn't supported by
                 the hardware, it must return GL_FALSE, otherwise
                 GL_TRUE

    PARAMETERS:          pname = GL_TEXTURE_ENV_MODE or GL_TEXTURE_ENV_COLOR
                         param = GL_MODULATE, GL_BLEND, GL_DECAL,
                                 or GL_REPLACE if pname=GL_TEXTURE_ENV_MODE
                                 else it is the costant
                                 color used for the GL_BLEND mode
    OPTIONAL: NO
*/
    ACLbool (* Env)
	( ACLContext ctx,
	  ACLglenum pname, const ACLfloat *param );

/*+*************************************************************



    NAME: Param
    ===============================

    DESCRIPTION: Handle glTexParameter() settings

    PARAMETERS:	pname = GL_TEXTURE_MIN_FILTER, GL_TEXTURE_MAG_FILTER,
			GL_TEXTURE_WRAP_S or GL_TEXTURE_WRAP_T
			param = dependant on pname:

			if GL_TEXTURE_MIN_FILTER or GL_TEXTURE_MAG_FILTER
				then GL_NEAREST or GL_LINEAR

			if GL_TEXTURE_WRAP_S or GL_TEXTURE_WRAP_T
				then GL_CLAMP or GL_REPEAT

    OPTIONAL: NO
*/
    ACLvoid (* Param)
	( ACLContext ctx,
	  ACLglenum target, ACLglenum pname, 
	  const ACLfloat *param, ACLvoid *dd);

/*+*************************************************************



    NAME: TexDel
    ===============================

    DESCRIPTION: Free of alloced memory

    PARAMETERS:	dd pointer to driver memory

    OPTIONAL: NO
*/
    ACLvoid * (* TexDel)
	( ACLContext ctx,
	  ACLvoid * dd );

/*+*************************************************************



    NAME: AllocTexObjData
    ===============================

    DESCRIPTION: Free of alloced memory

    PARAMETERS:	dd pointer to driver memory

    OPTIONAL: NO
*/
   ACLvoid * (* AllocTexObjData)
   	( ACLContext ctx );

/*+*************************************************************

    

    NAME: Palette
    ===============================

    DESCRIPTION: 

    PARAMETERS:	dd pointer to driver memory

    OPTIONAL: NO
*/
    ACLvoid (* Palette)
         (ACLContext aclctx, ACLuint size, ACLuint * data, ACLvoid *dd);


/*+*************************************************************

    

    NAME: SwitchToGlobalPalette
    ===============================

    DESCRIPTION: 

    PARAMETERS:

    OPTIONAL: NO
*/
   ACLvoid (*SwitchToGlobalPalette)
         (ACLContext aclctx);


/*+*************************************************************

    

    NAME: DownloadGlobalPalette
    ===============================

    DESCRIPTION: 

    PARAMETERS:

    OPTIONAL: NO
*/
   ACLvoid (*DownloadGlobalPalette)
         (ACLContext aclctx, ACLuint size, ACLuint * data, ACLvoid *dd);

/*+*************************************************************
TODO:
	TranslateSubTexImage
        UploadToSubTexMem
	and other
*/
} acl_tex_mod;



/*+*************************************************************
 ***************************************************************
 * DESCRIPTION:
 * Definition of function pointers for driver rendering modules
 *
 * OPTIONAL: NO
 ***************************************************************
 ***************************************************************/
typedef struct acl_modules {

/* The flags about presence of different modules (max 32) */
    ACLuint ModFlags;

/* For the color buffer (NOT OPTIONAL) */
    struct acl_color_mod Color;

/* For the accelerated rendering of primitives (optional, can be NULL) */
    ACLAccelMod Accel;

/* For the depth buffer (optional, can be NULL) */
    ACLDepthMod Depth;

/* Texturing */
    ACLTexMod Tex;

/******************************************
 * The folowing functions are obligatory: *
 ******************************************/
/* has to be called before any rendering, for screen setup */
    ACLint (*InitClient)
	( ACLContext, ACLSetup );

/* after rendering finished, this should return screen in default state */
    ACLvoid (*ExitClient)
	( ACLContext );

/* swaps rendering buffers */
    ACLvoid (*SwapBuffer)
	( ACLContext );

/* sets current rendering buffer */
    ACLbool (*DrawBuffer)
	( ACLContext, ACLglenum buffer );

    /*+ TODO: complete list of modules */
} acl_modules;



/*+*************************************************************
 ***************************************************************
 * DESCRIPTION:
 * Different data used for setup of graphic hardware and setting
 * of driver context
 *
 * OPTIONAL: NO
 ***************************************************************
 ***************************************************************/
typedef struct acl_setup {
/* visible screen area */
    ACLuint VisibleWidth;
    ACLuint VisibleHeight;

/* virtual screen area */
    ACLuint VirtualWidth;
    ACLuint VirtualHeight;

/* color depth */
    ACLubyte ScreenBpp;
} acl_setup;



/*+*************************************************************
 ***************************************************************
 * DESCRIPTION:
 * Definition of the accelerator context. This is the DriverCtx
 * if used with Mesa.
 *
 ***************************************************************
 ***************************************************************/
typedef struct acl_context {

/***********************************************************
 * Common hw information about current accelerated context *
 ***********************************************************/
/* The card/driver currently used */
    ACLint drvidx;

/* The modules bound to the driver context */
    ACLModules	mod;

/**************************
 * Rendering informations *
 **************************/
/* ACL_RENDER_TWOSIDE, ACL_RENDER_SMOOTH, ACL_RENDER_TEX */
    ACLuint Flags;

/****************************************************************
 * Other informations needed by core renderer / Mesa compatible *
 ****************************************************************/
/* the core context */
    ACLHyperContext hctx;

/* describes the color buffer */
    ACLVisual vis;

/* the ancillary buffers */
    ACLFramebuffer buff;

/* context of texture manager */
    aclTexmanCtx tmCtx;
    
/* other data */
    ACLuint clearColor, currentColor;

    ACLint width, height;

    ACLbool haveDoubleBuffer;

/* the current TexEnv() settings are supported ? */
    ACLbool isTexEnvHWSupported;
} acl_context;



/*+*************************************************************
 ***************************************************************
 *                 INITIALIZATION FUNCTIONS
 ***************************************************************
 ***************************************************************/
/*+
 * DESCRIPTION:
 * ============
 * The implementation of this function should initialize the acceleration
 * hardware and drivers. It should be called as first function before 
 * anything else is done with ACL resources. On success it should return
 * zero.
 *
 */
ACLint		AclEnter
		    ( ACLvoid );

/*+
 * DESCRIPTION:
 * ============
 * After rendering library initialization, with this query function we
 * can choose hardware resources the rendering client would use. The
 * driver context descriptor is returned, which is later used to acces
 * the particular hardware. The hardware of the same type ( 3 Permedia2
 * cards, for example) can be easy accessed through relative queries.
 * 
 *     Relative query (ACL_GET_NEXT and ACL_GET_PREV):
 *     ------------------------------------------------
 *     The parameter 'drvidx' must be specified, and parameter 'hw' is
 *     ignored. This query works cyclic, and at least original 'drvidx'
 *     is returned, if there is only one member in hardware group of
 *     'drvidx' type. At least one absoulte query should be started
 *     before using the relative query.
 *
 *     Absolute query (ACL_GET_FIRST and ACL_GET_LAST):
 *     ------------------------------------------------
 *     The 'drvidx' is ignored and the 'first' or 'last' driver context
 *     which is bound to the specified type of graphic hardware is
 *     returned, or (-1) if there is no such type of graphic hardware.
 *
 */
ACLint		AclQueryCard
		    ( ACLint drvidx, ACLenum hw, ACLenum position );

/*+
 * DESCRIPTION:
 * ============
 * AclCreateContext() function allocates new ACLModules and setups new
 * ACLContext with appropriate pointers. The ACLModules gets bound with
 * the specified driver context. The ACLVisual structure members should
 * be already set for used visual.
 *
 */
ACLContext	AclCreateContext
		    ( ACLint drvidx );

/*+
 * DESCRIPTION:
 * ============
 * Similar like AclEnter, the AclExit implementation should call the
 * function which frees rendering resources, previously used by client.
 *
 */
ACLvoid		AclExit
		    ( ACLContext aclctx );



/*+ TO DO: define all the functions for visual selection, etc. */
#endif

/*+****************************************************************************

                                ACL API END

 ******************************************************************************/
EOF




































###############################################################################
# this small perl script parses out comments and makes include file aclapi.h; #
# every c comment which looks like /*+ at beginning will be parsed out.       #
###############################################################################

$curr = `pwd`;
chomp $curr;

open OUT, ">$curr/aclapi.h";

while ( $text =~ s|^(.*\n)|| ) {
	$line = $1;
	if ($line =~ s|\s*\/\*\+.*?\*\/\s*||g) {
		($line =~ /^\s*$/) && next;
		($line !~ /\n$/) && ($line .= "\n");
	}
	elsif ($line =~ s|\s*\/\*\+.*?||) {
		while ( $text =~ s|^(.*\n)|| ) {
			$line = $1;
			($line =~ s|.*?\*\/\s*||) && last;
		}
		($line =~ /^\s*$/) && next;
	}
	print OUT $line;
}

close OUT;
