/*
 * -*- mode: C; tab-width:8;  -*-
 * 
 * aclDD.c - Mesa device driver functions
 */

/*
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#if defined(ACL)



#include "aclDrv.h"

TINIT(30)



/**********************************************************************/
/*****                 Miscellaneous functions                    *****/
/**********************************************************************/

/*
 * Return buffer size information 
 */
static void
aclDDBufferSize(GLcontext * ctx, GLuint * width, GLuint * height)
{
    ACLContext aclMesa = ( ACLContext ) ctx->DriverCtx;

TLOG("aclDDBufferSize(...)\n");

    *width = aclMesa->width;
    *height = aclMesa->height;
}



/*
 * Set current drawing color 
 */
static void
aclDDSetColor(GLcontext * ctx, GLubyte red, GLubyte green,
	     GLubyte blue, GLubyte alpha)
{
    ACLContext aclMesa = ( ACLContext ) ctx->DriverCtx;
    ACLColorMod colormod = &(aclMesa->mod->Color);
TOFF
LOG("aclDDSetColor(%d,%d,%d,%d)\n", red, green, blue, alpha);
TON
    colormod->Color(aclMesa, red, green, blue, alpha);
}



/*
 * Implements glClearColor() 
 */
static void
aclDDClearColor(GLcontext * ctx, GLubyte red, GLubyte green,
	       GLubyte blue, GLubyte alpha)
{
    ACLContext aclMesa = ( ACLContext ) ctx->DriverCtx;
    ACLAccelMod accelmod = aclMesa->mod->Accel;
TOFF
LOG("aclDDClearColor(%d,%d,%d,%d)\n", red, green, blue, alpha);
TON
    accelmod->ClearColor(aclMesa, red, green, blue, alpha);
}



static GLbitfield
aclDDClear(GLcontext *ctx, GLbitfield mask, GLboolean all,
		GLint x, GLint y, GLint width, GLint height )
{
    ACLContext aclMesa = ( ACLContext ) ctx->DriverCtx;
    ACLAccelMod accelmod = aclMesa->mod->Accel;
    ACLDepthMod depthmod = aclMesa->mod->Depth;

TLOG("aclDDClear\n");

    depthmod->ClearDepth(aclMesa, ctx->Depth.Clear);
    return accelmod->Clear(ctx, mask, all, x, y, width, height);
}


/*
 * Set the buffer used in double buffering 
 */
static GLboolean
aclDDSetBuffer(GLcontext * ctx, GLenum mode)
{
    ACLContext aclMesa = ( ACLContext ) ctx->DriverCtx;
    ACLModules modules = aclMesa->mod;

TLOG("aclDDSetBuffer\n");

    if( (! aclMesa->haveDoubleBuffer) && mode != GL_FRONT)
	return GL_FALSE;
    return ( modules->DrawBuffer(aclMesa, mode) );
}



static const char *
aclDDRendererString(void)
{
    return ("ACL API Device Driver " 
	    _ACL_API_VERSION_STRING_
	    "\n***********   Written by " _ACL_API_AUTHORS_STRING_ "\n");
}



static void
aclMesaDDUpdateTexPointers(GLcontext * ctx, ACLTexMod texmod)
{
    ACLContext aclMesa = ( ACLContext ) ctx->DriverCtx;

    texmod->UpdateFuncPointers(aclMesa);

    aclSetupDDTexPointers(ctx); 
}



static void
aclMesaDDUpdateAccelPointers(GLcontext * ctx, ACLAccelMod accelmod)
{
    ACLContext aclMesa = ( ACLContext ) ctx->DriverCtx;
    ACLuint flags = 0;

    if (!accelmod)
	return;

    if(ctx->Line.SmoothFlag) 
	flags |= ACL_RENDER_SMOOTH;

    if (ctx->Light.ShadeModel==GL_SMOOTH)
	flags |= ACL_RENDER_SMOOTH;

    if(ctx->Light.Model.TwoSide) 
	flags |= ACL_RENDER_TWOSIDE;

    if (ctx->Texture.Enabled) 
	flags |= ACL_RENDER_TEX;

    if (ctx->Color.ColorLogicOpEnabled) {
	accelmod->SetLogOp(aclMesa, GL_TRUE, ctx->Color.LogicOp);
    }
    
    accelmod->SetAlpha(aclMesa, ctx->Color.AlphaEnabled, ctx->Color.AlphaFunc, 
		       ctx->Color.AlphaRef);
    accelmod->SetBlend(aclMesa, ctx->Color.BlendEnabled, ctx->Color.BlendSrc,
		       ctx->Color.BlendDst);
    accelmod->SetFog(aclMesa, ctx->Fog.Enabled,
		     ctx->FogMode, ctx->Fog.Density,
		     ctx->Fog.Color[0], 
		     ctx->Fog.Color[1], 
		     ctx->Fog.Color[2],
		     ctx->Fog.Color[3]);

    aclMesa->Flags = flags;
    accelmod->UpdateFuncPointers(aclMesa);
}

void aclDDDither(GLcontext *ctx, GLboolean enable)
{
    ((ACLContext) ctx->DriverCtx)->mod->Accel->
	SetDither(( ACLContext ) ctx->DriverCtx, enable, 0);
}

static void
aclMesaSetupAclUnits(GLcontext * ctx, ACLDepthMod depthmod )
{
    ACLContext aclMesa = ( ACLContext ) ctx->DriverCtx;


    if (!depthmod) {
	ERR("DepthMod == NULL!\n");
	return;
    }

TLOG("aclMesaSetupAclUnits (start)\n");

    /*
     * setup depth test:
     */
    if(ctx->Depth.Test) {
	depthmod->Func(aclMesa, ctx->Depth.Func);

	if(ctx->Depth.Mask)
	    depthmod->Mask(aclMesa, GL_TRUE);
	else
	    depthmod->Mask(aclMesa, GL_FALSE);

	depthmod->Enabled(aclMesa, GL_TRUE);
    }
    else {
	depthmod->Func(aclMesa,  GL_ALWAYS);
	depthmod->Mask(aclMesa, GL_FALSE);
	/* depthmod->Mask(aclMesa, GL_TRUE); */
	depthmod->Enabled(aclMesa, GL_TRUE);
    }

}

static GLint aclDDGetParameteri(const GLcontext *ctx, GLint param)
{
  switch(param) {
  case DD_MAX_TEXTURE_SIZE:
    return 2048;
  case DD_MAX_TEXTURES:
      return 2048;
  case DD_HAVE_HARDWARE_FOG:
    return 1;
  case DD_MAX_TEXTURE_COORD_SETS:
      return 1;
  }
  return 0;
}

void aclDDSetNearFar(GLcontext *ctx, GLfloat n, GLfloat f)
{

}


char aclDDIsSupported(GLcontext *ctx)
{
    /* Stripple untersttzt die Hardware, das 
     mu noch implementiert werden */
    if (ctx->Polygon.StippleFlag) {
	MSG("Polygon have Stripples, not supported yet!\n");
	return GL_FALSE;
    }

    if (ctx->Texture.Enabled & (TEXTURE0_3D | TEXTURE1_3D)) {
	MSG("Texture is 3D not supported by Permedia2!\n");	
	return GL_FALSE;
    }

    if ((ctx->Point.Size!=1.0f) ||
       (!(ctx->Point.Params[0]==1.0 && 
	  ctx->Point.Params[1]==0.0 && 
	  ctx->Point.Params[2]==0.0))
	) {
	MSG("Pointparams != 0.0 !\n");	
	return GL_FALSE;
    }

    if (ctx->Texture.Enabled & (TEXTURE0_3D | TEXTURE1_3D)) {
	MSG("Texture3D not supported by Hardware!\n");	
      return GL_FALSE;
    }

    if (ctx->Texture.Enabled & (TEXTURE1_1D | TEXTURE1_2D | TEXTURE1_3D)) {
	MSG("Multitexturing not supported by Hardware!\n");	
	return GL_FALSE;
    }

    /* is supported */
    return GL_TRUE;
}

char aclDDIsPixelSupported(GLcontext *ctx)
{
    if((ctx->Point.Size!=1.0f) ||
     (!(ctx->Point.Params[0]==1.0 && 
	ctx->Point.Params[1]==0.0 && 
	ctx->Point.Params[2]==0.0))
     )
    return GL_FALSE;
}

void
aclSetupDDPointers(GLcontext * ctx)
{
    ACLContext aclMesa = ( ACLContext ) ctx->DriverCtx;
    ACLModules mod = aclMesa->mod;

    ACLColorMod colormod = &(mod->Color);
    ACLAccelMod accelmod = mod->Accel;
    ACLDepthMod depthmod = mod->Depth;
    ACLTexMod texmod = mod->Tex;

    TLOG("aclSetupDDPointers()\n");

    aclMesaSetupAclUnits( ctx, depthmod );

    ctx->Driver.UpdateState = aclSetupDDPointers;

    ctx->Driver.RendererString = aclDDRendererString;
    ctx->Driver.ExtensionString = NULL;

    ctx->Driver.Dither = aclDDDither;
    ctx->Driver.NearFar = NULL;
    ctx->Driver.GetParameteri = aclDDGetParameteri; 

    ctx->Driver.ClearIndex = NULL;
    ctx->Driver.ClearColor = aclDDClearColor;
    ctx->Driver.Clear = aclDDClear;

    ctx->Driver.Index = NULL;
    ctx->Driver.Color = aclDDSetColor;

    ctx->Driver.SetBuffer = aclDDSetBuffer;
    ctx->Driver.GetBufferSize = aclDDBufferSize;

    ctx->Driver.Bitmap = NULL;
    ctx->Driver.DrawPixels = NULL;

    ctx->Driver.Finish = NULL;
    ctx->Driver.Flush = NULL;

    ctx->Driver.RectFunc = NULL;
    ctx->Driver.RenderVB = NULL; 
    ctx->Driver.RasterSetup = NULL;

    aclMesaDDUpdateAccelPointers( ctx, accelmod );

    if (aclDDIsSupported(ctx)) {

	/* if (aclDDIsPixelSupported(ctx)) */
	ctx->Driver.PointsFunc = accelmod->Points;
	/* else  ctx->Driver.PointsFunc = NULL; */
	ctx->Driver.LineFunc = accelmod->Line;
	ctx->Driver.TriangleFunc = accelmod->Triangle;
	ctx->Driver.QuadFunc = accelmod->Quad;

	aclMesaDDUpdateTexPointers( ctx, texmod );

    } else {
	ctx->Driver.TexEnv = NULL;
	ctx->Driver.TexImage = NULL;
	ctx->Driver.TexParameter = NULL;
	ctx->Driver.BindTexture = NULL;
	ctx->Driver.DeleteTexture = NULL;
	ctx->Driver.TexSubImage = NULL;
	ctx->Driver.UpdateTexturePalette = NULL;
	ctx->Driver.UseGlobalTexturePalette = NULL;

	ctx->Driver.PointsFunc=NULL;
	ctx->Driver.LineFunc=NULL;
	ctx->Driver.TriangleFunc=NULL;
	ctx->Driver.QuadFunc=NULL;
	ctx->Driver.RenderVB=NULL;
	ctx->Driver.RasterSetup=NULL;
    }

    
    ctx->Driver.DepthTestSpan=colormod->WriteRGBADSpanGeneric;
    ctx->Driver.DepthTestPixels=colormod->WriteRGBADPixelsGeneric;
    /*
    ctx->Driver.ReadDepthSpanFloat=colormod->ReadDepthSpanFloat;
    ctx->Driver.ReadDepthSpanInt=colormod->ReadDepthSpanInt;
    */

    ctx->Driver.WriteRGBASpan = colormod->WriteRGBASpan;
    ctx->Driver.WriteRGBSpan = colormod->WriteRGBSpan;
    ctx->Driver.WriteMonoRGBASpan = colormod->WriteMonoRGBASpan;
    ctx->Driver.WriteRGBAPixels = colormod->WriteRGBAPixels;
    ctx->Driver.WriteMonoRGBAPixels = colormod->WriteMonoRGBAPixels;

    ctx->Driver.WriteCI8Span = NULL;
    ctx->Driver.WriteCI32Span = NULL;
    ctx->Driver.WriteMonoCISpan = NULL;
    ctx->Driver.WriteCI32Pixels = NULL;
    ctx->Driver.WriteMonoCIPixels = NULL;

    ctx->Driver.ReadRGBASpan = colormod->ReadRGBASpan;
    ctx->Driver.ReadRGBAPixels = colormod->ReadRGBAPixels;

    ctx->Driver.ReadCI32Span = NULL;
    ctx->Driver.ReadCI32Pixels = NULL;
}

#else

/*
 * Need this to provide at least one external definition.
 */

int
gl_acl_dummy_function_dd(void)
{
    return 0;
}

#endif /* ACL */
