
/*****************************************************************************
 *                                  G P M o d                                *
 *                    Generic kernel module for PCI hardware                 *
 *                           (C) SuSE GmbH 1997, 1998                        *
 *****************************************************************************/
/* Author: simon pogarcic, sim@suse.de */
/*
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "project_gpm.def"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <sys/mman.h>
#include <linux/pci.h>

#if 1
#define LOG_DEBUG
#endif

#define LOG_DBGINFO "<GPclient>"
#include "log_debug.h"

#include "client_gpm.h"

#define GPM_CLIENT_OBJECT
#include "scanpci_gpm.h"
#undef GPM_CLIENT_OBJECT



static char *   devpci = "/dev/" PROJECT_NAME "pci";
static char *   devmem = "/dev/" PROJECT_NAME "mem";
static Tint	fh_devpci;
static Tint	fh_devmem;


GPMCard_t *     GpcliSupportedCards = SupportedCards;
GPMCardInfo_t	GpcliCardsInfo[MAX_CARDS_ALOWED];
Tuint		GpcliCardsFound = 0;
pid_t		GpcliMyPid;



void *
gpcli_mmap_memshare(Tulong size, Tuint flags, Tint *dbnr)
{
    caddr_t vbase = 0;
    GPMMmapMem_t mmpar;

LOG("\n");
LOG("_mmap_memshare() ######\n");

    mmpar.action = MMAP_MEM_SHARE;
    mmpar.flags = flags;
    mmpar.buffer_nr = *dbnr;

    vbase = mmap((caddr_t) 0, size,
			PROT_READ | PROT_WRITE,
			MAP_SHARED,
			fh_devmem,
			(off_t) &mmpar);

    if ( !vbase || (Tlong) vbase == -1) {
        ERR("mmap_memshare: mmap() < %s >\n", strerror(errno));
	return NULL;
    }

    *dbnr = mmpar.buffer_nr;

LOG("_mmap_memshare(): mmaped: 0x%lx\n", (Tulong) vbase);
LOG("_mmap_memshare(): buff: %d, size: 0x%lx\n", mmpar.buffer_nr, size);

    return ((void *) vbase);
}



void *
gpcli_mmap_memalloc(Tulong size, Tuint flags, Tint *dbnr, Tuint *acnt)
{
    caddr_t vbase = 0;
    GPMMmapMem_t mmpar;

LOG("\n");
LOG("_mmap_memalloc() ######\n");

    mmpar.action = MMAP_MEM_ALLOC;
    mmpar.flags = flags;
    mmpar.buffer_nr = 0;

    vbase = mmap((caddr_t) 0, size,
			PROT_READ | PROT_WRITE,
			MAP_SHARED,
			fh_devmem,
			(off_t) &mmpar);

    if ( !vbase || (Tlong) vbase == -1) {
        ERR("mmap_memalloc: mmap() < %s >\n", strerror(errno));
	return NULL;
    }

    *dbnr = mmpar.buffer_nr;
    *acnt = mmpar.ret_areas_cnt;

LOG("_mmap_memalloc(): mmaped: 0x%lx\n", (Tulong) vbase);
LOG("_mmap_memalloc(): buff: %d, size: 0x%lx\n", mmpar.buffer_nr, size);

    return ((void *) vbase);
}



static void *
mmap_pciaddr( Tulong addr, Tulong size, Tulong aoffset )
{
    caddr_t vbase = 0;

LOG("\n");
LOG("mmap_pciaddr(): ######\n")

    size += aoffset;
    vbase = mmap((caddr_t) 0, size,
			PROT_READ | PROT_WRITE,
			MAP_SHARED,
			fh_devpci,
			addr);

    if( !vbase || (Tlong) vbase == -1) {
        ERR("mmap_pcidev: mmap() < %s >\n", strerror(errno));
	return NULL;
    }

LOG("mmap_pciaddr(): IO: 0x%lx to 0x%lx (+ 0x%lx), S: 0x%lx\n",
	addr-aoffset, (Tulong) vbase, aoffset, size);

    return (((void *) vbase) + aoffset);
}



void *
gpcli_mmap_pcidev(Tubyte act, Tubyte cardnr, Tubyte devnr)
{
    GPMCardInfo_t * mcard;
    GPMCardDev_t * cd;
    GPMBase_t *bi;

LOG("\n");
LOG("_mmap_pcidev(): ###### Card %d, Dev %d, BASE%d\n", cardnr, devnr, act);

    if(act > BASE_5) {
	ERR("Bad mmap action (%d)\n", act);
	return NULL;
    }

    if(cardnr >= GpcliCardsFound) {
	ERR("Bad card index (%d)\n", cardnr);
	return NULL;
    }

    mcard = GpcliCardsInfo + cardnr;

    if(devnr > mcard->maxdev) {
	ERR("Bad device index (%d)\n", devnr);
	return NULL;
    }

    cd = &(mcard->cd[devnr]);

    if( !(cd->basefound & BASE(act)) ) {
	ERR("PCI Base%d not found on this device\n", act);
	return NULL;
    }

    bi = &(cd->PCIBase[act]);

    if( bi->flags & PCI_BASE_ADDRESS_SPACE_IO ) {
	ERR("PCI Base%d (I/O region) cannot be mmapped\n", act);
	return NULL;
    }

    return mmap_pciaddr(bi->base, bi->size, bi->aoffset);
}



GPMIoctlArea_t *
gpcli_get_areainfo( Tint dbnr, Tuint acnt )
{
    GPMIoctlArea_t *areas;
    GPMIoctlGetArea_t apar;

LOG("\n");
LOG("_get_areainfo(): ######\n");

    if(!(areas = malloc(sizeof(GPMIoctlArea_t) * acnt))) {
	ERR("get_areas_info: malloc()\n");
	return NULL;
    }

    apar.dbnr = dbnr;
    apar.areas = areas;

    if (ioctl(fh_devpci, IOCTL_PCI_GETAREAS, &apar) == -1) {
	ERR("< %s >\n", strerror(errno) );
	return NULL;
    }

    return areas;
}



static Tint 
load_devices_data(void)
{
    GPMCardInfo_t *mcard;
    GPMCard_t *scard;
    Tuint type, cardnr;
    Tint cnt, dcnt;

LOG("\n");
LOG("load_devices_data(): ######\n");

    /*
     * mark all records as uninitialised 
     */
    for (cardnr = 0; cardnr < MAX_CARDS_ALOWED; cardnr++)
	GpcliCardsInfo[cardnr].idx = 0xffffffff;

    /*
     * now load data from kernel 
     */
    if (ioctl(fh_devpci, IOCTL_PCI_GETCARDDATA, GpcliCardsInfo) == -1) {
	ERR("< %s >\n", strerror(errno) );
	return 1;
    }

    cardnr = 0;
    while (cardnr < MAX_CARDS_ALOWED)
    {
	mcard = GpcliCardsInfo + cardnr;

	if(mcard->idx == 0xffffffff || mcard->idx >= NUMBER_OF_SUPPORTED_CARDS)
	    break;

	scard = &SupportedCards[mcard->idx];
	type = scard->chip_type[0];

LOG("load_devices_data(): * CARD #%d: %s\n",cardnr, scard->name);

	for(dcnt = 0; dcnt < mcard->maxdev; dcnt++) {
	    GPMCardDev_t *cd = &(mcard->cd[dcnt]);
	    for(cnt=0; cnt<6; cnt++) {
		GPMBase_t *bi = &(cd->PCIBase[cnt]);
		const char *atype = "MEM";
		if( !(cd->basefound & BASE(cnt)) )
			continue;
		if( bi->flags & PCI_BASE_ADDRESS_SPACE_IO )
			atype = "I/O";
LOG("load_devices_data(): Dev %d, BASE%d 0x%lx (%s)\n",
	dcnt, cnt, bi->base, atype);
	    }
	}
	cardnr++;
    }
    GpcliCardsFound = cardnr;

    return 0;
}



Tint
gpcli_install_isr(Tubyte cardnr)
{
LOG("\n");
LOG("_install_isr(): ###### (Card %d)\n", cardnr);

    if(cardnr >= GpcliCardsFound) {
	ERR("Bad card index (%d)\n", cardnr);
	return 1;
    }

    if (ioctl(fh_devpci, IOCTL_PCI_ISRINSTALL, cardnr) == -1) {
	ERR("< %s >\n", strerror(errno) );
	return 1;
    }
    return 0;
}



Tint
gpcli_uninstall_isr(Tubyte cardnr)
{
LOG("\n");
LOG("_uninstall_isr(): ###### (Card %d)\n", cardnr);

    if(cardnr >= GpcliCardsFound) {
	ERR("Bad card index (%d)\n", cardnr);
	return 1;
    }

    if (ioctl(fh_devpci, IOCTL_PCI_ISRUNINSTALL, cardnr) == -1) {
	ERR("< %s >\n", strerror(errno) );
	return 1;
    }
    return 0;
}



Tint
gpcli_config_isr(Tubyte cardnr, Tubyte devnr, Tuint irqtype, Tulong mask)
{
    GPMIsrCfg_t par;

LOG("\n");
LOG("_config_isr(): ###### (Card %d, Dev %d)\n", cardnr, devnr);
LOG("_config_isr(): Type: 0x%x  MASK: 0x%lx\n", irqtype, mask);

    if(cardnr >= GpcliCardsFound) {
	ERR("Bad card index (%d)\n", cardnr);
	return 1;
    }

    if(devnr > GpcliCardsInfo[cardnr].maxdev) {
	ERR("Bad device index (%d)\n", devnr);
	return 1;
    }

    par.cardnr = cardnr;
    par.devnr = devnr;
    par.irqtype = irqtype;
    par.mask = mask;

    if (ioctl(fh_devpci, IOCTL_PCI_ISRCONFIG, &par) == -1) {
	ERR("< %s >\n", strerror(errno) );
	return 1;
    }
    return 0;
}



Tint
gpcli_find_cardnr(Tuint type, Tubyte *cardnrptr)
{
    Tubyte cardnr;

LOG("\n");
LOG("_find_cardnr(): ###### (Type %d, Start card %d)\n", type, *cardnrptr);

    for(cardnr = *cardnrptr; cardnr < GpcliCardsFound; cardnr++)
	if(SupportedCards[GpcliCardsInfo[cardnr].idx].chip_type[0] == type) {
	    *cardnrptr = cardnr;
	    return 0;
	}

    ERR("find_cardnr: Card not found (type %d)\n", type);
    return 1;
}



Tint
gpcli_read_pcibase(GPMIoctlRw_t *data)
{
LOG("_read_pcibase(): ######\n");
    if (ioctl(fh_devpci, IOCTL_PCI_READBASE, data) == -1) {
	ERR("< %s >\n", strerror(errno) );
	return 1;
    }
    return 0;
}



Tint
gpcli_write_pcibase(GPMIoctlRw_t *data)
{
LOG("_write_pcibase(): ######\n");
    if (ioctl(fh_devpci, IOCTL_PCI_WRITEBASE, data) == -1) {
	ERR("< %s >\n", strerror(errno) );
	return 1;
    }
    return 0;
}



Tint
gpcli_get_pciconfig(GPMPciParam_t *par)
{
LOG("_get_pciconfig(): ######\n");
    if (ioctl(fh_devpci, IOCTL_PCI_GETCONFIG, par) == -1) {
	ERR("< %s >\n", strerror(errno) );
	return 1;
    }
    return 0;
}



Tint
gpcli_set_pciconfig(GPMPciParam_t *par)
{
LOG("_set_pciconfig(): ######\n");
    if (ioctl(fh_devpci, IOCTL_PCI_SETCONFIG, par) == -1) {
	ERR("< %s >\n", strerror(errno) );
	return 1;
    }
    return 0;
}



Tint
gpcli_enter_app( void )
{
    GpcliMyPid = getpid();

    MSG("\n");
    MSG("CLIENT pid: %d\n", GpcliMyPid );
    MSG("Opening %s... (access to PCI regions)\n", devpci);

    if((fh_devpci = open(devpci, O_RDWR)) == -1) {
	ERR("< %s >\n", strerror(errno) );
	return 1;
    }

    MSG("Opening %s... (access to memory management)\n", devmem);

    if((fh_devmem = open(devmem, O_RDWR)) == -1) {
	ERR("< %s >\n", strerror(errno) );
	return 1;
    }

    return load_devices_data();
}



void 
gpcli_exit_app( Tint ret, Tbool exitapp )
{
    MSG("\n");
    MSG("CLIENT pid: %d\n", GpcliMyPid );
    MSG("Closing %s...\n", devpci);

    if(close(fh_devpci) == -1) {
	ERR("< %s >\n", strerror(errno) );
    }
    
    MSG("Closing %s...\n", devmem);

    if(close(fh_devmem) == -1) {
	ERR("< %s >\n", strerror(errno) );
    }

    MSG("\n");

    if(exitapp) exit(ret);
}
