//  This file is part of ff3d - http://www.freefem.org/ff3d
//  Copyright (C) 2001, 2002, 2003 Stphane Del Pino

//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2, or (at your option)
//  any later version.

//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.

//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software Foundation,
//  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

//  $Id: FunctionExpression.cpp,v 1.28 2006/04/08 10:43:20 delpinux Exp $

#include <FEMFunction.hpp>
#include <FEMFunctionBuilder.hpp>

#include <FunctionExpression.hpp>
#include <Vector3Expression.hpp>
#include <BooleanExpression.hpp>

#include <MeshExpression.hpp>
#include <StringExpression.hpp>

#include <Variable.hpp>

#include <Mesh.hpp>
#include <Structured3DMesh.hpp>
#include <MeshOfHexahedra.hpp>
#include <MeshOfTetrahedra.hpp>

#include <Structured3DVector.hpp>

#include <Scene.hpp>

#include <Information.hpp>

#include <fstream>

#include <FiniteElementTraits.hpp>
#include <ConformTransformation.hpp>

#include <DomainExpression.hpp>
#include <Convection.hpp>

const ReferenceCounting<FunctionVariable>
FunctionExpressionVariable::variable() const
{
  return __functionVariable;
}

ReferenceCounting<FunctionVariable>
FunctionExpressionVariable::variable()
{
  return __functionVariable;
}

ReferenceCounting<FunctionExpression>
FunctionExpressionVariable::value()
{
  return __functionExpression;
}

real_t FunctionExpressionVariable::value(const real_t& x,
					 const real_t& y,
					 const real_t& z) const
{
  return (*__functionExpression).value(x,y,z);
}

void FunctionExpressionVariable::execute()
{
  __functionExpression = (*__functionVariable).expression();
  (*__functionExpression).execute();
}

FunctionExpressionVariable::
FunctionExpressionVariable(ReferenceCounting<FunctionVariable> functionVariable,
			   const DiscretizationType::Type& discretizationType)
  : FunctionExpression(FunctionExpression::variable),
    __functionVariable(functionVariable),
    __functionExpression(0),
    __discretizationType(discretizationType)
{
  ;
}

FunctionExpressionVariable::
FunctionExpressionVariable(const FunctionExpressionVariable& f)
  : FunctionExpression(f),
    __functionVariable(f.__functionVariable),
    __functionExpression(f.__functionExpression),
    __discretizationType(f.__discretizationType)
{
  ;
}

FunctionExpressionVariable::~FunctionExpressionVariable()
{
  ;
}

/*!
  FEM FunctionExpression.
*/


real_t FunctionExpressionFEM::value(const real_t& x,
				    const real_t& y,
				    const real_t& z) const
{
  return (*__femFunction)(x,y,z);
}


real_t FunctionExpressionFEM::dx(const real_t& x,
				 const real_t& y,
				 const real_t& z,
				 const FunctionExpressionDx::Direction& d) const
{
  const TinyVector<3, real_t> X(x,y,z);
  switch(d) {
  case FunctionExpressionDx::x: {
    return (*__femFunction).dx(X);
  }
  case FunctionExpressionDx::y: {
    return (*__femFunction).dy(X);
  }
  case FunctionExpressionDx::z: {
    return (*__femFunction).dz(X);
  }
  default: {
    throw ErrorHandler(__FILE__,__LINE__,
		       "unknown direction",
		       ErrorHandler::unexpected);

    return 0;
  }
  }
}


const FEMFunctionBase&
FunctionExpressionFEM::femFunction() const
{
  return *__femFunction;
}

FEMFunctionBase&
FunctionExpressionFEM::femFunction()
{
  return *__femFunction;
}

ReferenceCounting<MeshExpression>
FunctionExpressionFEM::mesh() const
{
  return __mesh;
}


void FunctionExpressionFEM::execute()
{
  if (__function != 0) {
    (*__mesh).execute();
    Information::instance().setMesh(__mesh);

    FEMFunctionBuilder builder;
    __femFunction = builder.build(DiscretizationType(__discretizationType),
				  *(*__mesh).mesh());

    (*__function).execute();
    ReferenceCounting<UserFunctionLanguage>
      f = new UserFunctionLanguage(__function);

    (*__femFunction) = *f;

    // function has now been evaluated.
    __function = 0;
    Information::instance().unsetMesh();
  }
}

FunctionExpressionFEM::
FunctionExpressionFEM(ReferenceCounting<MeshExpression> mesh,
		      ReferenceCounting<FunctionExpression> function,
		      const DiscretizationType::Type& discretizationType)
  : FunctionExpression(FunctionExpression::fem),
    __discretizationType(discretizationType),
    __mesh(mesh),
    __function(function)
{
  ;
}

FunctionExpressionFEM::
FunctionExpressionFEM(ReferenceCounting<MeshExpression> mesh,
		      ReferenceCounting<FEMFunctionBase> femFunction)
  : FunctionExpression(FunctionExpression::fem),
    __discretizationType((*femFunction).type()),
    __mesh(mesh),
    __femFunction(femFunction),
    __function(0)
{
  ;
}

FunctionExpressionFEM::
FunctionExpressionFEM(const FunctionExpressionFEM& femFunction)
  : FunctionExpression(femFunction),
    __discretizationType(femFunction.__discretizationType),
    __mesh(femFunction.__mesh),
    __function(femFunction.__function)
{
  ;
}

FunctionExpressionFEM::~FunctionExpressionFEM()
{
  ;
}


real_t FunctionExpressionNormalComponent::value(const real_t& x,
						const real_t& y,
						const real_t& z) const
{
  return (*__normal)(x,y,z);
}

void FunctionExpressionNormalComponent::execute()
{
  __normal = new Normal(__normalType);
}

ReferenceCounting<FunctionExpression> FunctionExpressionDx::value()
{
  return this;
}

real_t FunctionExpressionDx::value(const real_t& x,
				   const real_t& y,
				   const real_t& z) const
{
#warning BAD TRICK TO AVOID CONST PROBLEM
  FunctionExpression* f = __function;

  return static_cast<const FunctionExpressionFEM&>(*(*f).value()).dx(x,y,z,__direction);
}

void FunctionExpressionDx::execute()
{
  (*__function).execute();
  if ((*(*__function).value()).type() != FunctionExpression::fem) {

    std::stringstream errorMsg;

    errorMsg << "cannot compute derivative of non FEM functions :-(\n";
    errorMsg << "the function " << (*__function) << " is not of that kind"
	     << std::ends;

    throw ErrorHandler(__FILE__,__LINE__,
		       errorMsg.str(),
		       ErrorHandler::normal);
  }
}

FunctionExpressionDx::
FunctionExpressionDx(ReferenceCounting<FunctionExpression> function,
		     FunctionExpressionDx::Direction d)
  : FunctionExpression(FunctionExpression::dx),
    __direction(d),
    __function(function)
{
  ;
}

FunctionExpressionDx::
FunctionExpressionDx(const FunctionExpressionDx& f)
  : FunctionExpression(f),
    __direction(f.__direction),
    __function(f.__function)
{
  ;
}

FunctionExpressionDx::
~FunctionExpressionDx()
{
  ;
}

ReferenceCounting<FunctionExpression>
FunctionExpressionConvection::value()
{
  return this;
}

real_t FunctionExpressionConvection::value(const real_t& x,
					   const real_t& y,
					   const real_t& z) const
{
  if (__isToEvaluate) {
    throw ErrorHandler(__FILE__,__LINE__,
		       "cannot evaluate the function: no mesh found",
		       ErrorHandler::normal);
    return 0;
  } else {
    return (*__convectionOperator)(x,y,z);
  }
}

void FunctionExpressionConvection::execute()
{
  (*__convectedFunction).execute();
  if (Information::instance().usesMesh()) {
    __mesh = Information::instance().getMesh();

    (*__mesh).execute();
    (*__v0Function).execute();
    (*__v1Function).execute();
    (*__v2Function).execute();
    (*__timeStep).execute();

    __timeStepValue = (*__timeStep).realValue();

    __v0UserFunction = new UserFunctionLanguage(__v0Function);
    __v1UserFunction = new UserFunctionLanguage(__v1Function);
    __v2UserFunction = new UserFunctionLanguage(__v2Function);
    __convectedUserFunction = new UserFunctionLanguage(__convectedFunction);


    const Mesh& M = *(*__mesh).mesh();

    switch (M.type()) {
    case Mesh::cartesianHexahedraMesh: {
      __convectionOperator
	= new Convection<Structured3DMesh>(*__convectedUserFunction,
					   *__v0UserFunction,
					   *__v1UserFunction,
					   *__v2UserFunction,
					   __timeStepValue,
					   static_cast<const Structured3DMesh&>(M));
      break;
    }
    case Mesh::tetrahedraMesh: {
      __convectionOperator
	= new Convection<MeshOfTetrahedra>(*__convectedUserFunction,
					   *__v0UserFunction,
					   *__v1UserFunction,
					   *__v2UserFunction,
					   __timeStepValue,
					   static_cast<const MeshOfTetrahedra&>(M));
      break;
    }
    default: {
      throw ErrorHandler(__FILE__,__LINE__,
			 "convection is not implemented for this kind of mesh",
			 ErrorHandler::unexpected);
    }
    }

    __isToEvaluate = false;
  } else {
    __isToEvaluate = true;
  }
}


FunctionExpressionConvection::
FunctionExpressionConvection(ReferenceCounting<FunctionExpression> v0,
			     ReferenceCounting<FunctionExpression> v1,
			     ReferenceCounting<FunctionExpression> v2,
			     ReferenceCounting<RealExpression> dt,
			     ReferenceCounting<FunctionExpression> phi)
  : FunctionExpression(FunctionExpression::convection),
    __isToEvaluate(true),
    __timeStepValue(0),
    __v0Function(v0),
    __v1Function(v1),
    __v2Function(v2),
    __timeStep(dt),
    __convectedFunction(phi),
    __convectionOperator(0)
{
  ;
}


FunctionExpressionConvection::
FunctionExpressionConvection(const FunctionExpressionConvection& f)
  : FunctionExpression(f),
    __isToEvaluate(f.__isToEvaluate),
    __timeStepValue(f.__timeStepValue),
    __v0Function(f.__v0Function),
    __v1Function(f.__v1Function),
    __v2Function(f.__v2Function),
    __timeStep(f.__timeStep),
    __convectedFunction(f.__convectedFunction),
    __mesh(f.__mesh),
    __convectionOperator(f.__convectionOperator)
{
  ;
}

FunctionExpressionConvection::~FunctionExpressionConvection()
{
  ;
}



std::ostream& FunctionExpressionCharacteristic::put(std::ostream& os) const
{
  os << "one(<" << (*__reference).value(0)
     << ',' << (*__reference).value(1)
     << ',' << (*__reference).value(2) << ">)";
  return os;
}

real_t FunctionExpressionCharacteristic::value(const real_t& x,
					       const real_t& y,
					       const real_t& z) const
{
  TinyVector<3> V(x,y,z);
  for (std::list<Object* >::const_iterator i = __objects.begin();
       i != __objects.end(); ++i)
    if ((*(*i)).inside(V))
      return 1;
  return 0;
}

//! \todo optimize this region
void FunctionExpressionCharacteristic::execute()
{
  __scene = Information::instance().getScene();
  Scene& S = *__scene;

  (*__reference).execute();
  __objects.clear();

  TinyVector<3> r;
  for (size_t i=0; i<3; ++i)
    r[i] = (*__reference).value(i);

  for (size_t i =0; i<S.nbObjects() ; ++i) {
    if (S.object(i).reference() == r) {
      __objects.push_back(&S.object(i));
    }
  }

  if (__objects.size() == 0) {
    throw ErrorHandler(__FILE__,__LINE__,
		       "error: no object in the Scene has reference "
		       +stringify(r),
		       ErrorHandler::normal);
  }
}

FunctionExpressionCharacteristic
::FunctionExpressionCharacteristic(ReferenceCounting<Vector3Expression> ref)
  : FunctionExpression(FunctionExpression::characteristic),
    __reference(ref)
{
  ;
}

FunctionExpressionCharacteristic::
FunctionExpressionCharacteristic(const FunctionExpressionCharacteristic& f)
  : FunctionExpression(FunctionExpression::characteristic),
    __reference(f.__reference),
    __scene(f.__scene)
{
  ;
}

FunctionExpressionCharacteristic::
~FunctionExpressionCharacteristic()
{
  ;
}

// MeshReferences

FunctionExpressionMeshReferences::
FunctionExpressionMeshReferences(ReferenceCounting<MeshExpression> mesh,
				 ReferenceCounting<ReferencesSet> ref,
				 FunctionExpression::FunctionType t)
  : FunctionExpression(t),
    __mesh(mesh),
    __referenceSet(ref)
{
  ;
}

FunctionExpressionMeshReferences::
FunctionExpressionMeshReferences(const FunctionExpressionMeshReferences& f)
  : FunctionExpression(f),
    __mesh(f.__mesh),
    __referenceSet(f.__referenceSet)
{
  ;
}

FunctionExpressionMeshReferences::
~FunctionExpressionMeshReferences()
{
  ;
}


std::ostream& FunctionExpressionMeshElementsReferences::
put(std::ostream& os) const
{
  os << "reference(elements," << *__mesh << ',' << *__referenceSet << ')';
  return os;
}

template <typename MeshType> 
real_t FunctionExpressionMeshElementsReferences::
__evaluate(const real_t& x,
	   const real_t& y,
	   const real_t& z) const
{
  const MeshType& M = static_cast<const MeshType&>(*(*__mesh).mesh());
  typename MeshType::const_iterator i = M.find(x,y,z);

  if (i.end()) { // outside of the mesh!
    return 0;
  }

  const size_t ref = (*i).reference();
  std::map<size_t, ReferenceCounting<FunctionExpression> >::const_iterator
    f = this->__functionReferences.find(ref);
  if (f == this->__functionReferences.end()) { // reference was not found in list
    return 0;
  } else {
    return (*(*f).second).value(x,y,z);
  }
}

real_t FunctionExpressionMeshElementsReferences::
value(const real_t& x,
      const real_t& y,
      const real_t& z) const
{
  switch ((*(*__mesh).mesh()).type()) {
  case Mesh::cartesianHexahedraMesh: {
    return __evaluate<Structured3DMesh>(x,y,z);
    break;
  } 
  case Mesh::hexahedraMesh: {
    return __evaluate<MeshOfHexahedra>(x,y,z);
    break;
  } 
  case Mesh::tetrahedraMesh: {
    return __evaluate<MeshOfTetrahedra>(x,y,z);
    break;
  } 
  default: {
    throw ErrorHandler(__FILE__,__LINE__,
		       "references are not implemented for this kind of mesh",
		       ErrorHandler::unexpected);
    return 0;
  }
  }
}

void FunctionExpressionMeshElementsReferences::
execute()
{
  (*__mesh).execute();
  (*__referenceSet).execute();

  for (FunctionExpressionMeshReferences::ReferencesSet::FunctionReferences::
	 iterator i = (*__referenceSet).functionReferences().begin();
       i != (*__referenceSet).functionReferences().end(); ++i) {
    size_t ref = static_cast<size_t>((*(*i).first).realValue());
    __functionReferences[ref] = (*i).second;
  }
}

FunctionExpressionMeshElementsReferences::
FunctionExpressionMeshElementsReferences(ReferenceCounting<MeshExpression> mesh,
					 ReferenceCounting<ReferencesSet> ref)
  : FunctionExpressionMeshReferences(mesh, ref, meshElementsReferences)
{
  ;
}

FunctionExpressionMeshElementsReferences::
FunctionExpressionMeshElementsReferences(const FunctionExpressionMeshElementsReferences& f)
  : FunctionExpressionMeshReferences(f)
{
  ;
}

FunctionExpressionMeshElementsReferences::
~FunctionExpressionMeshElementsReferences()
{
  ;
}


std::ostream&
FunctionExpressionMeshVerticesReferences::
put(std::ostream& os) const
{
  os << "reference(vertices," << *__mesh << ',' << *__referenceSet << ')';
  return os;
}

ReferenceCounting<FunctionExpression>
FunctionExpressionMeshVerticesReferences::value()
{
  return static_cast<FunctionExpression*>(__femFunction);
}

real_t FunctionExpressionMeshVerticesReferences::
value(const real_t& x,
      const real_t& y,
      const real_t& z) const
{
  return (*__femFunction).value(x,y,z);
}

void FunctionExpressionMeshVerticesReferences::execute()
{
  (*__mesh).execute();
  (*__referenceSet).execute();

  for (FunctionExpressionMeshReferences::ReferencesSet::FunctionReferences::
	 iterator i = (*__referenceSet).functionReferences().begin();
       i != (*__referenceSet).functionReferences().end(); ++i) {
    size_t ref = static_cast<size_t>((*(*i).first).realValue());
    __functionReferences[ref] = (*i).second;
  }

  const Mesh& m = *(*__mesh).mesh();

  Vector<real_t> values(m.numberOfVertices());
  for (size_t i=0; i<values.size(); ++i) {
    const Vertex& v = m.vertex(i);
    std::map<size_t, ReferenceCounting<FunctionExpression> >::const_iterator
      f = this->__functionReferences.find(v.reference());
    if (f != this->__functionReferences.end()) {
      values[i] = (*(*f).second).value(v[0], v[1], v[2]);
    } else { // reference was not found in list
      values[i] = 0;
    }
  }

  FEMFunctionBuilder builder;
  DiscretizationType d(DiscretizationType::LagrangianFEM1);
  __femFunction = new FunctionExpressionFEM(__mesh, builder.build(d, m, values));
}

FunctionExpressionMeshVerticesReferences::
FunctionExpressionMeshVerticesReferences(ReferenceCounting<MeshExpression> mesh,
					 ReferenceCounting<ReferencesSet> ref)
  : FunctionExpressionMeshReferences(mesh, ref, meshVerticesReferences),
    __femFunction(0)
{
  ;
}

FunctionExpressionMeshVerticesReferences::
FunctionExpressionMeshVerticesReferences(const FunctionExpressionMeshVerticesReferences& f)
  : FunctionExpressionMeshReferences(f),
    __femFunction(f.__femFunction)
{
  ;
}

FunctionExpressionMeshVerticesReferences::
~FunctionExpressionMeshVerticesReferences()
{
  ;
}

// DOMAIN

std::ostream& FunctionExpressionDomainCharacteristic::put(std::ostream& os) const
{
  os << "one(" << (*__domain) << ')';

  return os;
}

real_t FunctionExpressionDomainCharacteristic::value(const real_t& x,
						     const real_t& y,
						     const real_t& z) const
{
  TinyVector<3> X(x,y,z);

  return (*(*__domain).domain()).inside(X);
}

//! \todo optimize this region
void FunctionExpressionDomainCharacteristic::execute()
{
  (*__domain).execute();
}

FunctionExpressionDomainCharacteristic
::FunctionExpressionDomainCharacteristic(ReferenceCounting<DomainExpression> D)
  : FunctionExpression(FunctionExpression::domainCharacteristic),
    __domain(D)
{
  ;
}

FunctionExpressionDomainCharacteristic::
FunctionExpressionDomainCharacteristic(const FunctionExpressionDomainCharacteristic& f)
  : FunctionExpression(FunctionExpression::domainCharacteristic),
    __domain(f.__domain)
{
  ;
}

FunctionExpressionDomainCharacteristic::
~FunctionExpressionDomainCharacteristic()
{
  ;
}


// DOMAIN END

std::ostream&
FunctionExpressionMeshCharacteristic::
put(std::ostream& os) const
{
  os << "one(" << *__meshExpression <<')';
  return os;
}

real_t
FunctionExpressionMeshCharacteristic::
value(const real_t& x,
      const real_t& y,
      const real_t& z) const
{
  return (*__mesh).inside(x,y,z);
}

void
FunctionExpressionMeshCharacteristic::
execute()
{
  (*__meshExpression).execute();
  if ((*(*__meshExpression).mesh()).family() == Mesh::volume) {
    __mesh = &(*(*__meshExpression).mesh());
  } else {
    throw ErrorHandler(__FILE__,__LINE__,
		       "cannot get characteristic function of none volume meshes",
		       ErrorHandler::normal);
  }
}

FunctionExpressionMeshCharacteristic::
FunctionExpressionMeshCharacteristic(ReferenceCounting<MeshExpression> ref)
  : FunctionExpression(FunctionExpression::meshCharacteristic),
    __meshExpression(ref)
{
  ;
}

FunctionExpressionMeshCharacteristic::
FunctionExpressionMeshCharacteristic(const FunctionExpressionMeshCharacteristic& re)
  : FunctionExpression(re),
    __mesh(re.__mesh),
    __meshExpression(re.__meshExpression)
{
  ;
}


FunctionExpressionMeshCharacteristic::
~FunctionExpressionMeshCharacteristic()
{
  ;
}



std::ostream& FunctionExpressionRead::put(std::ostream& os) const
{
  os << "read(" << (*__fileName).value() << ')';
  return os;
}

ReferenceCounting<FunctionExpression>
FunctionExpressionRead::value()
{
  return static_cast<FunctionExpression*>(__femFunction);
}

real_t FunctionExpressionRead::value(const real_t& x,
				     const real_t& y,
				     const real_t& z) const
{
  return (*__femFunction).value(x,y,z);
}

void FunctionExpressionRead::execute()
{
  (*__mesh).execute();
  (*__fileName).execute();

  const std::string& filename = (*__fileName).value();

  std::ifstream fin(filename.c_str());
  if (not(fin)) {
    throw ErrorHandler(__FILE__,__LINE__,
		       "error: cannot open file '"+filename+"'",
		       ErrorHandler::normal);
  }

  switch ((*__fileDescriptor).format()) {
  case FileDescriptor::raw: {
    const Mesh& M = (*(*__mesh).mesh());

    const size_t numberOfVertices =  M.numberOfVertices();
    
    Vector<real_t> v(numberOfVertices);

    for (size_t i=0; i<numberOfVertices; ++i) {
      fin >> v[i];
    }
    if (fin.bad()) {
      throw ErrorHandler(__FILE__,__LINE__,
			 "Cannot read all datas in file '"+filename+"'",
			 ErrorHandler::normal);
    }
    FEMFunctionBuilder builder;
    DiscretizationType d(DiscretizationType::LagrangianFEM1);
    __femFunction = new FunctionExpressionFEM(__mesh, builder.build(d, M, v));
    break;
  }
  case FileDescriptor::medit: {
    size_t a, b, c, numberOfVertices;
    fin >> a >> b >> numberOfVertices >> c;
    const Mesh& M = (*(*__mesh).mesh());

    if ((a!=3)or(b!=1)or(c!=2)) {
      throw ErrorHandler(__FILE__,__LINE__,
			 "Unknown file descriptor while reading file '"+filename+"'",
			 ErrorHandler::normal);
    }

    if (numberOfVertices != M.numberOfVertices()) {
      throw ErrorHandler(__FILE__,__LINE__,
			 "Number of vertices does not match mesh while reading '"
			 +filename+"'",
			 ErrorHandler::normal);
    }

    Vector<real_t> v(numberOfVertices);
    for (size_t i=0; i<numberOfVertices; ++i) {
      fin >> v[i];
    }

    if (fin.bad()) {
      throw ErrorHandler(__FILE__,__LINE__,
			 "cannot read all datas in file '"+filename+"'",
			 ErrorHandler::normal);
    }
    FEMFunctionBuilder builder;
    DiscretizationType d(DiscretizationType::LagrangianFEM1);
    __femFunction = new FunctionExpressionFEM(__mesh, builder.build(d,M,v));
    break;
  }
  default: {
    throw ErrorHandler(__FILE__,__LINE__,
		       "unknown file type",
		       ErrorHandler::normal);
  }
  }
}

FunctionExpressionRead::
FunctionExpressionRead(ReferenceCounting<FileDescriptor> descriptor,
		       ReferenceCounting<StringExpression> fileName,
		       ReferenceCounting<MeshExpression> mesh)
  : FunctionExpression(FunctionExpression::read),
    __fileDescriptor(descriptor),
    __fileName(fileName),
    __mesh(mesh)
{
  ;
}

FunctionExpressionRead::
FunctionExpressionRead(const FunctionExpressionRead& f)
  : FunctionExpression(f),
    __fileDescriptor(f.__fileDescriptor),
    __femFunction(f.__femFunction),
    __fileName(f.__fileName),
    __mesh(f.__mesh)
{
  ;
}

FunctionExpressionRead::~FunctionExpressionRead()
{
  ;
}

// BOOLEAN FUNCTION

void FunctionExpressionBooleanValue::execute()
{
  (*__booleanExpression).execute();
}

ReferenceCounting<FunctionExpression>
FunctionExpressionBooleanValue::value()
{
  return this;
}

real_t FunctionExpressionBooleanValue::
value(const real_t& x,
      const real_t& y,
      const real_t& z) const
{
  if((*__booleanExpression).boolValue()) {
    return 1;
  } else {
    return 0;
  }
}

bool FunctionExpressionBooleanValue::
boolean(const real_t& x,
	const real_t& y,
	const real_t& z) const
{
  return ((*__booleanExpression).boolValue());
}

FunctionExpressionBooleanValue::
FunctionExpressionBooleanValue(ReferenceCounting<BooleanExpression> boolExp)
  : __booleanExpression(boolExp)
{
  ;
}

FunctionExpressionBooleanValue::
FunctionExpressionBooleanValue(const FunctionExpressionBooleanValue& f)
  : FunctionExpressionBoolean(f)
{
  ;
}

FunctionExpressionBooleanValue::~FunctionExpressionBooleanValue()
{
  ;
}

