//  This file is part of ff3d - http://www.freefem.org/ff3d
//  Copyright (C) 2001, 2002, 2003 Stphane Del Pino

//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2, or (at your option)
//  any later version.

//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.

//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software Foundation,
//  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

//  $Id: Shape.cpp,v 1.2 2004/12/31 16:38:58 delpinux Exp $


#include <Shape.hpp>

#include <Translation.hpp>
#include <Rotation.hpp>
#include <Scale.hpp>
#include <TransformMatrix.hpp>

#include <Scene.hpp>

/*! Prints the type name of the Shape into a std::string.
  \todo Make a virtual function of this.
*/
const std::string Shape::typeName() const
{
  switch (__stype) {
  // Primitives Shapes
  case sphere: 
    return "Sphere";
  case cylinder:
    return "Cylinder";
  case cone:
    return "Cone";
  case cube:
    return "Cube";
  case torus:
    return "Torus";

  // Boolean Operation
  case union_:
    return "Union";
  case difference:
    return "Difference";
  case intersection:
    return "Itersection";
  case not_:
    return "Not";

  default:
    return "Unknown (please modify class Shape)";
  }
}


/*! Adds the transformation \a T to the Shape.
  \warning Should rename the method to AddTransform.
*/
void Shape::setTransform(const Transform& T)
{
  switch (T.type()) {
  case translation: {
    const Translation& T0 = dynamic_cast<const Translation&>(T);
    Translation* t = new Translation(T0);
    __trans.push_back(t);
    break;
  }
  case rotation: {
    const Rotation& T0 = dynamic_cast<const Rotation&>(T);
    Rotation* t = new Rotation(T0);
    __trans.push_back(t);
    break;
  }
  case scale: {
    const Scale& T0 = dynamic_cast<const Scale&>(T);
    Scale* t = new Scale(T0);
    __trans.push_back(t);
    break;
  }
  case matrix: {
    const TransformMatrix& M0 = dynamic_cast<const TransformMatrix&>(T);
    TransformMatrix* t = new TransformMatrix(M0);
    __trans.push_back(t);
    break;
  }
  default:
    throw ErrorHandler(__FILE__,__LINE__,
		       "not implemented",
		       ErrorHandler::unexpected);
  }
}

//! Outputs the Shape \a S to the stream \a os.
std::ostream& operator << (std::ostream& s, const Shape& shape)
{
  return shape.put(s);
}

//! Builds the transformations vector using informations stored in \a trans.
void Shape::parseTransform(const parsetrans& trans)
{
  for (int i=0; i<trans.number ; i++) {
    switch (trans.type[i]) {
    case matrix: {
      TransformMatrix M(trans.mat);
      setTransform(M);
      break;
    }
    case rotation: {
      TinyVector<3> v( trans.vect[0][i],
		       trans.vect[1][i],
		       trans.vect[2][i]);
      Rotation R(v);
      setTransform(R);
      break;
    }
    case translation: {
      TinyVector<3> v( trans.vect[0][i],
		       trans.vect[1][i],
		       trans.vect[2][i]);
      Translation T(v);
      setTransform(T);
      break;
    }
    case scale: {
      TinyVector<3> v( trans.vect[0][i],
		       trans.vect[1][i],
		       trans.vect[2][i]);
      Scale S(v);
      setTransform(S);
      break;
    }
    default: {
      throw ErrorHandler(__FILE__,__LINE__,
			 "unknown transformation",
			 ErrorHandler::unexpected);
    }
    }
  }
}

