//////////////////////////////////////////////////////////////////////////////
// safety.C

#include "dl.h"

//////////////////////////////////////////////////////////////////////////////
/**Check whether a variable t is "saved" by a built-in equality predicate
 * bt. This is the case if it is bound either by a constant (such as 
 * X==goofie or goofie==X) or by an integer (such as X==5 or 5==X).
 */
static bool isSavedByBuiltinEquality(const LITERAL &bt, const TERM &t)
    {
    assert( bt.isBuiltinEquality() && t.isVar() );

    const TERM &t1 = *( (bt).getParams()->begin() );
    const TERM &t2 = *( (bt).getParams()->begin()+1 );

    // Handle the case t = integer\constant.
    if(t1 == t)
        return (t2.getType() == TERM::String) 
                || (t2.getType() == TERM::Integer);

    // Handle the case integer\constant = t.
    if(t2 == t)
        return (t1.getType() == TERM::String) 
                || (t1.getType() == TERM::Integer);

    return false; 
    }

//////////////////////////////////////////////////////////////////////////////
// CONJUNCTION
//////////////////////////////////////////////////////////////////////////////

static bool Saves(
/**Return true if var appears as:
 * - a variable in a positive literal;
 * - a variable of an assignment builtin which is bound
 *   by a constant or integer for instance X=5 or X=goofie;
 * - a guard of an assigning aggregate in the 
 *   range [ conj.begin(), conj.end() ).
 */
    const CONJUNCTION& conj, 
    const TERM&       var )
    {
    bool saved=false;

    for( CONJUNCTION::const_iterator j=conj.begin();
         ! saved  &&  j != conj.end();
         j++ )
        {
        if ( (*j).isSaviour() )
            {
            if((*j).isAggregate())
                {
                assert((*j).getAggregate().isAssignment());
                saved = (*j).getAggregate().getLowerGuardAsVar() == var;
                }
            else
                {
                const TERMS* bparams=(*j).getParams();
                if( bparams )
                    { 
                    for( TERMS::const_iterator j1=bparams->begin();
                         ! saved  &&  j1 != bparams->end();
                         j1++ )
                        {
                        if( var == *j1 )
                            saved=true;
                        }
                    }
                }//else
            }
        else if( (*j).isBuiltinEquality() )
            {
            saved = isSavedByBuiltinEquality(*j,var);
            }
        }

    return saved;
    }

//////////////////////////////////////////////////////////////////////////////
// Check safety of aggregate.
inline static bool safeAggregate(
    const CONJUNCTION &conjunction,
    const LITERAL &agg_lit )
    {
    // Check safety of lower guard.
    if( agg_lit.getAggregate().getLowerGuardAsVar().isVar()
        && !Saves(conjunction,agg_lit.getAggregate().getLowerGuardAsVar()) )
                return false;

    // Check safety of user's upper guard.
    else if( agg_lit.getAggregate().getUpperGuardAsVar().isVar()
        && !Saves(conjunction,agg_lit.getAggregate().getUpperGuardAsVar()) )
            return false;

    // Check safety of aggregate Conjunction.
    for(CONJUNCTION::const_iterator lit =
        agg_lit.getAggregate().getConjunction().begin();
        lit != agg_lit.getAggregate().getConjunction().end();
        lit++ )
        {
        // For all negative literals, builtins,...
        if( ! (*lit).isSaviour() )
            {
            for(TERMS::const_iterator par = lit->getParams()->begin();
                par != lit->getParams()->end();
                par++ )
                {
                // If par is a variable, check whether it appears either in
                // a positive literal of conjunction, or in a positive literal
                // of the aggregate conjunction.
                if( (*par).isVar() && !Saves(conjunction,*par) )
                    if( ! Saves(agg_lit.getAggregate().getConjunction(),
                        *par ) )
                        return false;
                }
            }//if( ! (*lit).isSaviour() )
        }

    return true;
    }

//////////////////////////////////////////////////////////////////////////////
template<>
bool TCONJUNCTION<LITERAL>::isSafe() const
//
    {
    for( const_iterator i=begin();
         i != end();
         i++ )
        {
        // ...if it is a negative literal, a builtin, or an aggregate...
        if (! (*i).isSaviour() )
            {
            if( (*i).isAggregate())
                {
                if(! safeAggregate(*this,*i) )
                    return false;
                }
            else
                {
                assert((*i).isRegularAtom());
                const TERMS *nparams=(*i).getParams();
                if( nparams )
                    {
                    // ...for the the terms it contains:
                    for( TERMS::const_iterator i1=nparams->begin();
                        i1 != nparams->end();
                        i1++ )
                        {
                        if( (*i1).isVar() )
                             if( ! Saves(*this,*i1) )
                                  return false;
                        }
                    }
                }// else
            }
        }

    return true;
    }

//////////////////////////////////////////////////////////////////////////////
// RULE
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
bool TRULE<ATOM, LITERAL>::isSafe() const
//
    {   
    const CONJUNCTION* body=getBody();

    // For all head elements...
    for( ATOMS::const_iterator i=getHead().begin();
         i != getHead().end();
         i++ )
        {
        const TERMS* hparams=(*i).getParams();
        if( hparams )
            {
            // ...and the terms they contain:
            for( TERMS::const_iterator i1=hparams->begin();
                 i1 != hparams->end();
                 i1++ )
                {
                if( (*i1).isVar() )
                    {
                    if (!body || !Saves(*body,*i1))
			return false;
                    }
                }
            }
        }

    return !body || body->isSafe();
    }

//////////////////////////////////////////////////////////////////////////////
// WEIGHTS
//////////////////////////////////////////////////////////////////////////////
bool safeWeights(const WEIGHTS& weights, 
    const TCONJUNCTION<LITERAL>& conj)
    {
    //Check safety of the terms which weights contains.
    if( weights.first.isVar() && ! Saves(conj ,weights.first) )
        return false;
    if( weights.second.isVar() && ! Saves(conj, weights.second) )
        return false;

    return true;
    }

// Local Variables:
// mode: c++
// c-file-style: "dl"
// End:
