//////////////////////////////////////////////////////////////////////////////
// rewrite.C

#define WITH_GLOBAL_VARIABLES

#include "dl.h"
#include <stdio.h>

// FIXME: We should allocate sufficient space dynamically and allow for
//        arbitrarily long strings.
#define PREDNAME_LENGTH 50*IDENT_LENGTH

/////////////////////// SUBSUMPTION ///////////////////////////////////

typedef vector< pair<TERM,TERM> > Substitution;

inline ostream& operator<< (ostream& out,const pair<TERM,TERM> &iTheta)
    {
    out << iTheta.first << "/" << iTheta.second;
    return out;
    }

// Update a substitution theta w.r.t. the matching between the
// variables of the literal L with terms of the literal L1
//
// @params substitution theta
// @params literal L 
// @params literal L1
void updateSubstitution(Substitution &theta,
                        const LITERAL &L,
                        const LITERAL &L1)
    {
    assert( L.getIndex() == L1.getIndex() );
    assert( L.isNegative() == L1.isNegative() );
    assert( L.isPropositional() == L1.isPropositional() );
    if( L.isPropositional() && L1.isPropositional())
        return;
    assert( !L.isPropositional() && !L1.isPropositional() );
    const TERMS *params = L.getParams();
    const TERMS *params1 = L1.getParams();
    assert( params != 0 );
    assert( params1 != 0 );
    TERMS ::const_iterator i1 = params1->begin();
    for( TERMS ::const_iterator i = params->begin();
         i != params->end() && i1 != params1->end();
         i++, i1++)
        {
        if( (*i).isVar() )
            {
            pair<TERM,TERM> ith(*i,*i1);
            if( find(theta.begin(),theta.end(),ith) == theta.end() )
                theta.push_back(ith);
            }
        }
    }


typedef vector<LITERAL> VectorOfLiterals;

// Given a pair of terms ith, for instance <T1,T2>, check if
// substitution {T1/T2} is permissible w.r.t. a substitution theta.
//
// @param ith, a pair of terms 
// @param theta, a substitution 
bool isPermissibleSubstitutions(const pair<TERM,TERM> &ith,Substitution &theta)
    {
    // FIXME: We would need a method isConstant() here.
    if( !ith.first.isVar() ) 
        if( !ith.second.isVar() && (ith.first == ith.second ) )
            // the terms are the same constant  
            return true;
        else
            return false;
    else // ith.first.isVar() 
        {
        if( find(theta.begin(),theta.end(),ith) != theta.end() )
            // the i-th pair is already present in substitution theta
            return true;
        for( Substitution::const_iterator i = theta.begin(); 
             i != theta.end();
             i++ )
            {
            if( ith.first == i->first && ith.second != i->second )
                // variable ith.first has already been substituted by
                // another term
                return false;
            }
        return true;
        }
    }

// Update a substitution theta w.r.t. the matching between the
// variables of the literal L with terms of the literal L1
//
// @params theta, a substitution
// @params L, a literal  
// @params L1, a literal
void updateSubstitution(const pair<TERM,TERM> &ith,Substitution &theta)
    {
    assert( isPermissibleSubstitutions(ith,theta) );
    // add only if it is not already there
    if( find(theta.begin(),theta.end(),ith) == theta.end() )
        // the i-th pair is not present in substitution theta
        theta.push_back(ith);
    }

// Inequality, equality, addition, and multiplication have symmetric operands.
// Check whether the given literal is one of those.
//
// @param L, the literal to be checked
bool isSymmetricBuiltin(const LITERAL &L)
    {
    if( L.isBuiltin() )
        { 
        int id = L.getBuiltinID();
        return( id == BuiltinInequality ||
                id == BuiltinEquality || 
                id == BuiltinAddition || 
                id == BuiltinMultiplication );
        }
    return false;

    }

// check if there exists a matching between literal L and L1 w.r.t.
// substitution "theta".
//
// @param L, literal
// @param L1, literal
// @param theta, substitution
bool existsMatching(const LITERAL &L,
                    const LITERAL &L1,
                    const Substitution &theta)
    {
    // Subsumption checking for aggregates is not yet implemented.
        if( L.isAggregate() || L1.isAggregate() )
            return false;

    if( L.getIndex() != L1.getIndex() ||// different predicate name
        L.isNegative() != L1.isNegative() )// complementary literals
        return false;
    else
        {
        if(L.isPropositional() && L1.isPropositional())
            return true;
        else
            {
            const TERMS *params = L.getParams();
            const TERMS *params1 = L1.getParams();
            assert( params != 0 );
            assert( params1 != 0 );
            TERMS::const_iterator i1 = params1->begin();
            TERMS::const_iterator i = params->begin();
            
            // I need to instantiate a local copy of substitution
            // theta for modifying it in order to catch matching of
            // literals L to L1 even if L contains shared variables
            Substitution theta1(theta);
            for( ;
                 i != params->end() && i1 != params1->end() ;
                 i++, i1++)
                {
                pair<TERM,TERM> ith(*i,*i1);
                if( !isPermissibleSubstitutions(ith,theta1) )
                    return false;
                else 
                    // Substitution ith is permissible!  It must be
                    // added to theta1 to check the permissibility of
                    // a matching between two literals L and L1, where
                    // L could have non distinct variables.
                    updateSubstitution(ith,theta1);
                }
            }
        return true;
        }
    }

// Given a symmetric built-in literal, create a copy, swap its
// symmetric terms and return the swapped built-in
//
// @params L, the literal to be swapped
LITERAL swapSymmetricTerms(const LITERAL &L)
    {
    assert( isSymmetricBuiltin(L) );
    assert( L.getParams() );
    TERMS params(*(L.getParams()));
    int id = L.getBuiltinID();
    switch (id)
        {
        case BuiltinInequality:
        case BuiltinEquality:
            swap(*(params.begin()),*(params.begin()+1));
            break;
        case BuiltinAddition:
        case BuiltinMultiplication:
            swap(*(params.begin()),*(params.begin()+1));
            break;
        default:
            assert(0);
        }
    return LITERAL(L.isNegative(), ATOM(L,&params));
    }

// Given two literals, L and L1, and a substitution theta, return the
// vector of admissible substitutions for variables of L with terms of
// L1 w.r.t. theta.
// Note that pairs of symmetric builtin only can have more than one
// substitution.
//
// @param L, literal
// @param L1, literal
// @param theta, substitution
vector<Substitution> Match(const LITERAL &L,
                           const LITERAL &L1,
                           const Substitution &theta)
    {
    vector<Substitution> Substitutions;
    if( existsMatching(L,L1,theta) )
        {
        Substitution theta1(theta);
        updateSubstitution(theta1,L,L1); 
        Substitutions.push_back(theta1);
        }
    // Pairs of symmetric builtin can have more than one substitution.
    if( isSymmetricBuiltin(L) && isSymmetricBuiltin(L1) &&
        L.getBuiltinID() == L1.getBuiltinID() )
        {
        LITERAL sL1(swapSymmetricTerms(L1));
        if( existsMatching(L,sL1,theta) )
            {
            Substitution theta2(theta);
            updateSubstitution(theta2,L,sL1); 
            Substitutions.push_back(theta2);
            }
        }
    return Substitutions;
    }

// Given a literal L and a vector of literals literals1, count
// candidates of literal L occurring in literals1, that is the
// literals matching with L w.r.t. the substitution theta, and return,
// as output parameter, the reference pointer itL1 to last found
// candidate literal. If there are no candidates, pointer itL1 is
// set to literals.end().
//
// @param L, 
// @param literals1, the vector of literals to be checked for matching with L
// @param itL1, the reference to pointer to last found candidate
//       literal, if it exists, for L in literals1
// @param theta, the substitution applied.
template <class T_LITCOLLECTION>
unsigned countCandidates(const LITERAL &L,
                         const T_LITCOLLECTION &literals1,
                         typename T_LITCOLLECTION::const_iterator &itL1,
                         const Substitution &theta)
    {
    unsigned numC = 0;
    typename T_LITCOLLECTION::const_iterator i = literals1.begin();
    for(;
        i != literals1.end();
        i++ )
        {
        // there may be more than one match e.g. for symmetric builtins
        unsigned count = (Match(L,*i,theta)).size();
        numC += count;
        if( count > 0 ) 
            itL1 = i;
        }
    if( numC == 0 )
        itL1 = literals1.end(); 
    else 
        assert( itL1 != literals1.end() );
    assert( numC == 0 || itL1 != literals1.end() );
    return numC;
    }



// Given two literals L and L1 and a substitution theta, return true
// if L contains more distinct variables not yet instantiated
// w.r.t. substitution theta than L1.
//
// @param L,
// @param L1,
// @param theta
bool moreNotYetInstantiatedVars( const LITERAL &L,
                                 const LITERAL &L1,
                                 const Substitution &theta )
    {
    // L cannot have more variables than L1
    if( L.isPropositional() )
        return false;
    // L1 always has more variables than L
    if( L1.isPropositional() )
        return true;

    const TERMS *params = L.getParams();
    const TERMS *params1 = L1.getParams();
    assert( params != 0 );
    assert( params1 != 0 );
    TERMS vars;
    TERMS vars1;

    // put variables of L into vars
    for( TERMS::const_iterator p = params->begin();
         p != params->end(); 
         p++ )
        if( p->isVar() )
            vars.push_back(*p);

    // put variables of L1 into vars1
    for( TERMS::const_iterator p = params1->begin(); 
         p != params1->end();
         p++ )
        if( p->isVar() )
            vars1.push_back(*p);

    set<TERM> instantiatedVars;
    set<TERM> unboundVars,unboundVars1;

    for( Substitution::const_iterator it = theta.begin();
         it != theta.end(); 
         it++ )
        instantiatedVars.insert(it->first);

    // calculate the set unboundVars of instantiated variables of L
    // w.r.t. substitution theta
    insert_iterator<set<TERM> > inserter(unboundVars,unboundVars.begin());
    set_difference(vars.begin(),
                   vars.end(),
                   instantiatedVars.begin(),
                   instantiatedVars.end(),
                   inserter);
    
    // calculate the set unboundVars1 of instantiated variables of L1
    // w.r.t. substitution theta
    insert_iterator<set<TERM> > inserter1(unboundVars1,unboundVars1.begin());
    set_difference(vars1.begin(), 
                   vars1.end(),
                   instantiatedVars.begin(),
                   instantiatedVars.end(),
                   inserter1);
    
    return ( unboundVars.size() > unboundVars1.size());
    }

// Given two vectors of literals (literals and literals1), find the
// pointer itL to the literal occurring in literals having the
// smallest number of matched candidates (w.r.t. the substitution
// theta) in literals1; return also, as output parameter, the pointer
// iMin1 to the last found literal occurring in literals1, candidate to
// the matching with *iMin.
//
// @param literals,
// @param literals1,
// @param iMin,
// @param iMin1,
// @param theta,
template <class T_LITCOLLECTION>
unsigned calculateMinCandidates(const T_LITCOLLECTION &literals,
                                const T_LITCOLLECTION &literals1,
                                typename T_LITCOLLECTION::const_iterator &iMin,
                                typename T_LITCOLLECTION::const_iterator &iMin1,
                                const Substitution &theta)
    { 
    assert( literals.size() > 0 );
    iMin = literals.begin();
    typename T_LITCOLLECTION::const_iterator i = literals.begin();
    iMin1 = literals1.begin();
    typename T_LITCOLLECTION::const_iterator i1= literals1.begin();

    // initialize minC by computing the value for the first literal in
    // literals
    unsigned minC = countCandidates(*i,literals1,i1,theta);
    // If the minimum number of matching candidates is zero for the
    // first literal, literals cannot subsume literals1.
    if( minC == 0 )
        return minC;

    iMin = i;// Literal with the current minimum number of candidates.
    iMin1 = i1; // Last matching candidate literal.
    unsigned currC;

    for( i++ ; i != literals.end(); i++ ) 
        {
        currC = countCandidates(*i,literals1,i1,theta);
        // Select the literal with the minor number of matching
        // candidate literals ...
        if(  currC > 0)
            {
            assert( minC != 0 );
            assert( currC != 0 );
            if( currC < minC || 
                // ... and, if minC == currC, select the literal with the
                // greatest number of non-instantiated distinct variables
                ( minC == currC && 
                  moreNotYetInstantiatedVars(*i,*iMin,theta) ) )
                {
                assert( i != literals.end() );
                assert( i1 != literals1.end() );
                iMin = i;
                iMin1 = i1;
                minC = currC;
                }
            }
        else
            return currC;
        }
    return minC;
    }

// Given two rules r and r1, select the "best" literal of r with the
// smallest number of matched candidates in r1 and, if two literals
// have the same number of candidates select that one with the biggest
// number of distinct variables not yet instantiated w.r.t.
// substitution theta.
// If some literal L of r has been selected (i.e. a "best" literal
// exists), that is if at least a candidate in r1 matching with a
// literal of r exists, delete L from rule r, update the substitution
// theta w.r.t. the matching between L and an arbitrary matching
// candidate L1 in r1 and return true; return false otherwise.
//
// @param r, the rule where a literal must be selected
// @param r1, the rule where the candidates for a literal of r
//        must be found
// @param theta, the substitution for the variables of rule r
//        w.r.t. rule r1
bool bestMatchingLiteral(RULE &r,
                         const RULE &r1,
                         Substitution &theta)
    {
    // iterators to the literal pairs representing the best match in
    // head and body, resp.
    DISJUNCTION::const_iterator itH,itH1;
    CONJUNCTION::const_iterator itB,itB1;

    // try to find the "best" head literal
    // minH holds the minimum number of head literals in r1 matched by
    // a head literal in r
    unsigned minH = 0;
    bool found = false;
    // fromHead indicates whether the selected literal of r belongs to
    // its head or not
    bool fromHead = false;

    if( r.getHead().size() > 0 )
        { 
        if( ! r1.hasHead() ) 
            return false;
        // determine the literal pair representing the best match in
        // the head and compute the number of the matched candidates
        // of the best head literal in r
        minH = calculateMinCandidates(r.getHead(),r1.getHead(),itH,itH1,theta);

        if( minH > 0 )
            found = true;
        else
            // If there are no matching candidates in the head,
            // return false, without using variable "found"
            return false;

        if( found )
            {
            assert( itH != r.getHead().end() );
            assert( itH1 != r1.getHead().end() );
            }
        }

    // try to find the "best" body literal
    // minH holds the minimum number of body literals in r1 matched by
    // a body literal in r
    unsigned minB = 0;
   
    if(  r.hasBody() &&  r.getBody()->size() > 0 )
        {
        if( !r1.hasBody() )
            return false;

        // determine the literal pair representing the best match in
        // the body and compute the number of the matched candidates
        // of the best body literal in r
        minB = calculateMinCandidates(*r.getBody(),*r1.getBody(),itB,itB1,theta);

        if( minB > 0 )
            found = true;
        else
            return false;

        if( found )
            {
            assert( itB != r.getBody()->end() );
            assert( itB1 != r1.getBody()->end() );
            }
        }

    // select the "best" literal of the whole rule r between the best
    // body and the best head literal
    // choose the best head literal if a best head literal exists and
    // 1. no best body literal exists
    // 2. the minimum matched candidates for the best head literal are
    //    less than those for the best body literal
    // 3. the minimum matched candidates for best head and body
    //    literals are equal and the best head literal can bind more
    //    unbound variables w.r.t. theta
    if( minH > 0 
        && ( minB == 0 || minH < minB ||
             ( minH == minB && moreNotYetInstantiatedVars(*itH,*itB,theta) ) ) )
        {
        fromHead = true;
        }
    else
        {
        // otherwise, choose the best body literal
        assert( minB > 0 );
        assert( fromHead == false );
        }

    // remove the "best" literal from rule r
    if( found)
        {
        if( fromHead )
            {
            assert( find(r.getHead().begin(),r.getHead().end(),*itH )
                    != r.getHead().end() );
            assert( find(r1.getHead().begin(),r1.getHead().end(),*itH1 )
                    != r1.getHead().end() );

            // compute all matching substitutions
            vector<Substitution> Substitutions = Match(*itH,*itH1,theta);

            assert( ! Substitutions.empty() );
            // update theta by the first admissible matching between
            // *itH and *itH1
            theta = Substitutions.front();

            // remove L from r
            r.getHeadForModification().remove(
                       find(r.getHeadForModification().begin(),
                            r.getHeadForModification().end(),
                            *itH ));
            }
        else 
            {
            assert( find(r.getBody()->begin(),r.getBody()->end(),*itB )
                    != r.getBody()->end() );
            assert( find(r1.getBody()->begin(),r1.getBody()->end(),*itB1 )
                    != r1.getBody()->end() );

            // compute all matching substitutions
            vector<Substitution> Substitutions = Match(*itB,*itB1,theta);

            assert( ! Substitutions.empty() );
            // update theta by the first admissible matching between
            // *itB and *itB1
            theta = Substitutions.front();
            
            // remove L from r
            r.getBodyForModification()->
                remove(find(r.getBodyForModification()->pos_begin(),
                            r.getBodyForModification()->neg_end(),
                            *itB));
            }
        }
    return found;
    }

// Given two rules, r and r1, return true if rule r subsumes rule r1,
// false otherwise
//
// @param r, the rule that might subsume rule r1.
// @param r1, the rule that could be subsumed by r.
bool subsumes(const RULE &r,const RULE &r1)
    {
    RULE rule(r);
    Substitution theta;
    bool fail = false;

    if( TraceLevel >= 2 )
        {
        cdebug << "Check if  " << rule << "  subsumes " << r1 << endl;
        }

    // delete literals in rule which match other literals in r1 until
    // no such literal can be found
    while( !fail && ( rule.getHead().size() > 0 || 
                      (rule.hasBody() && rule.getBody()->size() > 0 ) ) )
        {
        if( !bestMatchingLiteral(rule,r1,theta) )
            fail = true;
        }
    return !fail;
    }


// Given a rule r, a program stored in a vector of rules and a range
// [ibegin , iend), over this vector, check if there is redundancy
// between r and the program, over the fixed range.
// If the program contains redundant rules w.r.t. r, erase them.
// Return true if the rule r is subsumed by a rule of the program,
// false if one or more rule of the program is subsumed by r, or if
// redundancy has been not detected
// 
// @param rule, the rule that could subsume one or more rule of the
//        program
// @param rules, the rules of the program
// @param ibegin, 
// @param iend,
bool subsumption(const RULE &r, 
                 RULES & rules,
                 RULES::iterator ibegin,
                 RULES::iterator iend)
    {
    assert( ibegin <= rules.end() );
    assert( iend <= rules.end() );

    if( !(ibegin < iend) )
        {
        return false;
        }

    if( TraceLevel >= 2 )
        {  
        cdebug << "*** CHECKING subsumption redundancy of rule  " 
               << r << endl;
        }

    bool result = false;
    for( RULES::iterator i = ibegin;
         i < iend && i != rules.end(); 
        )
        {
        // First, check if the given rule r is redundant w.r.t. the
        // current rule *i.
        if( subsumes(*i,r) )
            {
            if( TraceLevel >= 2 )
                {
                cdebug << "The rule  " << *i << "  subsumes  " << r << endl;
                }
            result = true; // the rule r is redundant
            break;
            }
        else
            {
            // Then, check if the current rule *i is redundant
            // w.r.t. the given rule r.
            if( subsumes(r,*i) )
                {
                result = false;
                if( TraceLevel >= 2 )
                    {   
                    cdebug << "The rule  " << r << "  subsumes  " << *i << endl;
                    cdebug << "Deleting it." << endl;
                    }
                // The i-th rule of the program is redundant w.r.t. r
                // and it must be erased.
                rules.erase(i);
                }
            else 
                {
                i++;
                }
            }
        }
    return result;
    }


// Given a rule r and a program rules, check if
// there is redundancy between r and the program.
// If the program contains redundant rules w.r.t. r erase them.
//
// @param r,
// @param rules,
bool subsumption(const RULE &r, RULES &rules)
    {
    return subsumption(r,rules,rules.begin(),rules.end()); 
    }

// Perform subsumption checking of IDB rules and delete redundant
// rules
bool SubsumptionCheckingRewriting()
    {
    if(TraceLevel >= 1)
        {
        cdebug << "Performing subsumption checking of IDB rules" << endl;
        print_list(cdebug,IDB,"\n");
        cdebug << endl <<" ---> "<< IDB.size() << " rules\n";
        cdebug << endl;
        }

    // used for determining the number of deleted rules
    unsigned originalSize = IDB.size();

    for(RULES::iterator i=IDB.begin();
        i != IDB.end();
        )
        {
        // Check whether *i is subsumed by the rest of the program.
        // If rules in the rest of the program are subsumed by *i,
        // they are deleted in subsumption().
        if( subsumption(*i,IDB,i+1,IDB.end()) )
            {
            if(TraceLevel >= 2)
                {    
                cdebug << "Rule  " << *i << "  is redundant." << endl;
                }
            IDB.erase(i);
            // the previous (i+1) is now i
            }
        else
            {
            if(TraceLevel >= 2)
                {
                cdebug << "Rule  " << *i << "  is not redundant." << endl;
                }
            i++;
            }
        }

    if(TraceLevel >= 1)
        {
        cdebug << "Detected and deleted " << originalSize - IDB.size() 
               << " redundant rules." << endl << endl
               << "Rewritten Program:\n";
        print_list(cdebug,IDB,"\n");
        cdebug << endl << endl;
        }
    
    return true;
    }

//////////////////////////////////////////////////////////////////////////////
void BasicRewriting(RULES &rules, CONSTRAINTS &constraints,
                    bool &hasBuiltins, bool &hasAggregates)
// 
// Note: It is important to process rules before constraints as we may
// rewrite a rule into a constraint, but not the other way around.
//
// On the fly, we determine whether any rule contains built-ins or
// aggregates. This is done here and not in the parser, since rules
// may also stem from frontends. General assumption: "Undef" atoms,
// built-in atoms and aggregate atoms are disjoint.
//
    {
    hasBuiltins=false;
    hasAggregates=false;

    // For each rule perform the following checks:

    for( RULES::iterator i=rules.begin(); i != rules.end(); )
        {
        CONJUNCTION* body = i->getBodyForModification();

        if( body )
            {
            bool didRemove=false;

            // If the (positive) body can never become true, simply
            // eliminate the rule and continue with the next rule.
            for( CONJUNCTION::iterator j=body->pos_begin();
                 ! didRemove  &&  j != body->pos_end();
                 j++ )
                {
                if( OptionBasicRewrite && OptionRewriteDeleteRules
                    && (*j).isUndef() )
                    {
                    if( TraceLevel >= 1 )
                        cdebug << "Removing   " << *i << endl
                               << "       " << *j << " is always false."
                               << endl;

                    rules.erase(i);
                    didRemove=true;
                    }
                else if( (*j).isBuiltin() )
                    hasBuiltins=true;
                else if( (*j).isAggregate() )
                    hasAggregates=true;
                }

            if( didRemove )
                continue;

            // Eliminate undefined negative literals from the body as
            // these are always true.
            
            bool didRewrite=false;

            for( CONJUNCTION::iterator j=body->neg_begin();
                 j != body->neg_end(); )
                {
                assert( ! (*j).isBuiltin() );

                if( OptionBasicRewrite && (*j).isUndef() )
                    {
                    if( TraceLevel >= 1 )
                        // Print this at most once for each rule.
                        if( ! didRewrite )
                            {
                            didRewrite=true;
                            cdebug << "Rewriting  " << *i << endl;
                            }
                    
                    j=body->remove(j);
                    
                    if( TraceLevel >= 1 )
                        cdebug << "       to  " << *i << endl;
                    }
                else
                    {
                    if( (*j).isAggregate() )
                        hasAggregates=true;
                    j++;
                    }
                }

            // If the body is empty now, remove it and proceed with the next 
            // rule.

            if( OptionBasicRewrite && body->begin() == body->end() )
                {
                if( TraceLevel >= 1 )
                    {
                    if( ! didRewrite )
                        {
                        didRewrite=true;
                        cdebug << "Rewriting  " << *i << endl;
                        }

                    cdebug << "       Removing empty body." << endl;
                    }

                (*i).removeBody();
                
                continue;
                }

            if( OptionBasicRewrite )
                {
                // Remove all atoms from the head that occur negatively in
                // the body.

                DISJUNCTION &head=(*i).getHeadForModification();
                for( DISJUNCTION::iterator j=head.begin();
                     j != head.end(); )
                    {
                    bool currentHeadLiteralRemoved=false;

                    // Abort the scan of the body after the first match.
                    for( CONJUNCTION::const_iterator n=body->neg_begin();
                         ! currentHeadLiteralRemoved  
                             &&  n != body->neg_end(); 
                         n++ )
                        {
                        if( *j == *static_cast<const ATOM*>(&(*n)) )
                            {
                            if( TraceLevel >= 1 )
                                // Print this at most once for each rule.
                                if( ! didRewrite )
                                    {
                                    didRewrite=true;
                                    cdebug << "Rewriting " << *i << endl;
                                    }

                            j=head.remove(j);
                            currentHeadLiteralRemoved=true;

                            if( TraceLevel >= 1 )
                                // If the head is empty now, the rule
                                // has become a constraint and we'll
                                // print it later anyway.
                                if( head.begin() != head.end() )
                                    cdebug << "       to " << *i << endl;
                            }
                        }

                    if( ! currentHeadLiteralRemoved )
                        j++;
                    }

                if( head.begin() == head.end() )
                    {
                    constraints.push_back(CONSTRAINT(*body,false));
                    rules.erase(i);

                    if( TraceLevel >= 1 )
                        cdebug << "       to " << constraints.back() << endl;

                    // Proceed to the next rule (without incrementing i, as we
                    // just have removed *i and moved *(i+1) to *i).
                    continue;
                    }
                }
            }

        i++;
        }


    // For each constraint perform the following checks:

    for( CONSTRAINTS::iterator i=constraints.begin(); i != constraints.end(); )
        {
        // If the constraint can never be violated, simply remove it...

        bool didRemove=false;

        for( CONJUNCTION::const_iterator j=(*i).pos_begin();
             ! didRemove  &&  j != (*i).pos_end();
             j++ )
            {
            if( OptionBasicRewrite && OptionRewriteDeleteRules 
                && (*j).isUndef() )
                {
                if( TraceLevel >= 1 )
                    cdebug << *j << " is always false." << endl
                           << "  Removing " << *i << endl;

                constraints.erase(i);
                didRemove=true;
                }
            else if( (*j).isBuiltin() )
                hasBuiltins=true;
            else if( (*j).isAggregate() )
                hasAggregates=true;
            }
           
        if( ! didRemove )
            {                
            // ...else eliminate undefined negative literals as these
            // are always true.

            for( CONJUNCTION::iterator j=(*i).neg_begin();
                 j != (*i).neg_end(); )
                {
                assert( ! (*j).isBuiltin() );

                if( OptionBasicRewrite && (*j).isUndef() )
                    {
                    if( TraceLevel >= 1 )
                        cdebug << "Rewriting  " << *i << endl;

                    (*i).remove(j);

                    if( TraceLevel >= 1 )
                        cdebug << "       to  " << *i << endl;
                    }
                else
                    {
                    if( (*j).isAggregate() )
                        hasAggregates=true;
                    j++;
                    }
                }

            // If the constraint is empty now, it's always violated.

            if( OptionBasicRewrite && (*i).begin() == (*i).end() )
                {
                if( TraceLevel >= 1 )
                    {
                    if( (*i).isInternal() )
                        cdebug << "Internal ";
                    cdebug << "Constraint  " << *i << "  is always violated." 
                           << endl;
                    }

                SafeExit(0);
                }
 
            i++;
            }
        }

    }


//////////////////////////////////////////////////////////////////////////////
// Projection Rewriting
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
static void addInfoAboutPar(
    string& s,
    const ATOM &anAtom, 
    const TERMS::const_iterator k,
    const unsigned* v2i)
    {
    unsigned index=static_cast<unsigned>(k-anAtom.getParams()->begin()) + 1;

    // FIXME as soon as sstream is available.
    // 20 chars is enough for 64bit uints
    char tmp[21];
    sprintf(tmp,"%u",index);

    s += tmp;

    if( v2i[(*k).getVar()] )
        {
        s += "=";
        sprintf(tmp,"%u",v2i[(*k).getVar()]);
        s += tmp;
        }
    }


//////////////////////////////////////////////////////////////////////////////
static void addInfoAboutNotProjectedPar(
    string& s, 
    const ATOM &anAtom,
    const TERMS::const_iterator k,
    const unsigned* v2i)
    {
    if( v2i[(*k).getVar()] )
        {
        s += ",[";
        addInfoAboutPar(s,anAtom,k,v2i);
        s += "]";
        }
    }


//////////////////////////////////////////////////////////////////////////////
static void addInfoAboutInstantiatedPar(
    string& s, 
    const ATOM &anAtom,
    const TERMS::const_iterator k)
    {
    assert( ! (*k).isVar() );

    unsigned index=static_cast<unsigned>(k-anAtom.getParams()->begin()) + 1;

    // FIXME as soon as sstream is available.
    // 20 chars is enough for 64bit uints
    char tmp[21];
    sprintf(tmp,"%u",index);

    s += tmp;
    s += "=";

    if( (*k).getType() == TERM::Integer )
        {
        s += "int[";

        sprintf(tmp,"%u",(*k).getInt());

        s += tmp;
        s += "]";
        }
    else
        s += (*k).getNameAsString();
    }


//////////////////////////////////////////////////////////////////////////////
static void addInfoAboutNotProjectedPar(
    string& s, 
    const ATOM &anAtom,
    const TERMS::const_iterator k)
    {
    s += ",[";

    addInfoAboutInstantiatedPar(s,anAtom,k);

    s += "]";
    }


//////////////////////////////////////////////////////////////////////////////
static void Project(const size_t i, const bool isRule)
//
// Let p(...) be a predicate and suppose we want to project some variable V
// of p. The internal name p' corresponding to the projected predicate is
// generated as follows:
//
// 1. p' := "_p" // initialization
// 2. FOR i:=1 TO nr. of arguments of p DO
//      V := i-th argument of p
//      IF V is variable THEN
//        IF V is projected out THEN
//          p' := p' + "," + itoa(i)
//          IF this is not the first occurrence of V in p THEN
//            k := the argument index of the last occurrence of V in p
//            p' := p' + "=" + itoa(k)
//        ELSE
//          IF this is not the first occurrence of V in p THEN
//            k := the argument index of the last occurrence of V in p
//            p' := p' + ",[" + itoa(i) + "=" + itoa(k) + "]"
//      ELSE // V is a constant or an integer
//        IF V is an integer THEN
//          p' := p' + ",[" + itoa(i) + "=int[" + itoa(V) + "]]"
//        ELSE
//          p' := p' + ",[" + itoa(i) + "=" + nameAsString(V) + "]"
//
//    FOR each builtin literal q taken in the new rule DO
//      IF q is negative THEN
//        p' := p' + "_"
//      p' := p' + "q"
//      FOR i:=1 TO nr. of arguments of q DO
//        V := i-th argument of q
//        IF V is variable THEN
//          k := the index of the argument of p containing V
//          p' := p' + "," + itoa(k)
//        ELSE
//          IF V is an integer THEN
//            p' := p' + ",[" + itoa(i) + "=int[" + itoa(V) + "]]"
//          ELSE
//            p' := p' + ",[" + itoa(i) + "=" + nameAsString(V) + "]"
//
// Duplicates of projections arise when both the following conditions are
// fulfilled:
// - the same positive literal is projected from different rules on the same
//   arguments, and
// - the same built-in literals (more than one) are taken in the projection
//   rule body with the same arguments, but in a different order (that
//   depends on the order of the built-in literals in the rule body).
//
// These duplications are caused by the automatic generation of the
// name of the "projection" literal which may generate different names
// for the same projection under the conditions above.
//
// However, in practice, the probability that both these conditions
// occur is extremely low. Therefore, the addition of further control
// mechanisms would only slow down the processing.
//
// Note that such duplications have no influence on the correctness of the
// results.
//
    {
    const CONJUNCTION* body;

    if( isRule )
	body=(*(IDB.begin()+i)).getBody();
    else
	body=&(*(Constraints.begin()+i));

    bool goAhead=true;
    if( body )
        {
        if( (body->pos_end() - body->pos_begin()) < 2 )
            goAhead=false;
        else
            {
            unsigned nValidLiterals=0;
            for( CONJUNCTION::const_iterator j=body->pos_begin(); 
                 j != body->pos_end(); 
                 j++ )
                if( ! (*j).isBuiltin() )
                    // *j must be either IDB or EDB
                    if( ! (*j).isEDB() || OptionRewriteCountEDB )
                        {
                        nValidLiterals++;
                        if( nValidLiterals > 1 )
                            break;
                        }
            if( nValidLiterals < 2 )
                goAhead=false;
            }
        }
    else
        goAhead=false;

    if( ! goAhead )
        {
        return;
        }

    ProjectionsExist = true;

    const DISJUNCTION* head=0;
    CONJUNCTION newBody;
    unsigned nLiterals = body->end() - body->begin();
    unsigned nBuiltinLiterals = 0;
    unsigned nPosLiterals = 0;
    unsigned (*aMatrix)[n_par+1] = new unsigned[nLiterals+2][n_par+1];

    // FIXME: Describe the purpose of i2l.
    // Note: We must not use iterators as these may become invalidated
    // when the underlying vector<> grows.
    unsigned* i2l = new unsigned[nLiterals];

    for( unsigned j=0;j<=(nLiterals+1);j++ )
        for( unsigned k=0;k<=n_par;k++ )
            aMatrix[j][k]=0;

    // Set to 2 the counters of the variables appearing in head atoms.

    if( isRule )
	{
        head=&( (IDB.begin()+i)->getHead() );
        for( DISJUNCTION::const_iterator j=head->begin();
             j != head->end(); j++ )
            {
	    if( (*j).getParams() )
                for( TERMS::const_iterator k  = (*j).getParams()->begin();
                     k != (*j).getParams()->end(); k++ )
                    if( (*k).isVar() )
                        // it's sure that this variable appears in at least two
                        // literals; the precise number of occurrences is not
                        // important (we store the constant 2), it's enough to
                        // know that this number is greater than 1
                        aMatrix[nLiterals][(*k).getVar()]=2;
            }
	}

    // Fill the matrix: the rows starting from index 0 on correspond
    // to positive body literals; the rows starting from index
    // nLiterals-1 back correspond to built-in literals; the row
    // nLiterals stores the value 1 for the variables occurring only
    // once in the body, and a value greater than 1 otherwise; the row
    // nLiterals+1 stores the order numbers (row indexes) of the last
    // positive atoms containing the variables. The columns starting
    // from index 0 til n_par-1 correspond to the variables. The last
    // column (with index n_par), (i) for the positive body literals
    // stores 1 for the atoms already tested for projection, and 0 for
    // the others, (ii) for the built-in literals stores 1 for the
    // literals used in at least one projection, and 0 for the others.
    //
    // Increment the counters of the variables appearing in body literals:

    for( CONJUNCTION::const_iterator j=body->begin(); j != body->end(); j++ )
        {
        if( (*j).isBuiltin() )
            {
            nBuiltinLiterals++;
            i2l[nLiterals-nBuiltinLiterals] = j-body->begin();
            }
        else
            if( ! (*j).isNegative() )
                {
                i2l[nPosLiterals] = j-body->begin();
                nPosLiterals++;
                }
        if( (*j).isAggregate() )
            {
            const AGGREGATEATOM &aggr = (*j).getAggregate();

            // We should not perform projections on aggregates.
            // Therefore we set the occurrence counter of the
            // variables to 2, such that the variables will not
            // be projected out later.
            if( aggr.getLowerGuard().first.isVar() )
                aMatrix[nLiterals][aggr.getLowerGuardAsVar().getVar()]=2;   
            if( aggr.getUpperGuard().first.isVar() )
                aMatrix[nLiterals][aggr.getUpperGuardAsVar().getVar()]=2;
            ATOM aux = aggr.getAuxAtom();
            for( TERMS::const_iterator k=aux.getParams()->begin();
                k != aux.getParams()->end(); k++ )
                {
                if( (*k).isVar() )
                    aMatrix[nLiterals][(*k).getVar()]=2;
                }
            }
	else if( (*j).getParams() )
            for( TERMS::const_iterator k  = (*j).getParams()->begin();
                 k != (*j).getParams()->end(); k++ )
                if( (*k).isVar() )
                    {
                    if( (*j).isBuiltin() )
                        {
//                        assert( ! aMatrix[nLiterals-nBuiltinLiterals][(*k).getVar()] );
                        aMatrix[nLiterals-nBuiltinLiterals][(*k).getVar()]=
                            k-(*j).getParams()->begin()+1;

                        // We should not perform projections on builtins.
                        // Therefore we set the occurrence counter of the
                        // variable to 2, such that the variable will not
                        // be projected out later.
                        // The last literal occurrence is set to a
                        // nonsensical value, assuming that it will never
                        // be accessed.
                        aMatrix[nLiterals][(*k).getVar()] = 2;
                        aMatrix[nLiterals+1][(*k).getVar()] 
                            = nLiterals+nBuiltinLiterals;
                        }
                    else
                        {
                        if( ! (*j).isNegative() ) 
                            {
//                            assert( ! aMatrix[nPosLiterals-1][(*k).getVar()] );
                            aMatrix[nPosLiterals-1][(*k).getVar()]=
                                    k-(*j).getParams()->begin()+1;

                            if( aMatrix[nLiterals+1][(*k).getVar()] != 
                                nPosLiterals )
                                {
                                aMatrix[nLiterals][(*k).getVar()]++;
                                aMatrix[nLiterals+1][(*k).getVar()]=
                                    nPosLiterals;
                                }
                            }
                        else
                            // it's sure that this variable appears in
                            // at least two literals; the precise
                            // number of occurrences is not important
                            // (we store the constant 2), it's enough
                            // to know that this number is greater
                            // than 1
                            aMatrix[nLiterals][(*k).getVar()]=2;
                        }
                    }
        }

    // Look for the projections where at least one variable is eliminated.

    unsigned* v2i = new unsigned[n_par+1];
    // v2i is an arry where the unsigned stored at index i is the last position 
    // in the list of arguments of the current positive literal where the 
    // variable with the order number i appeared (some variables may appear
    // more than once in the list of arguments of the same literal)
    for( unsigned j=0; j<n_par; j++ )
        if( (aMatrix[nLiterals][j] == 1) && (! aMatrix[aMatrix[nLiterals+1][j]-1][n_par]) )
            {
            // anAtom is the iterator of the body positive literal
            // which contains the single occurrence of the variable j
            CONJUNCTION::const_iterator anAtom=body->begin()+
		                               i2l[aMatrix[nLiterals+1][j]-1];

            assert( body->begin() <= anAtom && anAtom < body->end() );

            // Builtins should not be rewritten, and therefore we set
            // aMatrix[nLiterals][j] to 2 for each variable j that
            // occurs in a builtin, so this code should not be reached
            // for builtins.
            assert( ! (*anAtom).isBuiltin() );

            // Record that this atom has been projected.
            aMatrix[aMatrix[nLiterals+1][j]-1][n_par]=1;

            string predName("\"_");
            predName += (*anAtom).getPredName();

            TERMS args;

	    assert( (*anAtom).getParams() );
            for( TERMS::const_iterator k  = (*anAtom).getParams()->begin();
                 k != (*anAtom).getParams()->end(); k++ )
                if( (*k).isVar() )
                    {
                    v2i[(*k).getVar()]=0;
                    }
            for( TERMS::const_iterator k  = (*anAtom).getParams()->begin();
                 k != (*anAtom).getParams()->end(); k++ )
		{
                if( (*k).isVar() ) 
                    {
                    if( (aMatrix[nLiterals][(*k).getVar()] == 1) )
                        {
                        predName += ",";
                        addInfoAboutPar(predName,*anAtom,k,v2i);
                        }
                    else
                        {
                        addInfoAboutNotProjectedPar(predName,*anAtom,k,v2i);
                        args.push_back(*k);
                        }
                    v2i[(*k).getVar()]=k-(*anAtom).getParams()->begin()+1;
                    }
                else // *k must be a constant or an integer constant
                    {
                    addInfoAboutNotProjectedPar(predName,*anAtom,k);
                    }
                }

            CONJUNCTION newRuleBody;
            for( unsigned k=0; k<nBuiltinLiterals; k++ )
                {
                CONJUNCTION::const_iterator aBuiltinLiteral=body->begin()+
                                                            i2l[nLiterals-k-1];
                string s;

                bool toBeTaken=true;
		assert( (*aBuiltinLiteral).getParams() );
                for( TERMS::const_iterator l=(*aBuiltinLiteral).getParams()->begin();
                     l != (*aBuiltinLiteral).getParams()->end() && toBeTaken; l++ )
                    {
                    if( (*l).isVar() )
                        {
                        if( aMatrix[aMatrix[nLiterals+1][j]-1][(*l).getVar()] )
                            {
                            s += ",";
                            // FIXME as soon as sstream is available.
                            // 20 chars is enough for 64bit uints
                            char tmp[21];
                            sprintf(tmp,"%u",aMatrix[aMatrix[nLiterals+1][j]-1][(*l).getVar()]);
                            s += tmp;
                            }
                        else
                            toBeTaken=false;
                        }
                    else
                        {
                        addInfoAboutNotProjectedPar(s,*aBuiltinLiteral,l);
                        }
                    }
                if( toBeTaken )
                    {
                    if( (*aBuiltinLiteral).isNegative() )
                        predName += "_";
                    predName += (*aBuiltinLiteral).getPredName() + s;
                    newRuleBody.add(LITERAL(*aBuiltinLiteral));
                    // Record that this built-in was used in a
                    // projection and must be deleted from the rule body.
                    aMatrix[nLiterals-k-1][n_par]=1;  
                    }
                }

            predName += "\"";

            // We have to add the predicate name if it is not found in
            // the predicates datastructure.
	    bool toBeAdded = 
                ! (ATOM::Predicates.find(NAMES_ITEM(predName.c_str()))).second;

            ATOM headAtom(predName.c_str(),&args);

            if( toBeAdded )
                {
                DISJUNCTION newRuleHead;
                newRuleHead.add(headAtom);
                newRuleBody.add(LITERAL(*anAtom));
                IDB.push_back( RULE(&newRuleHead,&newRuleBody) );

                if( TraceLevel >= 1 )
                    cdebug << "   Adding "
                           << RULE(&newRuleHead,&newRuleBody) << endl;

		if( isRule )
                    {
                    // Obtain the body and the head of the current
                    // rule again, since the push_back() operation may
                    // invalidate the iterator of the current rule.
                    head = &( (IDB.begin()+i)->getHead() );
                    body = (IDB.begin()+i)->getBody();
		    }
		else
		    // Nothing to do because Constraints was not modified.
		    ;
                }

            newBody.add(headAtom);
            }

    // Look for the projections where no variable is eliminated.

    for( unsigned j=0; j<nPosLiterals; j++ )
        if( ! aMatrix[j][n_par] )
            {
            CONJUNCTION::const_iterator anAtom=body->begin()+i2l[j];
            if( ! anAtom->isRegularAtom() || ! anAtom->getParams() )
                continue;

            string predName("\"_");
            predName += (*anAtom).getPredName();

            CONJUNCTION newRuleBody;
            bool atLeastOneWasTaken=false;
            for( unsigned k=0; k<nBuiltinLiterals; k++ )
                {
                CONJUNCTION::const_iterator aBuiltinLiteral=body->begin()+
                                                            i2l[nLiterals-k-1];
                string s;

                bool toBeTaken=true;
		assert( (*aBuiltinLiteral).getParams() );
                for( TERMS::const_iterator l=(*aBuiltinLiteral).getParams()->begin();
                     l != (*aBuiltinLiteral).getParams()->end() && toBeTaken; l++ )
                    {
                    if( (*l).isVar() )
                        {
                        if( aMatrix[j][(*l).getVar()] )
                            {
                            s += ",";
                            // FIXME as soon as sstream is available.
                            // 20 chars is enough for 64bit uints
                            char tmp[21];
                            sprintf(tmp,"%u",aMatrix[j][(*l).getVar()]);
                            s += tmp;
                            }
                        else
                            toBeTaken=false;
                        }
                    else
                        {
                        addInfoAboutNotProjectedPar(s,*aBuiltinLiteral,l);
                        }
                    }
                if( toBeTaken )
                    {
                    atLeastOneWasTaken=true;
        	    assert( (*anAtom).getParams() );
                    for( TERMS::const_iterator z  = (*anAtom).getParams()->begin();
                         z != (*anAtom).getParams()->end(); z++ )
                        if( (*z).isVar() )
                            {
                            v2i[(*z).getVar()]=0;
                            }
                    for( TERMS::const_iterator z  = (*anAtom).getParams()->begin();
                         z != (*anAtom).getParams()->end(); z++ )
		        {
                        if( (*z).isVar() )
                            {
                            assert( aMatrix[nLiterals][(*z).getVar()] != 1 );
                            addInfoAboutNotProjectedPar(predName,*anAtom,z,v2i);
                            v2i[(*z).getVar()]=z-(*anAtom).getParams()->begin()+1;
                            }
                        else // *z must be a constant or an integer constant
                            {
                            addInfoAboutNotProjectedPar(predName,*anAtom,z);
                            }
                        }
                    if( (*aBuiltinLiteral).isNegative() )
                        predName += "_";
                    predName += (*aBuiltinLiteral).getPredName() + s;
                    newRuleBody.add(LITERAL(*aBuiltinLiteral));
                    // Record that this built-in was used in a projection
                    // and must be deleted from the rule body.
                    aMatrix[nLiterals-k-1][n_par]=1;  
                    }
                }

            if( atLeastOneWasTaken )
                {
                // Record that this atom has been projected.
                aMatrix[j][n_par]=1; 

                predName += "\"";

                bool toBeAdded =
                    ! ( ATOM::Predicates.find(NAMES_ITEM(predName.c_str()))).second;

                ATOM headAtom(predName.c_str(),(*anAtom).getParams());

                if( toBeAdded )
                    {
                    DISJUNCTION newRuleHead;
                    newRuleHead.add(headAtom);
                    newRuleBody.add(LITERAL(*anAtom));
                    IDB.push_back( RULE(&newRuleHead,&newRuleBody) );

		    if( isRule )
			// Obtain the body and the head of the current rule
                        // again since the push_back() operation may
                        // invalidate the iterator of the current rule.
			{
                        head=&( (IDB.begin()+i)->getHead() );
                        body=(IDB.begin()+i)->getBody();
			}
		    else
		        // Nothing to do because Constraints was not modified.
		        ;

                    if( TraceLevel >= 1 )
                        cdebug << "   Adding "
                               << RULE(&newRuleHead,&newRuleBody) << endl;
                    }

                newBody.add(headAtom);
                }

            }

    if( newBody.begin() != newBody.end() )
	{
        nBuiltinLiterals = 0;
        nPosLiterals = 0;
	assert( body );
        for( CONJUNCTION::const_iterator j=body->begin();
             j != body->end(); j++ )
            {
            if( (*j).isBuiltin() )
                {
                nBuiltinLiterals++;
                if( ! aMatrix[nLiterals-nBuiltinLiterals][n_par] )
		    newBody.add(*j);
                }
            else
		{
                if( (*j).isNegative() )
                    newBody.add(*j);
		else
		    {
                    if( ! aMatrix[nPosLiterals][n_par] )
			newBody.add(*j);
		    nPosLiterals++;
		    }
		}
	    }
	if( isRule )
	    {
	    DISJUNCTION newHead(*head);

            if( TraceLevel >= 1 )
                cdebug << "Rewriting " << *(IDB.begin()+i) << endl 
                       << "       to " << RULE(&newHead,&newBody) << endl;

            *(IDB.begin()+i) = RULE(&newHead,&newBody);
	    }
        else
            {
            if( TraceLevel >= 1 )
                cdebug << "Rewriting " << *(Constraints.begin()+i) << endl
                       << "       to " << CONSTRAINT(newBody,false) << endl;

            *(Constraints.begin()+i) = CONSTRAINT(newBody,false);
	    }
	}

    delete[] aMatrix;
    delete[] i2l;
    delete[] v2i;
    }

//////////////////////////////////////////////////////////////////////////////
void ProjectionRewriting(RULES &rules)
// 
// It is important to optimize rules and constraints after
// PreprocessProgram().
//  
    {
    const size_t initialSize = rules.size();

    for( size_t i=0; i < initialSize; i++ )
        Project(i,true);
    }

//////////////////////////////////////////////////////////////////////////////
void ProjectionRewriting(CONSTRAINTS &constraints, RULES&)
    {
    const size_t initialSize = constraints.size();

    for( size_t i=0; i < initialSize; i++ )
        Project(i,false);
    }

// Local Variables:
// c-file-style: "dl"
// End:
