//////////////////////////////////////////////////////////////////////////////
// grounding-prop.C

#include "dl.h"

//////////////////////////////////////////////////////////////////////////////
static bool transformConjunction(
//
// Transform a CONJUNCTION into a GCONJUNCTION taking into account the EDB
// (as provided as a parameter).
//
// @returns false if the CONJUNCTION is always false (and the transformation
//          has been aborted), true else.

    const CONJUNCTION  &in,
    const VATOMSET     &EDB,
          GCONJUNCTION &out )
    {
    for( CONJUNCTION::const_iterator j = in.begin();
         j != in.end();
         j++ ) 
        {
        assert( (*j).isGround() );
            
        if( (*j).isEDB() )
            {
            if( (*j).isNegative() )
                {
                if( EDB.contains(*j) )
                    return false;
                }
            else
                {
                if( ! EDB.contains(*j) )
                    return false;
                }
            }
        else
            out.addUnique( GLITERAL((*j).isNegative(),GATOM(*j)) );
        }

    return true;
    }


//////////////////////////////////////////////////////////////////////////////
bool GroundPropositionalInput(
//
// Transform RULES, CONSTRAINTS and WEAKCONSTRAINTS into GRULES, GCONSTRAINTS 
// and GWEAKCONSTRAINTS, respectively,
// taking into account the EDB (as provided as a parameter).
//
// Some rules/constraints may be dropped, if their body/they always evaluate
// to false.
// If a weak constraint is violated, it is stored in ViolatedWeakConstraints.
//
// Returns false if some constraint is always violated, true else.

    const RULES            &IDB,
    const CONSTRAINTS      &Constraints,
    const WEAKCONSTRAINTS  &WConstraints,
    const VATOMSET         &EDB,
          GRULES           &GroundIDB,
          GCONSTRAINTS     &GroundConstraints,
          GWEAKCONSTRAINTS &GroundWConstraints,
          GWEAKCONSTRAINTS &ViolatedWConstraints, 
          bool              printInstantiation)
    {
    for( RULES::const_iterator i=IDB.begin(); i != IDB.end(); i++)
        {
        if( GTraceLevel > 0 )
            cdebug << "INPUT: " << *i << endl;
            
        GDISJUNCTION head;
        for( DISJUNCTION::const_iterator j = i->getHead().begin();
             j != i->getHead().end();
             j++ ) 
            {
            assert( (*j).isGround() );

            head.addUnique(GATOM(*j));
            }

        bool added=false;

        if( (*i).getBody() )
            {
            GCONJUNCTION body;

            if( transformConjunction(*((*i).getBody()),EDB,body) == false)
                {
                if( GTraceLevel > 0 )
                    cdebug << "    dropped" << endl;
                }
            else
                {
                if( body.begin() == body.end() )
                    {
                    GRULE r(&head,0);
                    GroundIDB.push_back(r);
                    }
                else
                    {
                    GRULE r(&head,&body);
                    GroundIDB.push_back(r);
                    }

                added=true;
                }
            }
        else
            {
            GRULE r(&head,0);
            GroundIDB.push_back(r);

            added=true;
            }

        if( added )
            if( GTraceLevel > 0 )
                cdebug << "    -> " << GroundIDB.back() << endl;
        }

    for( CONSTRAINTS::const_iterator i=Constraints.begin();
         i != Constraints.end();
         i++ )
        {
        if( GTraceLevel > 0 )
            cdebug << "INPUT: " << *i << endl;

        GCONSTRAINT c;

        if( transformConjunction(*i,EDB,c) == false )
            {
            if( GTraceLevel > 0 )
                cdebug << "    dropped" << endl;
            }
        else
            {
            if( c.begin() == c.end() )
                {
                if ( printInstantiation )
                    {
                    cout << "% A constraint will always be violated, " << endl 
                         << "% so this program does not have any answer set." 
                         << endl
                         << ":- not a_Constant_Never_Occurring_In_Regular_Programs."  
                         << endl;
                    GroundIDB.erase(GroundIDB.begin(), GroundIDB.end()); 
                    GroundConstraints.erase(GroundConstraints.begin(), 
                                            GroundConstraints.end());
                    }
                return false;
                }
            else
                {
                GroundConstraints.push_back(c);

                if( GTraceLevel > 0 )
                    cdebug << "    -> " << c << endl;
                }
            }
        }
 
    for( WEAKCONSTRAINTS::const_iterator i=WConstraints.begin();
                                         i != WConstraints.end();
                                         i++ )
        {
        if( GTraceLevel > 0 )
            cdebug << "INPUT: " << *i << endl;
        // Initialise the ground constraint with the grounded terms.
        GWEAKCONSTRAINT c((*i).getWeights().first,(*i).getWeights().second);

        if( transformConjunction(*i,EDB,c) == false )
            {
            if( GTraceLevel > 0 )
                cdebug << "    dropped" << endl;
            }
        else
            {
            if( c.begin() == c.end() )
                // Add weak to the violated.
                ViolatedWConstraints.push_back(c);
            else
                {
                GroundWConstraints.push_back(c);

                if( GTraceLevel > 0 )
                    cdebug << "    -> " << c << endl;
                }
            }
        if(c.getLevel() > MaxWeakConstraintLevel)
            // Update the max number of priority level available
            MaxWeakConstraintLevel = c.getLevel(); 
        }
    return true;
    }

// Local Variables:
// c-file-style: "dl"
// End:
