//////////////////////////////////////////////////////////////////////////////
// grounding-body.h

#ifndef BODY_H
#define BODY_H

#include <limits.h> // for UINT_MAX

#include "grounding.h"
#include "grounding-dictionary.h"

// A global dictionary for variables selectivities
static DBDICTIONARY* dictionary;

// Functions used to calculate the size of builtin predicates
static unsigned computeSizeAddition(const int);
static unsigned computeSizeMultiplication(const int);

/** OrderedBody is a class which reorders the literals of a rule body so
 * that a more efficient instantiation is possible.
 */
class OrderedBody
    {
    /// oldBody references the original rule body.
    /// Note that mentions of "the i-th literal" and other indices relate to
    /// the original order, unless stated otherwise.
    const CONJUNCTION &oldBody;

    /// newOrderdBody stores the literals of the body in a new order,
    /// the order that we actually consider during the instantiation.
    CONJUNCTION::const_iterator* newOrderedBody;

    /// isExit stores whether our rule is an exit rule.
    bool isExit;

    /// recursive stores whether literals are recursive.
    /// recursive[i] is true, iff the i-th literal of the body is recursive
    /// with some predicate in the head.
    bool* recursive;

    /// range stores interpretations ranges of literals.
    /// range[i] determines which ground atoms are matched with the i-th
    /// literal.
    /// A value of interpretations range rI allows matching with atoms 
    /// generated in all but the last iteration (atoms in I).
    /// A value of range rDeltaNF allows matching with atoms generated in 
    /// the last iteration (atoms in deltaNF).
    /// A value of range rI_DeltaNF allows matching with atoms generated in 
    /// all iterations (atoms in I or atoms in deltaNF).
    /// Non-recursive literals always have range rI.
    INTERPRETATIONS_RANGE* range;

    /// indexInReordering maps original to new order.
    /// indexInReordering[i] contains the index in the new ordering of the
    /// i-th literal.
    unsigned* indexInReordering;

    /// token is the index of the current recursive literal.
    /// The current recursive literal is the one having range rDeltaNF.
    /// This is only useful in non-linear recursive rules, as they are the only
    /// ones having more than one recursive literal.
    int token;

    /// lastRecPred is the index of the last recursive literal.
    int lastRecPred;

    /// positiveSize is the number of positive literals in the body.
    unsigned positiveSize;

    /// totalSize is the number of all the literals in the body.
    /// This is also the size of the vectors newOrderedBody, recursive,
    /// range, and indexInReordering.
    unsigned totalSize;

    /// i is the index of the last matched literal.
    unsigned i;

    /// sizeToken is the size of deltaL of the current recursive literal.
    unsigned sizeToken;

    /// nrIsolatedPred stores the number of isolated literals in the body.
    /// A literal is isolated if it has no commune variables with other
    /// positive literals of the body. Isolated literals are pushed to the
    /// far right of the body, and are matched last.
    unsigned nrIsolatedPred;

    /// stores the number of ground instances of predicates. psize[i]
    /// contains the number of ground instances of the i-th predicate.
    const unsigned *psize;

    /// stores the number of occurances of predicates in deltaNF.       
    /// deltaSize[i] contains the number of ground instances of the i-th
    /// predicate found in deltaNF.
    const unsigned *pdeltaSize;

    /// stores whether predicates were already considered in grounding.
    /// done[i] is true, iff the i-th predicate is part of a preceeding
    /// component of the non-ground dependency graph, which has already been
    /// processed.
    /// If the i-th predicate is not IDB, done[i] is not used.
    /// In some cases (like the body of a constraint) done will be 0, so we
    /// cannot use it unconditionally.
    const bool *done;

    /// noChanceToBeGround stores whether deltaL of the current literal is {}.
    /// That means that for the current setting of token the body has no
    /// ground instances.
    bool noChanceToBeGrounded;

    /// noMore stores whether L and deltaL of the current literal are both {}.
    /// That means that for the current and further settings of token the body
    /// has no ground instances.
    bool noMore;

private:
    bool isIsolated(const unsigned[],CONJUNCTION::const_iterator,unsigned) const;
    bool isRecursivePred(CONJUNCTION::const_iterator) const;
    bool isPositivelyRecPred(CONJUNCTION::const_iterator) const;
    unsigned getSize(CONJUNCTION::const_iterator) const;
    unsigned getSizeBuiltin(CONJUNCTION::const_iterator,
                            const unsigned,
                            const bool*) const;
    void AdvancedReordering();
    void reorderBuiltins(const bool);

public:
    OrderedBody(const CONJUNCTION&, unsigned*, unsigned*, const bool*, bool, unsigned);
    ~OrderedBody();

    OrderedBody(const OrderedBody &b2) : oldBody(b2.oldBody)
	{
	assert( 0 );
	}

    void operator=(const OrderedBody&)
	{
	assert( 0 );
	}

    void Reorder();
    CONJUNCTION::const_iterator getLiteral(unsigned) const ;   
    CONJUNCTION::const_iterator begin() const;
    CONJUNCTION::const_iterator end() const; 
    CONJUNCTION::const_iterator getPrevious();
    CONJUNCTION::const_iterator getNext();
    INTERPRETATIONS_RANGE getRange() const;
    bool atEnd() const;
    bool noGround() const;
    inline int getIndexLiteral(CONJUNCTION::const_iterator) const;
 
    unsigned getPositiveSize() const
        {
        return positiveSize;
        }

    unsigned getTotalSize() const
        {
        return totalSize;
        }

    void invert(unsigned&, unsigned&) const;
    double weight_param (unsigned& ,unsigned&, const bool*, unsigned&, 
                         double*, double*, unsigned*) const; 
    bool precede(unsigned&, unsigned&, unsigned&, unsigned&,
                 double &, double& , unsigned&, unsigned&) const;  
    unsigned getNumberBound (CONJUNCTION::const_iterator, const bool*)const;
    void moveSimpleBuiltins() const;
    void moveAggregates() const;
    void updateSelectivity(LITERAL&, double*, bool*, 
                           const double*, unsigned&) const;
    void computeDomains(double* , unsigned*);
    };


/** builds an ordered copy of the given body.
 *
 * The literals are ordered on the growing size of the sets of ground atoms
 * that can match the literals. The isolated literals are put to the far
 * right, regardless of the size of their corresponding sets. Built-ins
 * are treated special. Simple built-ins must occur where all their variables
 * have already been bound.
 *
 * The exact behaviour depends on the value of OptionGroundingReorder:
 *
 * 0: Nothing is reordered - built-in predicates order is checked. When an
 *    incorrect order is detected, the program will terminate with an error
 *    message.
 *
 * 1: Normal reordering according to predicate sizes occurs. Built-ins are
 *    considered to have infinite size (which moves them to the right, just
 *    before any isolated predicates).
 *
 * 2: Just like 1, but built-in predicates are moved to the left (by fudging
 *    a size of 1 for them). Then AdvancedReorder() provides some further
 *    optimization. Finally, reorderBuiltins() is called to ensure proper
 *    built-in order.
 */
OrderedBody::OrderedBody(
    const CONJUNCTION &Body,
    unsigned *sizeV,
    unsigned *deltaSizeV,
    const bool *doneV,
    bool      isExitRule,
    unsigned  iteration ) 
    : oldBody(Body), isExit(isExitRule),
      token(-1), i(0), sizeToken(0), 
      psize(sizeV), pdeltaSize(deltaSizeV), done(doneV),
      noChanceToBeGrounded(false), noMore(false)
    {
    positiveSize=oldBody.pos_end()-oldBody.pos_begin();
    totalSize = oldBody.end()-oldBody.begin();

    assert( totalSize > 0 );

    recursive=new bool[totalSize];
    range=new INTERPRETATIONS_RANGE[totalSize];
    unsigned nrOccVar[n_par];
    for( unsigned i=0;i<n_par;i++ )
	nrOccVar[i]=0;
    indexInReordering=new unsigned[totalSize];
    newOrderedBody=new CONJUNCTION::const_iterator[totalSize];
    bool tokenIsSet=false;
    bool checkBuiltinOrder=false;
    int index=-1;
    for( CONJUNCTION::const_iterator pLit = oldBody.begin();
         pLit != oldBody.end();
         pLit++ )
        {
        if( (*pLit).isBuiltin() &&
            ((*pLit).getBuiltinID() >= NSimpleBuiltins) )
            if( MaxInteger == -1 )
                InternalError("#maxint undefined");
	index++;
	if( isExit )
	    {
	    recursive[index]=false;
	    range[index]=rI;
	    }
	else
            {
            if( isPositivelyRecPred(pLit) )
                {
                lastRecPred=index;
                recursive[index]=true;
                if( !tokenIsSet && OptionGroundingSemiNaive )
                    if(iteration)  
                        {
                        tokenIsSet=true;
                        token=index;
                        range[index]=rDeltaNF;
                        }
                    else
                        {
                        tokenIsSet=true;
                        token=index;
                        range[index]=rI_DeltaNF;
                        }
		else
		    range[index]=rI_DeltaNF;
		}
            else
                {
                recursive[index]=false;
                range[index]=rI;
                }
	   
            }   
	if( (*pLit).isRegularAtom() && (*pLit).getParams()
	    && OptionGroundingReorder >= 1 )
	    {
            for( TERMS::const_iterator i  = (*pLit).getParams()->begin();
                 i != (*pLit).getParams()->end();
                 i++ )
                if( (*i).isVar() )
                    nrOccVar[(*i).getVar()]++;
            }
        else if( (*pLit).isAggregate() )
            {
            // If guards are variables update nrOccVar.  
            const TERM &lower = (*pLit).getAggregate().getLowerGuardAsVar();
            const TERM &upper = (*pLit).getAggregate().getUpperGuardAsVar();
            if( lower.isVar() )
                nrOccVar[lower.getVar()]++;
            // If lower and upper guards are equal update nrOccVar only once.
            if( (upper != lower) && upper.isVar() )
                nrOccVar[upper.getVar()]++;
            // Update nrOccVar also for the variables of the auxiliary atom.
            const ATOM &aux = (*pLit).getAggregate().getAuxAtom();
            for( TERMS::const_iterator i = aux.getParams()->begin();
                 i != aux.getParams()->end();
                 i++ )
                if ( (*i).isVar() )
                    nrOccVar[(*i).getVar()]++;
            }
        
        if( GTraceLevel >= 2 )
            cdebug << "range[" << *pLit << "]=" << RangeName[range[index]]
            << endl;
	}

    index=-1;
    nrIsolatedPred=0;
    for( CONJUNCTION::const_iterator pLit = oldBody.pos_begin();
				     pLit != oldBody.pos_end();
				     pLit++ )
	{
	index++;
	unsigned size1=getSize(pLit);
	unsigned k=index-nrIsolatedPred;
	if( size1 == 0 )
	    {
	    if( GTraceLevel >= 2 )
		cdebug << "No chance to be grounded! size(" 
                       << (*pLit).getPredName() << ")=0";
            noChanceToBeGrounded=true;
	    if( (range[index] != rDeltaNF) || (token == lastRecPred) )
		{
		noMore=true;
		if( GTraceLevel >= 2 )
		    cdebug << " ; rule aborted! (noMore:=true)" << endl;
		return;
		}
	    else
		{
		if( GTraceLevel >= 2 )
		    cdebug << " ; maybe the next..." << endl;
		if( OptionGroundingReorder >= 1 )
		    {
		    if( ! isIsolated(nrOccVar,pLit,2) )
			while( k > 0 )
			    {
			    newOrderedBody[k]=newOrderedBody[k-1];
			    indexInReordering[newOrderedBody[k-1]
					     -oldBody.pos_begin()]=k;
			    k--;
			    }
		    else
			{
                        k = positiveSize-1-nrIsolatedPred;
			nrIsolatedPred++;
			if( GTraceLevel >=1 )
                            cdebug << " [" << *pLit << " is isolated.]" << endl;
			}
		    }
		}
	    }
	else
	    if( OptionGroundingReorder >= 1 )
		{
		if( ! isIsolated(nrOccVar,pLit,2) )
		    while( k > 0 )
			{
			if( size1 < getSize(newOrderedBody[k-1]) )
			    {
			    newOrderedBody[k]=newOrderedBody[k-1];
			    indexInReordering[newOrderedBody[k-1]
					     -oldBody.pos_begin()]=k;
			    k--;
			    }
			else
			    break;
			}
		else
                    {
                    k=positiveSize-1-nrIsolatedPred;
                    nrIsolatedPred++;
                    if( GTraceLevel >=1 )
                        cdebug << " [" << *pLit << " is isolated.]" << endl;
                    }
		}
	    else if ((*pLit).isBuiltin())
		{
		checkBuiltinOrder=true;
		}
	newOrderedBody[k]=pLit;
	indexInReordering[index]=k;
	if( token == index )
	    sizeToken = size1;
	}
    
    // Add negative literals at the end of the body.
    int j = positiveSize;
    for( CONJUNCTION::const_iterator pLit = oldBody.neg_begin();
         pLit != oldBody.neg_end();
         pLit++, j++ )
        {       
        newOrderedBody[j] = pLit;
        indexInReordering[pLit-oldBody.neg_begin()+positiveSize] = j;  
        }      
    if (checkBuiltinOrder)
        reorderBuiltins(false);

    if( (GTraceLevel >= 1) && !isExit && OptionGroundingSemiNaive )
	{
	if( token == -1 )
	    cdebug << "   again with token=0" << endl;
	else
	    cdebug << "   again with token="
		   << *(oldBody.pos_begin()+token)
		   << endl;
	}
    if( (GTraceLevel>=2) && !noChanceToBeGrounded
        && (OptionGroundingReorder>=1) )
        {
        cdebug << "rule REORDERED: ";
        for( unsigned j = 0; j < totalSize ; j++ )
            {
            cdebug << *newOrderedBody[j];
            if( j < totalSize-1 )
                cdebug << ", ";
            }
        cdebug << endl;
        }

    if( ! noChanceToBeGrounded )
        {
        if( OptionGroundingReorder >= 2 )
            AdvancedReordering();
        else
            reorderBuiltins(true);
        }
    }

/** destroys an OrderedBody.
 */
OrderedBody::~OrderedBody()
    {
//   oldBody=0;

    if( recursive )
	delete[] recursive;
    if( range )
	delete[] range;
    if( indexInReordering )
	delete[] indexInReordering;
    if( newOrderedBody )
	delete[] newOrderedBody;

    done=0;
    psize=0;
    pdeltaSize=0;
    }

/** checks if a literal has no variables in common with others in this rule.
 * @param nrOccVar the number of occurances of each variable.
 * @param pLit the literal to examine.
 * @param val the maximum number of occurances allowed for literal's variables.
 * @return whether the literal is isolated.
 */
bool OrderedBody::isIsolated(const unsigned nrOccVar[], 
                             CONJUNCTION::const_iterator pLit,
                             unsigned val) const
    {
    if( (*pLit).isRegularAtom() && (*pLit).getParams() )
        {
        for( TERMS::const_iterator i  = (*pLit).getParams()->begin();
                                    i != (*pLit).getParams()->end();
                                    i++ )
	    {
	    if( (*i).isVar() )
		{
		if( nrOccVar[(*i).getVar()] >= val )
		    return false;
		}
	    else
		{
		if( GTraceLevel >= 2 )
		    cdebug << *pLit
			   << " has a costant as parameter and it's not "
			   << "considered isolated..."
			   << endl;
		return false;
		}
	    }
	}
    else
	{
	if( GTraceLevel >= 2 )
	    cdebug << *pLit 
		   << " has no parameter, but it's not considered isolated..." 
		   << endl;
	return false;
	}
    return true;
    }

/** checks if a predicate is recursive.
 * This function depends on the components of NGDG and is valid only for
 * the IDB literals appearing in the subprogram of the current component.
 * @param pLit the literal to examine.
 * @return whether the literal is recursive.
 */
bool OrderedBody::isRecursivePred(CONJUNCTION::const_iterator pLit) const
    {
    if ((*pLit).isEDB() || (*pLit).isBuiltin())
	return false;
    else
	return ( ! done[(*pLit).getIndex()] );
    }

/** checks if a positive predicate is recursive.
 * @param pLit the literal to examine.
 * @return whether the positive literal is recursive.
 */
bool OrderedBody::isPositivelyRecPred(CONJUNCTION::const_iterator pLit) const
   {
   return ( ! (*pLit).isNegative()  &&  isRecursivePred(pLit) ); 
   }

/** computes the number of ground instances of a literal.
 * Ground instances are searched in appropriate databases, according to 
 * interpretations range.
 * If the literal is a builtin its size is computed 
 * by the function getSizeBuiltin.
 * @param pLit the literal to examine.
 * @return the size of the set of ground instances of this literal.
 */
unsigned OrderedBody::getSize(CONJUNCTION::const_iterator pLit) const
    {
    unsigned size1; 
    unsigned index = pLit-oldBody.pos_begin();
    if( (*pLit).isEDB() )
        size1 = psize[(*pLit).getIndex()];
    else
        if( (*pLit).isBuiltin() )
            // Compute the size of this builtin assuming there are no
            // parameters which are already bound. Pass a dummy as the third
            // parameter as the array of bound variables is useless in this
            // case.
            size1 = getSizeBuiltin(pLit, 0, 0);
        else if((*pLit).isAggregate())
            size1 = UINT_MAX;
        else
            {
            if (range[index] == rDeltaNF)
                size1 = pdeltaSize[(*pLit).getIndex()];
            else    
                {
                size1 = psize[(*pLit).getIndex()];
                if (range[index] == rI_DeltaNF)
                    size1 += pdeltaSize[(*pLit).getIndex()];
                }
            }
    return size1;
    }

/** Computes the number of instances of a builtin.
 * The extent of a builtin predicate depends on the kind of builtin,
 * and on the number of parameters which are already bound. This function
 * uses a parameter nBound that indicates the number of already bound
 * parameters (i.e., constants or bound variables) and an array
 * alreadyBound storing the variables that have been already instantiated.
 *
 * #int(X):
 * - nBound == 0: the size is the number of different instances for X,
 *   that is #maxint + 1 (0 included).
 * - nBound == 1: the size is 1.
 *  
 * #succ(X,Y):
 * - nBound == 0: the size is #maxint
 * - nBound == 1 or nBound = 2: the size is 1.
 *  
 * Z=X+Y:
 * - nBound == 0: the size is the number of possible instances for
 *   X and Y such that their sum is a number included
 *   between 0 and #maxint.
 *   Then, for X=0 there are #maxint+1 possible instances for Y
 *         for X=1 there are #maxint possible instances for Y
 *         for X=2 there are #maxint-1 possible instances for Y
 *                 ....
 *         for X=#maxint there is 1 possible instance for Y
 *   So, the size of Z=X+Y is
 *   1 + 2 + 3 + ... + #maxint-1 + #maxint + #maxint+1
 *   which is the partial sum of a famous series and equals
 *   (#maxint+1)*(#maxint+2)/2
 * - nBound == 1: If the bound term is Z and it is a constant "number"
 *   the size is (number+1)*(number+2)/2 while, if it is a bound variable,
 *   the size is (#maxint+1)*(#maxint+2)/2.
 *   If the bound term is X or Y and it is a constant "number" the size is
 *   #maxint-number+1, while it is #maxint if the term is a bound variable.
 *
 * Z=X*Y:
 * - nBound == 0: : the size is the number of possible instances for
 *   X and Y such that their multiplication is a number included
 *   between 0 and #maxint.
 *   Then, for X=0 there are #maxint+1 possible instances for Y
 *         for X=1 there are #maxint+1 possible instances for Y
 *         for X=2 there are #maxint/2 +1 possible instances for Y
 *         for X=3 there are #maxint/3 +1 possible instances for Y
 *                ....
 *         for X=#maxint there are 1 + 1 possible instance for Y
 *   With some tricks the sum
 *   (#maxint+1) + (#maxint+1) + (#maxint/2 + 1)+ (#maxint/3 +1) + ... +(1+1)
 *   can be rewritten as
 *   #maxint + #maxint/2 + #maxint/3 + ... + 1 + (2*#maxint+1)
 *   We can consider #maxint + #maxint/2 + #maxint/3 + ... + 1
 *   as the partial sum of the series
 *   #maxint * Sum (1/n) with n=0...#maxint
 *   This sum is not known but an estimation can be obtained using
 *   the logarithm.
 *   In particular, a quite good estimation for the size of Z=X*Y is
 *   #maxint*log(#maxint+1) + (2*#maxint + 1)
 * - nBound == 1: if the bound term is Z and it is a constant "number"
 *   the size is #number*log(#number+1) + (2*#number + 1) while, if it is
 *   a bound variable, the size is #maxint*log(#maxint+1) + (2*#maxint + 1).
 *   If the bound term is X or Y and it is a constant "number" the size is
 *   #maxint/number+1 while it is #maxint if the term is a bound variable.
 *
 * Simple builtins always have size 1. Their size is not important for the
 * ordering as they are treated like negative literals, i.e. placed as soon
 * as they are bound.
 *
 * @param pLit the builtin to examine
 * @param nBound the number of bound parameters of pLit
 * @param alreadyBound the array storing the variables that have been
 *        already instantiated.
 */
unsigned OrderedBody::getSizeBuiltin(
    CONJUNCTION::const_iterator pLit,
    const unsigned nBound,
    const bool *alreadyBound ) const
    {
    assert( (*pLit).isBuiltin() );
    unsigned sizeBuiltin;
    if( (OptionGroundingReorder>=2) )
        {
        switch( (*pLit).getBuiltinID() )
            {
            case BuiltinInt:
                if( nBound == 0 )
                    sizeBuiltin = static_cast<unsigned>(MaxInteger+1);
                else
                    sizeBuiltin = 1;
                break;
            case BuiltinSucc:
                if( nBound == 0 )
                    sizeBuiltin = static_cast<unsigned>(MaxInteger);
                else
                    sizeBuiltin = 1;
                break;
            case BuiltinAddition:
                if( nBound == 0 )
                    sizeBuiltin = computeSizeAddition(MaxInteger);
                else if( nBound == 1 )
                    {
                    // Traverse the parameters of the builtin until we find
                    // the term which is already bound.
                    TERMS::const_iterator j = (*pLit).getParams()->begin();
                    while( ! j->isInt()
                           && ! ( j->isVar() && alreadyBound[j->getVar()] ) )
                        j++;
                    if( (*j).isInt() )
                        {
                        int num = (*j).getInt();
                        // Check if the  bound term of Z=X+Y is Z
                        // i.e. the last one in the list of parameters
                        if( j == (*pLit).getParams()->end()-1 )
                            sizeBuiltin = computeSizeAddition(num);
                        else
                            // the bound term is X or Y
                            sizeBuiltin = static_cast <unsigned>
                                          (MaxInteger-num+1);
                        }
                    else
                        if( j == (*pLit).getParams()->end()-1 )
                            sizeBuiltin = computeSizeAddition(MaxInteger);
                        else
                            sizeBuiltin = static_cast <unsigned> (MaxInteger+1);
                    }
                else
                    sizeBuiltin = 1;
                break;
            case BuiltinMultiplication:
                if( nBound == 0 )
                    sizeBuiltin = computeSizeMultiplication(MaxInteger);
                else if( nBound == 1)
                    {
                    // Traverse the parameters of the builtin until we find
                    // the term which is already bound.
                    TERMS::const_iterator j = (*pLit).getParams()->begin();
                    while( ! j->isInt()
                           && ! ( j->isVar() && alreadyBound[j->getVar()] ) )
                        j++;
                    if( (*j).isInt() )
                        {
                        int num = (*j).getInt();
                        // Check if the  bound term of Z=X*Y is Z
                        // i.e. the last one in the list of parameters
                        if( j == (*pLit).getParams()->end()-1 )
                            sizeBuiltin = computeSizeMultiplication(num);
                        else
                            // the bound term is X or Y
                            sizeBuiltin = static_cast <unsigned>(MaxInteger)/
                                          static_cast <unsigned>(num)+1;
                        }
                    else
                        if( j == (*pLit).getParams()->end()-1 )
                            sizeBuiltin = computeSizeMultiplication(MaxInteger);
                        else
                            sizeBuiltin = static_cast <unsigned> (MaxInteger+1);
                    }
                else
                    sizeBuiltin = 1;
                break;
            default:
                sizeBuiltin = 1;
            }
        }
    else
        // Order built-ins just before isolated predicates.
        sizeBuiltin = UINT_MAX;
    return sizeBuiltin;
    }

/** computes the size of the bultin addition according to the
 *  possible number of value that the parameters of the builtin
 *  can assume. If this number is too big returns UINT_MAX.
 * @param num the number of possible values for the builtins parameters
*/
static unsigned computeSizeAddition(const int num)
    {
    assert( num != -1 );
    if( num < floor(sqrt(static_cast<double> (UINT_MAX))) )
        return static_cast<unsigned> (((num+1)*(num+2))/2);
    return UINT_MAX;
    }

/** computes the size of the bultin multiplication according to the
 *  possible number of values that the parameters of the builtin
 *  can assume. If this number is too big returns UINT_MAX.
 * @param num the number of possible values for the builtins parameters
*/
static unsigned computeSizeMultiplication(const int num)
    {
    assert( num != -1 );
    if( num < floor(sqrt(static_cast<double> (UINT_MAX))))
        return static_cast<unsigned> (2*num+1 + num*ceil(log(num+1.0)));
    return UINT_MAX;
    }

/** changes the token and reorders the positive literals in the body.
 * The literals are ordered on the growing size of the sets of ground atoms
 * that can match the literals. The isolated literals are put to the far
 * right, regardless of the size of their corresponding sets.
 */  
void OrderedBody::Reorder()
    {
    assert( !isExit );
    assert( OptionGroundingSemiNaive );
    assert( token < lastRecPred );
    assert( !atEnd() );
    noChanceToBeGrounded=false;

    CONJUNCTION::const_iterator pLit=oldBody.pos_begin()+token;
    range[token] = rI;
    unsigned size1=getSize(pLit);
    bool somethingChanged=false;
    if( size1 == 0 )
	{
	if( GTraceLevel >= 2 )
	    cdebug << "No chance to be grounded! size(" 
		   << (*pLit).getPredName() << " = ex-token)=0" 
		   << " ; rule aborted! (noMore:=true)" << endl;
	noChanceToBeGrounded=true;
	noMore=true;
	return;
	}
    unsigned k;
    if( OptionGroundingReorder >= 1 )
        {
        k=indexInReordering[token];
        if( k < positiveSize-nrIsolatedPred )
            {
	    if( size1 < sizeToken )
                while( k > 0 )
		    {
		    if( size1 < getSize(newOrderedBody[k-1]) )
			{
			newOrderedBody[k]=newOrderedBody[k-1];
			indexInReordering[newOrderedBody[k-1]
					 -oldBody.pos_begin()]=k;
			k--;
			}
		    else
			break;
		    }
	    else
		if( size1 > sizeToken )
                    while( k < (positiveSize-1-nrIsolatedPred) )
			{
			if( size1 > getSize(newOrderedBody[k+1]) )
			    {
			    newOrderedBody[k]=newOrderedBody[k+1];
			    indexInReordering[newOrderedBody[k+1]
					     -oldBody.pos_begin()]=k;
			    k++;
			    }
			else
			    break;
			}

	    if( k != indexInReordering[token] )
		{
		somethingChanged=true;
		newOrderedBody[k]=pLit;
		indexInReordering[token]=k;
		}
	    }
	}
    else
	k=0;			// to pacify egcs, which warns otherwise
    pLit++;
    
    int index=token+1;
    while( true )
        {
        assert( pLit !=  oldBody.end() );
        if( recursive[index] )
            {
            token=index;
            range[index]=rDeltaNF;
            break;
            }
        else
            {
            assert(range[index] == rI);
            pLit++;
            index++;
            }
        }   

    size1=getSize(pLit);
    //k=indexInReordering[index];
    if( size1 == 0 )
	{
	if( GTraceLevel >= 2 )
	    cdebug << "No chance to be grounded! size(" 
                   << (*pLit).getPredName() << ")=0";
	noChanceToBeGrounded=true;
	if( token == lastRecPred )
	    {
	    if( GTraceLevel >= 2 )
		cdebug << endl;
	    return;
	    }
	else
	    {
	    if( GTraceLevel >= 2 )
		cdebug << " ; maybe the next..." << endl;
	    if( OptionGroundingReorder >= 1 )
		{
		k=indexInReordering[index];
                if( k < positiveSize-nrIsolatedPred )
		    {
		    while( k > 0 )
			{
			newOrderedBody[k]=newOrderedBody[k-1];
			indexInReordering[newOrderedBody[k-1]
					 -oldBody.pos_begin()]=k;
			k--;
			}
		    }
		}
	    }
	}
    else
	if( OptionGroundingReorder >= 1 )
	    {
	    k=indexInReordering[index];
            if( k < positiveSize-nrIsolatedPred )
		{
		while( k > 0 )
		    {
		    if( size1 < getSize(newOrderedBody[k-1]) )
			{
			newOrderedBody[k]=newOrderedBody[k-1];
			indexInReordering[newOrderedBody[k-1]
					 -oldBody.pos_begin()]=k;
			k--;
			}
		    else
			break;
		    }
		}
	    }

    if( (OptionGroundingReorder >= 1) && (k != indexInReordering[index]) )
	{
	somethingChanged=true;
	newOrderedBody[k]=pLit;
	indexInReordering[index]=k;
	}
    sizeToken = size1;
    i=0;

    if( GTraceLevel >= 1 )
	cdebug <<"   again with token=" 
	       << *(oldBody.pos_begin()+token)<<endl;
    if( (GTraceLevel >= 2) && !noChanceToBeGrounded
	&& (OptionGroundingReorder>=1) )
	{
	cdebug << "Positive rule part REORDERED: ";
        for(unsigned j=0; j < positiveSize ; j++)
	    {
	    cdebug << *newOrderedBody[j];
            if( j < positiveSize-1 )
                cdebug << ", ";
            }
        cdebug << endl;
        }
    // FIXME: We should understand why the check on the variable
    // somethingChanged was there. At a glance it looks useless. 
    if( ! noChanceToBeGrounded )
    // && somethingChanged )
        {
        if( OptionGroundingReorder >= 2 )
            AdvancedReordering();
        else
            reorderBuiltins(true);
        }
    }

/** provides access to the first literal in the newly ordered body.
 * @return a constant iterator pointing at the start of newOrderedBody.
 */
CONJUNCTION::const_iterator OrderedBody::begin() const
    {
    if( OptionGroundingReorder == 0 )
	return (oldBody.pos_begin());
    else
	return newOrderedBody[0];
    }

/** provides access to the end of the newly ordered body.
 * @return a constant iterator pointing at the end of newOrderedBody.
 */
CONJUNCTION::const_iterator OrderedBody::end() const
    {
    return oldBody.end();
    }

/** provides access to the literal before the current one (in new order).
 * @return a constant iterator pointing at the previous literal (in new order).
 */
CONJUNCTION::const_iterator OrderedBody::getPrevious()
    {
    assert( i > 0 );
    i--;
    if( OptionGroundingReorder == 0 )
	return (oldBody.pos_begin()+i);
    else
	return newOrderedBody[i];
    }

/** provides access to the literal after the current one (in new order).
 * @return a constant iterator pointing at the next literal (in new order).
 */
CONJUNCTION::const_iterator OrderedBody::getNext()
    {
    i++;
    if( i == totalSize )
	return (oldBody.end());
    else
	{
	if( OptionGroundingReorder == 0 )
	    return (oldBody.pos_begin()+i);
	else
	    return newOrderedBody[i];
	}
    }

/** provides access to the interpretations range of the current literal.
 * @return the range of the current literal.
 */
INTERPRETATIONS_RANGE OrderedBody::getRange() const
    {
    assert( i < totalSize );
    if( OptionGroundingReorder == 0 )
	return range[i];
    else
	return range[newOrderedBody[i]-oldBody.pos_begin()];
    }

/** checks if there are no more literals.
 * @return whether the end of the body is reached.
 */
bool OrderedBody::atEnd() const
    {
    return( noMore || (token == lastRecPred) || !OptionGroundingSemiNaive );
    }

/** checks if there is no chance that the body can be grounded in this setting.
 * @return whether no grounding is possible with the current setting of token.
 */
bool OrderedBody::noGround() const
    {
    return noChanceToBeGrounded;
    }

/** Return an integer representing the index in the ordered body
 *  of the literal pointed by pLit.
 */
inline int OrderedBody::getIndexLiteral(
    CONJUNCTION::const_iterator pLit) const
    {
    return indexInReordering[pLit-oldBody.begin()];
    }

/** Implement a new ordering criterion, the `Combined criterion'.
 * For this criterion we exploit, as additional statistics,
 * the size of the (active) domains for the variables occurring in the rule.
 * We estimate this number, denoted by dom(X), by 
 * max(V(X,A1),...,V(X,An)) where A1,...,An are in B^+_r and V(X,Ai) 
 * denotes the number of distinct values for X over rel(Ai).
 * The combined criterion takes as the i-th atom in the ordered body the atom 
 * A in (B^+_r - B_{i-1}) that minimizes the selectivity index
 * sel_c(A) = sel_s(A) * sel_b(A), where
 *            T(B_{i-1} semijoin A)
 * sel_s(A)=  ---------------------
 *           dom(X1) * ... * dom(Xn) 
 * where X1,...,Xn are the variables that A has in common with some other
 * atom occurring in $B^+_r$ and
 * 
 *           {V(Y1,A)}*...*{V(Yn,A)}
 * sel_b(A)= -----------------------
 *         {dom(Y1)^2}*...*dom(Yn)^2}
 * where  Y1,...,Yn are the variables of A already bound.
 */

void OrderedBody::AdvancedReordering()
    {
    if( ( totalSize-nrIsolatedPred ) < 2)
        { 
        if ( GTraceLevel >= 2 )
            cdebug << "(size - nrIsolatedPred < 2) => AdvancedReordering() "
                   << "exits whithout doing anything!" << endl;
        return;
        }
   
    double selectivities[n_par], domains[n_par];;
    bool alreadyBound[n_par];
    unsigned occurrencesInBody[n_par];
    for (unsigned i = 0; i < n_par; i++)
	{
	alreadyBound[i] = false;
	domains[i] = 0;
	occurrencesInBody[i] = 0;
	}
    unsigned index = 0, cardLiteralPrev = 0;
    unsigned index1, index_min;
    unsigned boundLit1,cardLit1;
    unsigned boundLit2,cardLit2;
    double valuemin,value;
    CONJUNCTION::const_iterator min = newOrderedBody[index];
    // Move the Simple Builtins literals at the end of the body.
    moveSimpleBuiltins();
    // Compute the domains of the variables. 
    computeDomains(domains, occurrencesInBody);
    // Initialise the selectivities with the domains.
    for (unsigned i = 0; i < n_par; i++)
        selectivities[i] = domains[i];
    
    while( index < totalSize-nrIsolatedPred )
        {
        // Compute the selectivity index for the literal of index "index". 
        valuemin = weight_param(cardLit1, boundLit1, alreadyBound,
                                index, selectivities, domains,
                                occurrencesInBody); 
        index1 = index+1;
        index_min = index;
        while( index1 < totalSize-nrIsolatedPred )
            {  
            // Then, examine all the following literals in the body
            // but the simple builtins.
            if (!((*newOrderedBody[index1]).isBuiltin()) ||  
                ((*newOrderedBody[index1]).getBuiltinID() 
                 >= NSimpleBuiltins))  
                {
                // computes the selectivity index for the literal of position
                // "index1"
                value = weight_param(cardLit2, boundLit2, alreadyBound, 
                                     index1, selectivities, domains,
                                     occurrencesInBody); 
                // compare these values and choose the literal 
                // with smallest value
                if (!(precede(boundLit1, cardLit1, boundLit2, cardLit2, value,
                              valuemin, index, index1))) 
                    {
                    min = newOrderedBody[index1];
                    index_min = index1;
                    boundLit1 = boundLit2;
                    cardLit1 = cardLit2; 
                    valuemin = value;
                    }
                }
            index1++; 
            }
        // If the current literal is not the one with smallest value,
        // we swap it with that. 
        if (index_min != index)
            { 
            CONJUNCTION::const_iterator tmp = newOrderedBody[index];
            newOrderedBody[index_min] = newOrderedBody[index];
            newOrderedBody[index] = min;
            indexInReordering[tmp-oldBody.pos_begin()] = index_min; 
            indexInReordering[min-oldBody.pos_begin()] = index;
            }   
        cardLiteralPrev = getSize(newOrderedBody[index]);
        LITERAL lit = *newOrderedBody[index];
        
        if (GTraceLevel >= 2) 
            cdebug<<"-----> LITERAL "<< lit 
                  << " placed in the ORDERED BODY "<<endl; 
        // After we placed a literal L, we have to update the selectivities
        // of the variables of L.
        if( lit.isRegularAtom() && lit.getParams() )
            updateSelectivity(lit, selectivities, alreadyBound,
                              domains, cardLiteralPrev);
        index++;  
        } 
    if( (GTraceLevel >= 2) && !noChanceToBeGrounded )  
        {
        cdebug << "AdvancedReordering() results in: ";
        for(unsigned j=0; j <  totalSize; j++)   
            {
            cdebug << *newOrderedBody[j];  
            if( j <  totalSize-1 )
                cdebug << ", ";
            }
        cdebug << endl;
        }
    }
    
/* moves all built-in predicates to evaluable positions.
 * They are moved as far back as needed, but not farther.
 * @param doit if true, do as stated; otherwise exit with an error whenever
 *	an unevaluable predicate is encountered.
 */
void OrderedBody::reorderBuiltins(const bool doit)
    {
    const int Unbound=-1, UnboundNeeded=-2;

    // Initialise all variables as being unbound.
    int alreadyBound[n_par];
    for (unsigned i=0; i<n_par; i++)
	alreadyBound[i]=Unbound;

    // Iterate over the nonisolated atoms.
    unsigned index=0;
    while ( index < totalSize-nrIsolatedPred )
	{
        // Let p be the currently considered atom.
        CONJUNCTION::const_iterator p=newOrderedBody[index];
        
        if( p->isBuiltin() )
	    {
            // Find the number of unbound variables in p.
	    unsigned unbound=0;
            for( TERMS::const_iterator i=p->getParams()->begin();
                 i != p->getParams()->end();
                 i++ )
		if ((*i).isVar() && alreadyBound[(*i).getVar()]==Unbound)
		    {
		    unbound++;
		    alreadyBound[(*i).getVar()]=UnboundNeeded;
		    }

            // target is 0 for comparison built-ins and 1 for others.
            const unsigned target=p->isComplexBuiltin() ? 1 : 0;

            // If there are more unbound literals than the target
            // value, reordering might be necessary.
	    if (unbound>target)
		{
		if (doit)
		    {
                    // Move the builtin to a place where its unbound
                    // variables become bound enough, or, if this is
                    // not possible, to the end, immediately before
                    // the isolated predicates. Place complex builtins
                    // before simple builtins.

                    // FIXME: This should be made more readable
                    // (eliminate side effect in condition, remove
                    // goto,...).

		    unsigned k;
                    for( k=index;
                         unbound > target && ++k < totalSize-nrIsolatedPred; )
			{
                        // Each literal's variables are tested and it
                        // is recorded when they can bind the
                        // builtin's unbound variables.
			for (TERMS::const_iterator i=(*newOrderedBody[k]).
				 getParams()->begin();
			     i!=(*newOrderedBody[k]).getParams()->end();
			     i++) 
			    if ((*i).isVar()
				&& alreadyBound[(*i).getVar()]==UnboundNeeded)
                                // This variable can bind one of the
                                // builtin's unbound variables.
                                if( (*newOrderedBody[k]).isBuiltin()
                                    && p->isComplexBuiltin() )
                                    {
                                    // If the current literal is a
                                    // builtin as well and the literal
                                    // to be reordered is a complex
                                    // builtin (and they share at
                                    // least one variable), insert the
                                    // to be reordered literal before
                                    // the current literal.
                                    goto endBackwardMove;
                                    }
                                // The following condition is true if
                                // *newOrderedBody[k] is no builtin or
                                // a complex builtin. isComplexBuiltin()
                                // should only be called on builtins,
                                // so the condition relies on the fact
                                // that the second expression in the
                                // or is not evaluated if the first
                                // one is true.
                                else if( ( ! (*newOrderedBody[k]).isBuiltin() )
                                         || (*newOrderedBody[k]).isComplexBuiltin() )
                                    {
                                    // The following is equivalent to
                                    // the condition in the if above.
                                    assert(((*newOrderedBody[k]).isBuiltin() 
                                            && (*newOrderedBody[k]).isComplexBuiltin())
                                           || !(*newOrderedBody[k]).isBuiltin());
                                    unbound--;
                                    alreadyBound[(*i).getVar()]=Unbound;
                                    }

                        // Shift to the left.
			newOrderedBody[k-1]=newOrderedBody[k];
			indexInReordering[newOrderedBody[k]
					 -oldBody.pos_begin()]=k-1;
			}

endBackwardMove:
                    if (p->isComplexBuiltin())
			{
			if (unbound>target)
			    k--;
			}
		    else
			assert(unbound==0); // due to built-in safety

                    // Move the literal to be reordered (if necessary).
		    if (k!=index)
			{
                        newOrderedBody[k]=p;
                        indexInReordering[p-oldBody.pos_begin()]=k;
			index--;

                        if (GTraceLevel>=2)
                            {
                            cdebug << "reorderBuiltins() intermediate reordering: ";
                            for(unsigned j=0; j < totalSize ; j++)
                                {
                                cdebug << *newOrderedBody[j];
                                if( j < totalSize-1 )
                                    cdebug << ", ";
                                }
                            cdebug << endl; 
                            }

			}
		    }
		else
		    {
                    if( ! p->isComplexBuiltin() )
			cerr << "Simple built-in predicate used on unbound "
			    "parameters.";
		    else
			unbound=0;
		    if (unbound)
			{
                        cerr << endl
                             << "Could not compensate due to -OGo0 "
                                "- giving up." << endl;
			exit(1);
			}
		    }
		}
	    }
	else
            {
            // p is not a built-in.
            if( p->isRegularAtom() && p->getParams() )
                {
                for( TERMS::const_iterator i=p->getParams()->begin();
                     i != p->getParams()->end();
                     i++ )
		    if ((*i).isVar())
			alreadyBound[(*i).getVar()]=1;
		}
            }

	index++;
	}

    if (GTraceLevel>=2)
	{
	cdebug << "reorderBuiltins() results in: ";
        for(unsigned j=0; j < totalSize ; j++)
	    {
            cdebug << *newOrderedBody[j];
            if( j < totalSize-1 )
                cdebug << ", ";
            }
        cdebug << endl; 
        }
    }

// Invert the position of two literals
// @params firstLit and secondLit the positions of two literals.
void  OrderedBody::invert(unsigned& firstLit, unsigned& secondLit) const
    {
    CONJUNCTION::const_iterator tmp = newOrderedBody[firstLit];
    CONJUNCTION::const_iterator tmp1 = newOrderedBody[secondLit];
    newOrderedBody[secondLit] = newOrderedBody[firstLit];
    newOrderedBody[firstLit] = tmp1;
    indexInReordering[tmp1-oldBody.pos_begin()] = firstLit; 
    indexInReordering[tmp-oldBody.pos_begin()] = secondLit; 
    }

/** Provides the selectivity index for the atom at index `indexcurrent'
 * according to the criterion described in the comment to 
 * AdvancedReordering().
 * @param card, the cardinality for the current atom.
 * @param bound, the number of bound variables for the current atom.
 * @param alreadyBound, the vector of the bound variables.
 * @param indexcurrent the index of the atom to evaluate.
 * @param selectivities, the vector of the selectivities previously
 *  computed.
 * @param domains, the domains of the variables.
 * @param occurrencesInBody stores for each variable, the number of occurrences
 * in the body.
 */
double OrderedBody::weight_param( unsigned& card, 
                                  unsigned& bound,
                                  const bool *alreadyBound,
                                  unsigned& indexcurrent, 
                                  double* selectivities,
                                  double* domains,
                                  unsigned* occurrencesInBody) const 
    { 
    CONJUNCTION::const_iterator plit = (newOrderedBody[indexcurrent]);
    // Aggregates have to be placed at the end of the body. So, the 
    // assigned weight is MAX_MAXINT.  
    if( (*plit).isAggregate() )
        return MAX_MAXINT;
    // Compute the number of bound variables of *plit
    bound = getNumberBound(plit, alreadyBound);
    // Compute the cardinality of *plit.
    // If it's a builtin use the function getSizeBuiltin
    // wich computes the size taking in consideration the type
    // of builtin and how many variables are bound.
    // Otherwise use the function getSize().
    if ( (*plit).isBuiltin() )
        card = getSizeBuiltin(plit, bound, alreadyBound);
    else
        card = getSize(newOrderedBody[indexcurrent]);
    unsigned arity = (*plit).getArity();
    double value = 0;
    
    // Place the bound literals as soon as possible.  
    // Assign to the negative bound literals value -1 and -2 to the 
    // positives so that positive literals precede the negatives 
    // ones in the ordering.
    if (bound == arity) 
        if((*plit).isNegative())
            value = -1;
        else
            value=-2;
    else
        // If the literal is a builtin and it has less than 2 bound 
        // variables, assign it the value 0 so that it's placed  before 
        // each other unbound literal. Otherwise assign it a very 
        // big value to avoid to choose it.
        if ((*plit).isBuiltin())
            if((bound == 2) 
               || (bound == 1) && (*plit).getBuiltinID() == BuiltinSucc)
                value = 0;
            else
                // Unbound builtins and negative literals should be
                // placed at the end of the ordered body and the unbound
                // builtins should preceede the unbound negative literals.
                // So, assign to the builtins the value MAX_MAXINT-1 ...
                value = MAX_MAXINT-1;
        // ... and to the unbound negative literals the value MAX_MAXINT.
        else if ((*plit).isNegative())
            value=MAX_MAXINT;
        else
            { 
            // otherwise calculate the selectivity index
            value = card;
            double currSel,sel;
            if( (*plit).isRegularAtom() && (*plit).getParams() )
                { 
                if (GTraceLevel >= 2)
                    cdebug<<"cardinality of "<<*plit<<": "<<card<<endl;
                
                const TERMS &params = *((*plit).getParams());
               
                for( unsigned i = 0; i < params.size(); i++ )
                    {
                    if ( params[i].isVar() ) 
                        {
                        unsigned j = params[i].getVar();
                        if (occurrencesInBody[j] > 1)
                            {  
                            currSel = selectivities[j];
                            value = 
                                value * currSel / (domains[j]*domains[j]);
                            if (GTraceLevel >= 2)
                                cdebug<< " the previous selectivity of "
                                      << " the variable "<<(i)
                                      << " is "<< currSel<< endl;      
                            }
                        if ( alreadyBound[j] )
                            {
                            sel = 
                                dictionary->getSelectivity(*plit,done,i,card);
                            value = value * sel / (domains[j] * domains[j]); 
                            if ( GTraceLevel >= 2 )
                                cdebug<< " the selectivity of "
                                      <<params[i]  
                                      << " w.r.t the atom " 
                                      << *plit << " is " << sel <<endl;
                            }
                        }
                    }
                }
            
            }  
    if (GTraceLevel>=2)
         cdebug<< " the value assigned to the literal "<<*plit<< " is " 
               << value <<endl;
     return value;
    }

/** compute the number of the bound variables of a literal. 
  * @param pLit the literal to examine.
  * @param alreadyBound the vector of the bound variables
  */
unsigned OrderedBody::getNumberBound ( CONJUNCTION::const_iterator pLit,
                                       const bool *alreadyBound) const
    { 
    unsigned nbound=0;
    if( (*pLit).isRegularAtom() && (*pLit).getParams() )
        {
        for( TERMS::const_iterator i  = (*pLit).getParams()->begin();
             i != (*pLit).getParams()->end();
             i++ )
            if(!(*i).isVar() || (alreadyBound[(*i).getVar()]) )    
               nbound++; 
        
        if( GTraceLevel >= 2 )
            cdebug << *pLit 
                   << " has " << nbound << " bound variables" 
                   << endl;
        }
    else
        if( GTraceLevel >= 2 )
            cdebug << *pLit 
                   << " has no params " 
                   << endl;
    return nbound;
    }

// Given the statistics of two atoms, A1 and A2, returns true if A1 
// should precede A2 in the ordered body.
// @param boundAtom1 the number of bound variables for the first atom.
// @param cardAtom1 the cardinality of the first atom.
// @param boundAtom2 the number of bound variables for the second atom. 
// @param cardAtom2 the cardinality of the second atom.
// @param valueAtom1 the value assigned to the second atom.
// @param valueAtom2 the value assigned to the first atom.
// @param indexAtom2 the index of the second atom.
bool OrderedBody::precede( unsigned& boundAtom1,
                           unsigned& cardAtom1,
                           unsigned& boundAtom2, 
                           unsigned& cardAtom2,
                           double& valueAtom2,
                           double& valueAtom1,
                           unsigned& indexAtom1,
                           unsigned& indexAtom2) const 
    {    
    CONJUNCTION::const_iterator pAt1 = newOrderedBody[indexAtom1];
    CONJUNCTION::const_iterator pAt2 = newOrderedBody[indexAtom2];
    if ( valueAtom1 < valueAtom2 ) 
        return (true);
    else 
        // If two atoms have the same selectivity index, we choose that 
        // with smallest cardinality.
        if ((valueAtom1 == valueAtom2) && (cardAtom1 < cardAtom2)) 
            return (true);
        else
            // if the atoms have same value and same cardinality, we prefer
            // that having more free variables.
            {
            unsigned freeAtom1 = (*pAt1).getArity()-boundAtom1;
            unsigned freeAtom2 = (*pAt2).getArity()-boundAtom2;
            if ((valueAtom1 == valueAtom2) && (cardAtom1 == cardAtom2) 
                && (freeAtom2 < freeAtom1)) 
                return (true);
            }
    return (false); 
    }   

// Move the simple builtins at the end of the body.
void OrderedBody::moveSimpleBuiltins() const 
    {
    unsigned end = totalSize - nrIsolatedPred - 1;
    unsigned in = end;
    do  
        if ((*newOrderedBody[in]).isBuiltin() && 
            (*newOrderedBody[in]).getBuiltinID() < NSimpleBuiltins)
            {
            if( in != end)  
                invert(end,in);
            end--;
            }  
    while( in-- > 0 ); 
    
    if (GTraceLevel >= 2)
        {
        cdebug << "Simple Builtins are moved at the end of the body"
               <<endl;
        for(unsigned j = 0; j < totalSize; j++)   
            {
            cdebug << *newOrderedBody[j];  
            if( j <  totalSize-1 )
                cdebug << ", ";
            }
        }
    }

// After the choice of a literal L, we update the selectivity of the variables
// of L as follows: V(X,Bi)=V(X,Bi-1)*V(X,A)/dom(X).
// If X at step i is unbound, V(X,Bi-1) = dom(X) (as we initialise the vector
// selectivities with the domains values), so V(X,Bi)=V(X,A).
void OrderedBody::updateSelectivity(LITERAL& lit, double* selectivities,
                                    bool* alreadyBound, const double* domains,
                                    unsigned& cardLiteralPrev) const
    {
    double sel;
    const TERMS &params = *(lit.getParams());
    for( unsigned j  = 0; j < params.size(); j++ )
        if (params[j].isVar()) 
            { 
            unsigned var = params[j].getVar();
            alreadyBound[var] = true;
            if(lit.isBuiltin())
                sel = 1;
            else
                sel = dictionary->getSelectivity(lit,done,
                                                 j,cardLiteralPrev);
            selectivities[var] = 
                sel * selectivities[var]/domains[var];
            }
    }

// Computes the domains of all the variables in the body.
// domain(X) is estimated by max(V(X,A1),...,V(X,An)) where A1,...,Ar are in
// B^+(r) and V(X,Ai) denotes the number of distinct values for X w.r.t the 
// atom Ai.
// Therefore, count for each variable the number of times that appears
// in the body and store it in the vector occurrencesInBody.
void OrderedBody::computeDomains(
    double *domains,
    unsigned *occurrencesInBody )
    {
    // Scan just the positive literals.
    for(unsigned i = 0; i < positiveSize ; i++)
        {
        const LITERAL &currLit = *newOrderedBody[i];
        const unsigned card = getSize(newOrderedBody[i]);

        if( currLit.isRegularAtom() && currLit.getParams() )
            {
            const TERMS &params = *(currLit.getParams());
            for( unsigned j  = 0; j < params.size(); j++ )
                if ( params[j].isVar() )
                    {
                    double dom;
                    occurrencesInBody[params[j].getVar()]++;
                    if ( currLit.isBuiltin() )
                        dom = MaxInteger;
                    else
                        dom = 
                            dictionary->getSelectivity(currLit,done,j,card);

                    if ( domains[params[j].getVar()] < dom )
                        domains[params[j].getVar()] = dom;
                    
                    if ( GTraceLevel >= 2 ) 
                        cdebug<<" The domain of the variable "
                              <<params[j].getVar() <<" is " 
                              <<domains[params[j].getVar()]
                              <<endl;
                    }
            }
        }
    }



/** provides access to the i-th literal according to the new order.
 * @return an iterator pointing to that literal.
 */ 
CONJUNCTION::const_iterator OrderedBody::getLiteral(unsigned i) const 
    {
    assert( i < totalSize );
    
    if( OptionGroundingReorder == 0 )
        return oldBody.begin()+i;
    else
        return newOrderedBody[i];
    }

#endif

// Local Variables:
// mode: c++
// c-file-style: "dl"
// End:
