//////////////////////////////////////////////////////////////////////////////
// generate.h

#ifndef GENERATE_H
#define GENERATE_H

#include <deque>
#include "timer.h"
#include "indent.h"
#include "generate-counters.h"

bool sameComponentInHead( const DEPGRAPH<GATOM,GCONSTRAINTS,GPROGRAM> &, 
                          const GRULE &, const GATOM &);

//////////////////////////////////////////////////////////////////////////////

typedef pair<TruthValue,GATOM> QE;
typedef unsigned COMPONENT_INDEX;

//////////////////////////////////////////////////////////////////////////////
class QUEUE : private deque< QE >
//
// This queue structure is used by our linear time propagation machinery.
// GATOMs which we want to assign some TruthValue are pushed in that queue
// and then popped and processed one after the other.
//
// Each element in the queue (QE) thus consists of a GATOM and a TruthValue.
//
    {
public:
    size_t size() const
        {
        return deque< QE >::size();
        }

    bool empty() const
        {
        return size() == 0;
        }

    const QE& front() const
        {
        return deque< QE >::front();
        }


    void push(TruthValue tv, const GATOM& a)
        {
        push_back(QE(tv,a));
        }

    void pop()
        {
        pop_front();
        }

    void clear()
        {
        deque<QE>::clear();
        }
    };


//////////////////////////////////////////////////////////////////////////////
template<class T>
class STACK
//
// This structure is used by our linear time propagation machinery.
//
// Whenever we change one of those many counters there, we save its
// address and its old value on this stack so that we can easily
// restore the counter during backtracking.
//
// A special kind of element on this stack, where the address is 0 and
// the value denotes a (recursion) level, is used to group changes.
//
    {
public:
    class ELEMENT
        {
    public:
        T *address;
        T  value;

        ELEMENT(
            T* address2 = 0,
            T value2 = 0 ) 
            : address(address2), value(value2)
            { }

        ELEMENT(const ELEMENT &se)
            : address(se.address), value(se.value)
            { }

        void operator= (const ELEMENT &se)
            {
            assert( this != &se );

            address=se.address;
            value=se.value;
            }

        template<class U>
        friend ostream& operator<< ( ostream&,
                                     const typename STACK<U>::ELEMENT&);
        };

private:
    vector<ELEMENT> data;
    int current_level;

public:
    STACK() : data(), current_level(-1)
        { }

    STACK(const STACK&)
        {
        assert( 0 );
        }

    void operator=(const STACK&)
        {
        assert( 0 );
        }

    size_t size() const
        {
        return data.size();
        }

    bool empty() const
        {
        return data.empty();
        }

    void clear()
        {
        data.clear();
        current_level=-1;
        }

    void push(const unsigned level, T *p)
        {
        assert( current_level <= static_cast<int>(level) );

        while( current_level < static_cast<int>(level) )
            {
            current_level++;

            data.push_back( ELEMENT(0,current_level) );
            }

        data.push_back( ELEMENT(p,*p) );
        }

    void unroll(const unsigned level)
        {
        if( TraceLevel >= 2 )
            cdebug << indent(level) << "Restoring counters (from level "
                   << current_level << " to " << level << ')' << endl;

        // Do nothing if we already are at the desired level or even below.
        // (This is the case when a user of STACK operates at higher levels
        // without pushing anything on the STACK.)
        if( static_cast<int>(level) >= current_level )
            return;

        // Restore all values saved on the stack until we reach the marker
        // "closing" our desired level.
        for(;;)
            {
            assert( ! data.empty() );

            // Whenever encounter a marker...
            if( data.back().address == 0 )
                {
                const unsigned l=data.back().value;

                // ...simply pop it from the stack...
                data.pop_back();

                // ...and terminate if it is the one we've been looking for.
                if(  l == level+1 )
                    break;
                }
            else
                {
                *(data.back().address)=data.back().value;
                data.pop_back();
                }
            }

        current_level=level;
        }

    void restore()
        {
        if( TraceLevel >= 2 )
            cdebug << "Restoring counters." << endl;

        while( ! empty() )
            {
            if( data.back().address != 0 )
                *(data.back().address)=data.back().value;
            data.pop_back();
            }

        if( TraceLevel >= 3 )
            cdebug << "Done restoring counters." << endl;
        }

    template<class U>
    friend ostream& operator<< (ostream &out, const STACK<U>&);
    };

template<class T>
inline ostream& operator<< (ostream &out, const typename STACK<T>::ELEMENT &se)
    {
    return out << ":" << se.address << "=" << se.value;
    }

template<class T>
inline ostream& operator<< (ostream &out, const STACK<T> &s)
    {
    out << s.data.size() << " element(s) starting at " << &s << ": ";

    for( typename vector<typename STACK<T>::ELEMENT>::const_iterator i
         = s.data.begin();
         i != s.data.end(); 
         i++ )
        {
        if( i->address == 0 )
            out << "[level " << i->value << "] ";
        else
            out << i->address << ' ';
        }

    return out;
    }

// The class GRULEPTRS provides the possibility to insert rules in an
// 'ordered' fashion, meaning that rules, which are in the same
// component as the atom for which the object is created, appear
// before rules for which this condition does not hold.
// There is also the possibility to use it in an unordered fashion.
class GRULEPTRS : public vector< GRULEPTR >
    {
private:
    vector< GRULEPTR >::const_iterator rulesSeparator;

public:
    GRULEPTRS ()
        {
        rulesSeparator = end();
        }

    GRULEPTRS (const GRULEPTRS &G) : vector<GRULEPTR>(G)
        {
        rulesSeparator=begin()+(G.rulesSeparator-G.begin());
        }

    void operator = (const GRULEPTRS&)
        {
        assert(0);
        }

    vector< GRULEPTR >::const_iterator sameComponentBegin() const
        {
        return begin();
        }

    vector< GRULEPTR >::const_iterator sameComponentEnd() const
        {
        return rulesSeparator;
        }

    vector< GRULEPTR >::const_iterator otherComponentsBegin() const
        {
        return rulesSeparator;
        }

    vector< GRULEPTR >::const_iterator otherComponentsEnd() const
        {
        return end();
        }
    

    void insertOrdered(const DEPGRAPH<GATOM,GCONSTRAINTS,GPROGRAM> *g, 
                       const GATOM &atom, const GRULEPTR &r)
        {
        assert( begin() <= rulesSeparator && rulesSeparator <= end() );

        // The check below returns TRUE iff r has in the head at
        // least another atom belonging to the same component of
        // the current atom.
        bool sameComponent = sameComponentInHead(*g, *r, atom);

        // Something to remember
        size_t old_size = capacity();
        size_t old_separator_position = rulesSeparator - begin();

        // Insert the iterator to the rule in correct position
        if (sameComponent) 
            insert(begin(), r);
        else 
            push_back(r);

        // If the vector grows, the separator may have become invalid
        // due to relocation in memory.
        if (capacity() != old_size)
            rulesSeparator = begin() + old_separator_position;
           
        // Increment the separator if we have to.
        if (sameComponent)
            rulesSeparator++;

        assert( begin() <= rulesSeparator && rulesSeparator <= end() );
        } // insertOrdered()

    void insertUnordered(const GRULEPTR &r)
        {
        push_back(r);
        }
    }; // GRULEPTRS

// This class is used to store the components we have to compute 
// the GUS on. It is a queue that does not store duplicates.
class NoDupComponentQueue : public deque <COMPONENT_INDEX>
    {
    bool *alreadyQueued;
    unsigned nElements;

public:
    NoDupComponentQueue()
        : alreadyQueued(0), nElements(0)
        { }

    NoDupComponentQueue(const NoDupComponentQueue&)
        : deque<COMPONENT_INDEX>()
        {
        assert(0);
        }

    ~NoDupComponentQueue() 
        {
        if( alreadyQueued )
            delete[] alreadyQueued;
        }

    void operator = (const NoDupComponentQueue&) 
        {
        assert(0);
        }

    // This initialises the queue. It should be called only once,
    // since the computeGUS algorithm uses it in a way that grant the
    // consistency of the informations stored.
    void init(unsigned n)
        {
        assert( ! alreadyQueued );

        nElements = n;
        alreadyQueued = new bool[nElements];
        fill_n(alreadyQueued,nElements,false);
        }

    // This avoid duplicates, simply checking and updating an array of
    // flags.
    void push(const COMPONENT_INDEX &C)
        {
        assert( alreadyQueued );

        if( ! alreadyQueued[C] )
            {
            push_back(C);

            alreadyQueued[C] = true;
            
            // Debug
            if( TraceLevel >= 3 )
                cdebug << "GUS: component " << C+1
                       << " added to the componentQueue."
                       << endl;
                      
            } // if()
        else
            // Debug
            if( TraceLevel >= 3 )
                cdebug << "GUS: component " << C+1
                       << " was ALREADY in the componentQueue." 
                       << endl;
        } // push()

    // This has to properly update the flags that allow to avoid
    // duplicates.
    void pop() 
        {
        assert( alreadyQueued );
        assert( !empty() );
        
        COMPONENT_INDEX C = front();        

        assert( alreadyQueued[C] );

        alreadyQueued[C] = false;
        pop_front();
        } // pop()

    // Reset the duplicates flags and clear the queue.
    void clear() 
        {
        fill_n(alreadyQueued,nElements,false);
        deque<COMPONENT_INDEX>::clear();
        }
        
    }; // NoDupComponentQueue

struct HeuristicCounters;

//////////////////////////////////////////////////////////////////////////////
class MODEL_GENERATOR
//
    {
    class HeuristicPT;
    friend class HeuristicPT;

    class PT;
    friend class PT;

    friend ostream& operator << (ostream&, const MODEL_GENERATOR::PT&);

    //////////////////////////////////////////////////////////////////////////
    // Structures used during the propagation
    QUEUE Queue, LookaheadQueue;
    STACK<unsigned short> ShortStack;
    STACK<unsigned> UnsignedStack;
    STACK<int> IntegerStack;

    // Used for the GUS computation.
    QUEUE GUSqueue;
    NoDupComponentQueue HCFcomponentQueue, nonHCFcomponentQueue;

    // occursInAggregateFunction is a vector of vectors of 
    // pairs of unsigned values, having an entry for each GATOM 
    // (where the index of the GATOM is used as index into the first vector).
    // If the GATOM appears in some aggregateFunction then its entry in
    // occursInAggregateFunction will be the vector of pairs, where the first
    // element is the index of the aggregateFunction and the second element 
    // is the value assumed by the first free variable (to be aggregated 
    // over), in the GATOM itself; 
    // otherwise the entry will be the empty vector.
    //
    // occursInGatom is a vector of vectors of GATOMs, having an entry 
    // for each different aggregateFunction. In each entry there will be
    // the vector of the aggregate GATOMs in which the aggregateFunction 
    // appears.
    //
    // HeadOccurrences is a vector of vectors of iterators pointing
    // into GroundIDB. HeadOccurrences[index] is the vector of the
    // actual GATOM instance. The rules which are pointed to by the
    // iterators in HeadOccurrences[index] are those in which the
    // current GATOM occurs in the head.
    //
    // PosBodyOccurrences, NegBodyOccurrences, PosConstraintOccurrences and
    // NegConstraintOccurrences are analogous.
    //
    // If i is an iterator of ...Occurrences[index], **i yields the
    // rule/constraint represented by the iterator i.
    vector< vector< pair<unsigned,TERMS> > >  
                                occursInAggregateFunction;
    vector< vector<GATOM> >     occursInGatom;
    vector<GRULEPTRS>           HeadOccurrences;
    vector<GRULEPTRS>           PosBodyOccurrences;
    vector<GRULEPTRS>           NegBodyOccurrences;
    vector<GCONSTRAINTPTRS>     PosConstraintOccurrences;
    vector<GCONSTRAINTPTRS>     NegConstraintOccurrences;
    vector<GWEAKCONSTRAINTPTRS> PosWConstraintOccurrences;
    vector<GWEAKCONSTRAINTPTRS> NegWConstraintOccurrences;
    vector<unsigned>            potentiallySupportingRules;
    vector<bool>                alreadyConsideredAsPosPT;
    vector<bool>                alreadyConsideredAsNegPT;
    vector<unsigned>            binaryClauseOccurrencesPos;
    vector<unsigned>            binaryClauseOccurrencesNeg;
    //////////////////////////////////////////////////////////////////////////

    GRULES            &rules;
    GCONSTRAINTS      &constraints;
    GWEAKCONSTRAINTS  &wconstraints;

    const CONJUNCTION *originalQuery;
    const VATOMSET    *globalEDB;
    const INTERPRET   *globalI;

public:
    enum FLAGS { DEFAULT=0, KNOWN_HCF=1, PERFORM_CHECK=2, PERFORM_GUS=4 };

private:
    unsigned           optionModels;
    FLAGS              flags;
    bool               (*callback)(MODEL_GENERATOR*,
                                   const INTERPRET*, const GINTERPRET*);
    bool               (*callbackCheck)(MODEL_GENERATOR*,
                                   const INTERPRET*, const GINTERPRET*);

    DEPGRAPH<GATOM,GCONSTRAINTS,GPROGRAM> *posGroundDG;

    bool haveToExecuteComputeGUS;
    bool recursiveAggregatesExist;

    enum MODE { MODE_REGULAR, MODE_ABORT, MODE_CHECK_FAILED };

    MODE               mode;
    unsigned           level;

    unsigned           statMaxRecursion;
    vector<unsigned>   statRecursion;
    unsigned           statInconsistentChoices;
    unsigned           statChoicePoints;
    unsigned           statChoicePointsWithRealBranches;
    unsigned           statModelCandidates;
    unsigned           statTotalChecks;
    unsigned           statPartialChecksForwards;
    unsigned           statPartialChecksForwardsFailed;
    unsigned           statPartialChecksBacktrack;
    unsigned           statPartialChecksBacktrackQuick;
    unsigned           statModelsFound;
    unsigned           statModelsPrinted;
    unsigned           statLookahead;
    unsigned           statInconsistentLookahead;
    unsigned           statPruningLookahead;

    // This must be dynamic, for we do not know the size of the residual
    // Herbrand base upon startup.
    GATOMSET*     PositiveAssumptions;
    vector<GATOM> PositiveAssumptionsStack;

    GATOMSET           ufset;
    bool               ufsetInitialized;
    vector<const GRULE*> ufsetRules;

    unsigned turnPointLow, turnPointHigh;     // Model checking forwards

public:
    bool               inconsistency;
    COST               *costBound;
    COST               *wfCost;
    COST               *bestCost;
    GINTERPRET         *bestModel;
    bool               groundedWC;
    bool               parsedWC;
    bool               optTraceWC;
    bool               isOptimalCostKnown;
    bool               optCostBound;
    bool               increasedCost;
    unsigned           maxWeakConstraintLevel;
    unsigned           aggregateFunctionsCount;

    HeuristicCounters  HC, hcbuffer, hccomplbuffer;

    CriteriaSequence   heuristicSequence;
    CriteriaSequence   heuristicCombinationSequence;

    TIMER              timerTotal,
                       timerFirst,
                       timerStructInit,
                       timerWellFounded,
                       timerDepgraph,
                       timerTotalChecks,
                       timerPartialChecks,
                       timerPartialChecksQuick,
                       timerPT,
                       timerCallback;

#ifdef TIMERS
    TIMER              timerPropagate,
                       timerRecomputeCounters;
#endif

private:

    // This GINTERPRET is used by the computeGUS algorithm
    GINTERPRET GUSInterpret;

    MODEL_GENERATOR()
        // This is a cute hack to avoid warnings.
      : rules(rules),
        constraints(constraints),
        wconstraints(wconstraints)
        {
        assert( 0 );
        }

    MODEL_GENERATOR(const MODEL_GENERATOR&)
        // This is a cute hack to avoid warnings.
        : rules(rules),
          constraints(constraints),
          wconstraints(wconstraints)
        {
        assert( 0 );
        }

public:
    MODEL_GENERATOR(
        GRULES            &rules2,
        GCONSTRAINTS      &constraints2,
        GWEAKCONSTRAINTS  &wconstraints2,
        const CONJUNCTION *originalQuery2,
        const VATOMSET    *globalEDB2,
        const INTERPRET   *globalI2,
        const unsigned    &optionModels2,
        const COST        *costBound2,
        const FLAGS       flags2,
        bool              (*callback2)(MODEL_GENERATOR*, 
                                       const INTERPRET*, const GINTERPRET*),
        bool              (*callbackCheck2)(MODEL_GENERATOR*, 
                                       const INTERPRET*, const GINTERPRET*),
        bool              recursiveAggregatesExist,
        const COST        *wfCost2,
        bool              groundedWC2,
        bool              parsedWC2,
        bool              optTraceWC2,
        bool              optCostBound2,
        const unsigned    maxWeakConstraintLevel2,
        const unsigned    aggregateFunctionsCount2,
        const CriteriaSequence heuristicSequence2,
        const CriteriaSequence heuristicCombinationSequence2
        );

    ~MODEL_GENERATOR()
        {
        if( posGroundDG )
            delete posGroundDG;

        if(wfCost)
            delete wfCost;
        if(bestCost)
            delete bestCost;
        if(bestModel)
            delete bestModel;
        }

private:
    void init(GINTERPRET &I);
    void done();

public:
    void detcons();
    void run();
    void printStatistics(ostream&) const;
    void printTimers(ostream&) const;

    unsigned getNumberOfModelsComputed() const
        {
        return statModelsFound;
        }

    unsigned getNumberOfModelsPrinted() const
        {
        return statModelsPrinted;
        }

    void incrNumberOfModelsPrinted()
        {
        statModelsPrinted++;

        // Stop the timer for the first model, if this was the first
        // printed model.
        if( statModelsPrinted == 1 )
            timerFirst.stop();

        // If a specific number of models to print is requested, and
        // this number has just been reached, abort the computation.
        if( optionModels  &&  statModelsPrinted == optionModels ) 
            mode=MODE_ABORT;
        }

    // Invoke the callback without invoking run().
    bool runCallback(const INTERPRET *I, const GINTERPRET *GI)
        {
        // Start the timers.
        timerTotal.start();
        if( statModelsPrinted == 0 )
            timerFirst.start();
        timerCallback.start();

        // Run the callback.
        assert(callback);
        bool result = callback(this,I,GI);

        // Stop the timers.
        timerCallback.stop();
        if( statModelsPrinted == 0 )
            timerFirst.stop();
        timerTotal.stop();

        return result;
        }

private:

    // Inside addHeadRule() and addPosBodyrule() we use insertOrdered()
    // or insertUnordered(), depending on whether we have to compute the
    // GUS, in addPosBodyRule we always use insertUnordered().
    ////////////////////////////////////////////////////////////////////////

    /** stores the given reference to GRULE into the head-occurrences 
     *  datastructure of this GATOM.
     * @param an iterator to the rule which contains this GATOM in its head.
     */
    void addHeadRule(const GATOM& atom, const GRULEPTR &r)
        {
        if( haveToExecuteComputeGUS )
            {
            assert( posGroundDG );
            HeadOccurrences[atom.getIndex()].insertOrdered(posGroundDG, atom, r);
            }
        else
            HeadOccurrences[atom.getIndex()].insertUnordered(r);
        }

    /** stores the given reference to GRULE into the positive-body-occurrences
     *  datastructure of this GATOM.
     * @param an iterator to the rule which contains this GATOM in its 
     *  positive body.
     */
    void addPosBodyRule(const GATOM& atom, const GRULEPTR &r)
        {
        if( haveToExecuteComputeGUS )
            {
            assert( posGroundDG );
            PosBodyOccurrences[atom.getIndex()].insertOrdered(posGroundDG, atom, r);
            }
        else
            PosBodyOccurrences[atom.getIndex()].insertUnordered(r);
        }

    /** stores the given reference to GRULE into the negative-body-occurrences
     *  datastructure of this GATOM.
     * @param an iterator to the rule which contains this GATOM in its
     *  negative body.
     */
    void addNegBodyRule(const GATOM& atom, const GRULEPTR &r)
        {
        NegBodyOccurrences[atom.getIndex()].insertUnordered(r);
        }

    /** stores the given reference to GCONSTRAINT into the positive-constraint-
     *  -occurrences datastructure of this GATOM.
     * @param an iterator to the constraint which contains this GATOM
     *  positively.
     */
    void addPosConstraint(const GATOM& atom, const GCONSTRAINTPTR &c)
        {
        PosConstraintOccurrences[atom.getIndex()].push_back(c);
        }

    /** stores the given reference to GCONSTRAINT into the negative-constraint-
     *  -occurrences datastructure of this GATOM.
     * @param an iterator to the constraint which contains this GATOM
     *  negatively.
     */
    void addNegConstraint(const GATOM& atom, const GCONSTRAINTPTR &c)
        {
        NegConstraintOccurrences[atom.getIndex()].push_back(c);
        }

    /** stores the given reference to GWEAKCONSTRAINT into the positive-weak-
     *  constraint-occurrences datastructure of this GATOM.
     * @param an iterator to the weak constraint which contains this GATOM
     *  positively.
     */
    void addPosWeakConstraint(const GATOM& atom, const GWEAKCONSTRAINTPTR &c)
        {
        PosWConstraintOccurrences[atom.getIndex()].push_back(c);
        }

    /** stores the given reference to GWEAKCONSTRAINT into the negative-weak-
     * constraint-occurrences datastructure of this GATOM.
     * @param an iterator to the weak constraint which contains this GATOM
     *  negatively.
     */
    void addNegWeakConstraint(const GATOM& atom, const GWEAKCONSTRAINTPTR &c)
        {
        NegWConstraintOccurrences[atom.getIndex()].push_back(c);
        }

    
    /** decrements this GATOM's potentially-supporting-rules-count by one.
     */
    void decPotentialSupport(const GATOM& atom)
        {
        assert( potentiallySupportingRules[atom.getIndex()] > 0 );
    
        potentiallySupportingRules[atom.getIndex()]--;
        }

    /** increments this GATOM's potentially-supporting-rules-count by one.
     *  Called for recomputations.
     */
    void incPotentialSupport(const GATOM& atom)
        {
        potentiallySupportingRules[atom.getIndex()]++;
        }

    /** obtains this GATOM's potentially-supporting-rules-count.
     */
    unsigned getPotentialSupport(const GATOM& atom)
        {
        return potentiallySupportingRules[atom.getIndex()];
        }

    unsigned& getPotentialSupportAsReference(const GATOM& atom)
        {
        return potentiallySupportingRules[atom.getIndex()];
        }

    /** resets this GATOM's potentially-supporting-rules-count to 0.
     *  Called during recomputations for initialisation.
     */
    void resetPotentialSupport(const GATOM& atom)
        {
        potentiallySupportingRules[atom.getIndex()] = 0;
        }

    /** provides access to the iterators pointing into GroundIDB, referring
     *  to the rules in which this GATOM occurs in the head.
     * @return an iterator pointing to the beginning of this list.
     */
    GRULEPTRS::const_iterator headRules_begin(const GATOM& atom) const
        {
        return HeadOccurrences[atom.getIndex()].begin();
        }

    /** provides access to the iterators pointing into GroundIDB, referring
     *  to the rules in which this GATOM occurs in the head.
     * @return an iterator pointing beyond the end of this list.
     */
    GRULEPTRS::const_iterator headRules_end(const GATOM& atom) const
        {
        return HeadOccurrences[atom.getIndex()].end();
        }

    /** provides access to the iterators pointing into GroundIDB, referring
     *  to the rules in which this GATOM occurs in the positive body.
     * @return an iterator pointing to the beginning of this list.
     */
    GRULEPTRS::const_iterator posBodyRules_begin(const GATOM& atom) const
        {
        return PosBodyOccurrences[atom.getIndex()].begin();
        }

    /** provides access to the iterators pointing into GroundIDB, referring
     *  to the rules in which this GATOM occurs in the positive body.
     * @return an iterator pointing beyond the end of this list.
     */
    GRULEPTRS::const_iterator posBodyRules_end(const GATOM& atom) const
        {
        return PosBodyOccurrences[atom.getIndex()].end();
        }

    /** provides access to the iterators pointing into GroundIDB, referring
     *  to the rules in which this GATOM occurs in the negative body.
     * @return an iterator pointing to the beginning of this list.
     */
    GRULEPTRS::const_iterator negBodyRules_begin(const GATOM& atom) const
        {
        return NegBodyOccurrences[atom.getIndex()].begin();
        }

    /** provides access to the iterators pointing into GroundIDB, referring 
     *  to the rules in which this GATOM occurs in the negative body.
     * @return an iterator pointing beyond the end of this list.
     */
    GRULEPTRS::const_iterator negBodyRules_end(const GATOM& atom) const
        {
        return NegBodyOccurrences[atom.getIndex()].end();
        }

    /** provides access to the iterators pointing into GroundConstraints, 
     *  referring to the constraints in which this GATOM occurs positively.
     * @return an iterator pointing to the beginning of this list.
     */
    GCONSTRAINTPTRS::const_iterator posConstraints_begin(const GATOM& atom) const
        {
        return PosConstraintOccurrences[atom.getIndex()].begin();
        }

    /** provides access to the iterators pointing into GroundConstraints, 
     *  referring to the constraints in which this GATOM occurs positively.
     * @return an iterator pointing beyond the end of this list.
     */
    GCONSTRAINTPTRS::const_iterator posConstraints_end(const GATOM& atom) const
        {
        return PosConstraintOccurrences[atom.getIndex()].end();
        }

    /** provides access to the iterators pointing into GroundConstraints, 
     *  referring to the constraints in which this GATOM occurs negatively.
     * @return an iterator pointing to the beginning of this list.
     */
    GCONSTRAINTPTRS::const_iterator negConstraints_begin(const GATOM& atom) const
        {
        return NegConstraintOccurrences[atom.getIndex()].begin();
        }

    /** provides access to the iterators pointing into GroundConstraints, 
     *  referring to the constraints in which this GATOM occurs negatively.
     * @return an iterator pointing beyond the end of this list.
     */
    GCONSTRAINTPTRS::const_iterator negConstraints_end(const GATOM& atom) const
        {
        return NegConstraintOccurrences[atom.getIndex()].end();
        }

    /** provides access to the iterators pointing into GroundWeakConstraints, 
     *  referring to the Weak constraints in which this GATOM occurs positively.
     * @return an iterator pointing to the beginning of this list.
     */
    GWEAKCONSTRAINTPTRS::const_iterator posWeakConstraints_begin(const GATOM&
                                                                 atom) const
        {
        return PosWConstraintOccurrences[atom.getIndex()].begin();
        }

    /** provides access to the iterators pointing into GroundWeakConstraints, 
     *  referring to the Weak constraints in which this GATOM occurs positively.
     * @return an iterator pointing beyond the end of this list.
     */
    GWEAKCONSTRAINTPTRS::const_iterator posWeakConstraints_end(const GATOM&
                                                               atom) const
        {
        return PosWConstraintOccurrences[atom.getIndex()].end();
        }

    /** provides access to the iterators pointing into GroundWeakConstraints, 
     *  referring to the Weak constraints in which this GATOM occurs
     *  negatively.
     * @return an iterator pointing to the beginning of this list.
     */
    GWEAKCONSTRAINTPTRS::const_iterator negWeakConstraints_begin(const GATOM&
                                                                 atom) const
        {
        return NegWConstraintOccurrences[atom.getIndex()].begin();
        }

    /** provides access to the iterators pointing into GroundWeakConstraints, 
     *  referring to the Weak constraints in which this GATOM occurs
     *  negatively.
     * @return an iterator pointing beyond the end of this list.
     */
    GWEAKCONSTRAINTPTRS::const_iterator negWeakConstraints_end(const GATOM&
                                                               atom) const
        {
        return NegWConstraintOccurrences[atom.getIndex()].end();
        }

    // Initialise the datastructures keeping information on whether
    // some literal has already been considered as PT (i.e. its
    // heuristics has already been calculated).
    void initialiseAlreadyConsideredAsPT() 
        {
        fill_n(alreadyConsideredAsPosPT.begin(),
               alreadyConsideredAsPosPT.size(),false);
        fill_n(alreadyConsideredAsNegPT.begin(),
               alreadyConsideredAsNegPT.size(),false);
        }

    // Check whether a given atom has already been considered as a
    // positive/negative PT.
    bool isAlreadyConsideredAsPT(const GATOM &atom, bool neg) const
        {
        if( neg )
            return alreadyConsideredAsNegPT[atom.getIndex()];
        else
            return alreadyConsideredAsPosPT[atom.getIndex()];
        }

    // Indicate that a given atom has already been considered as a
    // positive/negative PT.
    void setAlreadyConsideredAsPT(const GATOM& atom, bool neg)
        {
        if( neg )
            {
            assert( ! alreadyConsideredAsNegPT[atom.getIndex()] );
            alreadyConsideredAsNegPT[atom.getIndex()] = true;
            }
        else
            {
            assert( ! alreadyConsideredAsPosPT[atom.getIndex()] );
            alreadyConsideredAsPosPT[atom.getIndex()] = true;
            }
        }

    // Clear the datastructures keeping information on the
    // number of binary clauses a literal occurs in.
    void clearBinaryClauseOccurrences() 
        {
        fill_n(binaryClauseOccurrencesPos.begin(),
               binaryClauseOccurrencesPos.size(),0);
        fill_n(binaryClauseOccurrencesNeg.begin(),
               binaryClauseOccurrencesNeg.size(),0);
        }

    void initialiseBinaryClauseOccurrences(const GATOM& atom, bool neg)
        {
        if( neg )
            {
            for( GRULEPTRS::const_iterator i = headRules_begin(atom);
                 i != headRules_end(atom);
                 i++ )
                {
                if( (**i).isBinaryClause() )
                    incBinaryClauseOccurrences(atom,neg);
                }
            for( GRULEPTRS::const_iterator i = negBodyRules_begin(atom);
                 i != negBodyRules_end(atom);
                 i++ )
                {
                if( (**i).isBinaryClause() )
                    incBinaryClauseOccurrences(atom,neg);
                }
            for( GCONSTRAINTPTRS::const_iterator i = negConstraints_begin(atom);
                 i != negConstraints_end(atom);
                 i++ )
                {
                if( (**i).isBinaryClause() )
                    incBinaryClauseOccurrences(atom,neg);
                }
            }
        else
            {
            for( GRULEPTRS::const_iterator i = posBodyRules_begin(atom);
                 i != posBodyRules_end(atom);
                 i++ )
                {
                if( (**i).isBinaryClause() )
                    incBinaryClauseOccurrences(atom,neg);
                }
            for( GCONSTRAINTPTRS::const_iterator i = posConstraints_begin(atom);
                 i != posConstraints_end(atom);
                 i++ )
                {
                if( (**i).isBinaryClause() )
                    incBinaryClauseOccurrences(atom,neg);
                }
            }            
        }
    
    unsigned getBinaryClauseOccurrences(const GATOM& atom, bool neg) const
        {
        if( neg )
            return binaryClauseOccurrencesNeg[atom.getIndex()];
        else
            return binaryClauseOccurrencesPos[atom.getIndex()];
        }

    void incBinaryClauseOccurrences(const GATOM& atom, bool neg)
        {
        if( neg )
            binaryClauseOccurrencesNeg[atom.getIndex()]++;
        else
            binaryClauseOccurrencesPos[atom.getIndex()]++;
        }

    // For each GATOM occurring in the aggregateSet of an aggregate GATOM,
    // store the index of the aggregateFunction in which it appears and 
    // the value assumed by the first free variable (to be aggregated over) 
    // in the GATOM itself, in its entry in vector occursInAggregateFunction
    // and then store the aggregate GATOM in the entry of the 
    // aggregateFunction in vector occursInGatom. 
    void updateAggregatesLinearStructures(const GATOM &aggregate)
        {
        const AGGREGATEATOM &g = aggregate.getAggregate();
        for( AGGREGATESET::const_iterator
            i = g.getAggregateSet().begin();
            i != g.getAggregateSet().end();
            i++)
            {
            // Get the index of the current GATOM in the aggregateSet. 
            unsigned u1=(*i).second.getIndex();
	    
            // Get the hash index of the aggregateFunction.
            unsigned u2=aggregate.getAggregate().getAggregateFunctionIndex();
	    
            // Create a pair composed by the aggregateFunction hash index 
            // and the value assumed by the first free variable (to be 
            // aggregated over) in the current GATOM in the aggregateSet. 
            pair<unsigned,TERMS> element( u2,(*i).first );
	    
            if( find( occursInAggregateFunction[u1].begin(),
                occursInAggregateFunction[u1].end(), element )
                == occursInAggregateFunction[u1].end() )
                occursInAggregateFunction[u1].push_back(element);

            assert( occursInGatom.size() );
            if( find( occursInGatom[u2].begin(),
                occursInGatom[u2].end(), aggregate )
                == occursInGatom[u2].end() )
                occursInGatom[u2].push_back(aggregate);
            }
        }

    void PrintGruleCounters(ostream&);
    void PrintGconstraintCounters(ostream&);
    void PrintGatomRulesConstraints(ostream&);

    bool initialiseRules(const GINTERPRET&);
    bool initialiseConstraints(const GINTERPRET&);
    void initialiseWeakConstraints(GINTERPRET&);
    bool initialiseLinearDataStructures(GINTERPRET&);
    void freeLinearDataStructures();

    vector<unsigned>* createPotentiallySupportingRulesCounters();
    bool checkPotentiallySupportingRulesCounters(vector<unsigned>*);

    bool RecomputeCounters(GRULE&, const GINTERPRET&, bool);
    bool RecomputeRuleCounters(const GINTERPRET&, bool);
    bool RecomputeCounters(const GINTERPRET&, bool);

    void findMinUndefTrueInAggregateSet(
        AGGREGATEATOM&, const GINTERPRET&) const;
    void findMaxUndefTrueInAggregateSet(
        AGGREGATEATOM&, const GINTERPRET&) const;
    void updateAggregate(
        const pair<unsigned,TERMS>&, const GINTERPRET &);
    void pushAggregateCounters(AGGREGATEFUNCTION &);    

    inline void Propagate_Derive_TrueFromUndefined(
        const GATOM&, GINTERPRET&);
    void Propagate_Derive_TrueFromMustBeTrue(
        const GATOM&, GINTERPRET&);
    inline void Propagate_Derive_True(
        const GATOM&, GINTERPRET&);
    inline void Propagate_Derive_MustBeTrue(
        const GATOM&, GINTERPRET&);
    inline void Propagate_Derive_False(
        const GATOM& a, GINTERPRET&);
    inline bool possibleInferenceFromWeakConstraint(
        const GWEAKCONSTRAINT&, 
        const COST&) const;
    inline bool inferenceFromWeakConstraints(
        GINTERPRET&);
    inline bool Propagate_OnePotentiallySupportingRule(
        const GATOM&, GINTERPRET&);
    inline bool Propagate_RuleSatisfaction_AdjustPotentialSupport(
        GRULE&, const GATOM*, GINTERPRET &I);
    inline bool Propagate_RuleBecomesSatisfiedByHead(
        GRULE&, const GATOM&, GINTERPRET&);
    inline bool Propagate_RuleBecomesSatisfiedByBody(
        GRULE&,GINTERPRET&);
    inline void Propagate_DeriveSingleUndefinedHeadLiteral(
        const GRULE&, GINTERPRET&);
    inline void Propagate_DeriveSingleUndefinedPosBodyLiteral(
        const GRULE&, GINTERPRET&);
    inline void Propagate_DeriveSingleUndefinedNegBodyLiteral(
        const GRULE&, GINTERPRET&);
    inline void Propagate_MBTDeriveSingleUndefinedHeadLiteral(
        const GRULE&, GINTERPRET&);
    inline void Propagate_MBTDeriveSingleUndefinedPosBodyLiteral(
        const GRULE&, GINTERPRET&);
    inline void Propagate_MBTDeriveSingleUndefinedNegBodyLiteral(
        const GRULE&, GINTERPRET&);
    inline void Propagate_HandlePosSingleUndefinedLiteral(
        const GCONSTRAINT&, GINTERPRET&);
    inline void Propagate_HandleNegSingleUndefinedLiteral(
        const GCONSTRAINT&, GINTERPRET&);
    inline bool Propagate_DeriveFromRule(
        const GRULE&, GINTERPRET&);
    inline bool Propagate_DeriveMBTFromHead(
        const GRULE&, GINTERPRET&);
    inline bool Propagate_DeriveFromConstraint(
        const GCONSTRAINT&, GINTERPRET&);
    inline void Propagate_DeriveFromWeakConstraint(
        const GWEAKCONSTRAINT&, GINTERPRET&);
    inline bool atLeastOneTrueAtom(
        const AGGREGATESET&, const GINTERPRET&) const;
    inline bool onlyOneUndefAndNoTrueAtomsGreaterThanUpperGuard(
        const AGGREGATEATOM&, const GINTERPRET&) const;
    inline bool noTrueAtomsBetweenGuards(
        const AGGREGATEATOM&, const GINTERPRET&) const;
    inline bool noTrueOrUndefAtomsBetweenGuards(
        const AGGREGATEATOM&, const GINTERPRET&) const;
    inline bool onlyOneTrueOrUndefAtomBetweenGuards(
        const AGGREGATEATOM&, const GINTERPRET&) const;
    inline bool onlyOneUndefAndNoTrueAtomsLessThanLowerGuard(
        const AGGREGATEATOM&, const GINTERPRET&) const;
    inline void Propagate_DeriveMustBeTrueFromAggregate(
        const GATOM&, GINTERPRET&);
    inline void Propagate_DeriveFalseFromAggregate(
        const GATOM&, GINTERPRET&);
    inline void PropagateMustBeTrueCount(
        const GATOM&, GINTERPRET&);
    inline void PropagateMustBeTrueMax(
        const GATOM&, GINTERPRET&);
    inline void PropagateMustBeTrueMin(
        const GATOM&, GINTERPRET&);
    inline void PropagateMustBeTrueSum(
        const GATOM&, GINTERPRET&);
    inline void PropagateMustBeTrueTimes(
        const GATOM&, GINTERPRET&);
    inline void PropagateFalseCount(
        const GATOM&, GINTERPRET&);
    inline void PropagateFalseMax(
        const GATOM&, GINTERPRET&);
    inline void PropagateFalseMin(
        const GATOM&, GINTERPRET&);
    inline void PropagateFalseSum(
        const GATOM&, GINTERPRET&);
    inline void PropagateFalseTimes(
        const GATOM&, GINTERPRET&);
    inline void PropagateMustBeTrueAggregate(
        const GATOM&, GINTERPRET&);
    inline void PropagateFalseAggregate(
        const GATOM&, GINTERPRET&);
    inline bool PropagateAggregateBoundUpdate(
        const unsigned, GINTERPRET&);
    inline bool PropagateInAggregate(
        const GATOM&, GINTERPRET&, bool);
    inline bool PropagatePositiveFromUndefined(
        const GATOM&, GINTERPRET&);
    bool PropagatePositiveFromMustBeTrue(
        const GATOM&, GINTERPRET&);
    bool PropagateNegative(
        const GLITERAL&, GINTERPRET&);
    bool PropagateMustBeTrue(
        const GATOM&, GINTERPRET&);
    bool Propagate(
        GINTERPRET&);
    bool LookaheadPropagate(
        GINTERPRET&);
    inline bool Lookahead(
        const GATOM&, const bool, const bool, GINTERPRET&);
    bool CalculateHeuristicValue(
        const GATOM&, const bool, GINTERPRET&,
        HeuristicCounters*, HeuristicCounters*);

    inline bool isPT(const GATOM&, const GINTERPRET&) const;
    inline bool isPositivePT(const GATOM&, const GINTERPRET&) const;
    inline bool isNegativePT(const GATOM&, const GINTERPRET&) const;

    bool ComputeWellFounded(GINTERPRET&);
    bool simplifyConjunction(const GINTERPRET&, GCONJUNCTION&, const GRULE*);
    bool simplifyProgram(const GINTERPRET&, bool&);
    bool Totalise(GINTERPRET&);
   
    // Some typedef for more confortable use
    typedef DEPGRAPH<GATOM,GCONSTRAINTS,GPROGRAM>::COMPONENT COMPONENT;
    typedef DEPGRAPH<GATOM,GCONSTRAINTS,GPROGRAM>::COMPONENTS COMPONENTS;
    
    // Methods for the computeGUS algorithm
    void pushComponent(const COMPONENT_INDEX &C, 
                       const GRULE &r, const GATOM &a);

    bool sameComponent(const GATOM &a1, const GATOM &a2) const;
    bool belongToComponent(const COMPONENT_INDEX &C, const GATOM &a) const;
    GDISJUNCTION::const_iterator findFirstTrue(const GDISJUNCTION &D, 
                                               const GINTERPRET &I) const;
    GDISJUNCTION::const_iterator findHeadGatomInC(const COMPONENT_INDEX &C, 
                                                  const GDISJUNCTION &D) const;
    bool trueByExternal(const GRULE &r, const COMPONENT_INDEX &C,
                        const GINTERPRET &I) const;
    bool usefulRule(const COMPONENT_INDEX &C, const GINTERPRET &I,
                    const GRULE &r) const;
    unsigned countPosBodyOccInC(const COMPONENT_INDEX &C, const GRULE &r) const;
    bool propagateGUS(const COMPONENT_INDEX &C, GINTERPRET &I);
    bool computeGUS(const COMPONENT_INDEX &C, GINTERPRET &I);

    void findModelsContaining(GINTERPRET&);
    void assumePTLiteral(bool&, bool, const GLITERAL&, GINTERPRET&);
    bool assumeComplementaryPTLiteral(bool, const GLITERAL&, GINTERPRET&);
    bool checkModel(GINTERPRET&, GATOMSET*, bool*);
    void foundModelCandidate(GINTERPRET&);
    void foundStableModel(const GINTERPRET&);
    };

#include "generate-pt.h"

#endif

// Local Variables:
// mode: c++
// c-file-style: "dl"
// End:
