//////////////////////////////////////////////////////////////////////////////
// generate-counters.h

#include <iterator>

#ifndef GENERATE_COUNTERS_H
#define GENERATE_COUNTERS_H


struct HeuristicCounters
    {
    unsigned MBTEliminated;
    unsigned MBTEliminatedLevel2;
    unsigned MBTEliminatedLevel3;

    unsigned MBTIntroduced;
    unsigned MBTIntroducedLevel2;
    unsigned MBTIntroducedLevel3;

    unsigned ClauseEliminated;

    unsigned PVClauseEliminated;
#ifdef WITH_PVCLEVELS
    unsigned PVClauseEliminatedLevel2;
    unsigned PVClauseEliminatedLevel3;
#endif

    unsigned PVClauseIntroduced;
#ifdef WITH_PVCLEVELS
    unsigned PVClauseIntroducedLevel2;
    unsigned PVClauseIntroducedLevel3;
#endif

    unsigned UndefinedEliminated;

    COST* HeuristicCost;
    COST* EliminatedWCCost;

    HeuristicCounters()
        {
        assert(0);
        }

    HeuristicCounters(unsigned int maxWeakConstraintLevel)
        : MBTEliminated(0), MBTEliminatedLevel2(0), MBTEliminatedLevel3(0),
          MBTIntroduced(0), MBTIntroducedLevel2(0), MBTIntroducedLevel3(0),
          ClauseEliminated(0), PVClauseEliminated(0),
#ifdef WITH_PVCLEVELS
          PVClauseEliminatedLevel2(0), PVClauseEliminatedLevel3(0),
#endif
          PVClauseIntroduced(0),
#ifdef WITH_PVCLEVELS
          PVClauseIntroducedLevel2(0), PVClauseIntroducedLevel3(0),
#endif
          UndefinedEliminated(0)
        {
        if(maxWeakConstraintLevel == 0)
            {
            HeuristicCost = 0;
            EliminatedWCCost = 0;
            }
        else
            {
            HeuristicCost = new COST(maxWeakConstraintLevel);
            EliminatedWCCost = new COST(maxWeakConstraintLevel);
            }
        }

    HeuristicCounters( unsigned e, unsigned e2, unsigned e3,
                       unsigned i, unsigned i2, unsigned i3,
                       unsigned ce, unsigned pvce,
#ifdef WITH_PVCLEVELS
                       unsigned pvce2, unsigned pvce3,
#endif
                       unsigned pvci,
#ifdef WITH_PVCLEVELS
                       unsigned pvci2, unsigned pvci3,
#endif
                       unsigned ue, COST &c, COST &ewc )
        : MBTEliminated(e), MBTEliminatedLevel2(e2), MBTEliminatedLevel3(e3),
          MBTIntroduced(i), MBTIntroducedLevel2(i2), MBTIntroducedLevel3(i3),
          ClauseEliminated(ce), PVClauseEliminated(pvce),
#ifdef WITH_PVCLEVELS
          PVClauseEliminatedLevel2(pvce2), PVClauseEliminatedLevel3(pvce3),
#endif
          PVClauseIntroduced(pvci),
#ifdef WITH_PVCLEVELS
          PVClauseIntroducedLevel2(pvci2), PVClauseIntroducedLevel3(pvci3),
#endif
          UndefinedEliminated(ue)
        {
        HeuristicCost = new COST(c);
        EliminatedWCCost = new COST(ewc);
        }

    HeuristicCounters( const HeuristicCounters& hc )
        : MBTEliminated(hc.MBTEliminated),
          MBTEliminatedLevel2(hc.MBTEliminatedLevel2),
          MBTEliminatedLevel3(hc.MBTEliminatedLevel3),
          MBTIntroduced(hc.MBTIntroduced),
          MBTIntroducedLevel2(hc.MBTIntroducedLevel2),
          MBTIntroducedLevel3(hc.MBTIntroducedLevel3),
          ClauseEliminated(hc.ClauseEliminated),
          PVClauseEliminated(hc.PVClauseEliminated),
#ifdef WITH_PVCLEVELS
          PVClauseEliminatedLevel2(hc.PVClauseEliminatedLevel2),
          PVClauseEliminatedLevel3(hc.PVClauseEliminatedLevel3),
#endif
          PVClauseIntroduced(hc.PVClauseIntroduced),
#ifdef WITH_PVCLEVELS
          PVClauseIntroducedLevel2(hc.PVClauseIntroducedLevel2),
          PVClauseIntroducedLevel3(hc.PVClauseIntroducedLevel3),
#endif
          UndefinedEliminated(hc.UndefinedEliminated)
        {
        if(hc.HeuristicCost==0)
            HeuristicCost=0;
        else
            {
            HeuristicCost = new COST(*(hc.HeuristicCost));
            }
        if(hc.EliminatedWCCost==0)
            EliminatedWCCost=0;
        else
            {
            EliminatedWCCost = new COST(*(hc.EliminatedWCCost));
            }
        }

    HeuristicCounters& operator= (const HeuristicCounters& hc)
        {
        if( this != &hc )
            {
            MBTEliminated = hc.MBTEliminated;
            MBTEliminatedLevel2 = hc.MBTEliminatedLevel2;
            MBTEliminatedLevel3 = hc.MBTEliminatedLevel3;
            MBTIntroduced = hc.MBTIntroduced;
            MBTIntroducedLevel2 = hc.MBTIntroducedLevel2;
            MBTIntroducedLevel3 = hc.MBTIntroducedLevel3;
            ClauseEliminated = hc.ClauseEliminated;
            PVClauseEliminated = hc.PVClauseEliminated;
#ifdef WITH_PVCLEVELS
            PVClauseEliminatedLevel2 = hc.PVClauseEliminatedLevel2;
            PVClauseEliminatedLevel3 = hc.PVClauseEliminatedLevel3;
#endif
            PVClauseIntroduced = hc.PVClauseIntroduced;
#ifdef WITH_PVCLEVELS
            PVClauseIntroducedLevel2 = hc.PVClauseIntroducedLevel2;
            PVClauseIntroducedLevel3 = hc.PVClauseIntroducedLevel3;
#endif
            UndefinedEliminated = hc.UndefinedEliminated;

            if(HeuristicCost && hc.HeuristicCost)
                *HeuristicCost = (*hc.HeuristicCost);
            else if(!HeuristicCost && hc.HeuristicCost)
                HeuristicCost = new COST(*(hc.HeuristicCost));
            else if(HeuristicCost && !hc.HeuristicCost)
                {
                delete HeuristicCost;
                HeuristicCost = 0;
                }
            // Else both are 0.

            if(EliminatedWCCost && hc.EliminatedWCCost)
                *EliminatedWCCost = (*hc.EliminatedWCCost);
            else if(!(EliminatedWCCost) && hc.EliminatedWCCost)
                EliminatedWCCost = new COST(*(hc.EliminatedWCCost));
            else if(EliminatedWCCost && !hc.EliminatedWCCost)
                {
                delete EliminatedWCCost;
                EliminatedWCCost = 0;
                }
            // Else both are 0.
     
            }
        return *this;
        }

    ~HeuristicCounters()
        {
        if( HeuristicCost)
            delete HeuristicCost;
        if(  EliminatedWCCost)
            delete EliminatedWCCost;
        }

    void reset()
        {
        MBTEliminated=0;
        MBTEliminatedLevel2=0;
        MBTEliminatedLevel3=0;

        MBTIntroduced=0;
        MBTIntroducedLevel2=0;
        MBTIntroducedLevel3=0;

        ClauseEliminated=0;

        PVClauseEliminated=0;
#ifdef WITH_PVCLEVELS
        PVClauseEliminatedLevel2=0;
        PVClauseEliminatedLevel3=0;
#endif
        PVClauseIntroduced=0;
#ifdef WITH_PVCLEVELS
        PVClauseIntroducedLevel2=0;
        PVClauseIntroducedLevel3=0;
#endif

        UndefinedEliminated=0;

        if(HeuristicCost)
            (*HeuristicCost).reset();
        if(EliminatedWCCost)
            (*EliminatedWCCost).reset();
        }

    HeuristicCounters operator+(const HeuristicCounters HC1) const
        {
        HeuristicCounters result(*this);

        result.MBTEliminated += HC1.MBTEliminated;
        result.MBTEliminatedLevel2 += HC1.MBTEliminatedLevel2;
        result.MBTEliminatedLevel3 += HC1.MBTEliminatedLevel3;

        result.MBTIntroduced += HC1.MBTIntroduced;
        result.MBTIntroducedLevel2 += HC1.MBTIntroducedLevel2;
        result.MBTIntroducedLevel3 += HC1.MBTIntroducedLevel3;

        result.ClauseEliminated += HC1.ClauseEliminated;

        result.PVClauseEliminated += HC1.PVClauseEliminated;
#ifdef WITH_PVCLEVELS
        result.PVClauseEliminatedLevel2 += HC1.PVClauseEliminatedLevel2;
        result.PVClauseEliminatedLevel3 += HC1.PVClauseEliminatedLevel3;
#endif
        result.PVClauseIntroduced += HC1.PVClauseIntroduced;
#ifdef WITH_PVCLEVELS
        result.PVClauseIntroducedLevel2 += HC1.PVClauseIntroducedLevel2;
        result.PVClauseIntroducedLevel3 += HC1.PVClauseIntroducedLevel3;
#endif

        result.UndefinedEliminated += HC1.UndefinedEliminated;

        if(HeuristicCost)
            {
            assert(HC1.HeuristicCost);
            *(result.HeuristicCost) += *(HC1.HeuristicCost);
            }
        if(EliminatedWCCost)
            {
            assert(HC1.EliminatedWCCost);
            *(result.EliminatedWCCost) += *(HC1.EliminatedWCCost);
            }

        return result;
        }

    HeuristicCounters operator+(const int scalar) const
        {
        HeuristicCounters result(*this);

        result.MBTEliminated += scalar;
        result.MBTEliminatedLevel2 += scalar;
        result.MBTEliminatedLevel3 += scalar;

        result.MBTIntroduced += scalar;
        result.MBTIntroducedLevel2 += scalar;
        result.MBTIntroducedLevel3 += scalar;

        result.ClauseEliminated += scalar;

        result.PVClauseEliminated += scalar;
#ifdef WITH_PVCLEVELS
        result.PVClauseEliminatedLevel2 += scalar;
        result.PVClauseEliminatedLevel3 += scalar;
#endif
        result.PVClauseIntroduced += scalar;
#ifdef WITH_PVCLEVELS
        result.PVClauseIntroducedLevel2 += scalar;
        result.PVClauseIntroducedLevel3 += scalar;
#endif

        result.UndefinedEliminated += scalar;

        if(HeuristicCost)
            {
            *(result.HeuristicCost) += scalar;
            }
        if(EliminatedWCCost)
            {
            *(result.EliminatedWCCost) += scalar;
            }

        return result;
        }

    HeuristicCounters operator*(const HeuristicCounters HC1) const
        {
        HeuristicCounters result(*this);

        result.MBTEliminated *= HC1.MBTEliminated;
        result.MBTEliminatedLevel2 *= HC1.MBTEliminatedLevel2;
        result.MBTEliminatedLevel3 *= HC1.MBTEliminatedLevel3;

        result.MBTIntroduced *= HC1.MBTIntroduced;
        result.MBTIntroducedLevel2 *= HC1.MBTIntroducedLevel2;
        result.MBTIntroducedLevel3 *= HC1.MBTIntroducedLevel3;

        result.ClauseEliminated *= HC1.ClauseEliminated;

        result.PVClauseEliminated *= HC1.PVClauseEliminated;
#ifdef WITH_PVCLEVELS
        result.PVClauseEliminatedLevel2 *= HC1.PVClauseEliminatedLevel2;
        result.PVClauseEliminatedLevel3 *= HC1.PVClauseEliminatedLevel3;
#endif
        result.PVClauseIntroduced *= HC1.PVClauseIntroduced;
#ifdef WITH_PVCLEVELS
        result.PVClauseIntroducedLevel2 *= HC1.PVClauseIntroducedLevel2;
        result.PVClauseIntroducedLevel3 *= HC1.PVClauseIntroducedLevel3;
#endif

        result.UndefinedEliminated *= HC1.UndefinedEliminated;

        if(HeuristicCost)
            {
            assert(HC1.HeuristicCost);
            *(result.HeuristicCost) *= *(HC1.HeuristicCost);
            }
        if(EliminatedWCCost)
            {
            assert(HC1.EliminatedWCCost);
            *(result.EliminatedWCCost) *= *(HC1.EliminatedWCCost);
            }

        return result;
        }

    HeuristicCounters operator*(const int scalar) const
        {
        HeuristicCounters result(*this);

        result.MBTEliminated *= scalar;
        result.MBTEliminatedLevel2 *= scalar;
        result.MBTEliminatedLevel3 *= scalar;

        result.MBTIntroduced *= scalar;
        result.MBTIntroducedLevel2 *= scalar;
        result.MBTIntroducedLevel3 *= scalar;

        result.ClauseEliminated *= scalar;

        result.PVClauseEliminated *= scalar;
#ifdef WITH_PVCLEVELS
        result.PVClauseEliminatedLevel2 *= scalar;
        result.PVClauseEliminatedLevel3 *= scalar;
#endif
        result.PVClauseIntroduced *= scalar;
#ifdef WITH_PVCLEVELS
        result.PVClauseIntroducedLevel2 *= scalar;
        result.PVClauseIntroducedLevel3 *= scalar;
#endif

        result.UndefinedEliminated *= scalar;

        if(HeuristicCost)
            {
            *(result.HeuristicCost) *= scalar;
            }
        if(EliminatedWCCost)
            {
            *(result.EliminatedWCCost) *= scalar;
            }

        return result;
        }

    };

inline ostream& operator<< (ostream& out, const HeuristicCounters& hc)
    {
    out << "-mbt/+mbt/-mbt2/+mbt2/-mbt3/+mbt3/-c/-pvc/-pvc2/-pvc3/+pvc/+pvc2/+pvc3/-undef/heurCost/elimWCCost: " 
        << hc.MBTEliminated
        << "/" << hc.MBTIntroduced << "/" << hc.MBTEliminatedLevel2
        << "/" << hc.MBTIntroducedLevel2 << "/" << hc.MBTEliminatedLevel3 
        << "/" << hc.MBTIntroducedLevel3 
        << "/" << hc.ClauseEliminated
        << "/" << hc.PVClauseEliminated
#ifdef WITH_PVCLEVELS
        << "/" << hc.PVClauseEliminatedLevel2
        << "/" << hc.PVClauseEliminatedLevel3
#endif
        << "/" << hc.PVClauseIntroduced
#ifdef WITH_PVCLEVELS
        << "/" << hc.PVClauseIntroducedLevel2
        << "/" << hc.PVClauseIntroducedLevel3
#endif
        << "/" << hc.UndefinedEliminated
        << "/";

    if(hc.HeuristicCost)
        out << (*hc.HeuristicCost);
    out << "/";
    if(hc.EliminatedWCCost)
        out << (*hc.EliminatedWCCost);

    return out;
    }

struct HeuristicItem
    {
    GLITERAL literal;
    HeuristicCounters hc;
    HeuristicCounters *hccompl;

    HeuristicItem(GLITERAL l, HeuristicCounters hc1,
                  HeuristicCounters *hc2 = 0)
        : literal(l), hc(hc1)
        {
        if(hc2)
            hccompl = new HeuristicCounters(*hc2);
        else
            hccompl = 0;
        }

    HeuristicItem()
        {
        assert(0);
        }

    HeuristicItem(const HeuristicItem& hi)
        : literal(hi.literal), hc(hi.hc)
        {
        if(hi.hccompl)
            hccompl = new HeuristicCounters(*(hi.hccompl));
        else
            hccompl = 0;
        }

    HeuristicItem& operator= (const HeuristicItem& hi)
        {
        // Currently not used.
        assert(0);
        if( this != &hi )
            {
            literal = hi.literal;
            hc = hi.hc;
            if(hi.hccompl)
                hccompl = new HeuristicCounters(*(hi.hccompl));
            else
                hccompl = 0;
            }
        return *this;
        }

    ~HeuristicItem()
        {
        if( hccompl )
            delete hccompl;
        }
    };

inline ostream& operator<< (ostream& out, const HeuristicItem& h)
    {
    out << h.literal << " (" << h.hc;
    if( h.hccompl )
        out << " | " << *(h.hccompl);
    out << ")";

    return out;
    }

struct GreaterEqualHeuristic : 
    public binary_function<HeuristicItem, HeuristicItem, bool> 
    {
    CriteriaSequence& heuristicSequence;
    CriteriaSequence& heuristicCombinationSequence;

    GreaterEqualHeuristic()
        : heuristicSequence(heuristicSequence),
          heuristicCombinationSequence(heuristicCombinationSequence)
        {
        assert(0);
        }

    GreaterEqualHeuristic(CriteriaSequence& hs, CriteriaSequence& hcs)
        : heuristicSequence(hs),
          heuristicCombinationSequence(hcs)
        {
        }

    GreaterEqualHeuristic(const GreaterEqualHeuristic& geh)
        : heuristicSequence(geh.heuristicSequence),
          heuristicCombinationSequence(geh.heuristicCombinationSequence)
        {
        }

    void operator=(const GreaterEqualHeuristic& geh)
        {
        if( this != &geh )
            {
            heuristicSequence = geh.heuristicSequence;
            heuristicCombinationSequence = geh.heuristicCombinationSequence;
            }
        }

    ~GreaterEqualHeuristic()
        {
        }

    int combinedMBTDifference(const unsigned combinationIndex,
                              const HeuristicItem& x) const
        {
        assert(x.hccompl);

        return ( CombineHeuristics(combinationIndex,
                                   x.hc.MBTEliminated - x.hc.MBTIntroduced,
                                   x.hccompl->MBTEliminated
                                   - x.hccompl->MBTIntroduced) );
        }

    int combinedMBT2Difference(const unsigned combinationIndex,
                               const HeuristicItem& x) const
        {
        assert(x.hccompl);

        return ( CombineHeuristics(combinationIndex,
                                   x.hc.MBTEliminatedLevel2
                                   - x.hc.MBTIntroducedLevel2,
                                   x.hccompl->MBTEliminatedLevel2
                                   - x.hccompl->MBTIntroducedLevel2) );
        }

    int combinedMBT3Difference(const unsigned combinationIndex,
                               const HeuristicItem& x) const
        {
        assert(x.hccompl);

        return ( CombineHeuristics(combinationIndex,
                                   x.hc.MBTEliminatedLevel3
                                   - x.hc.MBTIntroducedLevel3,
                                   x.hccompl->MBTEliminatedLevel3
                                   - x.hccompl->MBTIntroducedLevel3) );
        }

    int combinedPVCDifference(const unsigned combinationIndex,
                              const HeuristicItem& x) const
        {
        assert(x.hccompl);

        return ( CombineHeuristics(combinationIndex,
                                   x.hc.PVClauseEliminated
                                   - x.hc.PVClauseIntroduced,
                                   x.hccompl->PVClauseEliminated
                                   - x.hccompl->PVClauseIntroduced) );
        }

#ifdef WITH_PVCLEVELS
    int combinedPVC2Difference(const unsigned combinationIndex,
                              const HeuristicItem& x) const
        {
        assert(x.hccompl);

        return ( CombineHeuristics(combinationIndex,
                                   x.hc.PVClauseEliminatedLevel2
                                   - x.hc.PVClauseIntroducedLevel2,
                                   x.hccompl->PVClauseEliminatedLevel2
                                   - x.hccompl->PVClauseIntroducedLevel2) );
        }

    int combinedPVC3Difference(const unsigned combinationIndex,
                              const HeuristicItem& x) const
        {
        assert(x.hccompl);

        return ( CombineHeuristics(combinationIndex,
                                   x.hc.PVClauseEliminatedLevel3
                                   - x.hc.PVClauseIntroducedLevel3,
                                   x.hccompl->PVClauseEliminatedLevel3
                                   - x.hccompl->PVClauseIntroducedLevel3) );
        }
#endif

    int compareMBTDifference(const HeuristicCounters& hca,
                             const HeuristicCounters& hcb) const
        {
        return ( (hca.MBTEliminated - hca.MBTIntroduced) 
                 - (hcb.MBTEliminated - hcb.MBTIntroduced) );
        }

    int compareMBTDifference(const unsigned combinationIndex,
                             const HeuristicItem& x,
                             const HeuristicItem& y) const
        {
        return ( combinedMBTDifference(combinationIndex,x)
                 - combinedMBTDifference(combinationIndex,y) );
        }

    // The deltas should not be compared if exactly one of -mbt(A)
    // or -mbt(B) is zero.
    bool exactlyOneHCHasNoMBTEliminated(const HeuristicCounters& hca,
                                        const HeuristicCounters& hcb) const
        {
        return ( ( hca.MBTEliminated == 0 || hcb.MBTEliminated == 0 )
                 && ( hca.MBTEliminated > 0 || hcb.MBTEliminated > 0 ) );
        }

    // The deltas should not be compared if exactly one of -mbt(A)
    // or -mbt(B) is zero.
    bool exactlyOneHCHasNoMBTEliminated(const unsigned combinationIndex,
                                        const HeuristicItem& x,
                                        const HeuristicItem& y) const
        {
        return ( ( CombineHeuristics(combinationIndex,x).MBTEliminated == 0
                   || CombineHeuristics(combinationIndex,y).MBTEliminated
                      == 0 )
                 && ( CombineHeuristics(combinationIndex,x).MBTEliminated
                      > 0
                      || CombineHeuristics(combinationIndex,y).MBTEliminated
                         > 0 ) );
        }

    int compareMBTDifferenceUnlessExactlyOneHasNoMBTEliminated(
        const HeuristicCounters& hca,
        const HeuristicCounters& hcb) const
        {
        if( exactlyOneHCHasNoMBTEliminated(hca,hcb) )
            return 0;
        else
            return compareMBTDifference(hca,hcb);
        }

    int compareMBTDifferenceUnlessExactlyOneHasNoMBTEliminated(
        const unsigned combinationIndex,
        const HeuristicItem& x,
        const HeuristicItem& y) const
        {
        if( exactlyOneHCHasNoMBTEliminated(combinationIndex,x,y) )
            return 0;
        else
            return compareMBTDifference(combinationIndex,x,y);
        }

    int compareMBTifExactlyOneHChasNoMBTeliminated(
        const HeuristicCounters &hca, const HeuristicCounters &hcb) const
        {
        if( hca.MBTEliminated > 0 && hcb.MBTEliminated == 0 )
            // prefer the first atom ("hca")
            return hca.MBTEliminated;

        if( hcb.MBTEliminated > 0 && hca.MBTEliminated == 0 )
            // prefer the second atom ("hcb")
            return -hcb.MBTEliminated;

        return 0;
        }

    int compareMBTifExactlyOneHChasNoMBTeliminated(
        const unsigned combinationIndex,
        const HeuristicItem &x,
        const HeuristicItem &y) const
        {
        HeuristicCounters hca = CombineHeuristics(combinationIndex,x);
        HeuristicCounters hcb = CombineHeuristics(combinationIndex,y);

        return compareMBTifExactlyOneHChasNoMBTeliminated(hca,hcb);
        }

    int compareMBT2Difference(const HeuristicCounters& hca,
                              const HeuristicCounters& hcb) const
        {
        return ( (hca.MBTEliminatedLevel2 - hca.MBTIntroducedLevel2) 
                 - (hcb.MBTEliminatedLevel2 - hcb.MBTIntroducedLevel2) );
        }

    int compareMBT2Difference(const unsigned combinationIndex,
                              const HeuristicItem& x,
                              const HeuristicItem& y) const
        {
        return ( combinedMBT2Difference(combinationIndex,x)
                 - combinedMBT2Difference(combinationIndex,y) );
        }

    int compareMBT2DifferenceUnlessExactlyOneHasNoMBTEliminated(
        const HeuristicCounters& hca,
        const HeuristicCounters& hcb) const
        {
        if( exactlyOneHCHasNoMBTEliminated(hca,hcb) )
            return 0;
        else
            return compareMBT2Difference(hca,hcb);
        }

    int compareMBT2DifferenceUnlessExactlyOneHasNoMBTEliminated(
        const unsigned combinationIndex,
        const HeuristicItem& x,
        const HeuristicItem& y) const
        {
        if( exactlyOneHCHasNoMBTEliminated(combinationIndex,x,y) )
            return 0;
        else
            return compareMBT2Difference(combinationIndex,x,y);
        }

    int compareMBT3Difference(const HeuristicCounters& hca,
                              const HeuristicCounters& hcb) const
        {
        return ( (hca.MBTEliminatedLevel3 - hca.MBTIntroducedLevel3) 
                 - (hcb.MBTEliminatedLevel3 - hcb.MBTIntroducedLevel3) );
        }

    int compareMBT3Difference(const unsigned combinationIndex,
                              const HeuristicItem& x,
                              const HeuristicItem& y) const
        {
        return ( combinedMBT3Difference(combinationIndex,x)
                 - combinedMBT3Difference(combinationIndex,y) );
        }

    int compareMBT3DifferenceUnlessExactlyOneHasNoMBTEliminated(
        const HeuristicCounters& hca,
        const HeuristicCounters& hcb) const
        {
        if( exactlyOneHCHasNoMBTEliminated(hca,hcb) )
            return 0;
        else
            return compareMBT3Difference(hca,hcb);
        }

    int compareMBT3DifferenceUnlessExactlyOneHasNoMBTEliminated(
        const unsigned combinationIndex,
        const HeuristicItem& x,
        const HeuristicItem& y) const
        {
        if( exactlyOneHCHasNoMBTEliminated(combinationIndex,x,y) )
            return 0;
        else
            return compareMBT3Difference(combinationIndex,x,y);
        }

    int compareEliminatedClauses(const HeuristicCounters& hca,
                                 const HeuristicCounters& hcb) const
        {
        return ( hca.ClauseEliminated - hcb.ClauseEliminated );
        }

    int compareEliminatedClauses(const unsigned combinationIndex,
                                 const HeuristicItem& x,
                                 const HeuristicItem& y) const
        {
        HeuristicCounters hca = CombineHeuristics(combinationIndex,x);
        HeuristicCounters hcb = CombineHeuristics(combinationIndex,y);

        return compareEliminatedClauses(hca,hcb);
        }

    int compareEliminatedPVClauses(const HeuristicCounters& hca,
                                   const HeuristicCounters& hcb) const
        {
        return ( hca.PVClauseEliminated - hcb.PVClauseEliminated );
        }

    int compareEliminatedPVClauses(const unsigned combinationIndex,
                                   const HeuristicItem& x,
                                   const HeuristicItem& y) const
        {
        HeuristicCounters hca = CombineHeuristics(combinationIndex,x);
        HeuristicCounters hcb = CombineHeuristics(combinationIndex,y);

        return compareEliminatedPVClauses(hca,hcb);
        }

#ifdef WITH_PVCLEVELS
    int compareEliminatedPVClauses2(const HeuristicCounters& hca,
                                    const HeuristicCounters& hcb) const
        {
        return ( hca.PVClauseEliminatedLevel2 - hcb.PVClauseEliminatedLevel2 );
        }

    int compareEliminatedPVClauses2(const unsigned combinationIndex,
                                    const HeuristicItem& x,
                                    const HeuristicItem& y) const
        {
        HeuristicCounters hca = CombineHeuristics(combinationIndex,x);
        HeuristicCounters hcb = CombineHeuristics(combinationIndex,y);

        return compareEliminatedPVClauses2(hca,hcb);
        }

    int compareEliminatedPVClauses3(const HeuristicCounters& hca,
                                    const HeuristicCounters& hcb) const
        {
        return ( hca.PVClauseEliminatedLevel3 - hcb.PVClauseEliminatedLevel3 );
        }

    int compareEliminatedPVClauses3(const unsigned combinationIndex,
                                    const HeuristicItem& x,
                                    const HeuristicItem& y) const
        {
        HeuristicCounters hca = CombineHeuristics(combinationIndex,x);
        HeuristicCounters hcb = CombineHeuristics(combinationIndex,y);

        return compareEliminatedPVClauses3(hca,hcb);
        }
#endif

    int comparePVClausesDifference(const HeuristicCounters& hca,
                                   const HeuristicCounters& hcb) const
        {
        return ( (hca.PVClauseEliminated - hca.PVClauseIntroduced)
                 - (hcb.PVClauseEliminated - hcb.PVClauseIntroduced) );
        }

    int comparePVClausesDifference(const unsigned combinationIndex,
                                    const HeuristicItem& x,
                                    const HeuristicItem& y) const
        {
        return ( combinedPVCDifference(combinationIndex,x)
                 - combinedPVCDifference(combinationIndex,y) );
        }

#ifdef WITH_PVCLEVELS
    int comparePVClauses2Difference(const HeuristicCounters& hca,
                                    const HeuristicCounters& hcb) const
        {
        return ( (hca.PVClauseEliminatedLevel2 - hca.PVClauseIntroducedLevel2)
                 - (hcb.PVClauseEliminatedLevel2
                    - hcb.PVClauseIntroducedLevel2) );
        }

    int comparePVClauses2Difference(const unsigned combinationIndex,
                                    const HeuristicItem& x,
                                    const HeuristicItem& y) const
        {
        return ( combinedPVC2Difference(combinationIndex,x)
                 - combinedPVC2Difference(combinationIndex,y) );
        }

    int comparePVClauses3Difference(const HeuristicCounters& hca,
                                    const HeuristicCounters& hcb) const
        {
        return ( (hca.PVClauseEliminatedLevel3 - hca.PVClauseIntroducedLevel3)
                 - (hcb.PVClauseEliminatedLevel3
                    - hcb.PVClauseIntroducedLevel3) );
        }

    int comparePVClauses3Difference(const unsigned combinationIndex,
                                    const HeuristicItem& x,
                                    const HeuristicItem& y) const
        {
        return ( combinedPVC3Difference(combinationIndex,x)
                 - combinedPVC3Difference(combinationIndex,y) );
        }
#endif

    int compareEliminatedUndefined(const HeuristicCounters& hca,
                                   const HeuristicCounters& hcb) const
        {
        return ( hca.UndefinedEliminated - hcb.UndefinedEliminated );
        }

    int compareEliminatedUndefined(const unsigned combinationIndex,
                                 const HeuristicItem& x,
                                 const HeuristicItem& y) const
        {
        HeuristicCounters hca = CombineHeuristics(combinationIndex,x);
        HeuristicCounters hcb = CombineHeuristics(combinationIndex,y);

        return compareEliminatedUndefined(hca,hcb);
        }

    int compareHeuristicCost(const HeuristicCounters& hca,
                             const HeuristicCounters& hcb) const
        {
        // We have to prefer the PT that permit the smallest value of the
        // cost of the interpretation.
        // NOTE: Before comparing Heuristic Cost we have to be sure that it
        // is properly initialised.
        if( hca.HeuristicCost && hcb.HeuristicCost )
            {            
            if ( *(hca.HeuristicCost) < *(hcb.HeuristicCost) )
                return 1;
            else if ( *(hca.HeuristicCost) > *(hcb.HeuristicCost) )
                return -1;
            }

        return 0;
        }

    int compareHeuristicCost(const unsigned combinationIndex,
                             const HeuristicItem& x,
                             const HeuristicItem& y) const
        {
        HeuristicCounters hca = CombineHeuristics(combinationIndex,x);
        HeuristicCounters hcb = CombineHeuristics(combinationIndex,y);

        return compareHeuristicCost(hca,hcb);
        }

    int compareEliminatedWCCost(const HeuristicCounters& hca,
                                const HeuristicCounters& hcb) const
        {
        // We have to prefer the PT that permit the greatest value of the
        // sum of the costs of the weak constraints satisfied.
        // NOTE: Before comparing Heuristic Cost we have to be sure that it
        // is properly initialised.
        if( hca.EliminatedWCCost && hcb.EliminatedWCCost )
            if ( *(hca.EliminatedWCCost) > *(hcb.EliminatedWCCost) )
                return 1;
            else if ( *(hca.EliminatedWCCost) < *(hcb.EliminatedWCCost) )
                return -1;

        return 0;
        }

    int compareEliminatedWCCost(const unsigned combinationIndex,
                                const HeuristicItem& x,
                                const HeuristicItem& y) const
        {
        HeuristicCounters hca = CombineHeuristics(combinationIndex,x);
        HeuristicCounters hcb = CombineHeuristics(combinationIndex,y);

        return compareEliminatedWCCost(hca,hcb);
        }

    int compareMBTEliminated(const HeuristicCounters& hca,
                             const HeuristicCounters& hcb) const
        {
        return ( hca.MBTEliminated - hcb.MBTEliminated );
        }

    int compareMBTEliminated(const unsigned combinationIndex,
                                const HeuristicItem& x,
                                const HeuristicItem& y) const
        {
        HeuristicCounters hca = CombineHeuristics(combinationIndex,x);
        HeuristicCounters hcb = CombineHeuristics(combinationIndex,y);

        return compareMBTEliminated(hca,hcb);
        }

    int compareMBT2Eliminated(const HeuristicCounters& hca,
                              const HeuristicCounters& hcb) const
        {
        return ( hca.MBTEliminatedLevel2 - hcb.MBTEliminatedLevel2 );
        }

    int compareMBT2Eliminated(const unsigned combinationIndex,
                                const HeuristicItem& x,
                                const HeuristicItem& y) const
        {
        HeuristicCounters hca = CombineHeuristics(combinationIndex,x);
        HeuristicCounters hcb = CombineHeuristics(combinationIndex,y);

        return compareMBT2Eliminated(hca,hcb);
        }

    int compareMBT3Eliminated(const HeuristicCounters& hca,
                              const HeuristicCounters& hcb) const
        {
        return ( hca.MBTEliminatedLevel3 - hcb.MBTEliminatedLevel3 );
        }

    int compareMBT3Eliminated(const unsigned combinationIndex,
                                const HeuristicItem& x,
                                const HeuristicItem& y) const
        {
        HeuristicCounters hca = CombineHeuristics(combinationIndex,x);
        HeuristicCounters hcb = CombineHeuristics(combinationIndex,y);

        return compareMBT3Eliminated(hca,hcb);
        }

    int compareHardWired(const HeuristicCounters& hca,
                         const HeuristicCounters& hcb) const
        {
        register int diff;

        diff = (hca.MBTEliminated - hca.MBTIntroduced) 
               - (hcb.MBTEliminated - hcb.MBTIntroduced);
        if( diff != 0 )
            return diff;

        // (-mbt2(x) - +mbt2(x)) > (-mbt2(y) - +mbt2(y))

        diff = (hca.MBTEliminatedLevel2 - hca.MBTIntroducedLevel2) 
               - (hcb.MBTEliminatedLevel2 - hcb.MBTIntroducedLevel2);
        if( diff != 0 )
            return diff;

        // (-mbt3(x) - +mbt3(x)) > (-mbt3(y) - +mbt3(y))

        diff = (hca.MBTEliminatedLevel3 - hca.MBTIntroducedLevel3) 
               - (hcb.MBTEliminatedLevel3 - hcb.MBTIntroducedLevel3);
        if( diff != 0 )
            return diff;

        // -partially-violated-clauses(A) > -partially-violated-clauses(B)

        diff = hca.PVClauseEliminated - hcb.PVClauseEliminated;
        if( diff != 0 )
            return diff;

        // HeuristicCost(x) > HeuristicCost(y)
        // We have to prefer the PT that permit the smallest value of the
        // cost of the interpretation.
        // NOTE: Before comparing Heuristic Cost we have to be sure that it
        // is properly initialised.
        if((hca.HeuristicCost)&&(hcb.HeuristicCost))
            if ( *(hca.HeuristicCost) < *(hcb.HeuristicCost))
                return 1;
            else if ( *(hca.HeuristicCost) > *(hcb.HeuristicCost))
                return -1;
        
        // EliminatedWCCost(x) > EliminatedWCCost(y)
        // We have to prefer the PT that permit the greatest value of the
        // sum of the costs of the weak constraint satisfied.
        // NOTE: Before comparing Heuristic Cost we have to be sure that it
        // is properly initialised.
        if((hca.EliminatedWCCost)&&(hcb.EliminatedWCCost))
            if ( *(hca.EliminatedWCCost) > *(hcb.EliminatedWCCost))
                return 1;
            else if ( *(hca.EliminatedWCCost) < *(hcb.EliminatedWCCost))
                return -1;

        // -mbt(x) > -mbt(y)

        diff = hca.MBTEliminated - hcb.MBTEliminated;
        if( diff != 0 )
            return diff;

        // -mbt2(x) > -mbt2(y)

        diff = hca.MBTEliminatedLevel2 - hcb.MBTEliminatedLevel2;
        if( diff != 0 )
            return diff;

        // -mbt3(x) > -mbt3(y)

        diff = hca.MBTEliminatedLevel3 - hcb.MBTEliminatedLevel3;
        if( diff != 0 )
            return diff;

        return 1;

        }

    int compareHardWired(const unsigned combinationIndex,
                              const HeuristicItem& x,
                              const HeuristicItem& y) const
        {
        HeuristicCounters hca = CombineHeuristics(combinationIndex,x);
        HeuristicCounters hcb = CombineHeuristics(combinationIndex,y);

        return compareHardWired(hca,hcb);
        }


    int compareEliminatedMBTsExist(const HeuristicCounters& hca,
                                   const HeuristicCounters& hcb) const
        {
        // If hca.MBTEliminated is greater than 0 and
        // hcb.MBTEliminated is zero, 1 is returned, indicating that
        // hca is preferred.  If hca.MBTEliminated is zero and
        // hcb.MBTEliminated is greater than 0, -1 is returned,
        // indicating that hcb is preferred.
        // In all other cases 0 is returned, indicating that no preference
        // arises from this condition.
        return ( (hca.MBTEliminated > 0) - (hcb.MBTEliminated > 0) );
        }

    int compareEliminatedMBTsExist(const unsigned combinationIndex,
                                   const HeuristicItem& x,
                                   const HeuristicItem& y) const
        {
        HeuristicCounters hca = CombineHeuristics(combinationIndex,x);
        HeuristicCounters hcb = CombineHeuristics(combinationIndex,y);

        return compareEliminatedMBTsExist(hca,hcb);
        }


    int compareWeightedHardWired(const HeuristicCounters& hca,
                                 const HeuristicCounters& hcb) const
        {
        //3*MBT2 + 2*MBT3 + MBT-(MBT2+MBT3)

        int hca_mbt2 = (hca.MBTIntroducedLevel2 - hca.MBTEliminatedLevel2);
        int hca_mbt3 = (hca.MBTIntroducedLevel3 - hca.MBTEliminatedLevel3);
        int hca_mbt  = (hca.MBTIntroduced - hca.MBTEliminated);
        int hcb_mbt2 = (hcb.MBTIntroducedLevel2 - hcb.MBTEliminatedLevel2);
        int hcb_mbt3 = (hcb.MBTIntroducedLevel3 - hcb.MBTEliminatedLevel3);
        int hcb_mbt  = (hcb.MBTIntroduced - hcb.MBTEliminated);

        int hca_wsum = 3*hca_mbt2 + 2*hca_mbt3 + hca_mbt - hca_mbt2 - hca_mbt3;
        int hcb_wsum = 3*hcb_mbt2 + 2*hcb_mbt3 + hcb_mbt - hcb_mbt2 - hcb_mbt3;

        return hca_wsum - hcb_wsum;
        }

    int compareWeightedHardWired(const unsigned combinationIndex,
                                 const HeuristicItem& x,
                                 const HeuristicItem& y) const
        {
        //3*MBT2 + 2*MBT3 + MBT-(MBT2+MBT3)

        int hca_mbt2 = combinedMBT2Difference(combinationIndex,x);
        int hca_mbt3 = combinedMBT3Difference(combinationIndex,x);
        int hca_mbt  = combinedMBTDifference(combinationIndex,x);

        int hcb_mbt2 = combinedMBT2Difference(combinationIndex,y);
        int hcb_mbt3 = combinedMBT3Difference(combinationIndex,y);
        int hcb_mbt  = combinedMBTDifference(combinationIndex,y);

        int hca_wsum = 3*hca_mbt2 + 2*hca_mbt3 + hca_mbt - hca_mbt2 - hca_mbt3;
        int hcb_wsum = 3*hcb_mbt2 + 2*hcb_mbt3 + hcb_mbt - hcb_mbt2 - hcb_mbt3;

        return hca_wsum - hcb_wsum;
        }

#ifdef WITH_PVCLEVELS
    int comparePVCLinearCombination1(const HeuristicCounters& hca,
                                     const HeuristicCounters& hcb) const
        {
        // 2 * PVC2Diff + 1.5 * PVC3Diff + (PVCDiff - PVC2Diff - PVC3Diff)
        // 4 * PVC2Diff + 3 * PVC3Diff + 2 * (PVCDiff - PVC2Diff - PVC3Diff)

        int hca_pvc2 = (hca.PVClauseEliminatedLevel2 
                        - hca.PVClauseIntroducedLevel2);
        int hca_pvc3 = (hca.PVClauseEliminatedLevel3
                        - hca.PVClauseIntroducedLevel3);
        int hca_pvc  = (hca.PVClauseEliminated - hca.PVClauseIntroduced);
        int hcb_pvc2 = (hcb.PVClauseEliminatedLevel2
                        - hcb.PVClauseIntroducedLevel2);
        int hcb_pvc3 = (hcb.PVClauseEliminatedLevel3
                        - hcb.PVClauseIntroducedLevel3);
        int hcb_pvc  = (hcb.PVClauseEliminated - hcb.PVClauseIntroduced);

        int hca_wsum = 4 * hca_pvc2 + 3 * hca_pvc3
            + 2 * (hca_pvc - (hca_pvc2 + hca_pvc3));
        int hcb_wsum = 4 * hcb_pvc2 + 3 * hcb_pvc3 
            + 2 * (hcb_pvc - (hcb_pvc2 + hcb_pvc3));

        return hca_wsum - hcb_wsum;
        }

    int comparePVCLinearCombination1(const unsigned combinationIndex,
                                     const HeuristicItem& x,
                                     const HeuristicItem& y) const
        {
        // 2 * PVC2Diff + 1.5 * PVC3Diff + (PVCDiff - PVC2Diff - PVC3Diff)
        // 4 * PVC2Diff + 3 * PVC3Diff + 2 * (PVCDiff - PVC2Diff - PVC3Diff)

        int hca_pvc2 = combinedPVC2Difference(combinationIndex,x);
        int hca_pvc3 = combinedPVC3Difference(combinationIndex,x);
        int hca_pvc  = combinedPVCDifference(combinationIndex,x);
        int hcb_pvc2 = combinedPVC2Difference(combinationIndex,y);
        int hcb_pvc3 = combinedPVC3Difference(combinationIndex,y);
        int hcb_pvc  = combinedPVCDifference(combinationIndex,y);

        int hca_wsum = 4 * hca_pvc2 + 3 * hca_pvc3
            + 2 * (hca_pvc - (hca_pvc2 + hca_pvc3));
        int hcb_wsum = 4 * hcb_pvc2 + 3 * hcb_pvc3 
            + 2 * (hcb_pvc - (hcb_pvc2 + hcb_pvc3));

        return hca_wsum - hcb_wsum;
        }

    int comparePVCLinearCombination2(const HeuristicCounters& hca,
                                     const HeuristicCounters& hcb) const
        {
        // 3 * PVC2Diff + 2 * PVC3Diff + (PVCDiff - PVC2Diff - PVC3Diff)

        int hca_pvc2 = (hca.PVClauseEliminatedLevel2 
                        - hca.PVClauseIntroducedLevel2);
        int hca_pvc3 = (hca.PVClauseEliminatedLevel3
                        - hca.PVClauseIntroducedLevel3);
        int hca_pvc  = (hca.PVClauseEliminated - hca.PVClauseIntroduced);
        int hcb_pvc2 = (hcb.PVClauseEliminatedLevel2
                        - hcb.PVClauseIntroducedLevel2);
        int hcb_pvc3 = (hcb.PVClauseEliminatedLevel3
                        - hcb.PVClauseIntroducedLevel3);
        int hcb_pvc  = (hcb.PVClauseEliminated - hcb.PVClauseIntroduced);

        int hca_wsum = 3 * hca_pvc2 + 2 * hca_pvc3
            + 1 * (hca_pvc - (hca_pvc2 + hca_pvc3));
        int hcb_wsum = 3 * hcb_pvc2 + 2 * hcb_pvc3 
            + 1 * (hcb_pvc - (hcb_pvc2 + hcb_pvc3));

        return hca_wsum - hcb_wsum;
        }

    int comparePVCLinearCombination2(const unsigned combinationIndex,
                                     const HeuristicItem& x,
                                     const HeuristicItem& y) const
        {
        // 3 * PVC2Diff + 2 * PVC3Diff + (PVCDiff - PVC2Diff - PVC3Diff)

        int hca_pvc2 = combinedPVC2Difference(combinationIndex,x);
        int hca_pvc3 = combinedPVC3Difference(combinationIndex,x);
        int hca_pvc  = combinedPVCDifference(combinationIndex,x);
        int hcb_pvc2 = combinedPVC2Difference(combinationIndex,y);
        int hcb_pvc3 = combinedPVC3Difference(combinationIndex,y);
        int hcb_pvc  = combinedPVCDifference(combinationIndex,y);

        int hca_wsum = 3 * hca_pvc2 + 2 * hca_pvc3
            + 1 * (hca_pvc - (hca_pvc2 + hca_pvc3));
        int hcb_wsum = 3 * hcb_pvc2 + 2 * hcb_pvc3 
            + 1 * (hcb_pvc - (hcb_pvc2 + hcb_pvc3));

        return hca_wsum - hcb_wsum;
        }

    int compareLinearCombination3(const HeuristicCounters& hca,
                                     const HeuristicCounters& hcb) const
        {
        // 12 * MBT + 6 * PVC2Diff + 4 * PVC3Diff
        // + 3 * (PVCDiff - PVC2Diff - PVC3Diff)

        int hca_mbt  = (hca.MBTEliminated - hca.MBTIntroduced);
        int hcb_mbt  = (hcb.MBTEliminated - hcb.MBTIntroduced);

        int hca_pvc2 = (hca.PVClauseEliminatedLevel2 
                        - hca.PVClauseIntroducedLevel2);
        int hca_pvc3 = (hca.PVClauseEliminatedLevel3
                        - hca.PVClauseIntroducedLevel3);
        int hca_pvc  = (hca.PVClauseEliminated - hca.PVClauseIntroduced);
        int hcb_pvc2 = (hcb.PVClauseEliminatedLevel2
                        - hcb.PVClauseIntroducedLevel2);
        int hcb_pvc3 = (hcb.PVClauseEliminatedLevel3
                        - hcb.PVClauseIntroducedLevel3);
        int hcb_pvc  = (hcb.PVClauseEliminated - hcb.PVClauseIntroduced);

        int hca_wsum = 12 * hca_mbt + 6 * hca_pvc2 + 4 * hca_pvc3
            + 3 * (hca_pvc - (hca_pvc2 + hca_pvc3));
        int hcb_wsum = 12 * hcb_mbt + 6 * hcb_pvc2 + 4 * hcb_pvc3 
            + 3 * (hcb_pvc - (hcb_pvc2 + hcb_pvc3));

        return hca_wsum - hcb_wsum;
        }

    int compareLinearCombination3(const unsigned combinationIndex,
                                  const HeuristicItem& x,
                                  const HeuristicItem& y) const
        {
        // 12 * MBT + 6 * PVC2Diff + 4 * PVC3Diff
        // + 3 * (PVCDiff - PVC2Diff - PVC3Diff)

        int hca_mbt  = combinedMBTDifference(combinationIndex,x);
        int hcb_mbt  = combinedMBTDifference(combinationIndex,y);

        int hca_pvc2 = combinedPVC2Difference(combinationIndex,x);
        int hca_pvc3 = combinedPVC3Difference(combinationIndex,x);
        int hca_pvc  = combinedPVCDifference(combinationIndex,x);
        int hcb_pvc2 = combinedPVC2Difference(combinationIndex,y);
        int hcb_pvc3 = combinedPVC3Difference(combinationIndex,y);
        int hcb_pvc  = combinedPVCDifference(combinationIndex,y);

        int hca_wsum = 12 * hca_mbt + 6 * hca_pvc2 + 4 * hca_pvc3
            + 3 * (hca_pvc - (hca_pvc2 + hca_pvc3));
        int hcb_wsum = 12 * hcb_mbt + 6 * hcb_pvc2 + 4 * hcb_pvc3 
            + 3 * (hcb_pvc - (hcb_pvc2 + hcb_pvc3));

        return hca_wsum - hcb_wsum;
        }

    int compareLinearCombination4(const HeuristicCounters& hca,
                                     const HeuristicCounters& hcb) const
        {
        // 6 * PVC2Diff + 4 * PVC3Diff
        // + 3 * (PVCDiff - PVC2Diff - PVC3Diff)

        int hca_pvc2 = (hca.PVClauseEliminatedLevel2 
                        - hca.PVClauseIntroducedLevel2);
        int hca_pvc3 = (hca.PVClauseEliminatedLevel3
                        - hca.PVClauseIntroducedLevel3);
        int hca_pvc  = (hca.PVClauseEliminated - hca.PVClauseIntroduced);
        int hcb_pvc2 = (hcb.PVClauseEliminatedLevel2
                        - hcb.PVClauseIntroducedLevel2);
        int hcb_pvc3 = (hcb.PVClauseEliminatedLevel3
                        - hcb.PVClauseIntroducedLevel3);
        int hcb_pvc  = (hcb.PVClauseEliminated - hcb.PVClauseIntroduced);

        int hca_wsum = 6 * hca_pvc2 + 4 * hca_pvc3
            + 3 * (hca_pvc - (hca_pvc2 + hca_pvc3));
        int hcb_wsum = 6 * hcb_pvc2 + 4 * hcb_pvc3 
            + 3 * (hcb_pvc - (hcb_pvc2 + hcb_pvc3));

        return hca_wsum - hcb_wsum;
        }

    int compareLinearCombination4(const unsigned combinationIndex,
                                  const HeuristicItem& x,
                                  const HeuristicItem& y) const
        {
        // 6 * PVC2Diff + 4 * PVC3Diff
        // + 3 * (PVCDiff - PVC2Diff - PVC3Diff)

        int hca_pvc2 = combinedPVC2Difference(combinationIndex,x);
        int hca_pvc3 = combinedPVC3Difference(combinationIndex,x);
        int hca_pvc  = combinedPVCDifference(combinationIndex,x);
        int hcb_pvc2 = combinedPVC2Difference(combinationIndex,y);
        int hcb_pvc3 = combinedPVC3Difference(combinationIndex,y);
        int hcb_pvc  = combinedPVCDifference(combinationIndex,y);

        int hca_wsum = 6 * hca_pvc2 + 4 * hca_pvc3
            + 3 * (hca_pvc - (hca_pvc2 + hca_pvc3));
        int hcb_wsum = 6 * hcb_pvc2 + 4 * hcb_pvc3 
            + 3 * (hcb_pvc - (hcb_pvc2 + hcb_pvc3));

        return hca_wsum - hcb_wsum;
        }

    int compareLinearCombination5(const HeuristicCounters& hca,
                                  const HeuristicCounters& hcb) const
        {
        // 125*MBT + 25*PVC2 + 5*PVC3 + (PVC - PVC2 - PVC3)

        int hca_mbt  = (hca.MBTEliminated - hca.MBTIntroduced);
        int hcb_mbt  = (hcb.MBTEliminated - hcb.MBTIntroduced);

        int hca_pvc2 = (hca.PVClauseEliminatedLevel2 
                        - hca.PVClauseIntroducedLevel2);
        int hca_pvc3 = (hca.PVClauseEliminatedLevel3
                        - hca.PVClauseIntroducedLevel3);
        int hca_pvc  = (hca.PVClauseEliminated - hca.PVClauseIntroduced);
        int hcb_pvc2 = (hcb.PVClauseEliminatedLevel2
                        - hcb.PVClauseIntroducedLevel2);
        int hcb_pvc3 = (hcb.PVClauseEliminatedLevel3
                        - hcb.PVClauseIntroducedLevel3);
        int hcb_pvc  = (hcb.PVClauseEliminated - hcb.PVClauseIntroduced);

        int hca_wsum = 125*hca_mbt + 25*hca_pvc2 + 5*hca_pvc3
            + (hca_pvc - (hca_pvc2 + hca_pvc3));
        int hcb_wsum = 125*hcb_mbt + 25*hcb_pvc2 + 5*hcb_pvc3 
            + (hcb_pvc - (hcb_pvc2 + hcb_pvc3));

        return hca_wsum - hcb_wsum;
        }

    int compareLinearCombination5(const unsigned combinationIndex,
                                  const HeuristicItem& x,
                                  const HeuristicItem& y) const
        {
        // 125*MBT + 25*PVC2 + 5*PVC3 + (PVC - PVC2 - PVC3)

        int hca_mbt  = combinedMBTDifference(combinationIndex,x);
        int hcb_mbt  = combinedMBTDifference(combinationIndex,y);

        int hca_pvc2 = combinedPVC2Difference(combinationIndex,x);
        int hca_pvc3 = combinedPVC3Difference(combinationIndex,x);
        int hca_pvc  = combinedPVCDifference(combinationIndex,x);
        int hcb_pvc2 = combinedPVC2Difference(combinationIndex,y);
        int hcb_pvc3 = combinedPVC3Difference(combinationIndex,y);
        int hcb_pvc  = combinedPVCDifference(combinationIndex,y);

        int hca_wsum = 125*hca_mbt + 25*hca_pvc2 + 5*hca_pvc3
            + (hca_pvc - (hca_pvc2 + hca_pvc3));
        int hcb_wsum = 125*hcb_mbt + 25*hcb_pvc2 + 5*hcb_pvc3 
            + (hcb_pvc - (hcb_pvc2 + hcb_pvc3));

        return hca_wsum - hcb_wsum;
        }

    int compareLinearCombination6(const HeuristicCounters& hca,
                                  const HeuristicCounters& hcb) const
        {
        // 150*MBT2 + 100*MBT3 + 75*(MBT - MBT2 - MBT3)
        // + 25*PVC2 + 5*PVC3 + (PVC - PVC2 - PVC3)

        int hca_mbt2 = (hca.MBTIntroducedLevel2 - hca.MBTEliminatedLevel2);
        int hcb_mbt2 = (hcb.MBTIntroducedLevel2 - hcb.MBTEliminatedLevel2);

        int hca_mbt3 = (hca.MBTIntroducedLevel3 - hca.MBTEliminatedLevel3);
        int hcb_mbt3 = (hcb.MBTIntroducedLevel3 - hcb.MBTEliminatedLevel3);

        int hca_mbt  = (hca.MBTEliminated - hca.MBTIntroduced);
        int hcb_mbt  = (hcb.MBTEliminated - hcb.MBTIntroduced);

        int hca_pvc2 = (hca.PVClauseEliminatedLevel2 
                        - hca.PVClauseIntroducedLevel2);
        int hca_pvc3 = (hca.PVClauseEliminatedLevel3
                        - hca.PVClauseIntroducedLevel3);
        int hca_pvc  = (hca.PVClauseEliminated - hca.PVClauseIntroduced);
        int hcb_pvc2 = (hcb.PVClauseEliminatedLevel2
                        - hcb.PVClauseIntroducedLevel2);
        int hcb_pvc3 = (hcb.PVClauseEliminatedLevel3
                        - hcb.PVClauseIntroducedLevel3);
        int hcb_pvc  = (hcb.PVClauseEliminated - hcb.PVClauseIntroduced);

        int hca_wsum = 150*hca_mbt2 + 100*hca_mbt3
            + 75*(hca_mbt - (hca_mbt2 + hca_mbt3))
            + 25*hca_pvc2 + 5*hca_pvc3
            + (hca_pvc - (hca_pvc2 + hca_pvc3));
        int hcb_wsum = 150*hcb_mbt2 + 100*hcb_mbt3
            + 75*(hcb_mbt - (hcb_mbt2 + hcb_mbt3))
            + 25*hcb_pvc2 + 5*hcb_pvc3
            + (hcb_pvc - (hcb_pvc2 + hcb_pvc3));

        return hca_wsum - hcb_wsum;
        }

    int compareLinearCombination6(const unsigned combinationIndex,
                                  const HeuristicItem& x,
                                  const HeuristicItem& y) const
        {
        // 150*MBT2 + 100*MBT3 + 75*(MBT - MBT2 - MBT3)
        // + 25*PVC2 + 5*PVC3 + (PVC - PVC2 - PVC3)

        int hca_mbt2 = combinedMBT2Difference(combinationIndex,x);
        int hca_mbt3 = combinedMBT3Difference(combinationIndex,x);
        int hca_mbt  = combinedMBTDifference(combinationIndex,x);
        int hcb_mbt2 = combinedMBT2Difference(combinationIndex,y);
        int hcb_mbt3 = combinedMBT3Difference(combinationIndex,y);
        int hcb_mbt  = combinedMBTDifference(combinationIndex,y);

        int hca_pvc2 = combinedPVC2Difference(combinationIndex,x);
        int hca_pvc3 = combinedPVC3Difference(combinationIndex,x);
        int hca_pvc  = combinedPVCDifference(combinationIndex,x);
        int hcb_pvc2 = combinedPVC2Difference(combinationIndex,y);
        int hcb_pvc3 = combinedPVC3Difference(combinationIndex,y);
        int hcb_pvc  = combinedPVCDifference(combinationIndex,y);

        int hca_wsum = 150*hca_mbt2 + 100*hca_mbt3
            + 75*(hca_mbt - (hca_mbt2 + hca_mbt3))
            + 25*hca_pvc2 + 5*hca_pvc3
            + (hca_pvc - (hca_pvc2 + hca_pvc3));
        int hcb_wsum = 150*hcb_mbt2 + 100*hcb_mbt3
            + 75*(hcb_mbt - (hcb_mbt2 + hcb_mbt3))
            + 25*hcb_pvc2 + 5*hcb_pvc3
            + (hcb_pvc - (hcb_pvc2 + hcb_pvc3));

        return hca_wsum - hcb_wsum;
        }

    int compareLinearCombination7(const HeuristicCounters& hca,
                                  const HeuristicCounters& hcb) const
        {
        // 180*MBT2 + 120*MBT3 + 90*(MBT - MBT2 - MBT3)
        // + 25*PVC2 + 5*PVC3 + (PVC - PVC2 - PVC3)

        int hca_mbt2 = (hca.MBTIntroducedLevel2 - hca.MBTEliminatedLevel2);
        int hcb_mbt2 = (hcb.MBTIntroducedLevel2 - hcb.MBTEliminatedLevel2);

        int hca_mbt3 = (hca.MBTIntroducedLevel3 - hca.MBTEliminatedLevel3);
        int hcb_mbt3 = (hcb.MBTIntroducedLevel3 - hcb.MBTEliminatedLevel3);

        int hca_mbt  = (hca.MBTEliminated - hca.MBTIntroduced);
        int hcb_mbt  = (hcb.MBTEliminated - hcb.MBTIntroduced);

        int hca_pvc2 = (hca.PVClauseEliminatedLevel2 
                        - hca.PVClauseIntroducedLevel2);
        int hca_pvc3 = (hca.PVClauseEliminatedLevel3
                        - hca.PVClauseIntroducedLevel3);
        int hca_pvc  = (hca.PVClauseEliminated - hca.PVClauseIntroduced);
        int hcb_pvc2 = (hcb.PVClauseEliminatedLevel2
                        - hcb.PVClauseIntroducedLevel2);
        int hcb_pvc3 = (hcb.PVClauseEliminatedLevel3
                        - hcb.PVClauseIntroducedLevel3);
        int hcb_pvc  = (hcb.PVClauseEliminated - hcb.PVClauseIntroduced);

        int hca_wsum = 180*hca_mbt2 + 120*hca_mbt3
            + 90*(hca_mbt - (hca_mbt2 + hca_mbt3))
            + 25*hca_pvc2 + 5*hca_pvc3
            + (hca_pvc - (hca_pvc2 + hca_pvc3));
        int hcb_wsum = 180*hcb_mbt2 + 120*hcb_mbt3
            + 90*(hcb_mbt - (hcb_mbt2 + hcb_mbt3))
            + 25*hcb_pvc2 + 5*hcb_pvc3
            + (hcb_pvc - (hcb_pvc2 + hcb_pvc3));

        return hca_wsum - hcb_wsum;
        }

    int compareLinearCombination7(const unsigned combinationIndex,
                                  const HeuristicItem& x,
                                  const HeuristicItem& y) const
        {
        // 180*MBT2 + 120*MBT3 + 90*(MBT - MBT2 - MBT3)
        // + 25*PVC2 + 5*PVC3 + (PVC - PVC2 - PVC3)

        int hca_mbt2 = combinedMBT2Difference(combinationIndex,x);
        int hca_mbt3 = combinedMBT3Difference(combinationIndex,x);
        int hca_mbt  = combinedMBTDifference(combinationIndex,x);
        int hcb_mbt2 = combinedMBT2Difference(combinationIndex,y);
        int hcb_mbt3 = combinedMBT3Difference(combinationIndex,y);
        int hcb_mbt  = combinedMBTDifference(combinationIndex,y);

        int hca_pvc2 = combinedPVC2Difference(combinationIndex,x);
        int hca_pvc3 = combinedPVC3Difference(combinationIndex,x);
        int hca_pvc  = combinedPVCDifference(combinationIndex,x);
        int hcb_pvc2 = combinedPVC2Difference(combinationIndex,y);
        int hcb_pvc3 = combinedPVC3Difference(combinationIndex,y);
        int hcb_pvc  = combinedPVCDifference(combinationIndex,y);

        int hca_wsum = 180*hca_mbt2 + 120*hca_mbt3
            + 90*(hca_mbt - (hca_mbt2 + hca_mbt3))
            + 25*hca_pvc2 + 5*hca_pvc3
            + (hca_pvc - (hca_pvc2 + hca_pvc3));
        int hcb_wsum = 180*hcb_mbt2 + 120*hcb_mbt3
            + 90*(hcb_mbt - (hcb_mbt2 + hcb_mbt3))
            + 25*hcb_pvc2 + 5*hcb_pvc3
            + (hcb_pvc - (hcb_pvc2 + hcb_pvc3));

        return hca_wsum - hcb_wsum;
        }
#endif

    int compareLinearCombinationMBT1(const HeuristicCounters& hca,
                                     const HeuristicCounters& hcb) const
        {
        // 25 * MBT2 + 5 * MBT3 + MBT - MBT2 - MBT3

        int hca_mbt2 = (hca.MBTIntroducedLevel2 - hca.MBTEliminatedLevel2);
        int hcb_mbt2 = (hcb.MBTIntroducedLevel2 - hcb.MBTEliminatedLevel2);

        int hca_mbt3 = (hca.MBTIntroducedLevel3 - hca.MBTEliminatedLevel3);
        int hcb_mbt3 = (hcb.MBTIntroducedLevel3 - hcb.MBTEliminatedLevel3);

        int hca_mbt  = (hca.MBTEliminated - hca.MBTIntroduced);
        int hcb_mbt  = (hcb.MBTEliminated - hcb.MBTIntroduced);

        int hca_wsum = 25*hca_mbt2 + 5*hca_mbt3
            + (hca_mbt - (hca_mbt2 + hca_mbt3));
        int hcb_wsum = 25*hcb_mbt2 + 5*hcb_mbt3
            + (hcb_mbt - (hcb_mbt2 + hcb_mbt3));

        return hca_wsum - hcb_wsum;
        }

    int compareLinearCombinationMBT1(const unsigned combinationIndex,
                                     const HeuristicItem& x,
                                     const HeuristicItem& y) const
        {
        // 25 * MBT2 + 5 * MBT3 + MBT - MBT2 - MBT3

        int hca_mbt2 = combinedMBT2Difference(combinationIndex,x);
        int hca_mbt3 = combinedMBT3Difference(combinationIndex,x);
        int hca_mbt  = combinedMBTDifference(combinationIndex,x);
        int hcb_mbt2 = combinedMBT2Difference(combinationIndex,y);
        int hcb_mbt3 = combinedMBT3Difference(combinationIndex,y);
        int hcb_mbt  = combinedMBTDifference(combinationIndex,y);

        int hca_wsum = 25*hca_mbt2 + 5*hca_mbt3
            + (hca_mbt - (hca_mbt2 + hca_mbt3));
        int hcb_wsum = 25*hcb_mbt2 + 5*hcb_mbt3
            + (hcb_mbt - (hcb_mbt2 + hcb_mbt3));

        return hca_wsum - hcb_wsum;
        }

    int compareLinearCombinationMBT2(const HeuristicCounters& hca,
                                     const HeuristicCounters& hcb) const
        {
        // 9 * MBT2 + 3 * MBT3 + MBT - MBT2 - MBT3

        int hca_mbt2 = (hca.MBTIntroducedLevel2 - hca.MBTEliminatedLevel2);
        int hcb_mbt2 = (hcb.MBTIntroducedLevel2 - hcb.MBTEliminatedLevel2);

        int hca_mbt3 = (hca.MBTIntroducedLevel3 - hca.MBTEliminatedLevel3);
        int hcb_mbt3 = (hcb.MBTIntroducedLevel3 - hcb.MBTEliminatedLevel3);

        int hca_mbt  = (hca.MBTEliminated - hca.MBTIntroduced);
        int hcb_mbt  = (hcb.MBTEliminated - hcb.MBTIntroduced);

        int hca_wsum = 9*hca_mbt2 + 3*hca_mbt3
            + (hca_mbt - (hca_mbt2 + hca_mbt3));
        int hcb_wsum = 9*hcb_mbt2 + 3*hcb_mbt3
            + (hcb_mbt - (hcb_mbt2 + hcb_mbt3));

        return hca_wsum - hcb_wsum;
        }

    int compareLinearCombinationMBT2(const unsigned combinationIndex,
                                     const HeuristicItem& x,
                                     const HeuristicItem& y) const
        {
        // 9 * MBT2 + 3 * MBT3 + MBT - MBT2 - MBT3

        int hca_mbt2 = combinedMBT2Difference(combinationIndex,x);
        int hca_mbt3 = combinedMBT3Difference(combinationIndex,x);
        int hca_mbt  = combinedMBTDifference(combinationIndex,x);
        int hcb_mbt2 = combinedMBT2Difference(combinationIndex,y);
        int hcb_mbt3 = combinedMBT3Difference(combinationIndex,y);
        int hcb_mbt  = combinedMBTDifference(combinationIndex,y);

        int hca_wsum = 9*hca_mbt2 + 3*hca_mbt3
            + (hca_mbt - (hca_mbt2 + hca_mbt3));
        int hcb_wsum = 9*hcb_mbt2 + 3*hcb_mbt3
            + (hcb_mbt - (hcb_mbt2 + hcb_mbt3));

        return hca_wsum - hcb_wsum;
        }

    int compareLinearCombinationMBT3(const HeuristicCounters& hca,
                                     const HeuristicCounters& hcb) const
        {
        // 4 * MBT2 + 2 * MBT3 + MBT - MBT2 - MBT3

        int hca_mbt2 = (hca.MBTIntroducedLevel2 - hca.MBTEliminatedLevel2);
        int hcb_mbt2 = (hcb.MBTIntroducedLevel2 - hcb.MBTEliminatedLevel2);

        int hca_mbt3 = (hca.MBTIntroducedLevel3 - hca.MBTEliminatedLevel3);
        int hcb_mbt3 = (hcb.MBTIntroducedLevel3 - hcb.MBTEliminatedLevel3);

        int hca_mbt  = (hca.MBTEliminated - hca.MBTIntroduced);
        int hcb_mbt  = (hcb.MBTEliminated - hcb.MBTIntroduced);

        int hca_wsum = 4*hca_mbt2 + 2*hca_mbt3
            + (hca_mbt - (hca_mbt2 + hca_mbt3));
        int hcb_wsum = 4*hcb_mbt2 + 2*hcb_mbt3
            + (hcb_mbt - (hcb_mbt2 + hcb_mbt3));

        return hca_wsum - hcb_wsum;
        }

    int compareLinearCombinationMBT3(const unsigned combinationIndex,
                                     const HeuristicItem& x,
                                     const HeuristicItem& y) const
        {
        // 4 * MBT2 + 2 * MBT3 + MBT - MBT2 - MBT3

        int hca_mbt2 = combinedMBT2Difference(combinationIndex,x);
        int hca_mbt3 = combinedMBT3Difference(combinationIndex,x);
        int hca_mbt  = combinedMBTDifference(combinationIndex,x);
        int hcb_mbt2 = combinedMBT2Difference(combinationIndex,y);
        int hcb_mbt3 = combinedMBT3Difference(combinationIndex,y);
        int hcb_mbt  = combinedMBTDifference(combinationIndex,y);

        int hca_wsum = 4*hca_mbt2 + 2*hca_mbt3
            + (hca_mbt - (hca_mbt2 + hca_mbt3));
        int hcb_wsum = 4*hcb_mbt2 + 2*hcb_mbt3
            + (hcb_mbt - (hcb_mbt2 + hcb_mbt3));

        return hca_wsum - hcb_wsum;
        }

    int compareLinearCombinationMBT4(const HeuristicCounters& hca,
                                     const HeuristicCounters& hcb) const
        {
        // 9 * MBT2 + 6 * MBT3 + 4 * (MBT - MBT2 - MBT3)

        int hca_mbt2 = (hca.MBTIntroducedLevel2 - hca.MBTEliminatedLevel2);
        int hcb_mbt2 = (hcb.MBTIntroducedLevel2 - hcb.MBTEliminatedLevel2);

        int hca_mbt3 = (hca.MBTIntroducedLevel3 - hca.MBTEliminatedLevel3);
        int hcb_mbt3 = (hcb.MBTIntroducedLevel3 - hcb.MBTEliminatedLevel3);

        int hca_mbt  = (hca.MBTEliminated - hca.MBTIntroduced);
        int hcb_mbt  = (hcb.MBTEliminated - hcb.MBTIntroduced);

        int hca_wsum = 9*hca_mbt2 + 6*hca_mbt3
            + 4*(hca_mbt - (hca_mbt2 + hca_mbt3));
        int hcb_wsum = 9*hcb_mbt2 + 6*hcb_mbt3
            + 4*(hcb_mbt - (hcb_mbt2 + hcb_mbt3));

        return hca_wsum - hcb_wsum;
        }

    int compareLinearCombinationMBT4(const unsigned combinationIndex,
                                     const HeuristicItem& x,
                                     const HeuristicItem& y) const
        {
        // 9 * MBT2 + 6 * MBT3 + 4 * (MBT - MBT2 - MBT3)

        int hca_mbt2 = combinedMBT2Difference(combinationIndex,x);
        int hca_mbt3 = combinedMBT3Difference(combinationIndex,x);
        int hca_mbt  = combinedMBTDifference(combinationIndex,x);
        int hcb_mbt2 = combinedMBT2Difference(combinationIndex,y);
        int hcb_mbt3 = combinedMBT3Difference(combinationIndex,y);
        int hcb_mbt  = combinedMBTDifference(combinationIndex,y);

        int hca_wsum = 9*hca_mbt2 + 6*hca_mbt3
            + 4*(hca_mbt - (hca_mbt2 + hca_mbt3));
        int hcb_wsum = 9*hcb_mbt2 + 6*hcb_mbt3
            + 4*(hcb_mbt - (hcb_mbt2 + hcb_mbt3));

        return hca_wsum - hcb_wsum;
        }

    int compareLinearCombinationMBT5(const HeuristicCounters& hca,
                                     const HeuristicCounters& hcb) const
        {
        // 16 * MBT2 + 12 * MBT3 + 9 * (MBT - MBT2 - MBT3)

        int hca_mbt2 = (hca.MBTIntroducedLevel2 - hca.MBTEliminatedLevel2);
        int hcb_mbt2 = (hcb.MBTIntroducedLevel2 - hcb.MBTEliminatedLevel2);

        int hca_mbt3 = (hca.MBTIntroducedLevel3 - hca.MBTEliminatedLevel3);
        int hcb_mbt3 = (hcb.MBTIntroducedLevel3 - hcb.MBTEliminatedLevel3);

        int hca_mbt  = (hca.MBTEliminated - hca.MBTIntroduced);
        int hcb_mbt  = (hcb.MBTEliminated - hcb.MBTIntroduced);

        int hca_wsum = 16*hca_mbt2 + 12*hca_mbt3
            + 9*(hca_mbt - (hca_mbt2 + hca_mbt3));
        int hcb_wsum = 16*hcb_mbt2 + 12*hcb_mbt3
            + 9*(hcb_mbt - (hcb_mbt2 + hcb_mbt3));

        return hca_wsum - hcb_wsum;
        }

    int compareLinearCombinationMBT5(const unsigned combinationIndex,
                                     const HeuristicItem& x,
                                     const HeuristicItem& y) const
        {
        // 16 * MBT2 + 12 * MBT3 + 9 * (MBT - MBT2 - MBT3)

        int hca_mbt2 = combinedMBT2Difference(combinationIndex,x);
        int hca_mbt3 = combinedMBT3Difference(combinationIndex,x);
        int hca_mbt  = combinedMBTDifference(combinationIndex,x);
        int hcb_mbt2 = combinedMBT2Difference(combinationIndex,y);
        int hcb_mbt3 = combinedMBT3Difference(combinationIndex,y);
        int hcb_mbt  = combinedMBTDifference(combinationIndex,y);

        int hca_wsum = 16*hca_mbt2 + 12*hca_mbt3
            + 9*(hca_mbt - (hca_mbt2 + hca_mbt3));
        int hcb_wsum = 16*hcb_mbt2 + 12*hcb_mbt3
            + 9*(hcb_mbt - (hcb_mbt2 + hcb_mbt3));

        return hca_wsum - hcb_wsum;
        }

#ifdef WITH_PVCLEVELS
    int compareLinearCombinationPVC1(const HeuristicCounters& hca,
                                     const HeuristicCounters& hcb) const
        {
        // 25*PVC2 + 5*PVC3 + (PVC - PVC2 - PVC3)

        int hca_pvc2 = (hca.PVClauseEliminatedLevel2 
                        - hca.PVClauseIntroducedLevel2);
        int hca_pvc3 = (hca.PVClauseEliminatedLevel3
                        - hca.PVClauseIntroducedLevel3);
        int hca_pvc  = (hca.PVClauseEliminated - hca.PVClauseIntroduced);
        int hcb_pvc2 = (hcb.PVClauseEliminatedLevel2
                        - hcb.PVClauseIntroducedLevel2);
        int hcb_pvc3 = (hcb.PVClauseEliminatedLevel3
                        - hcb.PVClauseIntroducedLevel3);
        int hcb_pvc  = (hcb.PVClauseEliminated - hcb.PVClauseIntroduced);

        int hca_wsum = 25*hca_pvc2 + 5*hca_pvc3
            + (hca_pvc - (hca_pvc2 + hca_pvc3));
        int hcb_wsum = 25*hcb_pvc2 + 5*hcb_pvc3
            + (hcb_pvc - (hcb_pvc2 + hcb_pvc3));

        return hca_wsum - hcb_wsum;
        }

    int compareLinearCombinationPVC1(const unsigned combinationIndex,
                                     const HeuristicItem& x,
                                     const HeuristicItem& y) const
        {
        // 25*PVC2 + 5*PVC3 + (PVC - PVC2 - PVC3)

        int hca_pvc2 = combinedPVC2Difference(combinationIndex,x);
        int hca_pvc3 = combinedPVC3Difference(combinationIndex,x);
        int hca_pvc  = combinedPVCDifference(combinationIndex,x);
        int hcb_pvc2 = combinedPVC2Difference(combinationIndex,y);
        int hcb_pvc3 = combinedPVC3Difference(combinationIndex,y);
        int hcb_pvc  = combinedPVCDifference(combinationIndex,y);

        int hca_wsum = 25*hca_pvc2 + 5*hca_pvc3
            + (hca_pvc - (hca_pvc2 + hca_pvc3));
        int hcb_wsum = 25*hcb_pvc2 + 5*hcb_pvc3
            + (hcb_pvc - (hcb_pvc2 + hcb_pvc3));

        return hca_wsum - hcb_wsum;
        }

    int compareLinearCombinationPVCI1(const HeuristicCounters& hca,
                                      const HeuristicCounters& hcb) const
        {
        // 25*PVCI2 + 5*PVCI3 + (PVCI - PVCI2 - PVCI3)

        int hca_pvci2 = hca.PVClauseIntroducedLevel2;
        int hca_pvci3 = hca.PVClauseIntroducedLevel3;
        int hca_pvci  = hca.PVClauseIntroduced;
        int hcb_pvci2 = hcb.PVClauseIntroducedLevel2;
        int hcb_pvci3 = hcb.PVClauseIntroducedLevel3;
        int hcb_pvci  = hcb.PVClauseIntroduced;

        int hca_wsum = 25*hca_pvci2 + 5*hca_pvci3
            + (hca_pvci - (hca_pvci2 + hca_pvci3));
        int hcb_wsum = 25*hcb_pvci2 + 5*hcb_pvci3
            + (hcb_pvci - (hcb_pvci2 + hcb_pvci3));

        return hca_wsum - hcb_wsum;
        }

    int compareLinearCombinationPVCI1(const unsigned combinationIndex,
                                      const HeuristicItem& x,
                                      const HeuristicItem& y) const
        {
        HeuristicCounters hca = CombineHeuristics(combinationIndex,x);
        HeuristicCounters hcb = CombineHeuristics(combinationIndex,y);

        return compareLinearCombinationPVCI1(hca,hcb);
        }

    int compareLinearCombinationPVCI2(const HeuristicCounters& hca,
                                      const HeuristicCounters& hcb) const
        {
        // 125*MBT + 25*PVCI2 + 5*PVCI3 + (PVCI - PVCI2 - PVCI3)

        int hca_mbt   = hca.MBTIntroduced;
        int hca_pvci2 = hca.PVClauseIntroducedLevel2;
        int hca_pvci3 = hca.PVClauseIntroducedLevel3;
        int hca_pvci  = hca.PVClauseIntroduced;
        int hcb_mbt   = hcb.MBTIntroduced;
        int hcb_pvci2 = hcb.PVClauseIntroducedLevel2;
        int hcb_pvci3 = hcb.PVClauseIntroducedLevel3;
        int hcb_pvci  = hcb.PVClauseIntroduced;

        int hca_wsum = 125*hca_mbt + 25*hca_pvci2 + 5*hca_pvci3
            + (hca_pvci - (hca_pvci2 + hca_pvci3));
        int hcb_wsum = 125*hcb_mbt + 25*hcb_pvci2 + 5*hcb_pvci3
            + (hcb_pvci - (hcb_pvci2 + hcb_pvci3));

        return hca_wsum - hcb_wsum;
        }

    int compareLinearCombinationPVCI2(const unsigned combinationIndex,
                                      const HeuristicItem& x,
                                      const HeuristicItem& y) const
        {
        HeuristicCounters hca = CombineHeuristics(combinationIndex,x);
        HeuristicCounters hcb = CombineHeuristics(combinationIndex,y);

        return compareLinearCombinationPVCI2(hca,hcb);
        }
#endif

    int compareByIndex(const unsigned comparisonIndex,
                       const unsigned combinationIndex,
                       const HeuristicItem& x,
                       const HeuristicItem& y) const
        {
        assert( OptionHeuristicsConsiderComplementaryPT );
        if( comparisonIndex >= 1000 )
            return -compareByIndex(comparisonIndex-1000,combinationIndex,
                                   x,y);

        switch(comparisonIndex)
            {
            case 0:
                return compareMBTDifference(combinationIndex,x,y);
            case 1:
                return compareMBTDifferenceUnlessExactlyOneHasNoMBTEliminated(
                    combinationIndex,x,y);
            case 2:
                return compareMBT2Difference(combinationIndex,x,y);
            case 3:
                return compareMBT2DifferenceUnlessExactlyOneHasNoMBTEliminated(
                    combinationIndex,x,y);
            case 4:
                return compareMBT3Difference(combinationIndex,x,y);
            case 5:
                return compareMBT3DifferenceUnlessExactlyOneHasNoMBTEliminated(
                    combinationIndex,x,y);
            case 6:
                return compareEliminatedClauses(combinationIndex,x,y);
            case 7:
                return compareHeuristicCost(combinationIndex,x,y);
            case 8:
                return compareEliminatedWCCost(combinationIndex,x,y);
            case 9:
                return compareMBTEliminated(combinationIndex,x,y);
            case 10:
                return compareMBT2Eliminated(combinationIndex,x,y);
            case 11:
                return compareMBT3Eliminated(combinationIndex,x,y);
            case 12:
                return compareHardWired(combinationIndex,x,y);
            case 13:
                return compareEliminatedMBTsExist(combinationIndex,x,y);
            case 14:
                return compareWeightedHardWired(combinationIndex,x,y);
            case 15:
                return compareEliminatedPVClauses(combinationIndex,x,y);
            case 16:
                return comparePVClausesDifference(combinationIndex,x,y);
#ifdef WITH_PVCLEVELS
            case 17:
                return compareEliminatedPVClauses2(combinationIndex,x,y);
            case 18:
                return compareEliminatedPVClauses3(combinationIndex,x,y);
            case 19:
                return comparePVClauses2Difference(combinationIndex,x,y);
            case 20:
                return comparePVClauses3Difference(combinationIndex,x,y);
#endif
            case 21:
                return compareEliminatedUndefined(combinationIndex,x,y);
#ifdef WITH_PVCLEVELS
            case 22:
                return comparePVCLinearCombination1(combinationIndex,x,y);
            case 23:
                return comparePVCLinearCombination2(combinationIndex,x,y);
            case 24:
                return compareLinearCombination3(combinationIndex,x,y);
            case 25:
                return compareLinearCombination4(combinationIndex,x,y);
            case 26:
                return compareLinearCombination5(combinationIndex,x,y);
            case 27:
                return compareLinearCombination6(combinationIndex,x,y);
            case 28:
                return compareLinearCombination7(combinationIndex,x,y);
#endif
            case 29:
                return compareLinearCombinationMBT1(combinationIndex,x,y);
            case 30:
                return compareLinearCombinationMBT2(combinationIndex,x,y);
            case 31:
                return compareLinearCombinationMBT3(combinationIndex,x,y);
#ifdef WITH_PVCLEVELS
            case 32:
                return compareLinearCombinationPVC1(combinationIndex,x,y);
#endif
            case 33:
                return compareLinearCombinationMBT4(combinationIndex,x,y);
            case 34:
                return compareLinearCombinationMBT5(combinationIndex,x,y);
#ifdef WITH_PVCLEVELS
            case 35:
                return compareLinearCombinationPVCI1(combinationIndex,x,y);
            case 36:
                return compareLinearCombinationPVCI2(combinationIndex,x,y);
#endif
            // If you insert something here, don't forget to adapt
            // CRITERIA_MAX in dl.h and to add a string to
            // hCriteriaLabels in dl.C
            default:
                cerr << comparisonIndex 
                     << ": Invalid heuristic comparison index."
                     << endl;
                SafeExit(1);
            }
        // Dummy to avoid warning; all cases should be covered by the
        // switch.
        assert(0);
        return 0;
        }

    int compareByIndex(const unsigned comparisonIndex,
                       const HeuristicCounters& hca,
                       const HeuristicCounters& hcb) const
        {
        if( comparisonIndex >= 1000 )
            return -compareByIndex(comparisonIndex-1000,hca,hcb);

        switch(comparisonIndex)
            {
            case 0:
                return compareMBTDifference(hca,hcb);
            case 1:
                return compareMBTDifferenceUnlessExactlyOneHasNoMBTEliminated(
                    hca,hcb);
            case 2:
                return compareMBT2Difference(hca,hcb);
            case 3:
                return compareMBT2DifferenceUnlessExactlyOneHasNoMBTEliminated(
                    hca,hcb);
            case 4:
                return compareMBT3Difference(hca,hcb);
            case 5:
                return compareMBT3DifferenceUnlessExactlyOneHasNoMBTEliminated(
                    hca,hcb);
            case 6:
                return compareEliminatedClauses(hca,hcb);
            case 7:
                return compareHeuristicCost(hca,hcb);
            case 8:
                return compareEliminatedWCCost(hca,hcb);
            case 9:
                return compareMBTEliminated(hca,hcb);
            case 10:
                return compareMBT2Eliminated(hca,hcb);
            case 11:
                return compareMBT3Eliminated(hca,hcb);
            case 12:
                return compareHardWired(hca,hcb);
            case 13:
                return compareEliminatedMBTsExist(hca,hcb);
            case 14:
                return compareWeightedHardWired(hca,hcb);
            case 15:
                return compareEliminatedPVClauses(hca,hcb);
            case 16:
                return comparePVClausesDifference(hca,hcb);
#ifdef WITH_PVCLEVELS
            case 17:
                return compareEliminatedPVClauses2(hca,hcb);
            case 18:
                return compareEliminatedPVClauses3(hca,hcb);
            case 19:
                return comparePVClauses2Difference(hca,hcb);
            case 20:
                return comparePVClauses3Difference(hca,hcb);
#endif
            case 21:
                return compareEliminatedUndefined(hca,hcb);
#ifdef WITH_PVCLEVELS
            case 22:
                return comparePVCLinearCombination1(hca,hcb);
            case 23:
                return comparePVCLinearCombination2(hca,hcb);
            case 24:
                return compareLinearCombination3(hca,hcb);
            case 25:
                return compareLinearCombination4(hca,hcb);
            case 26:
                return compareLinearCombination5(hca,hcb);
            case 27:
                return compareLinearCombination6(hca,hcb);
            case 28:
                return compareLinearCombination7(hca,hcb);
#endif
            case 29:
                return compareLinearCombinationMBT1(hca,hcb);
            case 30:
                return compareLinearCombinationMBT2(hca,hcb);
            case 31:
                return compareLinearCombinationMBT3(hca,hcb);
#ifdef WITH_PVCLEVELS
            case 32:
                return compareLinearCombinationPVC1(hca,hcb);
#endif
            case 33:
                return compareLinearCombinationMBT4(hca,hcb);
            case 34:
                return compareLinearCombinationMBT5(hca,hcb);
#ifdef WITH_PVCLEVELS
            case 35:
                return compareLinearCombinationPVCI1(hca,hcb);
            case 36:
                return compareLinearCombinationPVCI2(hca,hcb);
#endif
            // If you insert something here, don't forget to adapt
            // CRITERIA_MAX in dl.h and to add a string to
            // hCriteriaLabels in dl.C
            default:
                cerr << comparisonIndex 
                     << ": Invalid heuristic comparison index."
                     << endl;
                SafeExit(1);
            }
        // Dummy to avoid warning; all cases should be covered by the
        // switch.
        assert(0);
        return 0;
        }

    inline HeuristicCounters CombineHeuristics(
        const unsigned combinationIndex,
        const HeuristicItem& x ) const
        {
        assert(x.hccompl);

        HeuristicCounters HC1 = x.hc;
        HeuristicCounters HC2 = *(x.hccompl);

        switch( combinationIndex )
            {
            case 0:
                return HC1 + HC2;
            case 1:
                return HC1 * HC2;
            case 2:
                return (HC1 * HC2 * 1024) + HC1 + HC2;
            case 3:
                return (HC1 + 1) * (HC2 + 1);
            case 4:
                return (HC1 * 2) + HC2;
            case 5:
                // minimum
                // if HC2 >= HC1
                if( (*this)(HC2,HC1) )
                    return HC1;
                else
                    return HC2;
            case 6:
                // maximum
                // if HC2 >= HC1
                if( (*this)(HC2,HC1) )
                    return HC2;
                else
                    return HC1;
            case 7:
                //     HC1 +   HC2 + 0.5*min(HC1,HC2)
                // = 2*HC1 + 2*HC2 +     min(HC1,HC2)
                if( (*this)(HC2,HC1) )
                    return (HC1 * 2) + (HC2 * 2) + HC1;
                else
                    return (HC1 * 2) + (HC2 * 2) + HC2;
            case 8:
                //     HC1 +   HC2 + (1/3)*min(HC1,HC2)
                // = 3*HC1 + 3*HC2 +      min(HC1,HC2)
                if( (*this)(HC2,HC1) )
                    return (HC1 * 3) + (HC2 * 3) + HC1;
                else
                    return (HC1 * 3) + (HC2 * 3) + HC2;
            default:
                assert(0);
            }
        // All cases should be covered by the switch.
        assert(0);
        // Dummy return to avoid a warning.
        return HeuristicCounters();
        }

    inline int CombineHeuristics(
        const unsigned combinationIndex,
        const int v1,
        const int v2) const
        {
        switch( combinationIndex )
            {
            case 0:
                return v1 + v2;
            case 1:
                return v1 * v2;
            case 2:
                if( v1 >= 0 && v2 >= 0)
                    return (v1 * v2 * 1024) + v1 + v2;
                else if( v1 + v2 < 0 )
                    // exactly one of v1, v2 is < 0
                    return v1 + v2;
                else
                    // v1 < 0 && v2 < 0
                    return - (v1 * v2 * 1024) + v1 + v2;
            case 3:
                return (v1 + 1) * (v2 + 1);
            case 4:
                return (v1 * 2) + v2;
            case 5:
                // minimum
                // if HC2 >= HC1
                if( v2 >= v1 )
                    return v1;
                else
                    return v2;
            case 6:
                // maximum
                // if HC2 >= HC1
                if( v2 >= v1 )
                    return v2;
                else
                    return v1;
            case 7:
                //     HC1 +   HC2 + 0.5*min(HC1,HC2)
                // = 2*HC1 + 2*HC2 +     min(HC1,HC2)
                if( v2 >= v1 )
                    return (v1 * 2) + (v2 * 2) + v1;
                else
                    return (v1 * 2) + (v2 * 2) + v2;
            case 8:
                //     HC1 +   HC2 + (1/3)*min(HC1,HC2)
                // = 3*HC1 + 3*HC2 +      min(HC1,HC2)
                if( v2 >= v1 )
                    return (v1 * 3) + (v2 * 3) + v1;
                else
                    return (v1 * 3) + (v2 * 3) + v2;
            default:
                assert(0);
            }
        // All cases should be covered by the switch.
        assert(0);
        // Dummy return to avoid a warning.
        return 0;
        }

    bool operator()(const HeuristicCounters& xhc,
                    const HeuristicCounters& yhc) const
        {
        register int diff;

        for( CriteriaSequence::const_iterator i 
                 = heuristicSequence.begin();
             i != heuristicSequence.end();
             i++ )
            {
            diff = compareByIndex(*i,xhc,yhc);

            if( diff != 0 )
                {
#ifdef HEURISTICSTATS
		statsHCriterionApplied[*i]++;
#endif
                if( diff > 0 )
                    return true;
                else
                    return false;
                }
            }
#ifdef HEURISTICSTATS
        statsHCriterionApplied[CRITERIA_MAX+1]++;
#endif
        return true;
        }

    bool operator()(const HeuristicItem& x, const HeuristicItem& y) const
        {
        register int diff;
        CriteriaSequence::const_iterator j
            // The next line is only here to suppress a warning.
            = heuristicSequence.begin();

        assert( ! OptionHeuristicsConsiderComplementaryPT
                || (heuristicSequence.size()
                    == heuristicCombinationSequence.size()));

        if( OptionHeuristicsConsiderComplementaryPT )
            j = heuristicCombinationSequence.begin();

        for( CriteriaSequence::const_iterator i
                 = heuristicSequence.begin();
             i != heuristicSequence.end();
             i++ )
            {
            if( OptionHeuristicsConsiderComplementaryPT )
                {
                assert(j != heuristicCombinationSequence.end());
                diff = compareByIndex(*i,*j,x,y);
                j++;
                }
            else
                diff = compareByIndex(*i,x.hc,y.hc);

            if( diff != 0 )
                {
#ifdef HEURISTICSTATS
		statsHCriterionApplied[*i]++;
#endif
                if( diff > 0 )
                    return true;
                else
                    return false;
                }
            }
#ifdef HEURISTICSTATS
        statsHCriterionApplied[CRITERIA_MAX+1]++;
#endif
        return true;
        }
    };

#endif

// Local Variables:
// mode: c++
// c-file-style: "dl"
// End:
