// depgraph.h -- contains the following classes
//   ADJACENCY_LIST
//   DEPGRAPH_SCC<GRAPH_REPRESENTATION,FN_SETCOMPONENT_TYPE>
//   COMPONENT_FACTORY
//   DG_ATOMS<ATOM_TYPE>
//   DG_P_ATOMS<ATOM_TYPE>
//   DEPGRAPH<ATOM_TYPE,LITERAL_TYPE,CONSTRAINTS_TYPE,PROGRAM_TYPE>

#ifndef DEPGRAPH_H
#define DEPGRAPH_H

//#define DEPGRAPH_BENCHMARK

#include <iostream>
#include <vector>
#ifdef DEPGRAPH_BENCHMARK
#include <sys/times.h>
#endif

// This type makes it easier to distinguish between node indices and other
// integers in the implementation of ADJACENCY_LIST.
typedef unsigned INDEX_T;

////////////////////////////////////////////////////////////////////////////////
class ADJACENCY_LIST
// Represent a directed graph.
//
// ADJ_BRANCH_TYPE stores the nodes directly reachable from a given node.
// Depending on its declaration, we consider (vector<>) or ignore (set<>)
// multiple edges between the same two nodes.
//
// Currently we use beginReachableNodes()/endReachableNodes() to access the
// concrete nodes reachable from a given node.
    {
public:
    typedef vector<INDEX_T> ADJ_BRANCH_TYPE;
    // typedef set<INDEX_T,less<INDEX_T> > ADJ_BRANCH_TYPE;

private:
    vector<ADJ_BRANCH_TYPE> adjlist;

public:
    // Return the number of nodes in the adjacency list.
    // In the case that we store sparse indices, this is the size of the
    // data structure.
    unsigned size() const
        {
        return adjlist.size();
        }

    ADJACENCY_LIST(unsigned n=0) : adjlist(n)
        // n is the number of nodes that the adjacency list will contain.
        {
        }

    ADJACENCY_LIST(const ADJACENCY_LIST&)
        {
        assert( 0 );
        }

    void operator=(const ADJACENCY_LIST&)
        {
        assert( 0 );
        }

    // Insert a new edge.  If a node index (from or to) is too large, we
    // could transparently grow internal structures, but don't do that yet.
    void connect(const INDEX_T &from, const INDEX_T &to)
        {
        assert( max(from,to)+1 <= size() );

        adjlist[from].push_back(to);
        }

    // Provides an iterator through all the edges departing from the node
    // with index idx.
    ADJ_BRANCH_TYPE::const_iterator beginReachableNodes(const INDEX_T idx)
        const
        { return adjlist[idx].begin(); }

    ADJ_BRANCH_TYPE::const_iterator endReachableNodes(const INDEX_T idx)
        const
        { return adjlist[idx].end(); }

    // Is there an edge from "from" to "to"?
    bool connected(const INDEX_T &from, const INDEX_T &to) const
        {
        for(ADJ_BRANCH_TYPE::const_iterator k = adjlist[from].begin();
            k != adjlist[from].end();
            k++ )
            {
            if( *k == to )
                return true;
            }
        return false;
        }

    // Output the adjacency matrix of the graph.  The complexity of
    // O(#nodes^3) is due to the representation as an adjacency list.
    void printMatrix() const
        {
        for(unsigned i = 0; i < adjlist.size(); i++)
            {
            for(unsigned j = 0; j < adjlist.size(); j++)
                {
                unsigned found=0;

                for(ADJ_BRANCH_TYPE::const_iterator k = beginReachableNodes(i);
                    k != endReachableNodes(i);
                    k++ )
                    {
                    if( *k == j )
                        found++;
                    }
                cout << found << " ";
                }
            cout << endl;
            }
        }
    };


////////////////////////////////////////////////////////////////////////////////
/**
   Computes the strongly connected components of a directed graph;
   overloads the function operator.

   GRAPHREP requirements: \\
    NODEIDX_ITERATOR_TYPE beginReachableNodes(INDEX_T); \\
    NODEIDX_ITERATOR_TYPE endReachableNodes(INDEX_T); \\
    unsigned size(); \\
   NODEIDX_ITERATOR_TYPE is a forward_iterator \\
    (global view:) INDEX_T operator*(NODEIDX_ITERATOR_TYPE) \\
   FN_SETCOMPONENT_TYPE requirements: \\
    formerly:
    void operator()(const INDEX_T nodeidx, const INDEX_T componentidx); \\
    now:
    void assignComponent()(const INDEX_T nodeidx,
                           const INDEX_T componentidx); \\
    void createInterComponentLink(const INDEX_T nodeidx1,
                                  const INDEX_T nodeidx2); \\

   Instances of this class are used as function objects.
   GRAPHREP is a representation of the graph which has to be already
   filled with data when the algorithm is called; the number of nodes
   is needed during construction.
   An object of this class can be created, initialized with a number of nodes
   and can be called several times with different graphs.
   There are two alternative contructors, one that just initializes the object
   and one that also starts the computation. In the first case, the computation
   is started by calling operator()(). Using the COMPONENT_FACTORY_TYPE argument
   in the constructor or in operator()(),
   storing of the results can be customized. Every time a component is found,
   the COMPONENT_FACTORY_TYPE object is treated as a function object and is
   called as many times as there are nodes in the component, with the nodeidx
   and the componentidx as arguments.

   The algorithm used (the Tarjan algorithm) has an important property,
   besides the fact that it has a complexity of O(N):
   The output computed has a special ordering, which is very useful for the
   computations that have to be done on the components. The graph of the
   components is always a DAG (directed acyclic graph), since all cycles
   were removed (if something were cyclic, it would be strongly connected
   and therefore be part of a component).
   The Tarjan algorithm orders components always in a way that a component
   that are pointed to by another component always come before these
   components that are "higher" in the graph. Since this is a DAG, this
   does not uniquely specify the full ordering of a general graph,
   The order of these nodes at the infimum in the ordering depends on the
   ordering of the rules in the input (transformed to an adjacency list
   on which this algorithm runs).

   The reason for this property is the way this algorithm works.
   It does depth-first search through the graph. It floods the
   graph, moving from a start node to not yet visited neighboring nodes as
   long as it can, ie. as long as there are edges accessible.
   From a starting node it follows the edges to higher nodes by always
   moving up, and to lower noder by always moving into the direction
   pointed to (to lower nodes).
*/
template<class GRAPH_REP, class NODEIDX_ITERATOR_TYPE,
    class FN_SETCOMPONENT_TYPE>
class DEPGRAPH_SCC
    {
private:
    unsigned theSize,numComponents;

    // this value isn't always initialized from the beginning -> pointer
    FN_SETCOMPONENT_TYPE *pSCT;

    // things needed by the algorithm; globally ...
    vector<unsigned> theStack;
    unsigned lastDfsNumber;        // some of these types should be
    unsigned *dfsNumber;        // changed to INDEX_T
    unsigned *highWater;
    bool *onStack;

    /**
       dfs() performs a depth first search for strongly connected
       components.
       This is the recursive part and is called by compute().
       Read the paper on the Tarjan algorithm for additional comments.
       THE ALGORITHM WORKS WITH MULTIPLE EDGES, TOO.
       The depth-first-search is the reason for the ordering of the
       components mentioned before.
       Returns true when dfs was finally able to build a cluster (in this
       execution of dfs, not in a recursive call).
    */
    bool dfs(const GRAPH_REP &gr,unsigned vertex);
protected:
    /**
       init() allocates memory.
       Since there have to be some statically sized arrays, the
       number of nodes in the graph have to be known from the beginning.
    */
    void init(unsigned graphsize);

    /**
       compute() initializes the variables needed for the algorithm and
       starts the recursion by calling dfs().
       Starting with the first node, every node is checked if it was already
       processed. If this is not the case, the depth-first search is
       started on it.
    */
    void compute(const GRAPH_REP &,FN_SETCOMPONENT_TYPE &);

    ///
    unsigned size(void) { return theSize; }
public:
    /// calls init()
    DEPGRAPH_SCC(unsigned graphsize)
        {
        init(graphsize);
        }
    /// calls init() and compute()
    DEPGRAPH_SCC(const GRAPH_REP &gr,FN_SETCOMPONENT_TYPE &sct)
        {
        init(gr.size());
        compute(gr,sct);
        }
    ///
    ~DEPGRAPH_SCC()
        {
        delete[] highWater;
        delete[] dfsNumber;
        delete[] onStack;
        }

    /// calls compute()
    void operator()(const GRAPH_REP &gr,FN_SETCOMPONENT_TYPE &sct)
        {
        compute(gr,sct);
        }
    };


////////////////////////////////////////////////////////////////////////////////
template<class GRAPH_REP, class NODEIDX_ITERATOR_TYPE,
    class FN_SETCOMPONENT_TYPE>
void DEPGRAPH_SCC<GRAPH_REP, NODEIDX_ITERATOR_TYPE, FN_SETCOMPONENT_TYPE>::
    init(unsigned graphsize)
    {
    theSize=graphsize;

    highWater = new unsigned[theSize];
    dfsNumber = new unsigned[theSize];
    onStack = new bool[theSize];
    }


////////////////////////////////////////////////////////////////////////////////
template<class GRAPH_REP, class NODEIDX_ITERATOR_TYPE,
    class FN_SETCOMPONENT_TYPE>
void DEPGRAPH_SCC<GRAPH_REP, NODEIDX_ITERATOR_TYPE, FN_SETCOMPONENT_TYPE>::
    compute(const GRAPH_REP &gr,FN_SETCOMPONENT_TYPE &sct)
    {
    pSCT=&sct;

    numComponents=0;
    lastDfsNumber=0;

    for(unsigned i = 0;i < theSize;i++)
        {
        dfsNumber[i] = 0;
        highWater[i] = 0;
        }
    for(unsigned i = 0;i < theSize;i++)
        {
        if(dfsNumber[i] == 0)
            {
            (void)dfs(gr,i);
            }
        }
    }


////////////////////////////////////////////////////////////////////////////////
template<class GRAPH_REP, class NODEIDX_ITERATOR_TYPE,
    class FN_SETCOMPONENT_TYPE>
bool DEPGRAPH_SCC<GRAPH_REP, NODEIDX_ITERATOR_TYPE, FN_SETCOMPONENT_TYPE>::
    dfs(const GRAPH_REP &gr,INDEX_T v)
    {
    lastDfsNumber++;          // an ever-increasing node index sequencer
    dfsNumber[v] = lastDfsNumber; // the soc. security number of the nodes
    highWater[v] = lastDfsNumber; // this may change, dfsNumber may not
    theStack.push_back(v);
    onStack[v] = true;

    for(NODEIDX_ITERATOR_TYPE        // INDEX_T *
        w = gr.beginReachableNodes(v);
        w != gr.endReachableNodes(v); w++)
        {
        if(dfsNumber[*w] == 0) // node hasn't been visited yet
            {
            if(dfs(gr,*w))
                {
                pSCT->createInterComponentLink(v, *w, true);
                }

            if(highWater[*w] < highWater[v])
                {
                // highWater is always decreasing, therefore
                // it's got a funny name ...
                highWater[v] = highWater[*w];
                }
            }
           // checking dfsNumber[v] should be unnecessary, since
           // highWater[v] is smaller or equal
        else if((dfsNumber[*w] < dfsNumber[v]) && onStack[*w] &&
            (dfsNumber[*w] < highWater[v]))
               // loop found !
            {
            highWater[v] = dfsNumber[*w];
            }
        else if((dfsNumber[*w] < dfsNumber[v]) && (!onStack[*w]) &&
            (dfsNumber[*w] < highWater[v]))
               // important for reordering: edge to already
               // processed node found
            {
            pSCT->createInterComponentLink(v, *w, false);
            }
        }

    if(highWater[v] == dfsNumber[v])    // found strongly conn.comp.
        {
        INDEX_T w;     // a node index - only needed for saving
                       // the component

        numComponents++;
        do
            {
            w = theStack.back();

            theStack.pop_back();

            pSCT->assignComponent(w,numComponents-1);

            onStack[w] = false;
            }
        while(w != v);

        return true;
        }
    else
        {
        return false;
        }
    }



#define INVALID_COMPONENT_INDEX ((unsigned)-1)


////////////////////////////////////////////////////////////////////////////////
/**
   Adaptor between DEPGRAPH and DEPGRAPH_SCC - an object of this class
   is passed on to DEPGRAPH_SCC; filled with data there; and produces
   the data structures needed in DEPGRAPH when DEPGRAPH_SCC is
   finished.

   An instance of COMPONENT_FACTORY has to be given to DEPGRAPH_SCC as an
   argument and the operator() of this class is called every time that
   DEPGRAPH_SCC finds a new component. The createInterComponentLink()
   method is necessary to reorder the components in a way that weakly
   connected components are kept together.
*/
class COMPONENT_FACTORY
    {
private:
    /// Represents components by nodes.
    /// It is the responsibility of the code that instantiates this class
    /// to deallocate nodeComponents!
    unsigned *nodeComponents, *weakNumber;

    ///
    struct WEAK_EDGE
    {
       ///
       INDEX_T from, to;
       ///
       bool direct;
    };

    ///
    vector<WEAK_EDGE> weak_edges;
    ///
    unsigned long numComponents, numNodes;
    ///
    bool preprocessed;
    /// known_atoms is passed on as a parameter to this class and still
    /// belongs to the outside world; we must not deallocate it.
    const bool *known_atoms;

    /** this is a monotonically increasing counter for those nodes
        (=components, because the nodes have no arcs connecting to them)
        which are "holes", which are not in known_atoms.
    */
    unsigned dead_node_count;

public:
    /// returns the number of components
    inline unsigned long size(void) const
        {
        return numComponents;
        }

    COMPONENT_FACTORY()
        {
        assert(0);
        }

    COMPONENT_FACTORY(const COMPONENT_FACTORY &)
        {
        assert(0);
        }

    void operator=(const COMPONENT_FACTORY &)
        {
        assert(0);
        }

    inline COMPONENT_FACTORY(unsigned long _numNodes, const bool *_known_atoms)
        : numComponents(0), numNodes(_numNodes), preprocessed(false),
        known_atoms(_known_atoms), dead_node_count(0)
        {
        // FIXME: assert( _numNodes > 0 ) causes tons of testsuite failures!

        nodeComponents = new unsigned[_numNodes];
        }

    /**
       This Method is called by DEPGRAPH_SCC. It handles the information
       gained by running the Tarjan algorithm, the components.
       Complexity: amortized constant ;)
    */
    inline void assignComponent(const INDEX_T nodeidx,const INDEX_T componentidx)
        {
        assert(!preprocessed);

        if(!known_atoms[nodeidx])
            {
            // s is hopefully a very large number that will throw an
            // assertion if it is accessed somewhere.
            nodeComponents[nodeidx] = INVALID_COMPONENT_INDEX;

            dead_node_count++;
            return;
            }

        nodeComponents[nodeidx] = componentidx - dead_node_count;

        if((componentidx - dead_node_count + 1) > numComponents)
            numComponents = componentidx - dead_node_count + 1;
        }

    /** This method is called by DEPGRAPH_SCC whenever an edge is detected
       that is not part of a strongly connected component.
       Note that the ordering of these edges is not arbitrary.
       Edges that are called "direct" here express a grouping of
       components that is automatically generated by the depth-first
       search. Only "indirect" edges link components that might have to
       be grouped together. On the other hand, the "direct" edges are
       here still necessary to detect a correct grouping.
       The Tarjan algorithm could mark mark weakly connected components
       as far as they are detected in every call of dfs (from compute)
       with a common integer value (similar to the highwater values for
       strongly connected components). This would make storing and
       processing the "direct" edges here unnecessary.
       This is currently not done because it would make the interface of
       DEPGRAPH_SCC inconsistent.
    */
    inline void createInterComponentLink(const INDEX_T nodeidx1,
                                  const INDEX_T nodeidx2, bool direct)
        {
        assert(!preprocessed);
        assert(known_atoms[nodeidx1] && known_atoms[nodeidx2]);

        weak_edges.push_back(WEAK_EDGE());
        weak_edges.back().from   = nodeidx1;
        weak_edges.back().to     = nodeidx2;
        weak_edges.back().direct = direct;
        }

private:
    /** This new implementation of preprocessComponents() should behave in
        exactly the same way as preprocessComponents_old(), but should be
        faster.
    */
    void preprocessComponents()
        {
        if(DTraceLevel >= 1)
            cdebug << "In COMPONENT_FACTORY::preprocessComponents()" << endl;

        if(size() == 0)
            {
            if(DTraceLevel >= 1)
                cdebug << "Leaving COMPONENT_FACTORY::preprocessComponents()"
                       << " (empty component graph)" << endl;
            return;
            }

        // Build graph of strongly connected components.
        ADJACENCY_LIST aux_adjlist(size());

        for(vector<WEAK_EDGE>::iterator i = weak_edges.begin();
                                        i != weak_edges.end(); i++)
            {
            assert(nodeComponents[i->from]
                != nodeComponents[i->to]);

            aux_adjlist.connect(nodeComponents[i->from], nodeComponents[i->to]);
            aux_adjlist.connect(nodeComponents[i->to], nodeComponents[i->from]);
            }

        bool *dummy = new bool[size()];
        for(unsigned i = 0; i < size(); i++)
            dummy[i] = true;

        COMPONENT_FACTORY aux_cf(size(), dummy);
    
        DEPGRAPH_SCC< ADJACENCY_LIST,
            ADJACENCY_LIST::ADJ_BRANCH_TYPE::const_iterator,
            COMPONENT_FACTORY> (aux_adjlist, aux_cf);

        // Now what we've got is a mapping of SCC indexes to WCC indexes.
        unsigned *wcc_by_scc = aux_cf.getNonorderedNodeComponents();

        vector<unsigned> *WCC = new vector<unsigned>[aux_cf.size()];
        for(unsigned scc_index = 0; scc_index < size(); scc_index++)
            {
            assert(wcc_by_scc[scc_index] < aux_cf.size());

            WCC[wcc_by_scc[scc_index]].push_back(scc_index);
            }
        // The SCCs are ordered in the right way inside each WCC.

        if(DTraceLevel >= 1)
            {
            cdebug << "(weakly) connected components:" << endl;

            for(unsigned i = 0; i < aux_cf.size(); i++)
                {
                cdebug << "WCC#" << i << " = { ";
                for(unsigned j = 0; j < WCC[i].size(); j++)
                    cdebug << "SCC#" << WCC[i][j] << " ";
                cdebug << "}" << endl;
                }
            }

        unsigned *componentMapping = new unsigned[size()];

        unsigned scc_index = 0;
        for(unsigned i = 0; i < aux_cf.size(); i++)
            {
            for(unsigned j = 0; j < WCC[i].size(); j++)
                {
                componentMapping[WCC[i][j]] = scc_index++;

                if(DTraceLevel >= 1)
                    cdebug << "componentMapping[" << WCC[i][j] << "] = "
                           << componentMapping[WCC[i][j]] << endl;
                } 
            }
        assert(scc_index == size());

        // this loop also reverses the ordering of atoms
        for(unsigned n = 0; n < numNodes; n++) // for each node
            {
            if(known_atoms[n])
                nodeComponents[n] = size() - 1
                                    - componentMapping[nodeComponents[n]];

            if(DTraceLevel >= 1)
                cdebug << "nodeComponents[" << n << "] = SCC#"
                       << nodeComponents[n] << "(rotated)"
                       << " known?: " << (known_atoms[n] ? "true" : "false")
                       << endl;
            }

        delete[] dummy;
        delete[] componentMapping;
        delete[] WCC;
        delete[] wcc_by_scc;

        if(DTraceLevel >= 1)
            cdebug << "Leaving COMPONENT_FACTORY::preprocessComponents()"
                   << endl;
        }

    /** reorders the internal node-to-component mapping.
       This method is called during the first execution of
       makeNodeComponents. After that, neither
       operator() nor createInterComponentLink may be called anymore.

       First, after some memory allocation and initialization steps,
       for every strongly connected component (SCC), an integer value
       is computed that is equal for all SCC in a weakly connected
       component (WCC).
       Next, these integers, which may be non-contiguous, are mapped to
       WCC indexes, which start with 0 and are contiguous.
       Finally, this mapping is used to reorder the components.
       Besides the reordering based on these indexes, the component
       order is reversed (since the Tarjan algorithm creates the
       components when coming back from depth-first recursion).
       Besides reordering the nodeComponents values, the
       COMPONENT_TYPE vector is filled which may be accessed through
       makeComponents.

       Currently, WCC are not explicitly extracted, although this could
       be done easily here.

       FIXME: This class (and the ordering of components in general) could
       probably be made more flexible if this method were take out of the
       class.
       This method is O(N^2) with N = number of strongly connected components.
    */
    void preprocessComponents_old()
        {
        assert(!preprocessed);

        // Initialize weakNumber and compDone arrays.
        weakNumber = new unsigned[size()];
        bool *compDone = new bool[size()];

        for(unsigned c = 0; c < size(); c++)
            {
            weakNumber[c] = c;
            compDone[c] = false;
            }

        // Compute weakNumber values (all strongly connected components in
        // a weakly connected component have the same weakNumber value).
        for(vector<WEAK_EDGE>::iterator i = weak_edges.begin();
            i != weak_edges.end(); i++)
            {
            assert(nodeComponents[i->from] < size());
            assert(nodeComponents[i->to] < size());

            // This constraint is assured by the Tarjan algorithm, which
            // operates in a depth-first manner. Since all edges are followed
            // from a given starting point and all reachable nodes are touched,
            // the component index of a from node of a "weak edge" (which is
            // outside of any strongly connected component) is always strictly
            // greater than the component index of the to node.
            assert(nodeComponents[i->from] > nodeComponents[i->to]);

            // these are for sure valid atom indexes
            unsigned from_weakNumber =
               weakNumber[nodeComponents[i->from]];
            unsigned to_weakNumber =
               weakNumber[nodeComponents[i->to]];

            if(from_weakNumber != to_weakNumber)
               for(unsigned j = 0; j < size(); j++) // for each SCC
                {
                if(weakNumber[j] == to_weakNumber)
                    {
                    weakNumber[j] = from_weakNumber;
                    }
                }
            }

        // Based on that, build a mapping on which the reordering is
        // based (from weakNumber values. Here, the component ordering
        // is also reversed - dfs nature of the Tarjan algorithm !)
        // also, make components.

        // componentMapping is a permutation on the component indexes.
        unsigned *componentMapping = new unsigned[size()];

        unsigned idx = 0;

        for(int c = size() - 1; c >= 0; c--)
           if(!compDone[c])
            {
            for(int d = c; d >= 0; d--)
               if(weakNumber[d] == weakNumber[c])
                {
                componentMapping[d] = idx++;
                compDone[d] = true;
                }
            }

        unsigned *transformed_weakNumber = new unsigned[size()];

        for(unsigned c = 0; c < size(); c++)
            transformed_weakNumber[componentMapping[c]] = weakNumber[c];

        for(unsigned n = 0; n < numNodes; n++) // for each node
            if(known_atoms[n])
                nodeComponents[n] = componentMapping[nodeComponents[n]];

        delete[] componentMapping;
        delete[] compDone;
        delete[] weakNumber;
        weakNumber = transformed_weakNumber;

        preprocessed = true;
        }

public:
    /** returns a pointer to an array of per-node component indexes.
       Currently, calling this method several times returns the same
       pointer every time, so this is no real factory class.

       It is the responsibility of the code that instantiates this class
       to deallocate nodeComponents.
    */
    inline unsigned *makeNodeComponents(void)
        {
        if(!preprocessed)
            preprocessComponents();

        return nodeComponents;
        }

    /** returns the same pointer as makeNodeComponents() !
    */
    inline unsigned *getNonorderedNodeComponents(void) const
        {
        assert(!preprocessed);
        return nodeComponents;
        }

    inline unsigned *makeWeakNumber(void)
        {
        if(!preprocessed)
            {
            preprocessComponents();
            }

        return weakNumber;
        }
    inline const vector<WEAK_EDGE> &getWeakLinks(void) const
        {
        return weak_edges;
        }
    };

////////////////////////////////////////////////////////////////////////////////

inline const GATOM &getNonAggrAtom(const GATOM &atom)
    {
    return atom;
    }

inline const ATOM &getNonAggrAtom(const ATOM &atom)
    {
    if(atom.isAggregate())
        return atom.getAggregate().getAuxAtom();
    else
        return atom;
    }

////////////////////////////////////////////////////////////////////////////////
/**
   Another container type for atoms.
   See DG_ATOMS for additional comments.
*/
template<class ATOM_TYPE>
class DG_P_ATOMS
    {
private:
    ///
    bool compareAOBN;  // compare atoms only by name

    /// if there is no atom for a certain index, the pointer is zero.
    vector<const ATOM_TYPE *> atoms;

public:
    ///
    DG_P_ATOMS(bool cAOBN) : compareAOBN(cAOBN) { }

    ///
    inline unsigned size(void) const   { return atoms.size(); }

    ///
    inline bool isKnown(const INDEX_T idx) const
        {
        return ((idx < atoms.size()) && (atoms[idx] != 0));
        }

    ///
    inline bool isKnown(const ATOM_TYPE &atom) const
        {
        return isKnown(getNonAggrAtom(atom).getIndex());
        }

    ///
    inline const ATOM_TYPE &operator[](const INDEX_T idx) const
        {
        assert(isKnown(idx));

        return *atoms[idx];
        }

    /**  areAtomsEqual() does not access any other members in the class.
         As soon as GATOM:operator== is implemented in the way it is needed
         here (comparing just the names), this function can be removed
    */
    bool areAtomsEqual(const ATOM_TYPE &a1,const ATOM_TYPE &a2) const
        {
        if(compareAOBN)
            return getNonAggrAtom(a1).getName() ==
                   getNonAggrAtom(a2).getName();
        else
            return getNonAggrAtom(a1) == getNonAggrAtom(a2);
        }

    ///
    void outName(ostream &o, const INDEX_T idx) const
        {
        assert(isKnown(idx));

        if(compareAOBN)
            o << atoms[idx]->getName();
        else
            o << (*atoms[idx]);
        }

    /// in the case that the atom is not found, -1 is returned
    inline unsigned getIdx(const ATOM_TYPE &atom) const
        {
        assert(isKnown(atom));

        return getNonAggrAtom(atom).getIndex();
        }

    /** insertUnique() looks up the index of an atom in the atoms vector that
        corresponds to the indexes used by DEPGRAPH_SCC and the algorithm
        to compute the strongly connected components.
        idx must be allowed to be invalid (has to be checked outside !!!)
    */
    INDEX_T insertUnique(const ATOM_TYPE &atom)
        {
        const ATOM_TYPE &atom2 = getNonAggrAtom(atom);

        // In the nonground case, atoms are identified by name.
        unsigned i = atom2.getIndex();

        // Append (i - atoms.size() + 1) null pointers.
        // ATTENTION: If atoms grows, these pointers may become invalid.
        while( i >= atoms.size() )
            atoms.push_back(0);

        if(atoms[i] == 0)
            atoms[i] = &atom2;

        return i;
        }
    };


////////////////////////////////////////////////////////////////////////////////
/**
   builds a dependency graph and provides various kinds of information
   on the strongly connected components of this graph.

   PROGRAM_TYPE is the container type that is used for storing the
   subprograms associated with the components. The rules that are given to the
   constructor have to be of the type
   vector<TRULE<ATOM_TYPE,TLITERAL<ATOM_TYPE> > > .

   DEPGRAPH transforms the incoming rules into a dependency
   graph and stores the mapping between the contained atoms and the index 
   representation used in DEPGRAPH_SCC.
   Computes stratifications/componentHasDisjunctions.
   Assigns rules and constraints to the components.

   The atoms of the dependency graph are stored either in DG_ATOMS or in
   DG_P_ATOMS. The difference between those two classes is
   that the first one actually stores the atoms, while the latter one only
   stores references on them. There's no template argument for this type
   because the outside world doesn't have to know about it and it does not
   make sense to use both of them. The first one should only be used when
   underlying data structures are changed and for debugging. The initialisation
   of the second class is more effective.
)
   DEPGRAPH can be used in two ways: with or without paying attention to
   negations and disjunctions when building the strongly connected components.
   In the first case, the argument includeAllEdges of the constructor has to be
   set to true (and it has to be set to false otherwise).
*/
template<class ATOM_TYPE, class CONSTRAINTS_TYPE, class PROGRAM_TYPE>
class DEPGRAPH
    {
    ADJACENCY_LIST *graph;
    DG_P_ATOMS<ATOM_TYPE> atoms;

    bool hasNonstandardEdges;

    /// an array with a bool for every component
    bool *componentHasNegations;

    /// an array with a bool for every component
    bool *componentHasDisjunctions;

    /// an array with a bool for every component
    bool *componentHasRecursions;

    /** an array with a bool for every rule.
       ruleIsRecursive is initialized in every case, even if it is never
       used (in the current implementation) because WITHSPECIALEDGES
       is set to true. The computation is cheap (no extra loop) and
       it requires only a small fixed-size array.
    */
    bool *ruleIsRecursive;

    bool *componentConstraintsArePositive;

    /// -1 : not initialized, 0: false, 1: true
    int programIsCyclic,
        programIsHCF,
        programIsPositive,
        programIsStronglyHCF;

    /** reads the rules to get the total number of atoms to set up the
       adjacency list.
       fills the atoms container to get the size of the matrix.
       It is not needed if the size of
       the atoms container doesn't have to be known before it is filled. \\

       Complexity: square, worst case=average case
    */
    void readAtomsFromRules(
        const vector<TRULE<ATOM_TYPE,TLITERAL<ATOM_TYPE> > > &rules);

    /** builds the graph.
       While reading the rules, edges are inserted both into
       the adjacency list and (if hasNonstandardEdges is true and the
       edges are disjunctive or negative) the 'edges' container. \\

       Complexity: O(L*M*N), L=number of rules,
       M=number of literals in conjunction, N=number of atoms in the head.
    */
    void buildGraph( const vector<TRULE<ATOM_TYPE, TLITERAL<ATOM_TYPE > > >
                     &rules );
    void buildGraphAdd(const ATOM &, const LITERAL &);
    void buildGraphAdd(const GATOM &, const GLITERAL &);

    /** inserts constraints.
       insertConstraintsIntoGraph() ignores EDB atoms.
       The others are connected to double chains like the
       atoms in the disjunctions of the rules (rather than inserting
       connections between every two atoms).
    */
    void insertConstraintsIntoGraph(const CONSTRAINTS_TYPE &constraints);

    /** marks the components that have negations or disjunctions.
       Uses the edges object which stores all the "special" edges. \\
       Complexity: Linear.
    */
    void computeSpecialEdges(void);

    /**
       computes the rules that belong to each of the components.
       For each rule, and for each atom in the head, add the rule to the
       component of the head atom. This method also checks if components
       contain recursions and if rules are recursive or exit rules.
    */
    void computeAllComponentSubprograms(
        const vector<TRULE<ATOM_TYPE,TLITERAL<ATOM_TYPE> > > &rules,
        vector<PROGRAM_TYPE> &);

    ///
    struct EDGE
        {
        INDEX_T from,to;
        enum TYPE { NEGATIVE, DISJUNCTIVE } type;

        EDGE()
            : from(0), to(0), type(NEGATIVE)
            { assert(0); }

        EDGE(unsigned f,unsigned t, TYPE tp)
            : from(f), to(t), type(tp)
            { }

        EDGE(const EDGE &e)
            : from(e.from), to(e.to), type(e.type)
            { }

        void operator=(const EDGE&)
            { assert(0); }
        };
    ///
    typedef vector<EDGE> EDGES;

    ///
    EDGES edges; // should be called specialEdges

public:
    /**
       WITHCONSTRAINTEDGES triggers the insertion of constraints into the
       dependency graph while SUPPRESSCONSTRAINTASSIGNMENT prevents the
       assignment of constraints to components. These two options work
       independently, and either one can be selected no matter what the
       value of the other option is.
    */
    enum OPTION { NONGROUND=0x0001, WITHSPECIALEDGES=0x0010,
                  WITHCONSTRAINTEDGES=0x0100,
                  SUPPRESSCONSTRAINTASSIGNMENT=0x1000,
                  SIMONASORDERING=0x00010000 };

    ///
    typedef vector<INDEX_T> COMPONENT;
    ///
    typedef vector<COMPONENT> COMPONENTS;

private:
    /// represents nodes by components
    COMPONENTS *ptrComponents;
    /// represents components by nodes
    INDEX_T *nodeComponents;
                    // INDEX_T (unsigned) is the type of the COMPONENT indexes
    ///
    vector<PROGRAM_TYPE> subprograms;

    /**
       The value of this pointer is set to NULL in the constructor; the
       vector is allocated in computeAllComponentConstraints() and deallocated
       in the destructor.
       The vector is allocated with a fixed size.
    */
    vector<CONSTRAINTS_TYPE> *ptrComponentConstraints;

    /**
       computes the constraints on all components.

       Complexity O(L*M*N), L=total number of constraints, see comment on
       computeConstraintComponent() for explanations of M and N.

       Old comment for computeConstraintComponent():
       matches the lowest component index of the atoms
       contained in a constraint to the constraint.
       The Tarjan algorithm traverses the graph by depth first
       search and the 'leaf' components have the lowest indices. \\

       In the case that there's only EDB predicates in the constraint,
       the result is an invalid index (ptrComponents->size(), which is by one
       greater than the largest index).

       Negative EDB predicates could be removed from the constraint but
       currently AREN'T !

       Worst-case complexity: O(N*M), N = number of literals in the
       constraint, M = number of atoms in the graph.
       Average case complexity: O(N*M/2) since the search in the inner
       for-loop is terminated when the atom is found.
    */
    void computeAllComponentConstraints(const CONSTRAINTS_TYPE &cc);

    /// Simona's ordering for positive dependency graphs: respects negation
    void special_ordering(COMPONENT_FACTORY &cf, const bool *known_atoms);
public:
    /// dummy method, fails
    DEPGRAPH(const DEPGRAPH<ATOM_TYPE, CONSTRAINTS_TYPE,
             PROGRAM_TYPE> &) :
        atoms(false)
        {
        assert(0);
        }

    /// dummy method, fails
    void operator=(const DEPGRAPH<ATOM_TYPE,
        CONSTRAINTS_TYPE, PROGRAM_TYPE> &)
        {
        assert(0);
        }

    /** Constructor.
       The constructor initializes the data structures,
       establishes the connections of the graph while going through the
       rules (by calling buildGraph()), computes the strongly connected
       components, and finds the rules and constraints that belong to each
       of the components.
       The procedures computeAllComponentSubprograms() and
       computeAllComponentConstraints() don't have to be called here, they
       just have to be called before their results are used, and after the
       computation of the strongly connected components. They should
       stay in here if all the rules and the constraints for every
       component will be used in every instance of DEPGRAPH, i.e. if in the
       future DEPGRAPH will be used only to do some of the tasks it is made
       for, it would be appropriate to remove those two procedures from the
       constructor and call them explicitly (on demand).
    */
    DEPGRAPH(const vector<TRULE<ATOM_TYPE,TLITERAL<ATOM_TYPE> > > &rules,
        const CONSTRAINTS_TYPE &constraints,
        unsigned long options);

    /// destructor
    ~DEPGRAPH();

    /** use the size() method of the returned reference to check for the
       total number of components.
    */
    vector<COMPONENT> &getComponents(void) const
        { return *ptrComponents; }

    /** returns the the index of the component which contains the given atom.
    */
    unsigned getAtomComponent(const ATOM_TYPE& atom, bool assert=true) const
        {
        if( assert )
            {
            assert( atoms.isKnown(atom) );
            assert( nodeComponents[atoms.getIdx(atom)] !=
                    INVALID_COMPONENT_INDEX );
            }
        else
            {
            if( ! atoms.isKnown(atom) )
                return INVALID_COMPONENT_INDEX;
            }

        return nodeComponents[atoms.getIdx(atom)];
        }

    /// provides access to atoms
    const DG_P_ATOMS<ATOM_TYPE> &getAtoms(void) const { return atoms; }


    // component-specific functions

    /**
       returns true if the component with the index componentidx is
       stratified.
       isStratified() is only applicable on graphs with negations and
       disjunctions.

       A program is stratified if all of its components are stratified.

       A component is stratified if it contains no negations.
       Components are built on the full graph, containing negative and
       positive edges, including disjunctions.

       { a :- b. b :- not c. } is stratified because b and c are in
       different components.
       { a :- b. b :- not a. } is not stratified because a and b are
       strongly connected (there is a cycle) and there is a negation.
       { a v b :- not c.  b :- not a. } is not stratified because
       a and b are in one component (because of the disjunction) and
       because of b :- not a. , {a b} is not stratified.
       The negation in a v b :- not c. is not relevant because c is in
       its own component.

       The atoms of a disjunction are always strongly connected to
       each other. (This is true for the full dependency graph. In the
       positive dependency graph, which is generated when the flag
       WITHSPECIALEDGES in the argument options of the constructor is
       not set, disjunctions are not inserted into the graph.)
    */
    bool isStratified(unsigned componentIdx);

    /** returns true if the component with the index componentidx is
       head-cycle free (HCF).
       A component is HCF if it contains no disjunctions or if it is acyclic.
       isHCF() is only applicable on graphs without negations and
       disjunctions.
       See isStronglyHCF() for examples and an explanation of head-cycle
       freedom.
    */
    bool isHCF(unsigned componentIdx);

    /// checks if all components are head-cycle free
    bool isHCF()
        {
        if(programIsHCF == -1)
            {
            programIsHCF = 1; // true

            for(unsigned i = 0; i < ptrComponents->size(); i++)
                {
                if(!isHCF(i))
                    {
                    programIsHCF = 0; // false
                    break;
                    }
                }
           }
        return (programIsHCF == 1);
        }

    /// returns true iff a component is positive (does not contain negations)
    bool isPositive(unsigned componentIdx) const
        {
        return !componentHasNegations[componentIdx];
        }

    /// checks is all components are positive
    bool isPositive()
        {
        if(programIsPositive == -1)
            {
            programIsPositive = 1; // true

            for(unsigned i = 0; i < ptrComponents->size(); i++)
                {
                if(componentHasNegations[i])
                    {
                    programIsPositive = 0; // false
                    break;
                    }
                }
            }
        return (programIsPositive == 1);
        }

    /// checks if the constraints of a component do not contain negated literals
    bool areConstraintsPositive(unsigned componentIdx)
        {
        assert(ptrComponentConstraints != 0);

        if(componentConstraintsArePositive == 0)
            {
            componentConstraintsArePositive = new bool[ptrComponents->size()];

            for(unsigned i = 0; i < ptrComponents->size(); i++)
                {
                componentConstraintsArePositive[i] = true;
                for(typename CONSTRAINTS_TYPE::const_iterator
                      j = (*ptrComponentConstraints)[i].begin();
                      j != (*ptrComponentConstraints)[i].end(); j++)
                    {
                    if(j->neg_begin() != j->neg_end())
                        {
                        componentConstraintsArePositive[i] = false;
                        break;
                        }
                    }
                }
            }

        return componentConstraintsArePositive[componentIdx];
        }

    /** returns true if and only if the component is strongly
       head-cycle free.

       Hcf and strongly hcf may only be applied to positive dependency
       graphs,
       which are dependency graphs containing no disjunctive or negative
       edges. For example, the program { a v b. a :- b, not c. } would
       result in a positive dependency graph equal to a full dependency
       graph built from the program { a :- b. }.

       A component is strongly head-cycle free if it is head-cycle
       free and there is no recursion caused by positive literals in
       the body.
       (Again, EVERY strongly hcf component is also hcf!)
       Recursion means that the head as well as the body contain the
       same literal (for example, { a v b :- a, c. } contains a in the
       head as well as in the body and is therefore recursive).

       Every component that does not contain any disjunctions is hcf and
       also strongly hcf (but this is trivial, and such a component does
       not have to be explicitly marked as hcf).

       The program { a v b :-.  b :- b. } is hcf but not strongly hcf
       because the rule b :- b. is recursive with b being positive in the
       body.
       The program { a v b :-. b :- not b, c. } is both hcf and strongly hcf
       because the is no recursion through positive literals in the body.
       { a v b :-. b :- c. } is hcf and strongly hcf (the components do not
       contain any disjunctions since there is a component for every atom).
       { a v b :-. a :- b. } is hcf and strongly hcf since again there is
       a STRONGLY CONNECTED component for every atom (In a strongly
       connected component, all atoms are in a cycle, and this is not the
       case here.).
       { a :- b. b :- a. } results in one component but is trivially hcf
       and strongly hcf because there is no disjunction between atoms of
       the component.
       { a v b :-.  a :- b.  b :- a. } is not hcf and therefore also not
       strongly hcf.
       { a v b v c :-. a :- b. b :- a. } is not hcf. c is in a component
       separate from { a b }, but this is of no relevance. There is a
       disjunction a v b and a and b are in the same component, and
       therefore { a b } is not hcf. If any of the components is not
       hcf, the whole program is not hcf, either. The same applies
       for strongly hcf.
    */   
    bool isStronglyHCF(unsigned componentidx)
        {
        assert(!hasNonstandardEdges);
        assert( componentHasRecursions != 0 );

        return (isHCF(componentidx) &&
           (!componentHasRecursions[componentidx]));
        }

    /// checks if all components are strongly head-cycle free
    bool isStronglyHCF()
        {
        assert(!hasNonstandardEdges);

        if(programIsStronglyHCF == -1)
            {
            programIsStronglyHCF = 1; // true

            for(unsigned i = 0; i < ptrComponents->size(); i++)
                {
                if(!isStronglyHCF(i))
                    {
                    programIsStronglyHCF = 0; // false
                    break;
                    }
                }
            }
        return (programIsStronglyHCF == 1);
        }

    /**
       returns true if the component with the index componentidx is
       disjunctive.
       isDisjunctive() is only applicable on graphs with negations and
       disjunctions.
       This is computed in the same way head-cycle freedom is computed,
       just on a different dependency graph.
    */
    bool isDisjunctive(unsigned componentIdx);

    /**
       a component is cyclic iff it contains more than one atom or it is
       recursive.
    */
    bool isCyclic(unsigned componentIdx) const
        {
        assert(!hasNonstandardEdges); // for the positive depgraph only

        return (componentHasRecursions[componentIdx] ||
                ((*ptrComponents)[componentIdx].size() > 1));
        }

    ///
    bool isCyclic()
        {
        assert(!hasNonstandardEdges); // for the positive depgraph only

        if(programIsCyclic == -1)
            {
            programIsCyclic = 0; // false

            for(unsigned i = 0; i < ptrComponents->size(); i++)
                {
                if(isCyclic(i))
                    {
                    programIsCyclic = 1; // true
                    break;
                    }
                }
            }
        return (programIsCyclic == 1); 
        }

    /// provides access to a component's rules
    const PROGRAM_TYPE &getComponentRules(unsigned component_index) const
        {
        assert(component_index < subprograms.size());
        return subprograms[component_index];
        }

    /** provides access to a component's constraints.
       The necessary steps to prepare the data accessed here are taken
       in the constructor. This could be changed if the constraints are
       only needed in some cases, to reduce overhead.
    */
    const CONSTRAINTS_TYPE &getComponentConstraints(
        unsigned component_index) const
        {
        assert(ptrComponentConstraints != 0); // zero if not allocated
        assert(component_index < ptrComponents->size());
        return (*ptrComponentConstraints)[component_index];
        }

    // global methods (which apply to the whole dependency graph


    /** This procedure is for debugging purposes only.
       It prints out the following information on the graph's components:
        index, contained atoms, rules, constraints,
        and whether they are stratified, head-cycle-free or contain
        disjunctions (in the hasNonstandardEdges case).
       Besides, for the whole graph, the number of atoms and components is
       given and it is tested if the head-cycle-free condition is true for
       all components of the graph.
    */
    void debug(ostream &o);
    };


////////////////////////////////////////////////////////////////////////////////
template<class ATOM_TYPE, class CONSTRAINTS_TYPE, class PROGRAM_TYPE>
void DEPGRAPH<ATOM_TYPE, CONSTRAINTS_TYPE, PROGRAM_TYPE>::
computeAllComponentConstraints(const CONSTRAINTS_TYPE &constraints)
    {
    // This method should not be called more than once.
    assert(ptrComponentConstraints == 0);

    ptrComponentConstraints =
        new vector<CONSTRAINTS_TYPE>(ptrComponents->size());

    // bug-fix for { a.   :-a. }
    if(ptrComponents->size() > 0)
        {
        for(typename CONSTRAINTS_TYPE::const_iterator c = constraints.begin();
            c != constraints.end(); c++)
            {
            bool valid_maxcompidx = false;
            INDEX_T maxcompidx = 0;

            for( typename TCONJUNCTION<TLITERAL<ATOM_TYPE> >::const_iterator
                 lit = c->begin(); 
                 lit != c->end();
                 lit++ )
                {
                if(atoms.isKnown(*lit))
                    {
                    unsigned atomidx = atoms.getIdx(*lit);
                    unsigned componentidx = nodeComponents[atomidx];

                    if(componentidx >= maxcompidx)
                        {
                        maxcompidx = componentidx;
                        valid_maxcompidx = true;
                        }
                    }
                else
                    // We either have an EDB predicate (resp. ground atom) or
                    // an IDB predicate (resp. ground atom) that does not occur
                    // in the head of any rule.
                    // (Note from 1999-09-07: I do not know if this is still
                    // true.
                    // but in the program
                    // { a(1) :- true. true. :- a(2). }
                    // a(2) is not known to the dependency graph, and it is IDB
                if (lit->isNegative())
                    {
                    // negative -> always true
                    // -> Literal could be removed from constraint.
                    }
                else
                    {
                    // positive -> always false
                    // -> Ignore entire constraint
                    // (which can never be violated).
                    // -> Abort search.

                    valid_maxcompidx = false;
                    break;
                    }
                }

            // constraint must not be empty, I guess
            assert(c->begin() != c->end());

            if(valid_maxcompidx)
                {
                (*ptrComponentConstraints)[maxcompidx].push_back(*c);
                }
            else
                {
                // The constraint only contains ``undefined'' literals.
                // If all of these are negative, it is always violated.
                if( (*c).pos_begin() == (*c).pos_end() )
                    {
                    if( TraceLevel >= 1 )
                        {
                        cdebug << "Constraint " << *c
                               << "is always violated [depgraph.h]." << endl;
                        }

                    // FIXME: We really shouldn't simply exit here.
                    SafeExit(0);
                    }
                }
            }
        }
    }


////////////////////////////////////////////////////////////////////////////////
template<class ATOM_TYPE, class CONSTRAINTS_TYPE, class PROGRAM_TYPE>
DEPGRAPH<ATOM_TYPE, CONSTRAINTS_TYPE, PROGRAM_TYPE>::
~DEPGRAPH()
    {
    if(ptrComponentConstraints)
        delete ptrComponentConstraints;

    if(componentHasNegations)
        delete[] componentHasNegations;

    if(componentHasDisjunctions)
        delete[] componentHasDisjunctions;

    if(componentHasRecursions)
        delete[] componentHasRecursions;

    if(ruleIsRecursive)
        delete[] ruleIsRecursive;

    if(componentConstraintsArePositive)
        delete[] componentConstraintsArePositive;

    delete graph;
    delete[] nodeComponents;
    delete ptrComponents;
    }

////////////////////////////////////////////////////////////////////////////////
template<class ATOM_TYPE, class CONSTRAINTS_TYPE, class PROGRAM_TYPE>
void DEPGRAPH<ATOM_TYPE, CONSTRAINTS_TYPE, PROGRAM_TYPE>::
computeSpecialEdges(void)
    {
    // initialize array of results
    componentHasNegations = new bool[ptrComponents->size()]; 
    componentHasDisjunctions = new bool[ptrComponents->size()]; 

    for(unsigned i = 0; i < ptrComponents->size(); i++)
        {
        componentHasNegations[i] = false;
        componentHasDisjunctions[i] = false;
        }

    // scan for negated edges
    // if components of the connected nodes are equal, not stratified

    for( typename EDGES::iterator e = edges.begin();
         e != edges.end();
         e++ )
        {
        if( e->type == EDGE::NEGATIVE )
            {
            if(nodeComponents[e->from]==nodeComponents[e->to])
                {
                componentHasNegations[
                   nodeComponents[e->from]] = true;
                }
            }
        else if( e->type == EDGE::DISJUNCTIVE )
            {
            if(nodeComponents[e->from] == nodeComponents[e->to])
                {
                componentHasDisjunctions[
                   nodeComponents[e->from]] = true;
                }
            }
        else
            {
            assert(0); // nothing else was ever inserted
            }
        }
    }


////////////////////////////////////////////////////////////////////////////////
template<class ATOM_TYPE, class CONSTRAINTS_TYPE, class PROGRAM_TYPE>
bool DEPGRAPH<ATOM_TYPE, CONSTRAINTS_TYPE, PROGRAM_TYPE>::
isStratified(unsigned componentIdx)
    {
    assert(hasNonstandardEdges);
    if(componentHasNegations==NULL)
        {
        computeSpecialEdges(); 
        }

    return !componentHasNegations[componentIdx];
    }


////////////////////////////////////////////////////////////////////////////////
template<class ATOM_TYPE, class CONSTRAINTS_TYPE, class PROGRAM_TYPE>
bool DEPGRAPH<ATOM_TYPE, CONSTRAINTS_TYPE, PROGRAM_TYPE>::
isHCF(unsigned componentIdx)
    {
    assert(!hasNonstandardEdges);
    if(componentHasNegations==NULL)
        {
        computeSpecialEdges(); 
        }

    // A component is HCF if it contains no disjunctions or if it is acyclic
    return ((!isCyclic(componentIdx)) ||
            (!componentHasDisjunctions[componentIdx]));
    }


////////////////////////////////////////////////////////////////////////////////
template<class ATOM_TYPE, class CONSTRAINTS_TYPE, class PROGRAM_TYPE>
bool DEPGRAPH<ATOM_TYPE, CONSTRAINTS_TYPE, PROGRAM_TYPE>::
isDisjunctive(unsigned componentIdx)
    {
    assert(hasNonstandardEdges);
    if(componentHasNegations==NULL)
        {
        computeSpecialEdges(); 
        }

    return componentHasDisjunctions[componentIdx];
    }


////////////////////////////////////////////////////////////////////////////////
template<class ATOM_TYPE, class CONSTRAINTS_TYPE, class PROGRAM_TYPE>
void DEPGRAPH<ATOM_TYPE, CONSTRAINTS_TYPE, PROGRAM_TYPE>::
computeAllComponentSubprograms(
    const vector<TRULE<ATOM_TYPE,TLITERAL<ATOM_TYPE> > > &rules,
    vector<PROGRAM_TYPE> &result)
    {
    assert( result.size() == 0 );

    // This should be initialized only once (namely here).
    assert (componentHasRecursions == 0 );

    componentHasRecursions = new bool[ptrComponents->size()];

    // An element of this array is true iff an atom of the component with
    // the index of the field in the array occurs in the positive body of
    // the current rule. Only valid if the componentindex is also contained
    // in components_that_receive_rule. And only then, the field
    // posBodyHasStdAtomOf[compidx] was initialized for this rule.
    bool *posBodyHasStdAtomOf = new bool[ptrComponents->size()];

    INDEX_T *components_that_receive_rule = new INDEX_T[ptrComponents->size()];
    unsigned nrRuleReceivers;

    // An element here is true if the component with the corresponding index
    // was already inserted into components_that_receive_rule.
    bool *does_component_receive_rule = new bool[ptrComponents->size()];

    for( unsigned componentidx = 0;
         componentidx < ptrComponents->size();
         componentidx++ )
        {
        result.push_back(PROGRAM_TYPE());
        componentHasRecursions[componentidx] = false;
        does_component_receive_rule[componentidx] = false;
        }

    unsigned ruleidx = 0;

    for( typename vector<TRULE<ATOM_TYPE,TLITERAL<ATOM_TYPE> > >::
         const_iterator r = rules.begin();
         r != rules.end();
         r++ )
        {
        // Count the number of components that receive the rule.
        nrRuleReceivers = 0;

        for(typename TDISJUNCTION<ATOM_TYPE>::const_iterator
            i = r->getHead().begin();
            i != r->getHead().end(); i++)
            {
            assert(atoms.isKnown(*i));
            // The index of the component of the current head atom.
            unsigned comp_index = nodeComponents[atoms.getIdx(*i)];

            if(ruleIsRecursive[ruleidx])
                componentHasRecursions[comp_index] = true;

            // If an atom A appears in the head of the current rule R and it
            // belongs to a component, then that component "receives" rule R.
            // Store in components_that_receive_rule the index of the
            // component of the current head atom and increase the
            // counter nrRuleReceivers.
            if(! does_component_receive_rule[comp_index])
                {
                components_that_receive_rule[nrRuleReceivers++]
                    = comp_index;
                posBodyHasStdAtomOf[comp_index] = false;
                does_component_receive_rule[comp_index] = true;
                }
            }

        // Traverse the body and update the array posBodyHasStdAtomOf.
        // Set to true each element of the array whose index corresponds
        // to the component index of the current body atom.
        if( r->hasBody() )
            {
            for(typename TCONJUNCTION<TLITERAL<ATOM_TYPE> >::const_iterator
                i =  (*(r->getBody())).pos_begin();
                i != (*(r->getBody())).pos_end();
                i++)
                {
                assert(atoms.isKnown(*i));
                
                // This field is only valid if comp_index was added to
                // components_that_receive_rule.
                if( (*i).isRegularAtom() )
                    posBodyHasStdAtomOf[nodeComponents[atoms.getIdx(*i)]]
                        = true;
                }
            }

        // Examine all and only the components that receive the current rule
        // and decide if the rule is an exit rule or a recursive rule.
        for(unsigned i = 0; i < nrRuleReceivers; i++)
            {
            // If the positive body contains one atom that belongs to a
            // component C of some of the head atoms then the rule has to be
            // added as recursive to component C. Otherwise, it has to be
            // added as exit rule.
            if(posBodyHasStdAtomOf[components_that_receive_rule[i]])
                result[components_that_receive_rule[i]].addRecursive(*r);
            else
                result[components_that_receive_rule[i]].addExit(*r);

            does_component_receive_rule[components_that_receive_rule[i]] =
                false;
            }

        ruleidx++;
        }

    delete[] posBodyHasStdAtomOf;
    delete[] components_that_receive_rule;
    delete[] does_component_receive_rule;
    }

////////////////////////////////////////////////////////////////////////////////
template<class ATOM_TYPE, class CONSTRAINTS_TYPE, class PROGRAM_TYPE>
void DEPGRAPH<ATOM_TYPE, CONSTRAINTS_TYPE, PROGRAM_TYPE>::
readAtomsFromRules(const vector<TRULE<ATOM_TYPE,TLITERAL<ATOM_TYPE> > > &rules)
    {
    for( typename vector<TRULE<ATOM_TYPE,TLITERAL<ATOM_TYPE> > >::
         const_iterator r=rules.begin();
         r != rules.end();
         r++ )
        {
        for( typename TDISJUNCTION<ATOM_TYPE>::const_iterator
             i = r->getHead().begin();
              i != r->getHead().end();
             i++ )
            {
            (void)atoms.insertUnique(*i);
            }

        if( r->hasBody() )
            for( typename TCONJUNCTION<TLITERAL<ATOM_TYPE> >::const_iterator
                 i = r->getBody()->begin();
                 i != r->getBody()->end();
                 i++ )
                {
                (void)atoms.insertUnique(*(const ATOM_TYPE *)(&*i));
                }
        }
    }


////////////////////////////////////////////////////////////////////////////////
    /** uses Tarjan algorithm itself, but not to get the strongly connected
        components. Instead, it is run just to get the entailed ordering of
        the components.

        Nicola's requirements:
        We need a function which provides, one at a time,
        the components of the positive DG (the DG we use for hcf) bottom-up.
        The difficulty here is that the ordering of these components should
        respect negation. In particular, if a component A depends on B
        negatively, while B does not depend on A, then A should be provided
        after B.
        If both A depends negatively on B and B depends negatively on A,
        (i.e., they are in the same component of the graph with negation)
        then the ordering is irrelevant.

        This method only makes sense with a positive depgraph, because
        otherwise, the negative edges are already included in the graph and
        are automatically taken care of in the ordering.

        NOTE THAT THIS REORDERING REQUIRES THE OTHER REORDERING TO BE EXECUTED
        FIRST (OTHERWISE A COMPONENT GRAPH A :- B; C :- D MAY LEAD TO AN
        ORDERING B, D, A, C.
    */
template<class ATOM_TYPE, class CONSTRAINTS_TYPE, class PROGRAM_TYPE>
void DEPGRAPH<ATOM_TYPE, CONSTRAINTS_TYPE, PROGRAM_TYPE>::
special_ordering(COMPONENT_FACTORY &cf, const bool *known_atoms)
    {
    // nodeComponents is set up in the constructor

    if(DTraceLevel >= 1)
        {
        cdebug << "In DEPGRAPH::special_ordering()" << endl;
        }

    ADJACENCY_LIST aux_adjlist(cf.size());

    bool reordering_necessary = false;

    // add negated edges
    for(unsigned e = 0; e < edges.size(); e ++)
        {
        if( edges[e].type == EDGE::NEGATIVE )
            {
            if( DTraceLevel >= 1 )
                {
                cdebug << "edge from " << edges[e].from
                       << "   to "     << edges[e].to
                       << "   nC[f] "  << nodeComponents[edges[e].from]
                       << "   nC[t] "  << nodeComponents[edges[e].to]
                       << endl;
                }

            if(nodeComponents[edges[e].from] != nodeComponents[edges[e].to])
                {
                // Note: connected(a,b) is O(#edges leaving a).
                if( ! aux_adjlist.connected(nodeComponents[edges[e].from],
                                            nodeComponents[edges[e].to])
                    && ! aux_adjlist.connected(nodeComponents[edges[e].to],
                                               nodeComponents[edges[e].from]) )
                    {
                    aux_adjlist.connect(nodeComponents[edges[e].from],
                                        nodeComponents[edges[e].to]);
                    reordering_necessary = true;
                    }
                }
            }
        }

    if(! reordering_necessary)
        {
        if( DTraceLevel >=  1 )
            {
            cdebug << "Leaving DEPGRAPH::special_ordering()"
                   << " (no reordering necessary)" << endl;
            }
        return;
        }

    // add positive edges
    for(unsigned e = 0; e < cf.getWeakLinks().size(); e++)
        {
        assert(nodeComponents[cf.getWeakLinks()[e].from]
            != nodeComponents[cf.getWeakLinks()[e].to]);

        aux_adjlist.connect(nodeComponents[cf.getWeakLinks()[e].from],
                            nodeComponents[cf.getWeakLinks()[e].to]);
        }

    if( DTraceLevel >=  1 )
        {
        cdebug << "cf.size() = " << cf.size()
               << "  aux_adjlist.size() = " << aux_adjlist.size() << endl;
        aux_adjlist.printMatrix();
        }

    // we don't need the holes facility because we do not execute
    // preprocessComponents()
    bool *dummy = new bool[cf.size()];
    for(unsigned i = 0; i < cf.size(); i++)
        {
        dummy[i] = true;
        }

    COMPONENT_FACTORY aux_cf(cf.size(), dummy);
    
    DEPGRAPH_SCC< ADJACENCY_LIST,
        ADJACENCY_LIST::ADJ_BRANCH_TYPE::const_iterator,
        COMPONENT_FACTORY>(aux_adjlist, aux_cf);

    unsigned *componentMapping = aux_cf.getNonorderedNodeComponents();

    delete[] dummy;

    if( DTraceLevel >=  1 )
        {
        cdebug << "componentMapping before offsetting" << endl;

        for(unsigned n = 0; n < cf.size(); n++)
            {
            cdebug << "componentMapping[SCC#" << n << "] = "
                   << componentMapping[n] << endl;
            }
        }

    vector<unsigned> *tmp = new vector<unsigned>[cf.size()];

    unsigned comp_count = 0;
    for(unsigned n = 0; n < cf.size(); n++)
        {
        tmp[componentMapping[n]].push_back(n);
        if(componentMapping[n] + 1 > comp_count)
            comp_count = componentMapping[n] + 1;
        }
    unsigned cnt = 0;
    for(unsigned n = 0; n < comp_count; n++)
        {
        for(int j = tmp[n].size() - 1; j >= 0; j--)
            {
            componentMapping[tmp[n][j]] = cnt++;
            }
        }

    delete[] tmp;

    if( DTraceLevel >=  1 )
        {
        cdebug << "componentMapping after offsetting" << endl;

        for(unsigned n = 0; n < cf.size(); n++)
            {
            cdebug << "componentMapping[SCC#" << n << "] = "
                   << componentMapping[n] << endl;
            }
        }

    // the result of Tarjan of course again has to be processed in the reverse
    // order since the direction of the arcs is opposite to the dependencies.
    for(unsigned n = 0; n < atoms.size(); n++) // for each node
        if(known_atoms[n])
            nodeComponents[n] = cf.size() - 1
                                - componentMapping[nodeComponents[n]];

    delete[] componentMapping;

    if( DTraceLevel >=  1 )
        {
        cdebug << "Leaving DEPGRAPH::special_ordering()" << endl;
        }
    }

////////////////////////////////////////////////////////////////////////////////
template<class ATOM_TYPE, class CONSTRAINTS_TYPE, class PROGRAM_TYPE>
DEPGRAPH<ATOM_TYPE, CONSTRAINTS_TYPE, PROGRAM_TYPE>::
DEPGRAPH(const vector<TRULE<ATOM_TYPE,TLITERAL<ATOM_TYPE> > > &rules,
     const CONSTRAINTS_TYPE &constraints,
     unsigned long options) : atoms(options & NONGROUND)
    {
    hasNonstandardEdges = options & WITHSPECIALEDGES; // Include all edges.
    componentHasNegations = NULL;
    componentHasDisjunctions = NULL;
    ruleIsRecursive = NULL;
    componentHasRecursions = NULL;
    ptrComponentConstraints = 0;
    componentConstraintsArePositive = NULL;

    programIsCyclic = -1;
    programIsHCF = -1;
    programIsPositive = -1;
    programIsStronglyHCF = -1;

    if(DTraceLevel >= 1)
        {
        cdebug << endl
               << "In DEPGRAPH::DEPGRAPH() -- "
               << rules.size() <<" rules, "
               << constraints.size() << " constraints, options: { ";

        if(options & NONGROUND)
           cdebug << "NONGROUND ";
        if(options & WITHSPECIALEDGES)
           cdebug << "WITHSPECIALEDGES ";
        if(options & WITHCONSTRAINTEDGES)
           cdebug << "WITHCONSTRAINTEDGES ";
        if(options & SUPPRESSCONSTRAINTASSIGNMENT)
           cdebug << "SUPPRESSCONSTRAINTASSIGNMENT ";
        if(options & SIMONASORDERING)
           cdebug << "SIMONASORDERING ";

        cdebug << "}" << endl; 
        }

#ifdef DEPGRAPH_BENCHMARK
    struct tms buf;
    long t = times(&buf);
#endif

    // side effect: atoms.size() grows
    readAtomsFromRules(rules);

    graph = new ADJACENCY_LIST(atoms.size());

#ifdef DEPGRAPH_BENCHMARK
    t = times(&buf) - t;
    cout << "Time consumed by readAtomsFromRules() = "
         << ((double)t)/100 << "s" << endl;
    t = times(&buf);
#endif

    buildGraph(rules);

#ifdef DEPGRAPH_BENCHMARK
    t = times(&buf) - t;
    cout << "Time consumed by buildgraph() = "
         << ((double)t)/100 << "s" << endl;
    t = times(&buf);
#endif

    if(options & WITHCONSTRAINTEDGES)
        insertConstraintsIntoGraph(constraints);

#ifdef DEPGRAPH_BENCHMARK
    t = times(&buf) - t;
    cout << "Time consumed by insertConstraints() = "
         << ((double)t)/100 << "s" << endl;
    t = times(&buf);
#endif

    bool *atom_known = new bool[atoms.size()];
    for(unsigned i = 0; i < atoms.size(); i++)
        atom_known[i] = atoms.isKnown(i);

    // Prepare the component-insertion handler.
    COMPONENT_FACTORY fno_sc(atoms.size(), atom_known);

// this is an alternative way of using DEPGRAPH_SCC
/*
    // function object-like call
    DEPGRAPH_SCC<ADJACENCY_LIST,
        ADJACENCY_LIST<unsigned>::ADJ_BRANCH_TYPE::const_iterator,
        COMPONENT_FACTORY>(atoms.size());

    scc(*adjlist,fno_sc);
*/
    // constructor call
    DEPGRAPH_SCC< ADJACENCY_LIST,
        ADJACENCY_LIST::ADJ_BRANCH_TYPE::const_iterator,
        COMPONENT_FACTORY>(*graph,fno_sc);

    if(DTraceLevel >= 1)
        {
        cdebug << "dependency graph contains " << atoms.size() << " atoms"
               << endl;

        nodeComponents = fno_sc.getNonorderedNodeComponents();
        cdebug << fno_sc.size() << " components" << endl;

        for(unsigned i = 0; i < atoms.size(); i++)
            {
            if(atoms.isKnown(i))
                cdebug << "nodeComponents[" << i << "] = " << nodeComponents[i]
                       << ", atoms[" << i << "] = " << atoms[i] << endl;
            else
                cdebug << "atom #" << i << " not known." << endl;
            }
        } 

#ifdef DEPGRAPH_BENCHMARK
    t = times(&buf) - t;
    cout << "Time consumed by DEPGRAPH_SCC() = "
         << ((double)t)/100 << "s" << endl;
    t = times(&buf);
#endif

    // The "normal" ordering has to be executed in any case, even if Simona's
    // ordering is done next.
    nodeComponents = fno_sc.makeNodeComponents();

    if(DTraceLevel >= 1)
        {
        cdebug << "Components after reordering:" << endl;

        for(unsigned i = 0; i < atoms.size(); i++)
            {
            if(atoms.isKnown(i))
                cdebug << "nodeComponents[" << i << "] = "
                       << nodeComponents[i]
                       << ", atoms[" << i << "] = " << atoms[i] << endl;
            else
                cdebug << "atom #" << i << " not known." << endl;
            }
        }

#ifdef DEPGRAPH_BENCHMARK
    t = times(&buf) - t;
    cout << "Time consumed by makeNodeComponents() = "
         << ((double)t)/100 << "s" << endl;
    t = times(&buf);
#endif

    if(options & SIMONASORDERING)
        {
        assert(! (options & WITHSPECIALEDGES));
        special_ordering(fno_sc, atom_known);

#ifdef DEPGRAPH_BENCHMARK
        t = times(&buf) - t;
        cout << "Time consumed by special_ordering() = "
             << ((double)t)/100 << "s" << endl;
        t = times(&buf);
#endif
        }

    ptrComponents = new COMPONENTS(fno_sc.size());
    for(unsigned n = 0; n < atoms.size(); n++)
        if(atom_known[n])
            (*ptrComponents)[nodeComponents[n]].push_back(n);

    delete[] atom_known;                        // We don't need this anymore.

#ifdef DEPGRAPH_BENCHMARK
    t = times(&buf) - t;
    cout << "Time consumed for creation of ptrComponents = "
         << ((double)t)/100 << "s" << endl;
    t = times(&buf);
#endif

    computeAllComponentSubprograms(rules,subprograms);

#ifdef DEPGRAPH_BENCHMARK
    t = times(&buf) - t;
    cout << "Time consumed by computeAllComponentSubprograms() = "
         << ((double)t)/100 << "s" << endl;
    t = times(&buf);
#endif

    if(! (options & SUPPRESSCONSTRAINTASSIGNMENT))
        computeAllComponentConstraints(constraints);

#ifdef DEPGRAPH_BENCHMARK
    t = times(&buf) - t;
    cout << "Time consumed by computeAllComponentConstraints() = "
         << ((double)t)/100 << "s" << endl;
#endif

    if(DTraceLevel >= 1)
        cdebug << "Leaving DEPGRPAPH::DEPGRAPH() (initialization finished)"
               << endl;
    }

///////////////////////////////////////////////////////////////////////////////
template<>
inline void DEPGRAPH<ATOM, CONSTRAINTS, PROGRAM>::buildGraphAdd(
    const ATOM &h,
    const LITERAL &b )
    {
    // Negative aggregates also imply a positive dependency.
    if( ! b.isNegative() || b.isAggregate() || hasNonstandardEdges )
        graph->connect( atoms.insertUnique(b),
                        atoms.insertUnique(h) );

    if( b.isNegative() )
        edges.push_back( EDGE(atoms.insertUnique(b),
                              atoms.insertUnique(h),
                              EDGE::NEGATIVE) );
    }

template<>
inline void DEPGRAPH<GATOM, GCONSTRAINTS, GPROGRAM>::buildGraphAdd(
    const GATOM &h,
    const GLITERAL &b )
    {
    if( ! b.isNegative() || hasNonstandardEdges )
        graph->connect( atoms.insertUnique(b),
                        atoms.insertUnique(h) );

    if( b.isNegative() )
        edges.push_back( EDGE(atoms.insertUnique(b),
                              atoms.insertUnique(h),
                              EDGE::NEGATIVE) );

    if( b.isAggregate() )
         {
         // Add positive dependencies from all atoms in an aggregate set
         // to the aggregate atom(s) this set appears in.

        const AGGREGATESET &aggrSet =
            b.getAggregate().getAggregateFunctionPtr()->getAggregateSet();

        for( AGGREGATESET::const_iterator i = aggrSet.begin();
             i != aggrSet.end();
             i++ )
            graph->connect( atoms.insertUnique(i->second),
                            atoms.insertUnique(b) );
        }
    }

///////////////////////////////////////////////////////////////////////////////
template<class ATOM_TYPE, class CONSTRAINTS_TYPE, class PROGRAM_TYPE>
void DEPGRAPH<ATOM_TYPE, CONSTRAINTS_TYPE, PROGRAM_TYPE>::buildGraph(
    const vector<TRULE<ATOM_TYPE,TLITERAL<ATOM_TYPE> > > &rules )
    {
    assert(ruleIsRecursive == 0); // Initialized 0 in the constructor and
          // only changed in this method, which is to be called only once.

    ruleIsRecursive = new bool[rules.size()];

    int ruleidx = -1;

    for( typename vector<TRULE<ATOM_TYPE,TLITERAL<ATOM_TYPE> > >::
         const_iterator r=rules.begin();
         r != rules.end();
         r++ )
        {
        ruleIsRecursive[++ruleidx] = false;

        // Process the rule head.
        //
        // In principle we could only consider neighbouring atoms (and
        // let closure of the graph take care of the rest), but this did
        // not work for examples like the following
        //   b v a v c v d v e.  c :- e.  b :- c.  e :- b.  c :- d.
        // which were incorrectly reported to be HCF. To avoid adding the
        // full, quadratic number of edges we could compute the strongly
        // connected components of a graph containing only disjunctive edges.
        // In that case the program is not HCF iff there is a component in
        // this graph which contains at least two atoms of a component in
        // the full dependency graph.

        for( typename TDISJUNCTION<ATOM_TYPE>::const_iterator i=
               r->getHead().begin();
             i != r->getHead().end();
             i++ )
            {
            // that's only retrieval, no insertion
            const unsigned idx1=atoms.insertUnique(*i);

            for( typename TDISJUNCTION<ATOM_TYPE>::const_iterator j = i+1;
                 j != r->getHead().end();
                 j++ )
                {
                // that's only retrieval, no insertion
                const unsigned idx2=atoms.insertUnique(*j);

                edges.push_back( EDGE(idx1,idx2,EDGE::DISJUNCTIVE) );
                edges.push_back( EDGE(idx2,idx1,EDGE::DISJUNCTIVE) );

                // Consider disjunctions (and negative dependencies)?
                if( hasNonstandardEdges )
                    {
                    graph->connect(idx2,idx1);
                    graph->connect(idx1,idx2);
                    }
                }
            }

        // Process the rule body.

        if( r->hasBody() )
            for( typename TCONJUNCTION<TLITERAL<ATOM_TYPE> >::
                 const_iterator b=r->getBody()->begin();
                 b != r->getBody()->end();
                 b++ )
                {
                for( typename TDISJUNCTION<ATOM_TYPE>::const_iterator
                     h=r->getHead().begin();
                     h != r->getHead().end(); h++ )
                    {
                    if( atoms.areAtomsEqual(*h,*b) )
                        ruleIsRecursive[ruleidx] = true;

                    buildGraphAdd(*h,*b);
                    }
                }
        }
    }

////////////////////////////////////////////////////////////////////////////////
template<class ATOM_TYPE, class CONSTRAINTS_TYPE, class PROGRAM_TYPE>
void DEPGRAPH<ATOM_TYPE, CONSTRAINTS_TYPE, PROGRAM_TYPE>::
insertConstraintsIntoGraph(const CONSTRAINTS_TYPE &constraints)
    {
    for(typename CONSTRAINTS_TYPE::const_iterator c=constraints.begin();
        c!=constraints.end();c++)
        {
        long lastidx=-2;

        for( typename TCONJUNCTION<TLITERAL<ATOM_TYPE> >::const_iterator
             i=c->begin(); 
             i != c->end();
             i++ )
            {
            if(atoms.isKnown(*i))
                {
                long idx = atoms.getIdx(*i);

                if(lastidx == -2)
                    {
                    lastidx = idx;
                    }
                else 
                    {
                    graph->connect(lastidx,idx);
                    graph->connect(idx,lastidx);
                    }
                }
            }
        }
    }


////////////////////////////////////////////////////////////////////////////////
template<class ATOM_T, class CONSTRAINTS_T, class PROGRAM_T>
void DEPGRAPH<ATOM_T, CONSTRAINTS_T, PROGRAM_T>::debug(ostream &o)
{
    // whole-graph info
    o << "DEPGRAPH: ";
    o << "(" << atoms.size() << " atoms, ";
    o << ptrComponents->size() << " components)" << endl;

    if( ! hasNonstandardEdges )
        {
        if( isCyclic() )
            o << "DEPGRAPH::isCyclic() : true" << endl;
        else
            o << "DEPGRAPH::isCyclic() : false" << endl;

        if( isHCF() )
            o << "DEPGRAPH::isHCF() : true" << endl;
        else
            o << "DEPGRAPH::isHCF() : false" << endl;

        if( isStronglyHCF() )
            o << "DEPGRAPH::isStronglyHCF() : true" << endl;
        else
            o << "DEPGRAPH::isStronglyHCF() : false" << endl;
        }

    o << endl;

    for(unsigned j = 0; j < getAtoms().size(); j++)
        {
        if( getAtoms().isKnown(j) )
            o << getAtoms()[j] << " ";
        }
    o << endl;
    graph->printMatrix();

    o << endl;

    // per-component info
    for(unsigned i = 0; i < ptrComponents->size(); i++)
        {
        o << "COMPONENT " << (i+1) << ":";

        if( hasNonstandardEdges ) // full dependency graph
            {
            if( isStratified(i) )
               o << "  stratified";
            else
               o << "  not stratified";

            if( isDisjunctive(i) )
               o << "  disjunctive";
            else
               o << "  not disjunctive";

// reactivate for testing implementation of areConstraintsPositive
/*
            if( isPositive(i) && areConstraintsPositive(i) )
               o << "  program and constraints positive";
            else
               o << "  program or constraints not positive";
*/
            }
        else // positive dependency graph
            {
            if( isHCF(i) )
               o << "  hcf";
            else
               o << "  not hcf";

            if( isStronglyHCF(i) )
               o << "  strongly hcf";
            else
               o << "  not strongly hcf";

            if( isCyclic(i) )
               o << "  cyclic";
            else
               o << "  acyclic";
            } 

        o << endl;

        // atoms
        o << "   { ";
        for(unsigned j=0;j<(*ptrComponents)[i].size();j++)
            {
            atoms.outName(o,(*ptrComponents)[i][j]);

            if(j < (*ptrComponents)[i].size() - 1) o << " , ";
            }
        o << " }" << endl;

        for(typename PROGRAM_T::const_iterator pi=
            getComponentRules(i).begin();
            pi != getComponentRules(i).end(); pi++)
            {
            o << "   " << (*pi) << endl;
            }

        for( typename CONSTRAINTS_T::const_iterator
             it = getComponentConstraints(i).begin();
             it != getComponentConstraints(i).end();
             it++ )
            {
            o << "   " << *it << endl;
            }
        }
    }

////////////////////////////////////////////////////////////////////////////////
/** DEPGRAPH test routine, for debugging purposes only.
   Tests the four different kinds of DEPGRAPHs with sets of grounded and non-
   grounded rules.
*/
inline void testDEPGRAPH(
    RULES        &nongroundRules,
    CONSTRAINTS  &nongroundConstraints,
    GRULES       &groundRules,
    GCONSTRAINTS &groundConstraints )
    {
    const char *txt[2]= { "with", "without" };

    DEPGRAPH<ATOM,CONSTRAINTS,PROGRAM>
        *pDG[2],
        dg1(nongroundRules,nongroundConstraints,
            DEPGRAPH<ATOM,CONSTRAINTS,PROGRAM>::NONGROUND |
            DEPGRAPH<ATOM,CONSTRAINTS,PROGRAM>::WITHSPECIALEDGES ),
        dg2(nongroundRules,nongroundConstraints,
            DEPGRAPH<ATOM,CONSTRAINTS,PROGRAM>::NONGROUND |
            DEPGRAPH<ATOM,CONSTRAINTS,PROGRAM>::SIMONASORDERING);

    DEPGRAPH<GATOM,GCONSTRAINTS,GPROGRAM>
        *pGDG[2],
        gdg1(groundRules,groundConstraints,
             DEPGRAPH<GATOM,GCONSTRAINTS,GPROGRAM>::WITHSPECIALEDGES ),
        gdg2(groundRules,groundConstraints, 0);

    pDG[0]=&dg1; pDG[1]=&dg2; pGDG[0]=&gdg1; pGDG[1]=&gdg2;

    cout << "The nonground DG without negation and disjunction uses Simona's"
            " ordering." << endl;

    for(unsigned i=0; i<2; i++)
        {
        cout << "DG " << txt[i] << " negations and disjunctions:"
             << endl << endl;
        pDG[i]->debug(cout);
        cout << endl << endl;
        }
    for(unsigned i=0; i<2; i++)
        {
        cout << "GDG " << txt[i] << " negations and disjunctions:"
             << endl << endl;
        pGDG[i]->debug(cout);
        cout << endl << endl;
        }
    }

#endif

// Local Variables:
// c-file-style: "dl"
// End:
