/*************************************************
* Lowest Level MPI Algorithms Header File        *
* (C) 1999-2006 The Botan Project                *
*************************************************/

#ifndef BOTAN_MP_ASM_INTERNAL_H__
#define BOTAN_MP_ASM_INTERNAL_H__

#include <botan/mp_asm.h>

namespace Botan {

/*************************************************
* Word Addition                                  *
*************************************************/
inline word word_add(word x, word y, word* carry)
   {
   asm(
        "rorl %[carry]                 \n\t" //x+=carry
        "adcl %[y],%[x]                \n\t" //x+=y
        "sbb  %[carry],%[carry]        \n\t"
        "neg  %[carry]                 \n\t"
        : [x]"=r"(x),
          [carry]"=r"(*carry)     //*carry is changed
        : "0"(x),                 //x is input and output on the same register
          [y]"rm"(y),                //y is a register or memory
          "1"(*carry)         //
        : "cc");
   return x;
   }

/*************************************************
* Eight Word Block Addition, Two Argument        *
*************************************************/
inline word word8_add2(word x[8], const word y[8], word carry)
   {
   asm(
        "ror  %[carry] \n\t"
        "movl (%[y]),%[carry]               \n\t"
        "adcl  %[carry],0(%[x])               \n\t"
        "movl 4(%[y]),%[carry]               \n\t"
        "adcl  %[carry],4(%[x])               \n\t"
        "movl 8(%[y]),%[carry]               \n\t"
        "adcl  %[carry],8(%[x])               \n\t"
        "movl 12(%[y]),%[carry]               \n\t"
        "adcl  %[carry],12(%[x])               \n\t"
        "movl 16(%[y]),%[carry]               \n\t"
        "adcl  %[carry],16(%[x])               \n\t"
        "movl 20(%[y]),%[carry]               \n\t"
        "adcl  %[carry],20(%[x])               \n\t"
        "movl 24(%[y]),%[carry]               \n\t"
        "adcl  %[carry],24(%[x])               \n\t"
        "movl 28(%[y]),%[carry]               \n\t"
        "adcl  %[carry],28(%[x])               \n\t"
        "sbbl  %[carry],%[carry]          \n\t"
        "negl  %[carry] \n\t"
        : [carry]"=r"(carry)         //*carry is changed
        : [x]"r"(x),                 //x is a register
          [y]"r"(y),                 //y is a register
          "0"(carry)         //
        : "cc","memory");
   return carry;
   }

/*************************************************
* Eight Word Block Addition, Three Argument      *
*************************************************/
inline word word8_add3(word z[8], const word x[8], const word y[8], word carry)
   {
   asm(
        "ror  %[carry] \n\t"
        "movl (%[y]),%[carry]               \n\t"
        "adcl (%[x]),%[carry]               \n\t"
        "movl %[carry],(%[z])               \n\t"
        "movl 4(%[y]),%[carry]               \n\t"
        "adcl 4(%[x]),%[carry]               \n\t"
        "movl %[carry],4(%[z])               \n\t"
        "movl 8(%[y]),%[carry]               \n\t"
        "adcl 8(%[x]),%[carry]               \n\t"
        "movl %[carry],8(%[z])               \n\t"
        "movl 12(%[y]),%[carry]               \n\t"
        "adcl 12(%[x]),%[carry]               \n\t"
        "movl %[carry],12(%[z])               \n\t"
        "movl 16(%[y]),%[carry]               \n\t"
        "adcl 16(%[x]),%[carry]               \n\t"
        "movl %[carry],16(%[z])               \n\t"
        "movl 20(%[y]),%[carry]               \n\t"
        "adcl 20(%[x]),%[carry]               \n\t"
        "movl %[carry],20(%[z])               \n\t"
        "movl 24(%[y]),%[carry]               \n\t"
        "adcl 24(%[x]),%[carry]               \n\t"
        "movl %[carry],24(%[z])               \n\t"
        "movl 28(%[y]),%[carry]               \n\t"
        "adcl 28(%[x]),%[carry]               \n\t"
        "movl %[carry],28(%[z])               \n\t"
        "sbbl  %[carry],%[carry]          \n\t"
        "negl  %[carry] \n\t"
        : [carry]"=r"(carry)         //*carry is changed
        : [x]"r"(x),                 //x is a register
          [y]"r"(y),                 //y is a register
          [z]"r"(z),                 //y is a register
          "0"(carry)         //
        : "cc","memory");
   return carry;
   }

/*************************************************
* Word Subtraction                               *
*************************************************/
inline word word_sub(word x, word y, word* carry)
   {
   asm(
        "rorl  %[carry]                 \n\t" //carry->CF
        "sbbl  %[y],%[x]                \n\t" //x+=y+carry
        "sbbl  %[carry],%[carry]        \n\t"
        "negl  %[carry]                 \n\t"
        : [x]"=r"(x),
          [carry]"=r"(*carry)     //*carry is changed
        : "0"(x),                 //x is input and output on the same register
          [y]"rm"(y),             //y is a register or memory
          "1"(*carry)             //
        : "cc");
   return x;
   }

/*************************************************
* Eight Word Block Subtraction, Two Argument     *
*************************************************/
inline word word8_sub2(word x[8], const word y[8], word carry)
   {
   asm(
        "ror  %[carry] \n\t"
        "movl (%[y]),%[carry]               \n\t"
        "sbbl  %[carry],0(%[x])               \n\t"
        "movl 4(%[y]),%[carry]               \n\t"
        "sbbl  %[carry],4(%[x])               \n\t"
        "movl 8(%[y]),%[carry]               \n\t"
        "sbbl  %[carry],8(%[x])               \n\t"
        "movl 12(%[y]),%[carry]               \n\t"
        "sbbl  %[carry],12(%[x])               \n\t"
        "movl 16(%[y]),%[carry]               \n\t"
        "sbbl  %[carry],16(%[x])               \n\t"
        "movl 20(%[y]),%[carry]               \n\t"
        "sbbl  %[carry],20(%[x])               \n\t"
        "movl 24(%[y]),%[carry]               \n\t"
        "sbbl  %[carry],24(%[x])               \n\t"
        "movl 28(%[y]),%[carry]               \n\t"
        "sbbl  %[carry],28(%[x])               \n\t"
        "sbbl  %[carry],%[carry]          \n\t"
        "negl  %[carry] \n\t"
        : [carry]"=r"(carry)         //*carry is changed
        : [x]"r"(x),                 //x is a register
          [y]"r"(y),                 //y is a register
          "0"(carry)         //
        : "cc","memory");
   return carry;
   }

/*************************************************
* Eight Word Block Subtraction, Three Argument   *
*************************************************/
inline word word8_sub3(word z[8], const word x[8], const word y[8], word carry)
   {
   asm(
        "ror  %[carry] \n\t"
        "movl (%[x]),%[carry]               \n\t"
        "sbbl (%[y]),%[carry]               \n\t"
        "movl %[carry],(%[z])               \n\t"
        "movl 4(%[x]),%[carry]               \n\t"
        "sbbl 4(%[y]),%[carry]               \n\t"
        "movl %[carry],4(%[z])               \n\t"
        "movl 8(%[x]),%[carry]               \n\t"
        "sbbl 8(%[y]),%[carry]               \n\t"
        "movl %[carry],8(%[z])               \n\t"
        "movl 12(%[x]),%[carry]               \n\t"
        "sbbl 12(%[y]),%[carry]               \n\t"
        "movl %[carry],12(%[z])               \n\t"

        "movl 16(%[x]),%[carry]               \n\t"
        "sbbl 16(%[y]),%[carry]               \n\t"
        "movl %[carry],16(%[z])               \n\t"
        "movl 20(%[x]),%[carry]               \n\t"
        "sbbl 20(%[y]),%[carry]               \n\t"
        "movl %[carry],20(%[z])               \n\t"
        "movl 24(%[x]),%[carry]               \n\t"
        "sbbl 24(%[y]),%[carry]               \n\t"
        "movl %[carry],24(%[z])               \n\t"
        "movl 28(%[x]),%[carry]               \n\t"
        "sbbl 28(%[y]),%[carry]               \n\t"
        "movl %[carry],28(%[z])               \n\t"

        "sbbl  %[carry],%[carry]          \n\t"
        "negl  %[carry] \n\t"
        : [carry]"=r"(carry)         //*carry is changed
        : [x]"r"(x),                 //x is a register
          [y]"r"(y),                 //y is a register
          [z]"r"(z),                 //y is a register
          "0"(carry)         //
        : "cc","memory");
   return carry;
   }

/*************************************************
* Eight Word Block Linear Multiplication         *
*************************************************/
inline word word8_linmul2(word x[8], word y, word carry)
   {
   asm(
      "movl (%[x]),%%eax       \n\t" //eax = x[0]
      "mull %[y]               \n\t" //edx:eax = x[0]*y
      "addl %[carry],%%eax     \n\t" //edx:eax+=carry (lo)
      "adcl $0,%%edx           \n\t" //edx:eax+=carry (hi)
      "movl %%edx,%[carry]     \n\t" //update carry
      "movl %%eax,(%[x])       \n\t" //store x[0]

      "movl 4(%[x]),%%eax      \n\t"
      "mull %[y]               \n\t"
      "addl %[carry],%%eax     \n\t"
      "adcl $0,%%edx           \n\t"
      "movl %%edx,%[carry]     \n\t"
      "movl %%eax,4(%[x])      \n\t"

      "movl 8(%[x]),%%eax      \n\t"
      "mull %[y]               \n\t"
      "addl %[carry],%%eax     \n\t"
      "adcl $0,%%edx           \n\t"
      "movl %%edx,%[carry]     \n\t"
      "movl %%eax,8(%[x])      \n\t"

      "movl 12(%[x]),%%eax      \n\t"
      "mull %[y]               \n\t"
      "addl %[carry],%%eax     \n\t"
      "adcl $0,%%edx           \n\t"
      "movl %%edx,%[carry]     \n\t"
      "movl %%eax,12(%[x])      \n\t"

      "movl 16(%[x]),%%eax      \n\t"
      "mull %[y]               \n\t"
      "addl %[carry],%%eax     \n\t"
      "adcl $0,%%edx           \n\t"
      "movl %%edx,%[carry]     \n\t"
      "movl %%eax,16(%[x])      \n\t"

      "movl 20(%[x]),%%eax      \n\t"
      "mull %[y]               \n\t"
      "addl %[carry],%%eax     \n\t"
      "adcl $0,%%edx           \n\t"
      "movl %%edx,%[carry]     \n\t"
      "movl %%eax,20(%[x])      \n\t"

      "movl 24(%[x]),%%eax      \n\t"
      "mull %[y]               \n\t"
      "addl %[carry],%%eax     \n\t"
      "adcl $0,%%edx           \n\t"
      "movl %%edx,%[carry]     \n\t"
      "movl %%eax,24(%[x])      \n\t"

      "movl 28(%[x]),%%eax     \n\t"
      "mull %[y]               \n\t"
      "addl %[carry],%%eax     \n\t"
      "adcl $0,%%edx           \n\t"
      "movl %%edx,%[carry]     \n\t"
      "movl %%eax,28(%[x])     \n\t"
      : [carry]"=r"(carry)
      : [x]"r"(x), [y]"rm"(y), "0"(carry)
      : "cc","%eax","%edx");

   return carry;
   }

/*************************************************
* Eight Word Block Linear Multiplication          *
*************************************************/
inline word word8_linmul3(word z[8], const word x[8], word y, word carry)
   {
   asm(
      "movl (%[x]),%%eax       \n\t" //eax = x[0]
      "mull %[y]               \n\t" //edx:eax = x[0]*y
      "addl %[carry],%%eax     \n\t" //edx:eax+=carry (lo)
      "adcl $0,%%edx           \n\t" //edx:eax+=carry (hi)
      "movl %%edx,%[carry]     \n\t" //update carry
      "movl %%eax,(%[z])       \n\t" //store z[0]

      "movl 4(%[x]),%%eax      \n\t"
      "mull %[y]               \n\t"
      "addl %[carry],%%eax     \n\t"
      "adcl $0,%%edx           \n\t"
      "movl %%edx,%[carry]     \n\t"
      "movl %%eax,4(%[z])      \n\t"

      "movl 8(%[x]),%%eax      \n\t"
      "mull %[y]               \n\t"
      "addl %[carry],%%eax     \n\t"
      "adcl $0,%%edx           \n\t"
      "movl %%edx,%[carry]     \n\t"
      "movl %%eax,8(%[z])      \n\t"

      "movl 12(%[x]),%%eax      \n\t"
      "mull %[y]               \n\t"
      "addl %[carry],%%eax     \n\t"
      "adcl $0,%%edx           \n\t"
      "movl %%edx,%[carry]     \n\t"
      "movl %%eax,12(%[z])      \n\t"

      "movl 16(%[x]),%%eax      \n\t"
      "mull %[y]               \n\t"
      "addl %[carry],%%eax     \n\t"
      "adcl $0,%%edx           \n\t"
      "movl %%edx,%[carry]     \n\t"
      "movl %%eax,16(%[z])      \n\t"

      "movl 20(%[x]),%%eax      \n\t"
      "mull %[y]               \n\t"
      "addl %[carry],%%eax     \n\t"
      "adcl $0,%%edx           \n\t"
      "movl %%edx,%[carry]     \n\t"
      "movl %%eax,20(%[z])      \n\t"

      "movl 24(%[x]),%%eax      \n\t"
      "mull %[y]               \n\t"
      "addl %[carry],%%eax     \n\t"
      "adcl $0,%%edx           \n\t"
      "movl %%edx,%[carry]     \n\t"
      "movl %%eax,24(%[z])      \n\t"

      "movl 28(%[x]),%%eax     \n\t"
      "mull %[y]               \n\t"
      "addl %[carry],%%eax     \n\t"
      "adcl $0,%%edx           \n\t"
      "movl %%edx,%[carry]     \n\t"
      "movl %%eax,28(%[z])     \n\t"
      : [carry]"=r"(carry)
      : [z]"r"(z), [x]"r"(x), [y]"rm"(y), "0"(carry)
      : "cc","%eax","%edx");
   return carry;
   }

/*************************************************
* Eight Word Block Multiply/Add                  *
*************************************************/
inline word word8_madd3(word z[8], const word x[8], word y, word carry)
   {
   asm(
      "movl (%[x]),%%eax       \n\t" //eax = x[0]
      "mull %[y]               \n\t" //edx:eax = x[0]*y
      "addl %[carry],%%eax     \n\t" //edx:eax+=carry (lo)
      "adcl $0,%%edx           \n\t" //edx:eax+=carry (hi)
      "addl (%[z]),%%eax     \n\t" //edx:eax+=carry (lo)
      "adcl $0,%%edx           \n\t" //edx:eax+=carry (hi)
      "movl %%edx,%[carry]     \n\t" //update carry
      "movl %%eax,(%[z])       \n\t" //store z[0]

      "movl 4(%[x]),%%eax      \n\t"
      "mull %[y]               \n\t"
      "addl %[carry],%%eax     \n\t"
      "adcl $0,%%edx           \n\t"
      "addl 4(%[z]),%%eax     \n\t" //edx:eax+=carry (lo)
      "adcl $0,%%edx           \n\t" //edx:eax+=carry (hi)
      "movl %%edx,%[carry]     \n\t"
      "movl %%eax,4(%[z])      \n\t"

      "movl 8(%[x]),%%eax      \n\t"
      "mull %[y]               \n\t"
      "addl %[carry],%%eax     \n\t"
      "adcl $0,%%edx           \n\t"
      "addl 8(%[z]),%%eax     \n\t" //edx:eax+=carry (lo)
      "adcl $0,%%edx           \n\t" //edx:eax+=carry (hi)
      "movl %%edx,%[carry]     \n\t"
      "movl %%eax,8(%[z])      \n\t"

      "movl 12(%[x]),%%eax      \n\t"
      "mull %[y]               \n\t"
      "addl %[carry],%%eax     \n\t"
      "adcl $0,%%edx           \n\t"
      "addl 12(%[z]),%%eax     \n\t" //edx:eax+=carry (lo)
      "adcl $0,%%edx           \n\t" //edx:eax+=carry (hi)
      "movl %%edx,%[carry]     \n\t"
      "movl %%eax,12(%[z])      \n\t"

      "movl 16(%[x]),%%eax      \n\t"
      "mull %[y]               \n\t"
      "addl %[carry],%%eax     \n\t"
      "adcl $0,%%edx           \n\t"
      "addl 16(%[z]),%%eax     \n\t" //edx:eax+=carry (lo)
      "adcl $0,%%edx           \n\t" //edx:eax+=carry (hi)
      "movl %%edx,%[carry]     \n\t"
      "movl %%eax,16(%[z])      \n\t"

      "movl 20(%[x]),%%eax      \n\t"
      "mull %[y]               \n\t"
      "addl %[carry],%%eax     \n\t"
      "adcl $0,%%edx           \n\t"
      "addl 20(%[z]),%%eax     \n\t" //edx:eax+=carry (lo)
      "adcl $0,%%edx           \n\t" //edx:eax+=carry (hi)
      "movl %%edx,%[carry]     \n\t"
      "movl %%eax,20(%[z])      \n\t"

      "movl 24(%[x]),%%eax      \n\t"
      "mull %[y]               \n\t"
      "addl %[carry],%%eax     \n\t"
      "adcl $0,%%edx           \n\t"
      "addl 24(%[z]),%%eax     \n\t" //edx:eax+=carry (lo)
      "adcl $0,%%edx           \n\t" //edx:eax+=carry (hi)
      "movl %%edx,%[carry]     \n\t"
      "movl %%eax,24(%[z])      \n\t"

      "movl 28(%[x]),%%eax     \n\t"
      "mull %[y]               \n\t"
      "addl %[carry],%%eax     \n\t"
      "adcl $0,%%edx           \n\t"
      "addl 28(%[z]),%%eax     \n\t" //edx:eax+=carry (lo)
      "adcl $0,%%edx           \n\t" //edx:eax+=carry (hi)
      "movl %%edx,%[carry]     \n\t"
      "movl %%eax,28(%[z])     \n\t"
      : [carry]"=r"(carry)
      : [z]"r"(z),[x]"r"(x),[y]"rm"(y),"0"(carry)
      : "cc","%eax","%edx");
   return carry;
   }

}

#endif
