/*
 * Photo Image Print System
 * Copyright (C) 2002-2005 EPSON AVASYS Corporation.
 * Copyright (C) SEIKO EPSON CORPORATION 2002-2005.
 *
 *  This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * As a special exception, EPSON AVASYS Corporation gives permission to
 * link the code of this program with libraries which are covered by
 * the EPSON AVASYS Public License and distribute their linked
 * combinations.  You must obey the GNU General Public License in all
 * respects for all of the code used other than the libraries which
 * are covered by EPSON AVASYS Public License.
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <malloc.h>
#include <search.h>
#include "definitions.h"
#include "utility.h"
#include "prtOpt.h"
#include "optBase.h"

void papersize_cpy(PaperSize* dest, const PaperSize* src)
{
	ASSERT(dest && src);
	dest->width = src->width;
	dest->height = src->height;
	strcpy(dest->unit, src->unit);
}

void papermargin_cpy(PaperMargin* dest, const PaperMargin* src)
{
	ASSERT(dest && src);
	dest->left = src->left;
	dest->top = src->top;
	dest->right = src->right;
	dest->bottom = src->bottom;
	strcpy(dest->unit, src->unit);
}

double convert_unit(double value, const UNIT src_unit, const UNIT dst_unit)
{
	double result = value;
	/* convert to dpi first */
	if (strcmp(src_unit, "mm") == 0)
	{
		result = value/0.070564516;
	}
	else if (strcmp(src_unit, "in") == 0)
	{
		result = value/0.002778131;
	}
	else if (strcmp(src_unit, "pts") == 0)
	{
		result = value/0.2;
	}
	/* convert the above result to destionation unit */
	if (strcmp(dst_unit, "mm") == 0)
	{
		result = result*0.070564516;
	}
	else if (strcmp(dst_unit, "in") == 0)
	{
		result = result*0.002778131;
	}
	else if (strcmp(dst_unit, "pts") == 0)
	{
		result = result*0.2;
	}
	return result;
}

PaperSize* convert_size_unit(
	PaperSize	*paper_size,
	const UNIT	dest_unit
)
{
	paper_size->width = convert_unit(paper_size->width, paper_size->unit, dest_unit);
	paper_size->height = convert_unit(paper_size->height, paper_size->unit, dest_unit);
	strcpy(paper_size->unit, dest_unit);
	return paper_size;
}

PaperMargin* convert_margin_unit(
	PaperMargin	*paper_margin,
	const UNIT	dest_unit
)
{
	paper_margin->left = convert_unit(paper_margin->left, paper_margin->unit, dest_unit);
	paper_margin->top = convert_unit(paper_margin->top, paper_margin->unit, dest_unit);
	paper_margin->right = convert_unit(paper_margin->right, paper_margin->unit, dest_unit);
	paper_margin->bottom = convert_unit(paper_margin->bottom, paper_margin->unit, dest_unit);
	strcpy(paper_margin->unit, dest_unit);
	return paper_margin;
}

char* ltrim(
	char	*target
)
{
	/* find first non-space character */
	char	*p = target;

	while (isspace(*p))
	{
		p++;
	}
	if (p != target)
	{
		memmove(target, p, (strlen(p)+1)*sizeof(char));
	}

	return target;
}

char* rtrim(
	char	*target
)
{
	/* find beginning of trailing spaces by starting at beginning */
	char	*p = target;
	char	*last = NULL;

	while (*p != '\0')
	{
		if (isspace(*p))
		{
			if (last == NULL)
				last = p;
		}
		else
			last = NULL;
		p++;
	}

	if (last != NULL)
	{
		/* truncate at trailing space start */
		*last = '\0';
	}

	return target;
}

char* abbreviate(
	const char	*src,
	char		*dest
)
{
	const char	*p_lead = src;		/* main traverser for src */
	const char	*p_trail = src;		/* point to each word start position */
	const char	*p_series = NULL;	/* traverser for first item in series word, e.g "RX650/RX700", p_series will traverse "RX650" */
	char 		*p_dest = dest;		/* traverser for dest */
	
	while (*p_lead != '\0')
	{
		if (!p_series)
		{
			if (p_lead == src || isupper(*p_lead) || isdigit(*p_lead) ||
				(p_lead > src && isspace(*(p_lead-1)) && isalpha(*p_lead)))
			{
				*p_dest = *p_lead;
				p_dest++;
			}
			else if (*p_lead == '/')
			{
				*p_dest = '_';
				p_dest++;
				p_series = p_trail;
			}
			if (p_lead > src && isspace(*(p_lead-1)))
			{
				/* p_trail is set at word start position */
				p_trail = p_lead;
			}
		}
		else
		{
			if (isspace(*p_lead))
			{
				p_series = NULL;
			}
			else
			{
				if (p_lead > src && *(p_lead-1) == '/')
					p_series = p_trail;			
				if (*p_lead == '/')
				{
					*p_dest = '_';
					p_dest++;
				}
				else if (*p_series != *p_lead || isdigit(*p_lead))
				{
					*p_dest = *p_lead;
					p_dest++;
				}
				p_series++;
			}
		}
		p_lead++;		
	}
	*p_dest = '\0';
	return dest;
}

void assert(char* msg, char* file, int line)
{
#ifdef PPD_DEBUG
	printf("ASSERT: error at: %s in file: %s on line %d!\n", msg, file, line);
	exit(0);
#endif
}

#ifdef UNIX
void *strupr(char *s)
{
	char *p = s;

	if (p)
	{
		while (*p)
		{
			*p = toupper(*p);
			p++;
		}
	}
	return s;
}

void *strlwr(char *s)
{
	char *p = s;

	if (p)
	{
		while(*p)
		{
			*p = tolower(*p);
			p++;
		}
	}
	return s;
}
#endif

/* ListInfo class */

ListHandle list_init(
	PVOID		*_index_list,
	PVOID		_first_item,
	int			_count,
	Done		_done,
	GetNext		_get_next,
	Compare		_compare)
{
	ListHandle list_handle = (ListHandle)calloc(1, sizeof(ListInfo));
	list_handle->index_list = _index_list;
	list_handle->first_item = _first_item;
	list_handle->last_item = _first_item;
	list_handle->count = _count;
	list_handle->done = _done;
	list_handle->get_next = _get_next;
	list_handle->compare = _compare;
	return list_handle;
}

void list_done(ListHandle *list_handle_addr)
{
	if (list_handle_addr && *list_handle_addr)
	{
		list_close_index(*list_handle_addr);
		list_release(*list_handle_addr);
		free(*list_handle_addr);
		*list_handle_addr = NULL;
	}
}

int		list_count(ListHandle list_handle)
{
	PVOID cur_item = list_handle->first_item;
	list_handle->count = 0;
	while (cur_item != NULL)
	{
		list_handle->count++;
		cur_item = *list_handle->get_next(cur_item);
	}
	return list_handle->count;
}

bool	list_create_index(ListHandle list_handle)
{
	PVOID cur_item = list_handle->first_item;
	int count = 0, sze;

	if (list_handle->count <= 0)
	{
		return false;
	}
	if (list_handle->index_list != NULL)
	{
		return false;
	}
	list_handle->index_list = (PVOID*)calloc(list_handle->count, sizeof(PVOID));

	while (cur_item != NULL)
	{
		list_handle->index_list[count] = cur_item;
		cur_item = *(list_handle->get_next(cur_item));
		count += 1;
	}
	sze = sizeof(PVOID);
	qsort(list_handle->index_list, (size_t)list_handle->count, sizeof(PVOID), (__compar_fn_t)list_handle->compare);

	return true;
}

typedef struct 
{
	ListHandle list_handle;
	PVOID	*key;
	PVOID*	found_item_handle;
} FindInfo;

int	find_item(PVOID* cur_item_handle, PVOID _find_nfo)
{
	FindInfo* find_nfo = (FindInfo*)_find_nfo;
	if (find_nfo->list_handle->compare(cur_item_handle, find_nfo->key) == 0)
	{
		find_nfo->found_item_handle = cur_item_handle;
		return false;
	}
	return true;
}

PVOID*	list_find_item(ListHandle list_handle, const PVOID* key)
{
	static FindInfo find_nfo;
	if (list_handle->index_list == NULL)
	{
		find_nfo.key = (PVOID*)key;
		find_nfo.list_handle = list_handle;
		find_nfo.found_item_handle = NULL;
		if (list_for_each(list_handle, find_item, &find_nfo) == false)
			return find_nfo.found_item_handle;
		return NULL;
	}
	return bsearch(key, list_handle->index_list, list_handle->count, sizeof(PVOID), (__compar_fn_t)list_handle->compare);
}

void	list_close_index(ListHandle list_handle)
{
	if (list_handle->index_list)
	{
		free(list_handle->index_list);
		list_handle->index_list = NULL;
	}
}

void	list_add_item(ListHandle list_handle, PVOID item)
{
	PVOID* next_item_handle = NULL;
	if (list_handle)
	{
		if (list_handle->first_item == NULL)
		{
			list_handle->first_item = item;
			list_handle->last_item = item;
			list_handle->count++;
		}
		else if ((next_item_handle = list_handle->get_next(list_handle->last_item)) != NULL)
		{
			*next_item_handle = item;			
			list_handle->last_item = item;
			list_handle->count++;
		}
	}

}

void	list_release(ListHandle list_handle)
{
	PVOID	cur_item = NULL;

	ASSERT(list_handle);
	while (list_handle->first_item != NULL)
	{
		cur_item = list_handle->first_item;
		list_handle->first_item = *list_handle->get_next(cur_item);
		if (list_handle->done)
			list_handle->done(&cur_item);
	}
	list_handle->count = 0;
}

bool	list_for_each(ListHandle list_handle, EachItem each_item_proc, PVOID user_data)
{
	int i;
	bool result = true;
	PVOID	cur_item = NULL;

	ASSERT(list_handle)
	if (list_handle->index_list == NULL)
	{
		cur_item = list_handle->first_item;
		while (cur_item != NULL)
		{
			if (each_item_proc(&cur_item, user_data) == false)
				return false;
			cur_item = *(list_handle->get_next(cur_item));
		}
		return true;
	}

	for (i = 0; i < list_handle->count && result; i++)
	{
		if (each_item_proc(&list_handle->index_list[i], user_data) == false)
			result = false;
	}
	return result;
}


/* Support functions */
char*  getRscName(int id, int value)
{
  int i, j;

  for(i = 0; opt_all[i].id != id; i++);
  for(j = 0; opt_all[i].data[j].value != END_ARRAY; j++)
    if(opt_all[i].data[j].value == value)
      return opt_all[i].data[j].rsc_name;
  return NULL;
}

OPTION_DATA*  getOptData(int id, int value)
{
  int i, j;

  for(i = 0; opt_all[i].id != id; i++);
  for(j = 0; opt_all[i].data[j].value != END_ARRAY; j++)
    if(opt_all[i].data[j].value == value)
	    return (&opt_all[i].data[j]);
  return NULL;
}
