/**
 *
 *   Copyright 2017 Red Hat, Inc, and individual contributors.
 *
 *   Licensed under the Apache License, Version 2.0 (the "License");
 *   you may not use this file except in compliance with the License.
 *   You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 *   Unless required by applicable law or agreed to in writing, software
 *   distributed under the License is distributed on an "AS IS" BASIS,
 *   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *   See the License for the specific language governing permissions and
 *   limitations under the License.
 *
 */
package org.wildfly.swarm.microprofile.jwtauth;

import org.wildfly.swarm.config.runtime.AttributeDocumentation;
import org.wildfly.swarm.spi.api.Defaultable;
import org.wildfly.swarm.spi.api.Fraction;
import org.wildfly.swarm.spi.api.annotations.Configurable;
import org.wildfly.swarm.spi.api.annotations.DeploymentModule;
import static org.wildfly.swarm.spi.api.Defaultable.bool;
import static org.wildfly.swarm.spi.api.Defaultable.integer;
import static org.wildfly.swarm.spi.api.Defaultable.string;

/**
 * A fraction that adds support for the MicroProfile 1.0 JWT RBAC authentication and authorization spec.
 */
@Configurable("thorntail.microprofile.jwt")
@DeploymentModule(name = "org.wildfly.swarm.microprofile.jwtauth", slot = "deployment", export = true, metaInf = DeploymentModule.MetaInfDisposition.IMPORT)
public class MicroProfileJWTAuthFraction implements Fraction<MicroProfileJWTAuthFraction> {

    @AttributeDocumentation("The URI of the JWT token issuer")
    @Configurable("thorntail.microprofile.jwt.token.issued-by")
    @Configurable("thorntail.microprofile.jwtauth.token.issuedBy")
    private Defaultable<String> tokenIssuer = string("http://localhost");

    @AttributeDocumentation("The public key of the JWT token signer. Can be prefixed 'file:' or 'classpath:' for key assets, otherwise the key contents are expected")
    @Configurable("thorntail.microprofile.jwt.token.signer-pub-key")
    @Configurable("thorntail.microprofile.jwtauth.token.signerPubKey")
    private String publicKey;

    @AttributeDocumentation("The JWT token expiration grace period in seconds ")
    @Configurable("thorntail.microprofile.jwt.token.exp-grace-period")
    @Configurable("thorntail.microprofile.jwtauth.token.expGracePeriod")
    private Defaultable<Integer> expGracePeriodSecs = integer(60);

    @AttributeDocumentation("The JWKS URI from which to load public keys (if 'signer-pub-key' is set, this setting is ignored).")
    @Configurable("thorntail.microprofile.jwt.token.jwks-uri")
    private String jwksUri;

    @AttributeDocumentation("The interval at which the JWKS URI should be queried for keys (in minutes).")
    @Configurable("thorntail.microprofile.jwt.token.jwks-refresh-interval")
    private Defaultable<Integer> jwksRefreshInterval = integer(60);

    @AttributeDocumentation("If a JAX-RS resource has no class-level security metadata, then if this property is set to `true` and at least one resource method has security metadata all other resource methods without security metadata have an implicit `@DenyAll`, otherwise resource methods without security metadata are not secured")
    @Configurable("thorntail.microprofile.jwt.default-missing-method-permissions-deny-access")
    private Defaultable<Boolean> defaultMissingMethodPermissionsDenyAccess = bool(true);

    public Defaultable<String> getTokenIssuer() {
        return tokenIssuer;
    }

    public void setTokenIssuer(Defaultable<String> tokenIssuer) {
        this.tokenIssuer = tokenIssuer;
    }

    public String getPublicKey() {
        return publicKey;
    }

    public void setPublicKey(String publicKey) {
        this.publicKey = publicKey;
    }

    public Defaultable<Integer> getExpGracePeriodSecs() {
        return expGracePeriodSecs;
    }

    public void setExpGracePeriodSecs(Defaultable<Integer> expGracePeriodSecs) {
        this.expGracePeriodSecs = expGracePeriodSecs;
    }

    public String getJwksUri() {
        return jwksUri;
    }

    public void setJwksUri(String jwksUri) {
        this.jwksUri = jwksUri;
    }

    public Defaultable<Integer> getJwksRefreshInterval() {
        return jwksRefreshInterval;
    }

    public void setJwksRefreshInterval(Defaultable<Integer> jwksRefreshInterval) {
        this.jwksRefreshInterval = jwksRefreshInterval;
    }

    public boolean isDefaultMissingMethodPermissionsDenyAccess() {
        return defaultMissingMethodPermissionsDenyAccess.get();
    }

}
